// Author:   Jeff Dicker
// For:      Dr. Yves Lucet
// Whatsit:  Unit tests for me_llt, me_llt2d, me_direct, me_direct2d

mode(-1);
__test_set = "ME_LLT: Moreau Envelope thru LLT";

function b = testME_llt1()
//compare direct with llt algorithm to compute
//the Moreau envelope of f(x)=x^2 on the
//grid {1,2,3,...,10}
	X = (1:10)';//primal grid
	S = X;//dual grid
	F=X.^2;//evaluate the function on the grid
	//direct Moreau envelope algorithm
	//quadratic time (for comparison only)
	desired = me_direct(X, F, S);
	//llt Moreau envelope algorithm
	result = me_llt(X, F, S);
	b = and(result == desired);//verify results
endfunction

function b = testME_llt2()
//Compare direct with llt algorithm to compute the 
//Moreau envelope of f(x)=2 x^2 with different
//primal and dual grids
	X = (1:10)';//primal grid
	S = (-2:2)';//dual grid
	F=2*X.^2;//evaluate the function on the grid
	//direct Moreau envelope algorithm
	//quadratic time (for comparison only)
	desired = me_direct(X, F, S);
	//llt Moreau envelope algorithm
	result = me_llt(X, F, S);
	b = and(result == desired);//verify results
endfunction


function b = testME_llt2d_QuadraticUnbalanced()
//Compare direct with llt algorithm to compute the 
//Moreau envelope of the bivariate function
//g(x1,x2)=x1^2+2*x2^2
	function f=f(lambda,x),f=lambda * x.^2,endfunction
	function g=g(lambda1,lambda2,x,y),g=f(lambda1,x)+f(lambda2,y),endfunction
	lambda1=1;lambda2=2;
	x1=(-10:10)';x2=(-5:5)';//primal grid
	//evaluate the function on the primal grid
	[X, Y]=ndgrid(x1,x2);F=g(lambda1,lambda2,X,Y);
	s1=(-4:4)';s2=(-5:6)';//dual grid
	//llt Moreau envelope algorithm for the bivariate function
	desired=me_llt2d(x1,x2,F,s1,s2);

	//check the results by using the fact g is separable
	//so its bivariate Moreau envelope is the sum of the 
	//univariate Moreau envelopes for each of its variables
	Ms1=me_direct(x1,f(lambda1,x1),s1);//univariate Moreau envelope for x1
	Ms2=me_direct(x2,f(lambda2,x2),s2);//univariate Moreau envelope for x2
	//build the bivariate result
	t1 = Ms1 * ones(1,size(Ms2,1));
	t2 = ones(size(Ms1,1),1) * Ms2';
	correct=t1+t2;

	b = and(correct == desired);//verify results
endfunction

function b = testMEFromExampleOne()
//test bivariate Moreau envelope algorithms by comparing
//the direct with the llt algorithm for a discrete function
	Xr = [1 2.2 3]';Xc = [1 2 3]';//primal grid
	Sr = Xr;Sc = Xc;//dual grid=primal grid
	F = [3   100 100
	     100 100 100
	     100 100 2  ];//values of the function on the grid
	desired = me_direct2d(Xr, Xc, F, Sr, Sc);//result with direct algorithm
	result = me_llt2d(Xr, Xc, F, Sr, Sc);//result with llt algorithm
	//verify the result taking into account roundoff errors
	b = norm(result - desired) < 1E-10;
endfunction

function b = testME_llt2d_M()
//test the bivariate llt Moreau envelope algorithm by comparing
//its results with the one obtained by manual computation
	Xr = [1 2.2 3]'; Xc = [1 2 3]'; //primal grid
	Sr = Xr;Sc = Xc;//dual grid
	F = [3   100 100;100 100 100;100 100 2  ];//values of the function on the grid
	//result obtained by manual computation
	desired_MY = [ 3. 4.44 6.00 ; 4.00 3.64 3.00 ; 6.00 2.64 2.00 ]';
	[MY,g,Conjpartial,Conj] = me_llt2d(Xr, Xc, F, Sr, Sc);//apply the llt algorithm
	//verify the result taking into account roundoff errors
	b = norm(desired_MY - MY)<1E-10;
endfunction

function b = testMEFromExampleTwo()
//test the bivariate llt Moreau envelope algorithm by comparing
//its results with the one obtained by manual computation
	F = [3   100 100 ; 100 100 100 ; 100 100 100];//values of the function on the grid
	Xr = [1 2 3]'; Xc = Xr;//primal grid
	Sr = Xr; Sc = Sr;//dual grid
	desired = [3 4 7 ; 4 5 8 ; 7 8 11];//result obtained by manual computation
	result = me_llt2d(Xr, Xc, F, Sr, Sc);//apply the llt algorithm
	//verify the result taking into account roundoff errors
	b = isequal(result, desired);
endfunction

function b = testMEFromExampleThree()
//test the bivariate llt Moreau envelope algorithms by comparing
//its results with the one obtained by manual computation
	F = [100 100 100 ; 100 100 100 ; 100 100 3];//values of the function on the grid
  	Xr = [1 2 3]'; Xc = Xr;//primal grid
	Sr = Xr; Sc = Sr;//dual grid
	desired = [11 8 7 ; 8  5 4 ; 7  4 3];//result obtained by manual computation
	result = me_llt2d(Xr, Xc, F, Sr, Sc);//apply the llt algorithm
	b = isequal(result, desired);
endfunction

function b = testMEFloats_simple()
//test the bivariate llt Moreau envelope algorithm by comparing
//its results with the one obtained by manual computation
//case when the function contains %inf values
//the result is the distance transform of the binary image formed by
//considering any pixel with value 0 as background and pixel with value 
//%inf as foreground.
	//values of the function on the grid
	//the function has %inf values at the center of the grid
	F = 	[0 0 0 0 0                                         
		 0 %inf %inf %inf 0                                     
		 0 %inf %inf %inf 0                                     
		 0 %inf %inf %inf 0                                     
		 0 0 0 0 0];
	Xr = [1 1.25 1.5 1.75 2]';Xc = Xr; //primal grid
	Sr = Xr; Sc = Sr;//dual grid
	result = me_llt2d(Xr, Xc, F, Sr, Sc);//apply the llt algorithm
	desired = [0 0 0 0 0
	           0 .0625 .0625 .0625 0
		   0 .0625 .25 .0625 0
		   0 .0625 .0625 .0625 0
		   0 0 0 0 0];//result obtained by manual computation
	b = isequal(result, desired);
endfunction

function b = testMEFloats_one()
//test bivariate Moreau envelope algorithms by comparing
//the direct with the llt algorithm for a discrete function
	F = [2   100 100 ; 100 100 100 ; 4   100 3];//values of the function on the grid
	Xr = [1 2.2 3]'; Xc = Xr;//primal grid
	Sr = [1 2   3]'; Sc = Sr;//dual grid
	//apply the direct algorithm (for comparison only)
	desired = me_direct2d(Xr, Xc, F, Sr, Sc);
	//apply the llt algorithm
	result = me_llt2d(Xr, Xc, F, Sr, Sc);
	b = isequal(result, desired);
endfunction

function b = testMEFloats_both()
//test bivariate Moreau envelope algorithms by comparing
//the direct with the llt algorithm for a discrete function
	F = [2   100 100 ; 100 100 100 ; 4   100 3];//values of the function on the grid
	Xr = [1 2.2 3]'; Xc = Xr;//primal grid
	Sr = [1 2.1 2.8]'; Sc = Sr;//dual grid
	//apply the direct algorithm (for comparison only)
	desired = me_direct2d(Xr, Xc, F, Sr, Sc);
	//apply the llt algorithm
	result = me_llt2d(Xr, Xc, F, Sr, Sc);
	//verify the result taking into account roundoff errors
	b = norm(desired - result) < 1E-10;
endfunction

function b = testMEUsingIndicator()
//test both bivariate Moreau envelope algorithms (llt and direct)
//by comparing their results with manual computation
//the result is the distance transform of the binary image formed by
//considering any pixel with value 0 as background and pixel with value 
//%inf as foreground.
	F = [0 0 0 0 0                                         
		  0 %inf %inf %inf 0                                     
		  0 %inf %inf %inf 0                                     
		  0 %inf %inf %inf 0                                     
		  0 0 0 0 0];//indicator function
	Xr = [1 2 3 4 5]';Xc = Xr; //primal grid
	Sr = Xr; Sc = Sr;//dual grid
	//apply the llt algorithm
	result_llt = me_llt2d(Xr, Xc, F, Sr, Sc);
	//apply direct algorithm (for comparison only)
	result_me_direct = me_direct2d(Xr, Xc, F, Sr, Sc);
	//result from manual computation
	desired = [0 0 0 0 0
	           0 1 1 1 0
		   0 1 4 1 0
		   0 1 1 1 0
		   0 0 0 0 0];
	b = isequal(result_llt, result_me_direct, desired);
endfunction

function b = testAbsoluteSmallS()
//Check that the llt and direct bivariate algorithms return the same
//result for the bivariate absolute value function f(x,y) = |x| + |y|
	Xr = -5:5; Xr = Xr'; Xc = Xr; //primal grid
	Sc = -4.5:4.5; Sc = Sc'; Sr = Sc;//dual grid
	F = abs(Xr) * ones(1, size(Xr, 1)) + ones(size(Xc, 1), 1) * abs(Xc');	
	result = me_llt2d(Xr, Xc,F, Sr, Sc);
	correct = me_direct2d(Xr, Xc,F, Sr, Sc);
	b = isequal(correct, result);
endfunction

function b = testAbsoluteAnother()
//Check that the llt and direct bivariate algorithms return the same
//result for the bivariate absolute value function f(x,y) = |x| + |y|
	Xr = 1:10; Xr = Xr'; Xc = Xr; //primal grid
	Sc = Xc; Sr = Sc;//dual grid
	F = abs(Xr) * ones(1, size(Xr, 1)) + ones(size(Xc, 1), 1) * abs(Xc');
	result = me_llt2d(Xr, Xc,F, Sr, Sc);
	correct = me_direct2d(Xr, Xc,F, Sr, Sc);
	b = isequal(correct, result);
endfunction

function b = testSquareRoot()
//Check that the llt and direct bivariate algorithms return the same
//result for the bivariate absolute value function f(x,y) = x*sqrt(y)
	Xr = 1:10; Xr = Xr'; Xc = Xr;//primal grid
	Sc = Xc; Sr = Sc;//dual grid
	F = Xr * ones(1, size(Xr, 1)) * ones(size(Xc, 1), 1) * sqrt(abs(Xc'));
	result = me_llt2d(Xr, Xc,F, Sr, Sc);
	correct = me_direct2d(Xr, Xc,F, Sr, Sc);	
	b = isequal(correct, result);
endfunction

function b = runTestFile()
	b = %t;
	b = checkForFail(testWrapper(testME_llt1, "testME_llt1"), b);
	b = checkForFail(testWrapper(testME_llt2, "testME_llt2"), b);	
	b = checkForFail(testWrapper(testME_llt2d_QuadraticUnbalanced, "testME_llt2d_QuadraticUnbalanced"), b);
	b = checkForFail(testWrapper(testMEFromExampleOne, "testMEFromExampleOne"), b);	
	b = checkForFail(testWrapper(testME_llt2d_M, "testME_llt2d_M"), b);
	b = checkForFail(testWrapper(testMEFromExampleTwo, "testMEFromExampleTwo"), b);
	b = checkForFail(testWrapper(testMEFromExampleThree, "testMEFromExampleThree"), b);
	b = checkForFail(testWrapper(testMEFloats_simple, "testMEFloats_simple"), b);
	b = checkForFail(testWrapper(testMEFloats_one, "testMEFloats_one"), b);
	b = checkForFail(testWrapper(testMEFloats_both, "testMEFloats_both"), b);

	b = checkForFail(testWrapper(testMEUsingIndicator, "testMEUsingIndicator"), b);
	b = checkForFail(testWrapper(testAbsoluteSmallS, "testAbsoluteSmallS"), b);
	b = checkForFail(testWrapper(testAbsoluteAnother, "testAbsoluteAnother"), b);
	b = checkForFail(testWrapper(testSquareRoot, "testSquareRoot"), b);
endfunction
