// The current lft.sci file loads the functions in the CCA toolbox to compute the
// Legendre-Fenchel transform. It can be used standalone but is also called from me.sci
// See the help files for more information on the functions

///////////////////// DIRECT ///////////BEGIN
//conjugate of an univariate function (quadratic cost)
//for comparison purposes only
function [Conj]=lft_direct(X,Y,S) 
	Conj = max(X * S' - Y* ones(1,size(S,1)),'r' )';
	//matlab syntax: max(X * S' - Y* ones(size(S,1))' )';
endfunction

//internal function
//conjugate of an univariate function on the regular integer grid
function [Conj] = lft_direct_d(Y)
	// X = 1, 2, 3, 4...n
	X = [1:1:size(Y, 1)]';	
	// Compute the conjugate at all the indices: 1, 2, 3, 4...
	S = X;	
	[Conj] = lft_direct(X, Y, S);
endfunction
///////////////////// DIRECT ///////////END

///////////////////// LLT ///////////BEGIN
//compute the lower convex envelope of a set of 
//points in the plane using the Beneath-Beyond algorithm
//in linear time
function [bbx,bby] = bb (X,Y) 
	// Initialisation
	if size(Y)~=size(X),error("unconsistent input size in bb");end;
	//convert input to column vector
	if size(X,2)>1 then
		X=X';Y=Y';
	end
	if size(X,2)>1 then error("bb only works for vectors, not matrices");end;
	[n, columns] = size(X);
	if n==1, bbx=X;bby=Y;return;end;
	I=[1:n];v=2;CX=%nan*ones(n,1);CY=CX;
	CX(1:2)=[X(1);X(2)];CY(1:2)=[Y(1);Y(2)];
	v=2;

	// We look at each point only once, this gives the linear-time algorithm.
	for i=3:n,
  	  y=((CY(v)-CY(v-1))/(CX(v)-CX(v-1)))*(X(i)-CX(v))+CY(v);
  	  while (v>2 & Y(i)<=y), // Erase points which are not vertices of the convex hull 
	    v=v-1;
	    y=((CY(v)-CY(v-1))/(CX(v)-CX(v-1)))*(X(i)-CX(v))+CY(v);
  	  end
  	  if v>2 
  	    if Y(i)==y 
	      CX(v)=X(i);CY(v)=Y(i);
	    else 
	      CX(v+1)=X(i);
	      CY(v+1)=Y(i);
	      v=v+1;
	    end
  	  else 
	    if Y(i)>y // Trivial convex hull
  	      CX(3)=X(i);CY(3)=Y(i);v=3;
	    else 
	      CX(2)=X(i);CY(2)=Y(i);
	    end
  	  end;
	end
	bbx=CX(1:v);bby=CY(1:v);
endfunction

//internal function very fast but O(n log n)
function [fH]=fusionsci(C,S)
	// fusionsci	merge two increasing sequences.
	//		fusionsci(C,S) gives the resulting indices fH where each slope supports the epigraph. 
	//		All in all, it amounts to finding the first indice i such that C(i-1)<=S(j)<C(i)
	//
	// 		This function uses scilab syntax to speed up computation, see fusion for a more classical
	//		programming of the same function
	// 
	// Input parameters:
	//  C,S - Increasing column vectors. We want to know the conjugate at
	//	S(j). C gives the slope of planar input points.
	//  
	// Output parameter:
	//  fH - Index at which the slope support the epigraph. Column vector.
	//
	// Example:
	//  X=[-5:0.5:5]'
	//  Y=X.^2
	//  C=(Y(2:size(Y,1))-Y(1:size(Y,1)-1))./(X(2:size(X,1))-X(1:size(X,1)-1))
	//  S=C-0.25*ones(size(C, 1))
	//  [fH]=fusionsci(C,S)

	// The BUILT-IN sort function is faster than looking at each slope with a for loop
	[Z I]=gsort([C;S],'g','i')
	I=find(I>size(C,1))';
	J=I(2:size(I,1))-I(1:size(I,1)-1); 
	K=J-ones(size(J,1));
	L= cumsum(K)+ones(size(K,1));

	H=[I(1) L'+I(1)-1]';
	fS=S;fH=H;
	// This (maybe) obscure computation ONLY aims at using Scilab syntax
	// to do exactly the same as fusion whose programming sticks to the
	// LLT as written in ["Faster than the Fast Legendre Transform, the
	// Linear-time Legendre Transform", Y. Lucet]
endfunction

//internal function. Slower than fusionsci due to the interpreted
//Scilab commands but linear time.
function [fH]=fusion(C,S)
	// C and S must be column vectors
	n=size(C,1)-1;m=size(S,1);
	i=1;j=1;fS=[S(1);S(2)];clear fH;
	isEcho=%F;
	S(size(S,1)+1)=0; //Useful to make the last test of the loop
	if (isEcho) printf("starting...n=%i, m=%i\n",n,m);end;
	if (isEcho) printf("start...i=%i, j=%i\n",i,j);end;
	// Now we deal with the remaining slopes
	while (j<=m & i<=n+1 & S(j)<=C(n+1)),
		while S(j)>C(i),
			i=i+1; 
		end
		if (isEcho) printf("     ...i=%i, j=%i\n",i,j);end;
		fH(j)=i;
		fS(j)=S(j);
		j=j+1;
	end
	if (isEcho) printf("out  ...i=%i, j=%i\n",i,j);end;
	if (j<=m & i<=n+1 & S(j)>C(n+1))
		if (isEcho) printf("if   ...i=%i, j=%i\n",i,j);end;
		for l=j:m,
			fH(l)=n+2;//last point in the graph. Slopes are from i to i+1 so last slope is C(n+1), so last point is index n+2
			fS(l)=S(l);
			if (isEcho) printf("for  ...i=%i, j=%i\n",i,l);end;
		end
	end
endfunction

//Main function: Conjugate of an univariate function
function [Conj]=lft_llt(X,Y,S,fusionopt)
	if (argn(2)>3 & fusionopt~=1) 
		fusioncall=fusion;//scilab syntax for fast execution
	else
		fusioncall=fusionsci;//programming syntax for linear-time execution
	end
	if (Y == %inf.*Y | Y == %inf.*(-1.*Y))
		Conj = %inf.*(-1.*Y);
	else
		[XX,YY]=bb(X,Y); // Compute the convex hull (input sensitive algorithm).	
		h=size(XX,1);
		C=(YY(2:h)-YY(1:h-1))./(XX(2:h)-XX(1:h-1)); // Compute the slopes associated with the primal points
		[H]=fusioncall(C,S);// Merge sequences C and S
		// j=H(i,1) is the first indice of the point X(j) such that the line
		// with slope S(i) supports the epigraph of the function at X(j).
		Conj= S .* XX(H(:,1)) -YY(H(:,1));
	end
endfunction

//internal function: Conjugate of an univariate function
//on the regular integer grid
function [Conj] = lft_llt_d(Y)
	// X = 1, 2, 3, 4...n
	X = [1:1:size(Y, 1)]';	
	// Compute the conjugate at all the indices: 1, 2, 3, 4...
	S = X;	
	[Conj] = lft_llt(X, Y, S);
endfunction
///////////////////// LLT ///////////END
