/*
** TRI_QLTY.C
**
** AUTHORS: 
**
**   Maria MORANDI CECCHI               Stefano DE MARCHI,            
**   University of Padua, Italy         University of Udine, Italy
**   email: mcecchi@math.unipd.it       email: demarchi@dimi.uniud.it
**   
**                       Damiano FASOLI
**                       Verona Software Srl, Italy 
**                       email: vrsoft@sis.it
**
**
**
** REVISION DATE: May, 1999  
**
** MODULES CALLED: NONE
**
** -------------------------------------------------------------------
** SYNOPSIS:  tri_qlty file1 file2
**
** This program computes the minimum angle in degrees and
** the area for all triangles of a planar triangulation.
**
** 'file1' is the data file made up of the 3-dimensional points.
** The coordinates of the points may be separated by any
** combination of spaces, tabs and carriage returns.
** We recommend to use a row for each point, for clearness.
**
** 'file2' is the triangulation data file in the format generated by
** the program 'do_trian'. The file has three sections.
** The first, starting with a single capital letter `C' in the first row,
** lists in subsequent blocks separated by -1 the index of
** a node and those of its neighbours in a counterclockwise
** order.
** Note: this program uses only this first section and the heading of the
**       second section, that is a single capital letter `B' in the row.
**
** Output is formed by 5 columns:
**
**   1)        minimum angle in degrees;
**   2,3,4)    indexes of the vertices of triangles;
**   5)        area of triangles.
**
** Output format is handled by the two constants WIDTH and PREC
** that are the width of the output field, and the number of decimal digits.
**
** The program supports at most MAX_NUM_NODI (#DEFINE constant) input points.
** If more points are given, it exits printing a warning message.
**
** -------------------------------------------------------------------
** SUBROUTINES AND FUNCTIONS
**
** >>>>>>>>>>>>>>>>
**
** double triangle_area(int iv1,int iv2,int iv3,double[] x,double[] y)
**
** Compute the area of the triangle with vertices the points
** iv1,iv2,iv3.
**
**
** >>>>>>>>>>>>>>>>
**
** double triangle_min_angle(int iv1,int iv2,int iv3,double[] x,double[] y)
**
** This function computes in degrees the smallest angle among the three ones
** of the triangle with vertices the points iv1,iv2,iv3.
**
**
---------------------------------------------------------------------------------
*/

#include<stdio.h>
#include<stdlib.h>
#include<math.h>
#define MAX_NUM_NODI 100000
#define WIDTH 20   /* Format output width of real numbers written to
files */
#define PREC 10    /* Decimal precision width of real numbers written to files */

double triangle_min_angle();
double triangle_area();

double triangle_area(iv1,iv2,iv3,x,y)
long int iv1,iv2,iv3;
double x[],y[];
{
 double x1,y1,x2,y2,x3,y3;

 x1=x[iv1]; x2=x[iv2]; x3=x[iv3];
 y1=y[iv1]; y2=y[iv2]; y3=y[iv3];
 return (fabs((x2*y3-x3*y2+x3*y1-x1*y3+x1*y2-x2*y1)/2.));
}

double triangle_min_angle(iv1,iv2,iv3,x,y)
long int iv1,iv2,iv3;
double x[],y[];
{
 double t1,t2,t3;
 double x1,y1,x2,y2,x3,y3;

 x1=x[iv1]; x2=x[iv2]; x3=x[iv3];
 y1=y[iv1]; y2=y[iv2]; y3=y[iv3];
 t1=acos(1./(sqrt(pow(x2-x1,2.)+pow(y2-y1,2.))*sqrt(pow(x3-x1,2.)+pow(y3-y1,2.)))*((x2-x1)*(x3-x1)+(y2-y1)*(y3-y1)));

 t2=acos(1./(sqrt(pow(x3-x2,2.)+pow(y3-y2,2.))*sqrt(pow(x1-x2,2.)+pow(y1-y2,2.)))*((x3-x2)*(x1-x2)+(y3-y2)*(y1-y2)));

 t3=acos(1./(sqrt(pow(x1-x3,2.)+pow(y1-y3,2.))*sqrt(pow(x2-x3,2.)+pow(y2-y3,2.)))*((x1-x3)*(x2-x3)+(y1-y3)*(y2-y3)));

 if (t2<t1) t1=t2;
 if (t3<t1) t1=t3;
 return(t1/M_PI*180.);
}

void main(argc,argv)
int argc;
char *argv[];
{
 FILE *infile, *trifile;
 double x[MAX_NUM_NODI],y[MAX_NUM_NODI],z_trash;
 long int iv1,iv2,iv3,num_nodo;
 char c;
 char base_buffer[20];
 long int nt;

 infile=NULL; trifile=NULL;
 if (argc==1)
   {
    printf("\nStart the program with two parameters:\n");
    printf("First parameter: data file name.\n");
    printf("Second parameter: triangulation file name.\n");
    exit(1);
   }
 infile=fopen(argv[1],"r");
 if (infile==NULL)
   {
    printf("Cannot open file: %s.\n",argv[1]);
    exit(1);
   }
 if (argc==2)
   {
    printf("Missing triangulation file.\n");
    exit(1);
   }
 /* data reading */
 printf("Wait for data reading.\n");
 num_nodo=0;
 while (!feof(infile))
 {
  if (num_nodo<MAX_NUM_NODI)
   {
    fscanf(infile," %lf %lf %lf ",&x[num_nodo],&y[num_nodo],&z_trash);
      /* spaces are essential part of the input format string */
    num_nodo=num_nodo+1;
   }
   else
   {
    printf("Too many input data.\n");
    exit(1);
   }
 }
 if (num_nodo<3)
    {
     printf("I need at least three input points.\n");
     exit(1);
    }
 fclose(infile);
 /* end of data reading */
 /* opening triangulation file */
 trifile=fopen(argv[2],"r");
 if (trifile==NULL)
   {
    printf("Cannot open file: %s.\n",argv[2]);
    exit(1);
   }
 do {c=fgetc(trifile);} while ((c!='C') && (!feof(trifile)));
 if (feof(trifile))
  {
   printf("File %s in a no correct format.\n",argv[2]);
   exit(1);
  }
 fgetc(trifile); /* carriage return */

 /* Main cycle over triangles  */
 printf("Minimum angle (degrees) - Vertex indexes - Triangle area\n\n");

 fscanf(trifile,"%d ",&iv1);
 nt=1; /* Counter of the triangle being assembled */
 do
   {
    fscanf(trifile,"%d %d ",&iv2,&iv3);
    while(iv3!=-1)
     {
      if ((iv1<iv2)&&(iv1<iv3)) /* Each triangle must be considered once

*/
 {
           printf("%*.*f %5ld %5ld %5ld %*.*f\n",
           WIDTH,PREC,triangle_min_angle(iv1,iv2,iv3,x,y),iv1,iv2,iv3,
           WIDTH,PREC,triangle_area(iv1,iv2,iv3,x,y));
           nt++;
 }
      iv2=iv3;
      fscanf(trifile,"%d ",&iv3);
     }  /* End of the reading of the current row in the triangulation
file */
    fscanf(trifile,"%s ",base_buffer);
    iv1=atol(base_buffer);
   }  /* End of the reading of the triangulation file when the cycle
exits */
 while (base_buffer[0]!='B');
 fclose(trifile);
}



