/*
** PWSTOOLS.C
**
** AUTHORS: 
**
**   Maria MORANDI CECCHI               Stefano DE MARCHI,            
**   University of Padua, Italy         University of Udine, Italy
**   email: mcecchi@math.unipd.it       email: demarchi@dimi.uniud.it
**   
**                       Damiano FASOLI
**                       Verona Software Srl, Italy 
**                       email: vrsoft@sis.it
**
**
**
** REVISION DATE: May, 1999  
**
**
** MODULES CALLED: NONE
**
** -------------------------------------------------------------------------
** SUBROUTINES AND FUNCTIONS DESCRIPTION:
**
** >>>>>>>>>>>>>>>>
**
** double compute_for_matrix(int pwsb_mode,double At,
**                           double At1,double At2,double At3,
**                           int ip,int jp,double[] cpt0)
**
** GLOBAL VARIABLES: IndTA IndTB
**                   u4,v4 
**                   (circumcenter if ABflag==0, barycenter if ABflag==1)
**                   u12,v12,u23,v23,u31,v31, 
**                   (middle edge points of the macro-triangle V1 V2 V3)
**                   u41,v41,u42,v42,u43,v43 
**                   (if ABflag==1 intersections between lines connecting 
**                    middle edge points of macro-triangle and lines connecting
**                    barycenter with vertices of macro-triangle) 
**                   where Vi=(ui,vi) i=1,2,3,12,23,31,4,41,42,43 are the vertices.
**
** Let Q be a (quadratic) Powell-Sabin polynomial patch over the macro-triangle T
** with vertices V1,V2,V3 and 'cpt0' an array made up of 108 or 216 entries
** as follows.
**
** Case 'pwsb_mode==0':  Powell-Sabin subdivision of A-type
** Apart from a constant factor 1/(2A^2), where A is the area of the subtriangle,
** and At the one of the macro-triangle,  
**    cpt0[ 0*6+k]==(d Qxx/d v_k) over V4 V1 V12 (having A==At1/2)
**    cpt0[ 1*6+k]==(d Qxy/d v_k) over V4 V1 V12
**    cpt0[ 2*6+k]==(d Qyy/d v_k) over V4 V1 V12
**    cpt0[ 3*6+k]==(d Qxx/d v_k) over V4 V12 V2 (having A==At1/2)
**    cpt0[ 4*6+k]==(d Qxy/d v_k) over V4 V12 V2
**    cpt0[ 5*6+k]==(d Qyy/d v_k) over V4 V12 V2
**    cpt0[ 6*6+k]==(d Qxx/d v_k) over V4 V2 V23 (having A==At2/2)
**    cpt0[ 7*6+k]==(d Qxy/d v_k) over V4 V2 V23
**    cpt0[ 8*6+k]==(d Qyy/d v_k) over V4 V2 V23
**    cpt0[ 9*6+k]==(d Qxx/d v_k) over V4 V23 V3 (having A==At2/2) 
**    cpt0[10*6+k]==(d Qxy/d v_k) over V4 V23 V3
**    cpt0[11*6+k]==(d Qyy/d v_k) over V4 V23 V3
**    cpt0[12*6+k]==(d Qxx/d v_k) over V4 V3 V31 (having A==At3/2)
**    cpt0[13*6+k]==(d Qxy/d v_k) over V4 V3 V31
**    cpt0[14*6+k]==(d Qyy/d v_k) over V4 V3 V31
**    cpt0[15*6+k]==(d Qxx/d v_k) over V4 V31 V1 (having A==At3/2)
**    cpt0[16*6+k]==(d Qxy/d v_k) over V4 V31 V1
**    cpt0[17*6+k]==(d Qyy/d v_k) over V4 V31 V1
** where v_k ( k = 0, ..., 5 ) are the independent variables representing
** Qx(V1),Qy(V1),Qx(V2),Qy(V2),Qx(V3),Qy(V3).
**
** Case 'pwsb_mode==1':  Powell-Sabin subdivision of B-type
** Apart from a constant factor 1/(2A^2), where A is the area of the subtriangle
** and At the one of the macro-triangle, 
**    cpt0[ 0*6+k]==(d Qxx/d v_k) over V4 V41 V12 (having A==At/24) 
**    cpt0[ 1*6+k]==(d Qxy/d v_k) over V4 V41 V12
**    cpt0[ 2*6+k]==(d Qyy/d v_k) over V4 V41 V12
**    cpt0[ 3*6+k]==(d Qxx/d v_k) over V41 V1 V12 (having A==At/8)
**    cpt0[ 4*6+k]==(d Qxy/d v_k) over V41 V1 V12
**    cpt0[ 5*6+k]==(d Qyy/d v_k) over V41 V1 V12
**    cpt0[ 6*6+k]==(d Qxx/d v_k) over V4 V12 V42 (having A==At/24)
**    cpt0[ 7*6+k]==(d Qxy/d v_k) over V4 V12 V42
**    cpt0[ 8*6+k]==(d Qyy/d v_k) over V4 V12 V42
**    cpt0[ 9*6+k]==(d Qxx/d v_k) over V42 V12 V2 (having A==At/8)
**    cpt0[10*6+k]==(d Qxy/d v_k) over V42 V12 V2
**    cpt0[11*6+k]==(d Qyy/d v_k) over V42 V12 V2
**    cpt0[12*6+k]==(d Qxx/d v_k) over V4 V42 V23 (having A==At/24)
**    cpt0[13*6+k]==(d Qxy/d v_k) over V4 V42 V23
**    cpt0[14*6+k]==(d Qyy/d v_k) over V4 V42 V23
**    cpt0[15*6+k]==(d Qxx/d v_k) over V42 V2 V23 (having A==At/8)
**    cpt0[16*6+k]==(d Qxy/d v_k) over V42 V2 V23
**    cpt0[17*6+k]==(d Qyy/d v_k) over V42 V2 V23
**    cpt0[18*6+k]==(d Qxx/d v_k) over V4 V23 V43 (having A==At/24)
**    cpt0[19*6+k]==(d Qxy/d v_k) over V4 V23 V43
**    cpt0[20*6+k]==(d Qyy/d v_k) over V4 V23 V43
**    cpt0[21*6+k]==(d Qxx/d v_k) over V43 V23 V3 (having A==At/8)
**    cpt0[22*6+k]==(d Qxy/d v_k) over V43 V23 V3
**    cpt0[23*6+k]==(d Qyy/d v_k) over V43 V23 V3
**    cpt0[24*6+k]==(d Qxx/d v_k) over V4 V43 V31 (having A==At/24)
**    cpt0[25*6+k]==(d Qxy/d v_k) over V4 V43 V31
**    cpt0[26*6+k]==(d Qyy/d v_k) over V4 V43 V31
**    cpt0[27*6+k]==(d Qxx/d v_k) over V43 V3 V31 (having A==At/8)
**    cpt0[28*6+k]==(d Qxy/d v_k) over V43 V3 V31
**    cpt0[29*6+k]==(d Qyy/d v_k) over V43 V3 V31
**    cpt0[30*6+k]==(d Qxx/d v_k) over V4 V31 V41 (having A==At/24)
**    cpt0[31*6+k]==(d Qxy/d v_k) over V4 V31 V41
**    cpt0[32*6+k]==(d Qyy/d v_k) over V4 V31 V41
**    cpt0[33*6+k]==(d Qxx/d v_k) over V41 V31 V1 (having A==At/8)
**    cpt0[34*6+k]==(d Qxy/d v_k) over V41 V31 V1
**    cpt0[35*6+k]==(d Qyy/d v_k) over V41 V31 V1
** where v_k ( k = 0, ..., 5 ) are the independent variables representing
** Qx(V1),Qy(V1),Qx(V2),Qy(V2),Qx(V3),Qy(V3).
**
** Then this function computes:
**
** 2* Integral [ ( (d Qxx/d v_ip) (d Qxx/d v_jp)  ) dx dy ]  +
**       T
**
** 2* Integral [ ( 2(d Qxy/d v_ip)(d Qxy/d v_jp)  ) dx dy ]  +
**       T
** 
** 2* Integral [ ( (d Qyy/d v_ip) (d Qyy/d v_jp)  ) dx dy ]  .
**       T
** 
**
** >>>>>>>>>>>>>>>>
** 
** double compute_for_vectconst(int pwsb_mode,
**                              double At,double At1,double At2,double At3,
**                              double cpt0)
**
** GLOBAL VARIABLES: IndTA IndTB
**                   u4,v4 
**                   (circumcenter if ABflag==0, barycenter if ABflag==1)
**                   u12,v12,u23,v23,u31,v31, 
**                   (middle edge points of the macro-triangle V1 V2 V3)
**                   u41,v41,u42,v42,u43,v43 
**                   (if ABflag==1 intersections between lines connecting 
**                    middle edge points of macro-triangle and lines connecting
**                    barycenter with vertices of macro-triangle) 
**                   where Vi=(ui,vi) i=1,2,3,12,23,31,4,41,42,43 are the vertices.
**
** Let Q be a (quadratic) Powell-Sabin polynomial patch over the macro-triangle T
** with vertices V1,V2,V3 and 'cpt0' an array made up of 18 or 36 entries
** as follows.
**
** Case 'pwsb_mode==0':  Powell-Sabin subdivision of A-type
** Apart from a constant factor 1/(2A^2) where A is the area of the subtriangle and
** At the one of the macro-triangle,
**    cpt0[ 0]==Qxx over V4 V1 V12 (having A==At1/2)
**    cpt0[ 1]==Qxy over V4 V1 V12
**    cpt0[ 2]==Qyy over V4 V1 V12
**    cpt0[ 3]==Qxx over V4 V12 V2 (having A==At1/2)
**    cpt0[ 4]==Qxy over V4 V12 V2
**    cpt0[ 5]==Qyy over V4 V12 V2
**    cpt0[ 6]==Qxx over V4 V2 V23 (having A==At2/2)
**    cpt0[ 7]==Qxy over V4 V2 V23
**    cpt0[ 8]==Qyy over V4 V2 V23
**    cpt0[ 9]==Qxx over V4 V23 V3 (having A==At2/2) 
**    cpt0[10]==Qxy over V4 V23 V3
**    cpt0[11]==Qyy over V4 V23 V3
**    cpt0[12]==Qxx over V4 V3 V31 (having A==At3/2)
**    cpt0[13]==Qxy over V4 V3 V31
**    cpt0[14]==Qyy over V4 V3 V31
**    cpt0[15]==Qxx over V4 V31 V1 (having A==At3/2)
**    cpt0[16]==Qxy over V4 V31 V1
**    cpt0[17]==Qyy over V4 V31 V1
**
** Case 'pwsb_mode==1': Powell-Sabin subdivision of B-type
** Apart from a constant factor 1/(2A^2) where A is the area of the subtriangle and
** At the one of the macro-triangle, 
**    cpt0[ 0]==Qxx over V4 V41 V12 (having A==At/24) 
**    cpt0[ 1]==Qxy over V4 V41 V12
**    cpt0[ 2]==Qyy over V4 V41 V12
**    cpt0[ 3]==Qxx over V41 V1 V12 (having A==At/8)
**    cpt0[ 4]==Qxy over V41 V1 V12
**    cpt0[ 5]==Qyy over V41 V1 V12
**    cpt0[ 6]==Qxx over V4 V12 V42 (having A==At/24)
**    cpt0[ 7]==Qxy over V4 V12 V42
**    cpt0[ 8]==Qyy over V4 V12 V42
**    cpt0[ 9]==Qxx over V42 V12 V2 (having A==At/8)
**    cpt0[10]==Qxy over V42 V12 V2
**    cpt0[11]==Qyy over V42 V12 V2
**    cpt0[12]==Qxx over V4 V42 V23 (having A==At/24)
**    cpt0[13]==Qxy over V4 V42 V23
**    cpt0[14]==Qyy over V4 V42 V23
**    cpt0[15]==Qxx over V42 V2 V23 (having A==At/8)
**    cpt0[16]==Qxy over V42 V2 V23
**    cpt0[17]==Qyy over V42 V2 V23
**    cpt0[18]==Qxx over V4 V23 V43 (having A==At/24)
**    cpt0[19]==Qxy over V4 V23 V43
**    cpt0[20]==Qyy over V4 V23 V43
**    cpt0[21]==Qxx over V43 V23 V3 (having A==At/8)
**    cpt0[22]==Qxy over V43 V23 V3
**    cpt0[23]==Qyy over V43 V23 V3
**    cpt0[24]==Qxx over V4 V43 V31 (having A==At/24)
**    cpt0[25]==Qxy over V4 V43 V31
**    cpt0[26]==Qyy over V4 V43 V31
**    cpt0[27]==Qxx over V43 V3 V31 (having A==At/8)
**    cpt0[28]==Qxy over V43 V3 V31
**    cpt0[29]==Qyy over V43 V3 V31
**    cpt0[30]==Qxx over V4 V31 V41 (having A==At/24)
**    cpt0[31]==Qxy over V4 V31 V41
**    cpt0[32]==Qyy over V4 V31 V41
**    cpt0[33]==Qxx over V41 V31 V1 (having A==At/8)
**    cpt0[34]==Qxy over V41 V31 V1
**    cpt0[35]==Qyy over V41 V31 V1
**
** Then this function computes:
**
** Integral [ (Qxx^2 + 2*Qxy^2 +Qyy^2) dx dy ]
**    T
**
** 
** >>>>>>>>>>>>>>>>
**
** double uu(int ABflag,int ntriang,int nvertex)
**
** GLOBAL VARIABLES: u4,v4 
**                   (circumcenter if ABflag==0, barycenter if ABflag==1)
**                   u12,v12,u23,v23,u31,v31, 
**                   (middle edge points of the macro-triangle V1 V2 V3)
**                   u41,v41,u42,v42,u43,v43 
**                   (if ABflag==1 intersections between lines connecting 
**                    middle edge points of macro-triangle and lines connecting
**                    barycenter with vertices of macro-triangle) 
**
** This function returns the x-coordinate of the 'nvertex'-th vertex
** (among 1,2,3) of the 'ntriang'-th subtriangle of a
** Powell-Sabin subdivision of A-type if 'ABflag==0' or of a
** Powell-Sabin subdivision of B-type if 'ABflag==1'.
**
** For a subdivision of A-type the order of subtriangles used by 'ntriang' is
** the following (in the range from 0 to 5):
** V4 V1 V12
** V4 V12 V2
** V4 V2 V23
** V4 V23 V3
** V4 V3 V31
** V4 V31 V1
**
** where Vi=(ui,vi) i=1,2,3,12,23,31,4 are the vertices.
**
** For a subdivision of B-type the order of subtriangles used by 'ntriang' is
** the following (in the range from 0 to 11):
** V4 V41 V12
** V41 V1 V12
** V4 V12 V42
** V42 V12 V2
** V4 V42 V23
** V42 V2 V23
** V4 V23 V43
** V43 V23 V3
** V4 V43 V31
** V43 V3 V31
** V4 V31 V41
** V41 V31 V1
**
** where Vi=(ui,vi) i=1,2,3,12,23,31,4,41,42,43 are the vertices.
**
** 
** >>>>>>>>>>>>>>>>
**
** double vv(int ABflag,int ntriang,int nvertex)
**
** GLOBAL VARIABLES: u4,v4 
**                   (circumcenter if ABflag==0, barycenter if ABflag==1)
**                   u12,v12,u23,v23,u31,v31, 
**                   (middle edge points of the macro-triangle V1 V2 V3)
**                   u41,v41,u42,v42,u43,v43 
**                   (if ABflag==1 intersections between lines connecting 
**                    middle edge points of macro-triangle and lines connecting
**                    barycenter with vertices of macro-triangle) 
**
** This function returns the y-coordinate of the 'nvertex'-th vertex
** (among 1,2,3) of the 'ntriang'-th subtriangle of a
** Powell-Sabin subdivision of A-type if 'ABflag==0' or of a
** Powell-Sabin subdivision of B-type if 'ABflag==1'.
**
** For a subdivision f A-type the order of subtriangles used by 'ntriang' is
** the following (in the range from 0 to 5):
** V4 V1 V12
** V4 V12 V2
** V4 V2 V23
** V4 V23 V3
** V4 V3 V31
** V4 V31 V1
**
** where Vi=(ui,vi) i=1,2,3,12,23,31,4 are the vertices.
**
** For a subdivision of B-type the order of subtriangles used by 'ntriang' is
** the following (in the range from 0 to 11):
** V4 V41 V12
** V41 V1 V12
** V4 V12 V42
** V42 V12 V2
** V4 V42 V23
** V42 V2 V23
** V4 V23 V43
** V43 V23 V3
** V4 V43 V31
** V43 V3 V31
** V4 V31 V41
** V41 V31 V1
**
** where Vi=(ui,vi) i=1,2,3,12,23,31,4,41,42,43 are the vertices.
**
** 
** >>>>>>>>>>>>>>>>
**
** double build_der(int ABflag,in nder,int ntype,int ntriang)
**
** GLOBAL VARIABLES: IndTA IndTB
**                   u4,v4 
**                   (circumcenter if ABflag==0, barycenter if ABflag==1)
**                   u12,v12,u23,v23,u31,v31, 
**                   (middle edge points of the macro-triangle V1 V2 V3)
**                   u41,v41,u42,v42,u43,v43 
**                   (if ABflag==1 intersections between lines connecting 
**                    middle edge points of macro-triangle and lines connecting
**                    barycenter with vertices of macro-triangle) 
**                   where Vi=(ui,vi) i=1,2,3,12,23,31,4,41,42,43 are the vertices.
**
** Let 'Q_ntriang' be the quadratic polynomial Powell-Sabin patch  
** over the 'ntriang'-th sub-triangle of a
** Powell-Sabin subdivision of A-type if 'ABflag==0' or of a
** Powell-Sabin subdivision of B-type if 'ABflag==1'
** and let 'A_ntriang' be the area of that sub-triangle.
**
** The sub-triangles, pointed out by the index 'ntriang', are sorted as follows:
**
** V4 V1 V12
** V4 V12 V2
** V4 V2 V23    for a subdivision of A-type
** V4 V23 V3
** V4 V3 V31
** V4 V31 V1
**
** or
**
** V4 V41 V12
** V41 V1 V12
** V4 V12 V42
** V42 V12 V2
** V4 V42 V23
** V42 V2 V23   for a subdivision of B-type
** V4 V23 V43
** V43 V23 V3
** V4 V43 V31
** V43 V3 V31
** V4 V31 V41
** V41 V31 V1.
**
** 
** Then, apart from the constant factor (1/2 A_ntriang^2), the function computes
**
** d           d Q_ntriang
** -------- (  -----------  )    if ntype==1,      
** d v_nder       d xx
**
** d           d Q_ntriang
** -------- (  -----------  )    if ntype==2,      
** d v_nder       d xy
**
** d           d Q_ntriang
** -------- (  -----------  )    if ntype==3,      
** d v_nder       d yy
**
** where v_k ( k = 0, ..., 5 ) are the independent variables representing
** Qx(V1),Qy(V1),Qx(V2),Qy(V2),Qx(V3),Qy(V3) and Q is the quadratic
** polynomial Powell-Sabin patch over the whole macro-triangle
** with vertices V1,V2,V3.   
**
** 
** >>>>>>>>>>>>>>>>
**
** double build_calc(int ABflag,int nder,int ntype,int ntriang)
**
** GLOBAL VARIABLES: IndTA IndTB
**                   u4,v4 
**                   (circumcenter if ABflag==0, barycenter if ABflag==1)
**                   u12,v12,u23,v23,u31,v31, 
**                   (middle edge points of macro-triangle V1 V2 V3)
**                   u41,v41,u42,v42,u43,v43 
**                   (if ABflag==1 intersections between lines connecting 
**                    middle edge points of macro-triangle and lines connecting
**                    barycenter with vertices of macro-triangle) 
**                   where Vi=(ui,vi) i=1,2,3,12,23,31,4,41,42,43 are the vertices.
**
** Let 'Q_ntriang' be the quadratic polynomial Powell-Sabin patch over the
** 'ntriang'-th sub-triangle of a
** Powell-Sabin subdivision of A-type if 'ABflag==0' or of a
** Powell-Sabin subdivision of B-type if 'ABflag==1'
** and let 'A_ntriang' be the area of the sub-triangle.
**
** The sub-triangles pointed out by the index 'ntriang', are sorted as follows:
**
** V4 V1 V12
** V4 V12 V2
** V4 V2 V23    for a subdivision of A-type
** V4 V23 V3
** V4 V3 V31
** V4 V31 V1
**
** or
**
** V4 V41 V12
** V41 V1 V12
** V4 V12 V42
** V42 V12 V2
** V4 V42 V23
** V42 V2 V23   for a subdivision of B-type
** V4 V23 V43
** V43 V23 V3
** V4 V43 V31
** V43 V3 V31
** V4 V31 V41
** V41 V31 V1.
**
** 
** Then, apart from the constant factor (1/2 A_ntriang^2), this function computes
**
**    d Q_ntriang    |
** (  -----------  ) |                             if ntype==1,      
**       d xx        | (v0,v1,v2,v3,v4,v5)=e_i
**
**    d Q_ntriang    |
** (  -----------  ) |                             if ntype==2,      
**       d xy        | (v0,v1,v2,v3,v4,v5)=e_i
**
**    d Q_ntriang    |
** (  -----------  ) |                             if ntype==3,      
**       d yy        | (v0,v1,v2,v3,v4,v5)=e_i
**
** where e_k (k= 0, ..., 5) are the vectors of the canonical basis of R^6,
** v_k ( k = 0, ..., 5 ) are the independent variables representing
** Qx(V1),Qy(V1),Qx(V2),Qy(V2),Qx(V3),Qy(V3) and Q is the quadratic
** polynomial Powell-Sabin patch over the whole macro-triangle
** with vertices V1,V2,V3.   
**
**
** >>>>>>>>>>>>>>>>
**
** double build_calc_null(int ABflag,int ntype,int ntriang)
**
** GLOBAL VARIABLES: IndTA IndTB
**                   u4,v4 
**                   (circumcenter if ABflag==0, barycenter if ABflag==1)
**                   u12,v12,u23,v23,u31,v31, 
**                   (middle edge points of the macro-triangle V1 V2 V3)
**                   u41,v41,u42,v42,u43,v43 
**                   (if ABflag==1 intersections between lines connecting 
**                    middle edge points of macro-triangle and lines connecting
**                    barycenter with vertices of macro-triangle) 
**                   where Vi=(ui,vi) i=1,2,3,12,23,31,4,41,42,43 are the vertices.
**
** Let 'Q_ntriang' be the quadratic polynomial Powell-Sabin patch over the 
** 'ntriang'-th sub-triangle of a
** Powell-Sabin subdivision of A-type if 'ABflag==0' or of a
** Powell-Sabin subdivision of B-type if 'ABflag==1'
** and let 'A_ntriang' be the area of that sub-triangle.
**
** The sub-triangles pointed out by the index 'ntriang', are sorted as follows:
**
** V4 V1 V12
** V4 V12 V2
** V4 V2 V23    for a subdivision of A-type
** V4 V23 V3
** V4 V3 V31
** V4 V31 V1
**
** or
**
** V4 V41 V12
** V41 V1 V12
** V4 V12 V42
** V42 V12 V2
** V4 V42 V23
** V42 V2 V23   for a subdivision of B-type
** V4 V23 V43
** V43 V23 V3
** V4 V43 V31
** V43 V3 V31
** V4 V31 V41
** V41 V31 V1.
**
** 
** Then, apart from the constant factor (1/2 A_ntriang^2), this function computes
**
**    d Q_ntriang    |
** (  -----------  ) |                                      if ntype==1,      
**       d xx        | (v0,v1,v2,v3,v4,v5)=(0,0,0,0,0,0)
**
**    d Q_ntriang    |
** (  -----------  ) |                                      if ntype==2,      
**       d xy        | (v0,v1,v2,v3,v4,v5)=(0,0,0,0,0,0)
**
**    d Q_ntriang    |
** (  -----------  ) |                                      if ntype==3,      
**       d yy        | (v0,v1,v2,v3,v4,v5)=(0,0,0,0,0,0)
**
** where v_k ( k = 0, ..., 5 ) are the independent variables representing
** Qx(V1),Qy(V1),Qx(V2),Qy(V2),Qx(V3),Qy(V3) and Q is the quadratic
** polynomial Powell-Sabin patch over the whole macro-triangle
** with vertices V1,V2,V3.   
**
** 
** >>>>>>>>>>>>>>>>
**
** int circoscrivi(double x1,double y1,double x2,double y2,double x3,double y3,
**                 double xc,double yc,double r)
**
** It computes the circle passing through the points (x1,y1), (x2,y2), (x3,y3).
** 
** The function returns
**     i)  the center (xc,yc) and  the radius 'r' of the circle  
**         if the function result is 'circoscrivi==0';
**    ii)  if 'r==NULL' the computation of the radius is not performed;
**   iii)  if 'circoscrivi==1' the three points are aligned or nearby aligned ones
**          (there will be a denominator smaller than EPS_ALLIN)
**
** 
**
** >>>>>>>>>>>>>>>>
**
** double triangle_max_angle(double x1,double y1,double x2,double y2,
**                           double x3,double y3)
**
** It computes the greatest angle (in degrees) among the three ones
** of the triangle with vertices (x1,y1), (x2,y2) and (x3,y3).
*/ 

#include<stdio.h>
#include<math.h>

#define EPS_ALLIN 1.e-13
#define PIGRECO 3.14159265358979323846

extern double z1,z2,z3;
extern double u1,v1,u2,v2,u3,v3,u4,v4,t1,t2,t3;
extern double u12,v12,u23,v23,u31,v31,u41,v41,u42,v42,u43,v43;
extern int IndTA[6][3][3], IndTB[12][3][3];
extern double uu(), vv(),
       derivecptpwsa_v(), derivecptpwsb_v(),
       compute_d_j_pwsa(), compute_d_j_pwsb(),
       compute_d_j_pwsa_null(), compute_d_j_pwsb_null();

double compute_for_matrix(pwsb_mode,At,At1,At2,At3,ip,jp,cpt0)
int pwsb_mode,ip,jp;
double At,At1,At2,At3,cpt0[];
{
 double temp_calc,temp_calc1,temp_calc2;

	      if (pwsb_mode==0)
	       {
                /* The constant 1./(4.*pow(a,3.)) comes from
                   (1/2a^2)^2 * (2a / ( (0+1)*(0+2)) )
                   where a is the area of the micro-triangle
                */
		temp_calc=
                  2/(4*pow(At1/2.,3.))*
		      (cpt0[ip]*cpt0[jp]+2*cpt0[ip+6]*cpt0[jp+6]+
		       cpt0[ip+12]*cpt0[jp+12]+cpt0[ip+18]*cpt0[jp+18]+
		       2*cpt0[ip+24]*cpt0[jp+24]+cpt0[ip+30]*cpt0[jp+30]);
		temp_calc+=
                  2/(4*pow(At2/2.,3.))*
		       (cpt0[ip+36]*cpt0[jp+36]+2*cpt0[ip+42]*cpt0[jp+42]+
		       cpt0[ip+48]*cpt0[jp+48]+cpt0[ip+54]*cpt0[jp+54]+
		       2*cpt0[ip+60]*cpt0[jp+60]+cpt0[ip+66]*cpt0[jp+66]);
		temp_calc+=
                  2/(4*pow(At3/2.,3.))*
		      (cpt0[ip+72]*cpt0[jp+72]+2*cpt0[ip+78]*cpt0[jp+78]+
		       cpt0[ip+84]*cpt0[jp+84]+cpt0[ip+90]*cpt0[jp+90]+
		       2*cpt0[ip+96]*cpt0[jp+96]+cpt0[ip+102]*cpt0[jp+102]);
	       }
	      else
	       {
		temp_calc1=
                cpt0[ip]*cpt0[jp]+
		2*cpt0[ip+6]*cpt0[jp+6]+
		cpt0[ip+12]*cpt0[jp+12];
                temp_calc2=
		cpt0[ip+18]*cpt0[jp+18]+
		2*cpt0[ip+24]*cpt0[jp+24]+
		cpt0[ip+30]*cpt0[jp+30];
		
                temp_calc1+=
		cpt0[ip+36]*cpt0[jp+36]+
		2*cpt0[ip+42]*cpt0[jp+42]+
		cpt0[ip+48]*cpt0[jp+48];
                temp_calc2+= 
		cpt0[ip+54]*cpt0[jp+54]+
		2*cpt0[ip+60]*cpt0[jp+60]+
		cpt0[ip+66]*cpt0[jp+66];
		
                temp_calc1+=
		cpt0[ip+72]*cpt0[jp+72]+
		2*cpt0[ip+78]*cpt0[jp+78]+
		cpt0[ip+84]*cpt0[jp+84];
                temp_calc2+=
		cpt0[ip+90]*cpt0[jp+90]+
		2*cpt0[ip+96]*cpt0[jp+96]+
		cpt0[ip+102]*cpt0[jp+102];
		
                temp_calc1+=
		cpt0[ip+108]*cpt0[jp+108]+
		2*cpt0[ip+114]*cpt0[jp+114]+
		cpt0[ip+120]*cpt0[jp+120];
                temp_calc2+=
		cpt0[ip+126]*cpt0[jp+126]+
		2*cpt0[ip+132]*cpt0[jp+132]+
		cpt0[ip+138]*cpt0[jp+138];
		
                temp_calc1+=
		cpt0[ip+144]*cpt0[jp+144]+
		2*cpt0[ip+150]*cpt0[jp+150]+
		cpt0[ip+156]*cpt0[jp+156];
                temp_calc2+=
		cpt0[ip+162]*cpt0[jp+162]+
		2*cpt0[ip+168]*cpt0[jp+168]+
		cpt0[ip+174]*cpt0[jp+174];
		
                temp_calc1+=
		cpt0[ip+180]*cpt0[jp+180]+
		2*cpt0[ip+186]*cpt0[jp+186]+
		cpt0[ip+192]*cpt0[jp+192];
                temp_calc2+=
		cpt0[ip+198]*cpt0[jp+198]+
		2*cpt0[ip+204]*cpt0[jp+204]+
		cpt0[ip+210]*cpt0[jp+210];

                /* The constant 1./(4.*pow(a,3.)) comes from
                   (1/2a^2)^2 * (2a / ( (0+1)*(0+2)) )
                   where a is the area of the micro-triangle
                */	
                temp_calc1=2/(4*pow(At/24.,3.))*temp_calc1;
                temp_calc2=2/(4*pow(At/8.,3.))*temp_calc2;
                temp_calc=temp_calc1+temp_calc2;
	       }
	      return(temp_calc);
}

double compute_for_vectconst(pwsb_mode,At,At1,At2,At3,cpt0)
int pwsb_mode;
double At,At1,At2,At3,cpt0[];
{
 double temp_calc,temp_calc1,temp_calc2;

	  if (pwsb_mode==0)
	   {
            /* The constant 1./(4.*pow(a,3.)) comes from
               (1/2a^2)^2 * (2a / ( (0+1)*(0+2)) )
               where a is the area of the micro-triangle
            */
	    /* Areas are At1/2 At2/2 At3/2 */
	    temp_calc= 1/(4*pow(At1/2.,3.))*
                              (cpt0[0]*cpt0[0]+2*cpt0[1]*cpt0[1]+
			      cpt0[2]*cpt0[2]+cpt0[3]*cpt0[3]+
			      2*cpt0[4]*cpt0[4]+cpt0[5]*cpt0[5]);
	    temp_calc+= 1/(4*pow(At2/2.,3.))*
                              (cpt0[6]*cpt0[6]+2*cpt0[7]*cpt0[7]+
			      cpt0[8]*cpt0[8]+cpt0[9]*cpt0[9]+
			      2*cpt0[10]*cpt0[10]+cpt0[11]*cpt0[11]);
	    temp_calc+= 1/(4*pow(At3/2.,3.))*
                              (cpt0[12]*cpt0[12]+2*cpt0[13]*cpt0[13]+
			      cpt0[14]*cpt0[14]+cpt0[15]*cpt0[15]+
			      2*cpt0[16]*cpt0[16]+cpt0[17]*cpt0[17]);
	   }
	   else
	   {
            /* The constant 1./(4.*pow(a,3.)) comes from
               (1/2a^2)^2 * (2a / ( (0+1)*(0+2)) )
               where a is the area of the micro-triangle
            */
            /* Triangles using temp_calc1 have area value At/24 */
            /* Triangles using temp_calc2 have area value 3*At/24 that is At/8 */
	    temp_calc1= cpt0[0]*cpt0[0]+2*cpt0[1]*cpt0[1]+cpt0[2]*cpt0[2];
	    temp_calc2=	cpt0[3]*cpt0[3]+2*cpt0[4]*cpt0[4]+cpt0[5]*cpt0[5];
	    temp_calc1+=cpt0[6]*cpt0[6]+2*cpt0[7]*cpt0[7]+cpt0[8]*cpt0[8];
	    temp_calc2+=cpt0[9]*cpt0[9]+2*cpt0[10]*cpt0[10]+cpt0[11]*cpt0[11];
	    temp_calc1+=cpt0[12]*cpt0[12]+2*cpt0[13]*cpt0[13]+cpt0[14]*cpt0[14];
	    temp_calc2+=cpt0[15]*cpt0[15]+2*cpt0[16]*cpt0[16]+cpt0[17]*cpt0[17];
	    temp_calc1+=cpt0[18]*cpt0[18]+2*cpt0[19]*cpt0[19]+cpt0[20]*cpt0[20];
	    temp_calc2+=cpt0[21]*cpt0[21]+2*cpt0[22]*cpt0[22]+cpt0[23]*cpt0[23];
	    temp_calc1+=cpt0[24]*cpt0[24]+2*cpt0[25]*cpt0[25]+cpt0[26]*cpt0[26];
	    temp_calc2+=cpt0[27]*cpt0[27]+2*cpt0[28]*cpt0[28]+cpt0[29]*cpt0[29];
	    temp_calc1+=cpt0[30]*cpt0[30]+2*cpt0[31]*cpt0[31]+cpt0[32]*cpt0[32];
            temp_calc2+=cpt0[33]*cpt0[33]+2*cpt0[34]*cpt0[34]+cpt0[35]*cpt0[35];
            temp_calc1= 1/(4*pow(At/24.,3.))*temp_calc1;
            temp_calc2= 1/(4*pow(At/8.,3.))*temp_calc2;
	    temp_calc=temp_calc1+temp_calc2;
	   }
	  return(temp_calc);
}

double uu(ABflag,ntriang,nvertex)
 int ABflag; /* 0 A-type Powell-Sabin   1 B-type Powell-Sabin */
 int ntriang,nvertex;
{
 switch(ABflag)
 {
  case 0: switch(nvertex)
	  {
	   case 1: return(u4);
		   break;
	   case 2: if (ntriang==0) return(u1);
		   else if (ntriang==1) return(u12);
		   else if (ntriang==2) return(u2);
		   else if (ntriang==3) return(u23);
		   else if (ntriang==4) return(u3);
		   else if (ntriang==5) return(u31);
		   break;
	   case 3: if (ntriang==0) return(u12);
		   else if (ntriang==1) return(u2);
		   else if (ntriang==2) return(u23);
		   else if (ntriang==3) return(u3);
		   else if (ntriang==4) return(u31);
		   else if (ntriang==5) return(u1);
		   break;
	  }  /* end switch over nvertex */
	  break;
  case 1: switch(nvertex)
	   {
	    case 1: if (ntriang==0) return(u4);
		    else if (ntriang==1) return(u41);
		    else if (ntriang==2) return(u4);
		    else if (ntriang==3) return(u42);
		    else if (ntriang==4) return(u4);
		    else if (ntriang==5) return(u42);
		    else if (ntriang==6) return(u4);
		    else if (ntriang==7) return(u43);
		    else if (ntriang==8) return(u4);
		    else if (ntriang==9) return(u43);
		    else if (ntriang==10) return(u4);
		    else if (ntriang==11) return(u41);
		    break;
	    case 2: if (ntriang==0) return(u41);
		    else if (ntriang==1) return(u1);
		    else if (ntriang==2) return(u12);
		    else if (ntriang==3) return(u12);
		    else if (ntriang==4) return(u42);
		    else if (ntriang==5) return(u2);
		    else if (ntriang==6) return(u23);
		    else if (ntriang==7) return(u23);
		    else if (ntriang==8) return(u43);
		    else if (ntriang==9) return(u3);
		    else if (ntriang==10) return(u31);
		    else if (ntriang==11) return(u31);
		    break;
	    case 3: if (ntriang==0) return(u12);
		    else if (ntriang==1) return(u12);
		    else if (ntriang==2) return(u42);
		    else if (ntriang==3) return(u2);
		    else if (ntriang==4) return(u23);
		    else if (ntriang==5) return(u23);
		    else if (ntriang==6) return(u43);
		    else if (ntriang==7) return(u3);
		    else if (ntriang==8) return(u31);
		    else if (ntriang==9) return(u31);
		    else if (ntriang==10) return(u41);
		    else if (ntriang==11) return(u1);
		    break;
	   } /* End switch over nvertex */
	   break;
 } /* End switch over ABflag */
}

double vv(ABflag,ntriang,nvertex)
 int ABflag; /* A-type Powell-Sabin   1 B-type Powell-Sabin */
 int ntriang,nvertex;
{
 switch(ABflag)
 {
  case 0: switch(nvertex)
	  {
	   case 1: return(v4);
		   break;
	   case 2: if (ntriang==0) return(v1);
		   else if (ntriang==1) return(v12);
		   else if (ntriang==2) return(v2);
		   else if (ntriang==3) return(v23);
		   else if (ntriang==4) return(v3);
		   else if (ntriang==5) return(v31);
		   break;
	   case 3: if (ntriang==0) return(v12);
		   else if (ntriang==1) return(v2);
		   else if (ntriang==2) return(v23);
		   else if (ntriang==3) return(v3);
		   else if (ntriang==4) return(v31);
		   else if (ntriang==5) return(v1);
		   break;
	  }  /* end switch over nvertex */
	  break;
  case 1: switch(nvertex)
	   {
	    case 1: if (ntriang==0) return(v4);
		    else if (ntriang==1) return(v41);
		    else if (ntriang==2) return(v4);
		    else if (ntriang==3) return(v42);
		    else if (ntriang==4) return(v4);
		    else if (ntriang==5) return(v42);
		    else if (ntriang==6) return(v4);
		    else if (ntriang==7) return(v43);
		    else if (ntriang==8) return(v4);
		    else if (ntriang==9) return(v43);
		    else if (ntriang==10) return(v4);
		    else if (ntriang==11) return(v41);
		    break;
	    case 2: if (ntriang==0) return(v41);
		    else if (ntriang==1) return(v1);
		    else if (ntriang==2) return(v12);
		    else if (ntriang==3) return(v12);
		    else if (ntriang==4) return(v42);
		    else if (ntriang==5) return(v2);
		    else if (ntriang==6) return(v23);
		    else if (ntriang==7) return(v23);
		    else if (ntriang==8) return(v43);
		    else if (ntriang==9) return(v3);
		    else if (ntriang==10) return(v31);
		    else if (ntriang==11) return(v31);
		    break;
	    case 3: if (ntriang==0) return(v12);
		    else if (ntriang==1) return(v12);
		    else if (ntriang==2) return(v42);
		    else if (ntriang==3) return(v2);
		    else if (ntriang==4) return(v23);
		    else if (ntriang==5) return(v23);
		    else if (ntriang==6) return(v43);
		    else if (ntriang==7) return(v3);
		    else if (ntriang==8) return(v31);
		    else if (ntriang==9) return(v31);
		    else if (ntriang==10) return(v41);
		    else if (ntriang==11) return(v1);
		    break;
	   } /* End switch over nvertex */
	   break;
 } /* End switch over ABflag */
}

double build_der(ABflag,nder,ntype,ntriang)
 int ABflag; /* 0 A-type Powell-Sabin   1 B-type Powell-Sabin */
 int nder,ntype,ntriang;
{
 double temp_calc,uu1,uu2,uu3,vv1,vv2,vv3;

 switch(ABflag)
 {
  case 0:
  uu1=uu(0,ntriang,1); vv1=vv(0,ntriang,1);
  uu2=uu(0,ntriang,2); vv2=vv(0,ntriang,2);
  uu3=uu(0,ntriang,3); vv3=vv(0,ntriang,3);
  switch(ntype)
  {
   case 1:
   temp_calc=derivecptpwsa_v(IndTA[ntriang][2][0],nder)*(vv3-vv2)*(vv3-vv2);
   temp_calc+=derivecptpwsa_v(IndTA[ntriang][0][2],nder)*(vv1-vv3)*(vv1-vv3);
   temp_calc+=derivecptpwsa_v(IndTA[ntriang][0][0],nder)*(vv2-vv1)*(vv2-vv1);
   temp_calc+=derivecptpwsa_v(IndTA[ntriang][1][1],nder)*2*(vv3-vv2)*(vv1-vv3);
   temp_calc+=derivecptpwsa_v(IndTA[ntriang][1][0],nder)*2*(vv3-vv2)*(vv2-vv1);
   temp_calc+=derivecptpwsa_v(IndTA[ntriang][0][1],nder)*2*(vv1-vv3)*(vv2-vv1);
   break;

   case 2: 
   temp_calc=-derivecptpwsa_v(IndTA[ntriang][2][0],nder)*(vv3-vv2)*(uu3-uu2);
   temp_calc-=derivecptpwsa_v(IndTA[ntriang][0][2],nder)*(vv1-vv3)*(uu1-uu3);
   temp_calc-=derivecptpwsa_v(IndTA[ntriang][0][0],nder)*(vv2-vv1)*(uu2-uu1);
   temp_calc-=derivecptpwsa_v(IndTA[ntriang][1][1],nder)*
	      ((vv3-vv2)*(uu1-uu3)+(vv1-vv3)*(uu3-uu2));
   temp_calc-=derivecptpwsa_v(IndTA[ntriang][1][0],nder)*
	      ((vv3-vv2)*(uu2-uu1)+(vv2-vv1)*(uu3-uu2));
   temp_calc-=derivecptpwsa_v(IndTA[ntriang][0][1],nder)*
	      ((vv1-vv3)*(uu2-uu1)+(vv2-vv1)*(uu1-uu3));
   break;

   case 3: 
   temp_calc=derivecptpwsa_v(IndTA[ntriang][2][0],nder)*(uu3-uu2)*(uu3-uu2);
   temp_calc+=derivecptpwsa_v(IndTA[ntriang][0][2],nder)*(uu1-uu3)*(uu1-uu3);
   temp_calc+=derivecptpwsa_v(IndTA[ntriang][0][0],nder)*(uu2-uu1)*(uu2-uu1);
   temp_calc+=derivecptpwsa_v(IndTA[ntriang][1][1],nder)*2*(uu3-uu2)*(uu1-uu3);
   temp_calc+=derivecptpwsa_v(IndTA[ntriang][1][0],nder)*2*(uu3-uu2)*(uu2-uu1);
   temp_calc+=derivecptpwsa_v(IndTA[ntriang][0][1],nder)*2*(uu1-uu3)*(uu2-uu1);
   break;
  } /* end switch over ntype */
  break;

  case 1:
  uu1=uu(1,ntriang,1); vv1=vv(1,ntriang,1);
  uu2=uu(1,ntriang,2); vv2=vv(1,ntriang,2);
  uu3=uu(1,ntriang,3); vv3=vv(1,ntriang,3);
  switch(ntype)
  {
   case 1: 
   temp_calc=derivecptpwsb_v(IndTB[ntriang][2][0],nder)*(vv3-vv2)*(vv3-vv2);
   temp_calc+=derivecptpwsb_v(IndTB[ntriang][0][2],nder)*(vv1-vv3)*(vv1-vv3);
   temp_calc+=derivecptpwsb_v(IndTB[ntriang][0][0],nder)*(vv2-vv1)*(vv2-vv1);
   temp_calc+=derivecptpwsb_v(IndTB[ntriang][1][1],nder)*2*(vv3-vv2)*(vv1-vv3);
   temp_calc+=derivecptpwsb_v(IndTB[ntriang][1][0],nder)*2*(vv3-vv2)*(vv2-vv1);
   temp_calc+=derivecptpwsb_v(IndTB[ntriang][0][1],nder)*2*(vv1-vv3)*(vv2-vv1);
   break;

   case 2: 
   temp_calc=-derivecptpwsb_v(IndTB[ntriang][2][0],nder)*(vv3-vv2)*(uu3-uu2);
   temp_calc-=derivecptpwsb_v(IndTB[ntriang][0][2],nder)*(vv1-vv3)*(uu1-uu3);
   temp_calc-=derivecptpwsb_v(IndTB[ntriang][0][0],nder)*(vv2-vv1)*(uu2-uu1);
   temp_calc-=derivecptpwsb_v(IndTB[ntriang][1][1],nder)*
	      ((vv3-vv2)*(uu1-uu3)+(vv1-vv3)*(uu3-uu2));
   temp_calc-=derivecptpwsb_v(IndTB[ntriang][1][0],nder)*
	      ((vv3-vv2)*(uu2-uu1)+(vv2-vv1)*(uu3-uu2));
   temp_calc-=derivecptpwsb_v(IndTB[ntriang][0][1],nder)*
	      ((vv1-vv3)*(uu2-uu1)+(vv2-vv1)*(uu1-uu3));
   break;

   case 3: 
   temp_calc=derivecptpwsb_v(IndTB[ntriang][2][0],nder)*(uu3-uu2)*(uu3-uu2);
   temp_calc+=derivecptpwsb_v(IndTB[ntriang][0][2],nder)*(uu1-uu3)*(uu1-uu3);
   temp_calc+=derivecptpwsb_v(IndTB[ntriang][0][0],nder)*(uu2-uu1)*(uu2-uu1);
   temp_calc+=derivecptpwsb_v(IndTB[ntriang][1][1],nder)*2*(uu3-uu2)*(uu1-uu3);
   temp_calc+=derivecptpwsb_v(IndTB[ntriang][1][0],nder)*2*(uu3-uu2)*(uu2-uu1);
   temp_calc+=derivecptpwsb_v(IndTB[ntriang][0][1],nder)*2*(uu1-uu3)*(uu2-uu1);
   break;
  } /* end switch over ntype */
 } /* end switch over ABFlag */
 return (temp_calc);
}

double build_calc(ABflag,nder,ntype,ntriang)
 int ABflag; /* 0 A-type Powell-Sabin   1 B-type Powell-Sabin */
 int nder,ntype,ntriang;
{
 double temp_calc,uu1,uu2,uu3,vv1,vv2,vv3;

 switch(ABflag)
 {
  case 0:
  uu1=uu(0,ntriang,1); vv1=vv(0,ntriang,1);
  uu2=uu(0,ntriang,2); vv2=vv(0,ntriang,2);
  uu3=uu(0,ntriang,3); vv3=vv(0,ntriang,3);
  switch(ntype)
  {
   case 1: 
   temp_calc=compute_d_j_pwsa(IndTA[ntriang][2][0],nder)*(vv3-vv2)*(vv3-vv2);
   temp_calc+=compute_d_j_pwsa(IndTA[ntriang][0][2],nder)*(vv1-vv3)*(vv1-vv3);
   temp_calc+=compute_d_j_pwsa(IndTA[ntriang][0][0],nder)*(vv2-vv1)*(vv2-vv1);
   temp_calc+=compute_d_j_pwsa(IndTA[ntriang][1][1],nder)*2*(vv3-vv2)*(vv1-vv3);
   temp_calc+=compute_d_j_pwsa(IndTA[ntriang][1][0],nder)*2*(vv3-vv2)*(vv2-vv1);
   temp_calc+=compute_d_j_pwsa(IndTA[ntriang][0][1],nder)*2*(vv1-vv3)*(vv2-vv1);
   break;
   case 2: 
   temp_calc=-compute_d_j_pwsa(IndTA[ntriang][2][0],nder)*(vv3-vv2)*(uu3-uu2);
   temp_calc-=compute_d_j_pwsa(IndTA[ntriang][0][2],nder)*(vv1-vv3)*(uu1-uu3);
   temp_calc-=compute_d_j_pwsa(IndTA[ntriang][0][0],nder)*(vv2-vv1)*(uu2-uu1);
   temp_calc-=compute_d_j_pwsa(IndTA[ntriang][1][1],nder)*
	      ((vv3-vv2)*(uu1-uu3)+(vv1-vv3)*(uu3-uu2));
   temp_calc-=compute_d_j_pwsa(IndTA[ntriang][1][0],nder)*
	      ((vv3-vv2)*(uu2-uu1)+(vv2-vv1)*(uu3-uu2));
   temp_calc-=compute_d_j_pwsa(IndTA[ntriang][0][1],nder)*
	      ((vv1-vv3)*(uu2-uu1)+(vv2-vv1)*(uu1-uu3));
   break; 
   case 3:
   temp_calc=compute_d_j_pwsa(IndTA[ntriang][2][0],nder)*(uu3-uu2)*(uu3-uu2);
   temp_calc+=compute_d_j_pwsa(IndTA[ntriang][0][2],nder)*(uu1-uu3)*(uu1-uu3);
   temp_calc+=compute_d_j_pwsa(IndTA[ntriang][0][0],nder)*(uu2-uu1)*(uu2-uu1);
   temp_calc+=compute_d_j_pwsa(IndTA[ntriang][1][1],nder)*2*(uu3-uu2)*(uu1-uu3);
   temp_calc+=compute_d_j_pwsa(IndTA[ntriang][1][0],nder)*2*(uu3-uu2)*(uu2-uu1);
   temp_calc+=compute_d_j_pwsa(IndTA[ntriang][0][1],nder)*2*(uu1-uu3)*(uu2-uu1);
   break;
  } /* end switch over ntype */
  break;
  case 1:
  uu1=uu(1,ntriang,1); vv1=vv(1,ntriang,1);
  uu2=uu(1,ntriang,2); vv2=vv(1,ntriang,2);
  uu3=uu(1,ntriang,3); vv3=vv(1,ntriang,3);
  switch(ntype)
  {
   case 1:
   temp_calc=compute_d_j_pwsb(IndTB[ntriang][2][0],nder)*(vv3-vv2)*(vv3-vv2);
   temp_calc+=compute_d_j_pwsb(IndTB[ntriang][0][2],nder)*(vv1-vv3)*(vv1-vv3);
   temp_calc+=compute_d_j_pwsb(IndTB[ntriang][0][0],nder)*(vv2-vv1)*(vv2-vv1);
   temp_calc+=compute_d_j_pwsb(IndTB[ntriang][1][1],nder)*2*(vv3-vv2)*(vv1-vv3);
   temp_calc+=compute_d_j_pwsb(IndTB[ntriang][1][0],nder)*2*(vv3-vv2)*(vv2-vv1);
   temp_calc+=compute_d_j_pwsb(IndTB[ntriang][0][1],nder)*2*(vv1-vv3)*(vv2-vv1);
   break;
   case 2:
   temp_calc=-compute_d_j_pwsb(IndTB[ntriang][2][0],nder)*(vv3-vv2)*(uu3-uu2);
   temp_calc-=compute_d_j_pwsb(IndTB[ntriang][0][2],nder)*(vv1-vv3)*(uu1-uu3);
   temp_calc-=compute_d_j_pwsb(IndTB[ntriang][0][0],nder)*(vv2-vv1)*(uu2-uu1);
   temp_calc-=compute_d_j_pwsb(IndTB[ntriang][1][1],nder)*
	      ((vv3-vv2)*(uu1-uu3)+(vv1-vv3)*(uu3-uu2));
   temp_calc-=compute_d_j_pwsb(IndTB[ntriang][1][0],nder)*
	      ((vv3-vv2)*(uu2-uu1)+(vv2-vv1)*(uu3-uu2));
   temp_calc-=compute_d_j_pwsb(IndTB[ntriang][0][1],nder)*
	      ((vv1-vv3)*(uu2-uu1)+(vv2-vv1)*(uu1-uu3));
   break;
   case 3:
   temp_calc=compute_d_j_pwsb(IndTB[ntriang][2][0],nder)*(uu3-uu2)*(uu3-uu2);
   temp_calc+=compute_d_j_pwsb(IndTB[ntriang][0][2],nder)*(uu1-uu3)*(uu1-uu3);
   temp_calc+=compute_d_j_pwsb(IndTB[ntriang][0][0],nder)*(uu2-uu1)*(uu2-uu1);
   temp_calc+=compute_d_j_pwsb(IndTB[ntriang][1][1],nder)*2*(uu3-uu2)*(uu1-uu3);
   temp_calc+=compute_d_j_pwsb(IndTB[ntriang][1][0],nder)*2*(uu3-uu2)*(uu2-uu1);
   temp_calc+=compute_d_j_pwsb(IndTB[ntriang][0][1],nder)*2*(uu1-uu3)*(uu2-uu1);
   break;
  } /* end switch over ntype */
  break;
 } /* end switch over ABflag */
 return(temp_calc);
}

double build_calc_null(ABflag,ntype,ntriang)
 int ABflag; /* 0 A-type Powell-Sabin   1 B-type Powell-Sabin */
 int ntype,ntriang;
{
 double temp_calc,uu1,uu2,uu3,vv1,vv2,vv3;

 switch(ABflag)
 {
  case 0:
  uu1=uu(0,ntriang,1); vv1=vv(0,ntriang,1);
  uu2=uu(0,ntriang,2); vv2=vv(0,ntriang,2);
  uu3=uu(0,ntriang,3); vv3=vv(0,ntriang,3);
  switch(ntype)
  {
   case 1:
   temp_calc=compute_d_j_pwsa_null(IndTA[ntriang][2][0])*(vv3-vv2)*(vv3-vv2);
   temp_calc+=compute_d_j_pwsa_null(IndTA[ntriang][0][2])*(vv1-vv3)*(vv1-vv3);
   temp_calc+=compute_d_j_pwsa_null(IndTA[ntriang][0][0])*(vv2-vv1)*(vv2-vv1);
   temp_calc+=compute_d_j_pwsa_null(IndTA[ntriang][1][1])*2*(vv3-vv2)*(vv1-vv3);
   temp_calc+=compute_d_j_pwsa_null(IndTA[ntriang][1][0])*2*(vv3-vv2)*(vv2-vv1);
   temp_calc+=compute_d_j_pwsa_null(IndTA[ntriang][0][1])*2*(vv1-vv3)*(vv2-vv1);
   break;
   case 2:
   temp_calc=-compute_d_j_pwsa_null(IndTA[ntriang][2][0])*(vv3-vv2)*(uu3-uu2);
   temp_calc-=compute_d_j_pwsa_null(IndTA[ntriang][0][2])*(vv1-vv3)*(uu1-uu3);
   temp_calc-=compute_d_j_pwsa_null(IndTA[ntriang][0][0])*(vv2-vv1)*(uu2-uu1);
   temp_calc-=compute_d_j_pwsa_null(IndTA[ntriang][1][1])*
	      ((vv3-vv2)*(uu1-uu3)+(vv1-vv3)*(uu3-uu2));
   temp_calc-=compute_d_j_pwsa_null(IndTA[ntriang][1][0])*
	      ((vv3-vv2)*(uu2-uu1)+(vv2-vv1)*(uu3-uu2));
   temp_calc-=compute_d_j_pwsa_null(IndTA[ntriang][0][1])*
	      ((vv1-vv3)*(uu2-uu1)+(vv2-vv1)*(uu1-uu3));
   break;
   case 3:
   temp_calc=compute_d_j_pwsa_null(IndTA[ntriang][2][0])*(uu3-uu2)*(uu3-uu2);
   temp_calc+=compute_d_j_pwsa_null(IndTA[ntriang][0][2])*(uu1-uu3)*(uu1-uu3);
   temp_calc+=compute_d_j_pwsa_null(IndTA[ntriang][0][0])*(uu2-uu1)*(uu2-uu1);
   temp_calc+=compute_d_j_pwsa_null(IndTA[ntriang][1][1])*2*(uu3-uu2)*(uu1-uu3);
   temp_calc+=compute_d_j_pwsa_null(IndTA[ntriang][1][0])*2*(uu3-uu2)*(uu2-uu1);
   temp_calc+=compute_d_j_pwsa_null(IndTA[ntriang][0][1])*2*(uu1-uu3)*(uu2-uu1);
   break;
  } /* end switch over ntype */
  break;
  case 1:
  uu1=uu(1,ntriang,1); vv1=vv(1,ntriang,1);
  uu2=uu(1,ntriang,2); vv2=vv(1,ntriang,2);
  uu3=uu(1,ntriang,3); vv3=vv(1,ntriang,3);
  switch(ntype)
  {
   case 1:
   temp_calc=compute_d_j_pwsb_null(IndTB[ntriang][2][0])*(vv3-vv2)*(vv3-vv2);
   temp_calc+=compute_d_j_pwsb_null(IndTB[ntriang][0][2])*(vv1-vv3)*(vv1-vv3);
   temp_calc+=compute_d_j_pwsb_null(IndTB[ntriang][0][0])*(vv2-vv1)*(vv2-vv1);
   temp_calc+=compute_d_j_pwsb_null(IndTB[ntriang][1][1])*2*(vv3-vv2)*(vv1-vv3);
   temp_calc+=compute_d_j_pwsb_null(IndTB[ntriang][1][0])*2*(vv3-vv2)*(vv2-vv1);
   temp_calc+=compute_d_j_pwsb_null(IndTB[ntriang][0][1])*2*(vv1-vv3)*(vv2-vv1);
   break;
   case 2:
   temp_calc=-compute_d_j_pwsb_null(IndTB[ntriang][2][0])*(vv3-vv2)*(uu3-uu2);
   temp_calc-=compute_d_j_pwsb_null(IndTB[ntriang][0][2])*(vv1-vv3)*(uu1-uu3);
   temp_calc-=compute_d_j_pwsb_null(IndTB[ntriang][0][0])*(vv2-vv1)*(uu2-uu1);
   temp_calc-=compute_d_j_pwsb_null(IndTB[ntriang][1][1])*
	      ((vv3-vv2)*(uu1-uu3)+(vv1-vv3)*(uu3-uu2));
   temp_calc-=compute_d_j_pwsb_null(IndTB[ntriang][1][0])*
	      ((vv3-vv2)*(uu2-uu1)+(vv2-vv1)*(uu3-uu2));
   temp_calc-=compute_d_j_pwsb_null(IndTB[ntriang][0][1])*
	      ((vv1-vv3)*(uu2-uu1)+(vv2-vv1)*(uu1-uu3));
   break;
   case 3:
   temp_calc=compute_d_j_pwsb_null(IndTB[ntriang][2][0])*(uu3-uu2)*(uu3-uu2);
   temp_calc+=compute_d_j_pwsb_null(IndTB[ntriang][0][2])*(uu1-uu3)*(uu1-uu3);
   temp_calc+=compute_d_j_pwsb_null(IndTB[ntriang][0][0])*(uu2-uu1)*(uu2-uu1);
   temp_calc+=compute_d_j_pwsb_null(IndTB[ntriang][1][1])*2*(uu3-uu2)*(uu1-uu3);
   temp_calc+=compute_d_j_pwsb_null(IndTB[ntriang][1][0])*2*(uu3-uu2)*(uu2-uu1);
   temp_calc+=compute_d_j_pwsb_null(IndTB[ntriang][0][1])*2*(uu1-uu3)*(uu2-uu1);
   break;
  } /* end switch over ntype */
  break;
 } /* fine switch su ABflag */
 return(temp_calc);
}

int circoscrivi(x1,y1,x2,y2,x3,y3,xc,yc,r)
double x1,y1,x2,y2,x3,y3,*xc,*yc,*r;
{
double Denom;

Denom=(2*(x1*(y2-y3)+x2*(y3-y1)+x3*(y1-y2)));
if (fabs(Denom)<EPS_ALLIN) return(1);
*xc=(pow(x1,2.)*(y2-y3)
     +pow(x2,2.)*(y3-y1)
     +(y1-y2)*(pow(x3,2.)
	       +(y2-y3)*(y1-y3)) )
    /Denom;
*yc=-(pow(x1,2.)*(x2-x3)
      -x1*(pow(x2,2.)
      -pow(x3,2.)
      +(y2-y3)*(y2+y3))
      +pow(x2,2.)*x3
      -x2*(pow(x3,2.)
	   +(y3-y1)*(y1+y3))
      +x3*(y2-y1)*(y1+y2) )
     /Denom;
if (r!=NULL) *r=sqrt(pow((*xc-x1),2.)+pow((*yc-y1),2.));
return(0);
}

double triangle_max_angle(x1,y1,x2,y2,x3,y3)
double x1,y1,x2,y2,x3,y3;
{
 double t1,t2,t3;

 t1=acos(1./(sqrt(pow(x2-x1,2.)+pow(y2-y1,2.))*
	     sqrt(pow(x3-x1,2.)+pow(y3-y1,2.)))*
	 ((x2-x1)*(x3-x1)+(y2-y1)*(y3-y1)));
 t2=acos(1./(sqrt(pow(x3-x2,2.)+pow(y3-y2,2.))*
	     sqrt(pow(x1-x2,2.)+pow(y1-y2,2.)))*
	 ((x3-x2)*(x1-x2)+(y3-y2)*(y1-y2)));
 t3=acos(1./(sqrt(pow(x1-x3,2.)+pow(y1-y3,2.))*
	     sqrt(pow(x2-x3,2.)+pow(y2-y3,2.)))*
	 ((x1-x3)*(x2-x3)+(y1-y3)*(y2-y3)));
 if (t2>t1) t1=t2;
 if (t3>t1) t1=t3;
 return(t1/PIGRECO*180.);
}







