/*
** PWSGLDER.C
**
** AUTHORS: 
**
**   Maria MORANDI CECCHI               Stefano DE MARCHI,            
**   University of Padua, Italy         University of Udine, Italy
**   email: mcecchi@math.unipd.it       email: demarchi@dimi.uniud.it
**   
**                       Damiano FASOLI
**                       Verona Software Srl, Italy 
**                       email: vrsoft@sis.it
**
**
**
** REVISION DATE: May, 1999  
**
** MODULES CALLED: PWS_D_A.C
**                 PWS_D_B.C
**                 PWSTOOLS.C
**                 LINSIST.C
**
** ------------------------------------------------------------------------
**
** BACKGROUND
**
** Let (xi,yi,qi) i=0,...,N-1 be N points in the space R^3. 
** Let us consider their projections (xi,yi) onto the plane z=0 and 
** let T be a triangulation of them. 
** A triangulation T of a set of points is the set of triangles whose vertices 
** are the points and such that two distinct triangles intersect at 
** most in a common edge or in a common vertex. 
**
** The aim is the interpolation of the points (xi,yi,qi) with a bivariate
** C^1 function.
**
** The Powell-Sabin (PS) interpolation method builds up such a function 
** using a Powell-Sabin polynomial patch on each triangle, 
** assuming that the first and second partial derivatives are known
** at each vertex of the triangulation T.
** The Powell-Sabin polynomial element is based on two different splittings
** of the triangle (the macro-triangle) so that we can have two types of  
** PS-elements: A-type and B-type. This latter if the macro-triangle has
** an angle greater than 75 degrees. 
**
** The Powell-Sabin polynomial element of A-type,
** splits a triangle into six sub-triangles 
** (using the circumcenter, the vertices of the macro-triangles and the
** middle edge points)
** over which it builds six quadratic polynomial patches.
**
** The Powell-Sabin polynomial element of B-type,
** splits a triangle into twelve sub-triangles
** (using the barycenter, the vertices of the macro-triangles,
** the middle edge points and the intersections between lines
** connecting them and lines connecting the barycenter and the 
** vertices of the macro-triangle) over which it builds twelve
** quadratic polynomial patches.
**
** The aim of this program is to compute such parameters (in the following
** we shall refer to them as to qix,qiy i=0,...,N-1, or more often
** as to the 2N-dimensional vector v) in the
** "smoothest" possible way for a Powell-Sabin interpolating function 
** over the whole triangulation T. 
**
** To evaluate the "smoothness" of a Powell-Sabin
** interpolating function it is minimised the functional
**
** phi(v)=   sum  [ Integral[ (Qxx^2 + 2*Qxy^2 + Qyy^2  ) dx dy ] ]     
**          t in T      t      
** 
** with respect to v.
** 
** 
** METHODOLOGY 
**
**   phi(v)= 1/2 * v' A v + g' v + c 
**
** is minimum when v solves the system
**
**          A v + g = 0.
** 
** The program computes 
**
**         c=phi(0), A, gi=phi(ei)-1/2*aii-c, 
**
** where ei is the i-th vector of the canonical basis of R^(2N), 
** and solves the resulting linear system with the Cholesky factorisation.
**
** 
** REFERENCES
**
**  M. Morandi Cecchi, S. De Marchi, D. Fasoli,
**     "A Package for Representing C^1 Interpolating Surfaces: 
**      Application to the Lagoon of Venice's Bed."
**      Numer. Algorithms, 1999.
**
**  D. Fasoli,
**      Laurea's Thesis, University of Padua, 1995 (in Italian). 
**
**  P. Alfeld:
**    "Derivative Generation for multivariate 
**     scattered data functional minimization", 
**     Comp. Aided Graph. Design,2: 281-296, 1985.
**
**  G. Farin:
**     "Triangular Bernstein-Bezier patches",
**      Comput. Aided Geom. Design,3:83-127, 1986.
**
** ---------------------------------------------------------------
** 
** SYNOPSIS:   pwsglder file1 file2 [file3]
**
** file1 is the data file made up of 3-dimensional points. 
** The coordinates of the points may be separated by any 
** combination of spaces, tabs and carriage returns, 
** even though it is recommended to use a row for each point. 
** 
** file2 is the triangulation data file in the format generated by
** 'do_trian' program. This file has three sections; 
** the first, starting with a single capital letter C in the first row, 
** lists in subsequent blocks separated by -1 the index of
** a node and the indexes of its neighbours in a counterclockwise
** order. 
** The present program uses only this first section 
** and the heading of the second section, 
** which is made of a single capital letter B.
**
** file3 is the output file, the file of the derivatives.
** Omit it to use standard output. 
** The values in a same row refer to the same vertex of the triangulation.
** The first column is the x-derivative, the second column corresponds to
** the y-derivative, the third, the fourth and the fifth columns are 0.
** The format of the output defined by the constants WIDTH and PREC
** that are the width of the output field, and the number of decimal digits.
**        
** The program supports at most MAX_NUM_NODI (#DEFINE constant) input points.
** If more points are given, a warning message will be printed..
** 
** ---------------------------------------------------------------------
** SUBROUTINES AND FUNCTIONS
**
** int global_index(int i,int iv1,int iv2,int iv3) 
**
** This function transforms the index 'i', a local index in the range 0...5 
** corresponding to the sequence of parameters 
**
**    qx(V1), qy(V1),
**    qx(V2), qy(V2), 
**    qx(V3), qy(V3) 
**
** and the triangle with verteces V1,V2,V3 
** into a global index according to
** the vertex among V1,V2,V3 to which it refers to.
** 
** Global indexes are sorted by the index of the node in the triangulation
** and the derivation sequence qx,qy.  
**
** Function result: 
**           the global index corresponding to the local index 'i'.
**
** ------------------------------------------------------------------------------
*/

#include<stdio.h>
#include<stdlib.h>
#include<math.h>
#define MAX_NUM_NODI 100000
#define WIDTH 20   /* Format width of real numbers written to files */
#define PREC 10    /* Decimal precision width of real numbers written to files */

double derivecptpwsa_v(),compute_d_j_pwsa(),compute_d_j_pwsa_null(),
       derivecptpwsb_v(),compute_d_j_pwsb(),compute_d_j_pwsb_null(),
       build_der(),build_calc(),build_calc_null(),uu(),vv(),
       triangle_max_angle(),compute_for_matrix(),compute_for_vectconst();
int global_index(),circoscrivi();

/* Declarations for global variables */
double z1,z2,z3;
double u1,v1,u2,v2,u3,v3,u4,v4,t1,t2,t3;
double u12,v12,u23,v23,u31,v31,u41,v41,u42,v42,u43,v43;

  /* The value 31 in the following declarations
  ** is not used by the algorithm */
  int IndTA[6][3][3]={
  {{2,1,0},{13,12,31},{18,31,31}},
  {{4,3,2},{14,13,31},{18,31,31}},
  {{6,5,4},{15,14,31},{18,31,31}},
  {{8,7,6},{16,15,31},{18,31,31}},
  {{10,9,8},{17,16,31},{18,31,31}},
  {{0,11,10},{12,17,31},{18,31,31}}
  };
  int IndTB[12][3][3]={
  {{2,13,24},{14,27,31},{30,31,31}},
  {{2,1,0},{13,12,31},{24,31,31}},
  {{25,15,2},{28,14,31},{30,31,31}},
  {{4,3,2},{16,15,31},{25,31,31}},
  {{6,17,25},{18,28,31},{30,31,31}},
  {{6,5,4},{17,16,31},{25,31,31}},
  {{26,19,6},{29,18,31},{30,31,31}},
  {{8,7,6},{20,19,31},{26,31,31}},
  {{10,21,26},{22,29,31},{30,31,31}},
  {{10,9,8},{21,20,31},{26,31,31}},
  {{24,23,10},{27,22,31},{30,31,31}},
  {{0,11,10},{12,23,31},{24,31,31}}
  };

void main(argc,argv)
int argc;
char *argv[];
{
  struct triangle {long int v1; long int v2; long int v3;};
  
  size_t size_double;
  FILE *infile, *trifile, *outfile;
  char base_buffer[20];
  int num_nodo,iv1,iv2,iv3;
  double x[MAX_NUM_NODI],y[MAX_NUM_NODI],z[MAX_NUM_NODI];
  double At,At1,At2,At3;
  int i,j,ip_global,jp_global,nt,ip,jp,err;
  int ct1,ct2,pwsb_mode; /* 0 tipo A  1 tipo B */
  double cpt0[216],temp_calc;
  /* 216 comes from 6*3*12, that is from
  ** (unknown derivatives) * (xx xy yy contributes) * (triangles) */
  int stop; /* For loops */
  double *A, *v, *gg, *DiagL, c;

 infile=NULL; trifile=NULL; outfile=NULL;
 A=NULL; v=NULL; gg=NULL; DiagL=NULL;
 size_double=sizeof(double);
 if (argc==1)
   {
    printf("\nStart the program with three parameters:\n");
    printf("First parameter: data file name.\n");
    printf("Second parameter: triangulation file name.\n");
    printf("Third parameter: derivative file name.\n");
    printf("                 (Omit it for screen output.)\n");
    exit(1);
   }
 /* Reading */
 infile=fopen(argv[1],"r");
 if (infile==NULL)
   {
    printf("Cannot open file: %s.\n",argv[1]);
    exit(1);
   }
 if (argc==2)
   {
    printf("Triangulation file missing.\n");
    exit(1);
   }
 /* data reading */
 printf("Wait for data reading\n"); 
 num_nodo=0;
 while (!feof(infile))
 {
  if (num_nodo<MAX_NUM_NODI)
   {
    fscanf(infile," %lf %lf %lf ",&x[num_nodo],&y[num_nodo],&z[num_nodo]);
      /* spaces are essential part of input format string */
    num_nodo=num_nodo+1;
   }
   else
   {
    printf("Too many input data.\n");
    exit(1);
   }
 }
 if (num_nodo<3)
    {
     printf("I need at least three input points.\n");
     exit(1);
    }
 fclose(infile);
 /* end of data reading */
 /* opening triangulation file */
 trifile=fopen(argv[2],"r");
 if (trifile==NULL)
   {
    printf("Cannot open file: %s.\n",argv[2]);
    exit(1);
   }
 if (fgetc(trifile)!='C')
   {
    printf("File %s in a no correct format.\n",argv[2]);
    exit(1);
   }
 else fgetc(trifile); /* carriage return */
 /* opening output file */
 if (argc==3) outfile=stdout; else outfile=fopen(argv[3],"w");
 if (outfile==NULL)
   {
    printf("Cannot open file: %s.\n",argv[3]);
    exit(1);
   }
 /* HANDLING THE PARTICULAR CASE OF ONLY ONE TRIANGLE */
 if (num_nodo==3)
 {
  fclose(trifile);
  temp_calc=x[0]*(y[1]-y[2])+x[1]*(y[2]-y[0])+x[2]*(y[0]-y[1]);
  u1=-(y[0]*(z[1]-z[2])+y[1]*(z[2]-z[0])+y[2]*(z[0]-z[1]))/temp_calc;
  v1=(x[0]*(z[1]-z[2])+x[1]*(z[2]-z[0])+x[2]*(z[0]-z[1]))/temp_calc;
  /* Derivatives of the plane where the triangle lies */
  fprintf(outfile,"%*.*f %*.*f %*.*f %*.*f %*.*f\n",
	  WIDTH,PREC,u1,WIDTH,PREC,v1,
	  WIDTH,PREC,0.,WIDTH,PREC,0.,WIDTH,PREC,0.);
  fprintf(outfile,"%*.*f %*.*f %*.*f %*.*f %*.*f\n",
	  WIDTH,PREC,u1,WIDTH,PREC,v1,
	  WIDTH,PREC,0.,WIDTH,PREC,0.,WIDTH,PREC,0.);
  fprintf(outfile,"%*.*f %*.*f %*.*f %*.*f %*.*f\n",
	  WIDTH,PREC,u1,WIDTH,PREC,v1,
	  WIDTH,PREC,0.,WIDTH,PREC,0.,WIDTH,PREC,0.);
  fclose(outfile);
  return; /* regular exit from main function */
 }
 /* Creation of dynamical structures needed by the program */
 A=malloc(sizeof(double)*4*num_nodo*num_nodo);
 if (A==NULL) {printf("Not enough memory\n"); exit(1); };
 gg=malloc(sizeof(double)*2*num_nodo);
 if (gg==NULL) {printf("Not enough memory\n"); exit(1); };
 v=malloc(sizeof(double)*2*num_nodo);
 if (v==NULL) {printf("Not enough memory\n"); exit(1); };
 DiagL=malloc(sizeof(double)*2*num_nodo);
 if (DiagL==NULL) {printf("Not enough memory\n"); exit(1); };
 /* data initialization */
  c=0.;
  for (i=0;i<2*num_nodo;i++)
   {
    for (j=0;j<=i;++j) A[i*2*num_nodo+j]=0.;
    gg[i]=0.;
   } 
 /* MAIN LOOP OF COMPUTATIONS OVER TRIANGLES */
 printf("Beginning computations for the data of each triangle.\n");
 fscanf(trifile,"%d ",&iv1);
 nt=1;
 /* Counter of the triangle being assembled */
 do
   {
    fscanf(trifile,"%d %d ",&iv2,&iv3);
    while(iv3!=-1)
     {
      if ((iv1<iv2)&&(iv1<iv3))
	{/* New triangle: beginning computations and assembling phase */
	 u1=x[iv1]; u2=x[iv2]; u3=x[iv3];
	 v1=y[iv1]; v2=y[iv2]; v3=y[iv3];
	 z1=z[iv1]; z2=z[iv2]; z3=z[iv3];
	 u12=(u1+u2)/2.; v12=(v1+v2)/2.;
	 u23=(u2+u3)/2.; v23=(v2+v3)/2.;
	 u31=(u3+u1)/2.; v31=(v3+v1)/2.;
	 /* selection: 0 uses the circumcenter, 1 uses the barycenter */
	 if (triangle_max_angle(u1,v1,u2,v2,u3,v3)>=75.)
	   pwsb_mode=1; else pwsb_mode=0;
	 if (pwsb_mode==0)  
           /* I do not need to check for the alignment  (err==1) */
	   circoscrivi(u1,v1,u2,v2,u3,v3,&u4,&v4,NULL);
	  else  /* pwsb_mode==1 */
	  {
	   u4=(u1+u2+u3)/3.; v4=(v1+v2+v3)/3.;
	   t1=-((u3-u2)*(u2-u4)+(v3-v2)*(v2-v4))/
	       (pow(u3-u2,2.)+pow(v3-v2,2.));
	   t2=-((u1-u3)*(u3-u4)+(v1-v3)*(v3-v4))/
	       (pow(u1-u3,2.)+pow(v1-v3,2.));
	   t3=-((u2-u1)*(u1-u4)+(v2-v1)*(v1-v4))/
	       (pow(u2-u1,2.)+pow(v2-v1,2.));
	   u41=(u31+u12)/2.; v41=(v31+v12)/2.;
	   u42=(u12+u23)/2.; v42=(v12+v23)/2.;
	   u43=(u23+u31)/2.; v43=(v23+v31)/2.;
	  }
	 if (pwsb_mode==0)
	 {
	  At1=fabs((u1*(v2-v4)+u2*(v4-v1)+u4*(v1-v2))/2.);
	  At2=fabs((u2*(v3-v4)+u3*(v4-v2)+u4*(v2-v3))/2.);
	  At3=fabs(-(u1*(v3-v4)+u3*(v4-v1)+u4*(v1-v3))/2.);
	  /* Really, there are six areas: 
             At1/2, At1/2, At2/2, At2/2, At3/2, At3/2 */
	 }
	 else At=fabs((u2*v3-u3*v2+u3*v1-u1*v3+u1*v2-u2*v1)/2.);
	 /* Computations and updating assembling phase */
         /* c constant */
	   for (ct1=0;ct1<6*(1+pwsb_mode);ct1++)
	      for (ct2=1;ct2<=3;ct2++)
		 cpt0[3*ct1+ct2-1]=build_calc_null(pwsb_mode,ct2,ct1);
	 temp_calc=
	  compute_for_vectconst(pwsb_mode,At,At1,At2,At3,cpt0);
         c=c+temp_calc;
         /* First phase for gg vector */
	 for (i=0;i<2*num_nodo;++i)
	 {
	  if(iv1!=i/2 && iv2!=i/2 && iv3!=i/2)
	  {
	   gg[i]+=temp_calc;
	  }
	 }
         /* Second phase for gg vector */
	 for (ip=0;ip<=5;++ip)
	 {
	  for (ct1=0;ct1<6*(1+pwsb_mode);ct1++)
	     for (ct2=1;ct2<=3;ct2++)
		cpt0[3*ct1+ct2-1]=
		  build_calc(pwsb_mode,ip,ct2,ct1);
	  temp_calc=
	   compute_for_vectconst(pwsb_mode,At,At1,At2,At3,cpt0);
          gg[global_index(ip,iv1,iv2,iv3)]+=temp_calc;  
	 } /* end for over ip */
         /* A matrix
         ** The computation is splitted in two parts in order to perform
         ** the first one only 6 times instead of 21 times as we should do
         ** using only the second part.
         */
	 for (ip=0;ip<=5;ip++)
	 {
	   for (ct1=0;ct1<6*(1+pwsb_mode);ct1++)
	      for (ct2=1;ct2<=3;ct2++)
		 cpt0[ip+6*(3*ct1+(ct2-1))]=
		    build_der(pwsb_mode,ip,ct2,ct1);
	 } /* end for over ip */
	 for (ip=0;ip<=5;++ip)
           {
            ip_global=global_index(ip,iv1,iv2,iv3);
	    for (jp=0;jp<=ip;++jp)
	     {
              jp_global=global_index(jp,iv1,iv2,iv3);
	      /* Determination */
	      /* (We use the area instead of its middle 
                  because there is a 2 factor) */
	      temp_calc=
	       compute_for_matrix(pwsb_mode,At,At1,At2,At3,ip,jp,cpt0);
	      if(ip_global>=jp_global)
	        A[ip_global*2*num_nodo+jp_global]+=temp_calc;
	       else
	        A[jp_global*2*num_nodo+ip_global]+=temp_calc;
	     } /* End for over jp */
            } /* End for over ip */
	 if (nt%25==0)
	   printf("Assembled %d triangles.\n",nt); nt++;
	} /* End of computations and of assembling phase for the current triangle */
      iv2=iv3;
      fscanf(trifile,"%d ",&iv3);
     }  /* End of the reading of the row in triangulation file */
    fscanf(trifile,"%s ",base_buffer);
    iv1=atol(base_buffer);
   }  /* End of reading triangulation file at the loop exit */
 while (base_buffer[0]!='B');
 fclose(trifile);
 /* transforming phi (computed in gg) to -g (computed in gg too) */
  for (i=0;i<2*num_nodo;++i) gg[i]=-gg[i]+A[i*2*num_nodo+i]/2.+c;
  printf("Solving linear system.\n");
  /* getting v as the solution of the linear system Av=gg (remember that -g=gg) */
  err=cholesky(A,v,gg,2*num_nodo,1,2*num_nodo,1,DiagL);
  if (err==1) 
  {
   printf("Singular matrix or badly conditioned one.\n");
   exit(1);
  }
  printf("Output copying.\n");
  for (i=0;i<num_nodo;++i)
  {
   fprintf(outfile,"%*.*f %*.*f %*.*f %*.*f %*.*f\n",
	   WIDTH,PREC,v[2*i],WIDTH,PREC,v[2*i+1],
	   WIDTH,PREC,0.,WIDTH,PREC,0.,WIDTH,PREC,0.);
  }
  fclose(outfile);
} /* end of function */
 

int global_index(i,iv1,iv2,iv3)
int i,iv1,iv2,iv3;
{
 switch(i/2)
 {
  /* return statement allows me to avoid break statements 
     in the various cases */  
  case 0: /* Relative to the first vertex of the triangle */
	  return (2*iv1+i%2);
  case 1: /* Relative to the second vertex of the triangle */
	  return (2*iv2+i%2);
  case 2: /* Relative to the third vertex of the triangle */
	  return (2*iv3+i%2);
  default: printf("The conversion of indexes is not possible.\n"); exit(1);
 }
}


