/*
** PWS_DER.C
**
** AUTHORS: 
**
**   Maria MORANDI CECCHI               Stefano DE MARCHI,            
**   University of Padua, Italy         University of Udine, Italy
**   email: mcecchi@math.unipd.it       email: demarchi@dimi.uniud.it
**   
**                       Damiano FASOLI
**                       Verona Software Srl, Italy 
**                       email: vrsoft@sis.it
**
**
**
** REVISION DATE: May, 1999  
**
** MODULES CALLED: PWS_D_A.C
**                 PWS_D_B.C
**                 PWSTOOLS.C
**                 LINSIST.C 
**
** --------------------------------------------------------------------
**
** BACKGROUND
**
** Let (xi,yi,qi) i=0,...,N-1 be N points in R^3. Let us consider their
** projections (xi,yi) onto the plane z=0 and let T be a triangulation
** of these points. A triangulation of them is a set of triangles having
** for vertices all these points (and only these ones) so that two
** distinct triangles intersect at most in a common edge or in a common
** vertex.
**
** The aim is to build the bivariate C1-function that interpolates
** at the points (xi,yi,qi).
**
** The Powell-Sabin (PS) interpolation method builds up such a function 
** using a Powell-Sabin polynomial patch on each triangle, 
** assuming that the first and second partial derivatives are known
** at each vertex of the triangulation T.
** The Powell-Sabin polynomial element is based on two different splittings
** of the triangle (the macro-triangle) so that we can have two types of  
** PS-elements: A-type and B-type. This latter if the macro-triangle has
** an angle greater than 75 degrees. 
**
** The Powell-Sabin polynomial element of A-type,
** splits a triangle into six sub-triangles 
** (using the circumcenter, the vertices of the macro-triangles and the
** middle edge points)
** over which it builds six quadratic polynomial patches.
**
** The Powell-Sabin polynomial element of B-type,
** splits a triangle into twelve sub-triangles
** (using the barycenter, the vertices of the macro-triangles,
** the middle edge points and the intersections between lines
** connecting them and lines connecting the barycenter and the 
** vertices of the macro-triangle) over which it builds twelve
** quadratic polynomial patches.
**
** The aim of this program is to compute such parameters (in the following
** we shall refer to them as to qix,qiy i=0,...,N-1, or more often
** as to the 2N-dimensional vector v).
** For every fixed 'i' in the range from 0 to N-1, this program computes
** these parameters in the "smoothest" possible way for a Powell-Sabin 
** interpolating function over the triangulation TLoc(i) made up of 
** all triangles having the i-th interpolation point, Vi, 
** for vertex (we called it 'the cell centered in Vi).
** Then it considers only the values qix,qiy where i is now fixed. 
**   
** To evaluate the "smoothness" of a Powell-Sabin
** interpolating function it is minimised the functional
**
** phi(vLoc(i))=   sum     [ Integral[ (Qxx^2 + 2*Qxy^2 + Qyy^2  ) dx dy ] ]     
**               t in TLoc(i)   t      
** 
** with respect to vLoc(i), that is the same of v when T becomes TLoc(i).
**
**
** METHODOLOGY 
**
** For every fixed i in the range 0,...,N-1,
**
**    phi(vLoc(i))= 1/2 * vLoc(i)' A vLoc(i) + g' vLoc(i) + c 
**
** is minimum when vLoc(i) solves the system
**
**    A vLoc(i) + g = 0.
**
** Let M be the number of nodes in the cell TLoc(i).
** The program computes 
**
**    c=phi(0), A, gj=phi(ej)-1/2*ajj-c, 
**
** where ej is the j-th vector of the canonical basis of R^(2M), 
** and solves the resulting linear system with the Cholesky factorisation. 
**
** Hence, it considers only the parameters relative to Vi, the center of TLoc(i).
**
** 
** REFERENCES
**
**  M. Morandi Cecchi, S. De Marchi, D. Fasoli,
**     "A Package for Representing C^1 Interpolating Surfaces: 
**      Application to the Lagoon of Venice's Bed."
**      Numer. Algorithms, 1999.
**
**  D. Fasoli,
**      Laurea's Thesis, University of Padua, 1995 (in Italian). 
**
**  R. J. Renka and A. K. Cline:
**     "A triangle-based C^1 interpolation method"
**      Rocky Mountain J. Math, 14(1):223-237, 1984.
**
**  P. Alfeld:
**    "Derivative Generation for multivariate 
**     scattered data functional minimization", 
**     Comp. Aided Graph. Design,2: 281-296, 1985.
**
**  G. Farin:
**     "Triangular Bernstein-Bezier patches",
**      Comput. Aided Geom. Design,3:83-127, 1986.
**
**
** --------------------------------------------------------------------
**
** SYNOPSIS:  pws_der file1 file2 [file3]
**
** file1 is the data file made up of the 3-dimensional points. 
** The coordinates of the points may be separated by any 
** combination of spaces, tabs and carriage returns. 
** We recommend to use a row for each point, for clearness. 
** 
** file2 is the triangulation data file in the format generated by
** the program 'do_trian'. The file has three sections. 
** The first, starting with a single capital letter `C' in the first row, 
** lists in subsequent blocks separated by -1 the index of
** a node and those of its neighbours in a counterclockwise
** order. 
** Note: this program uses only this first section and the heading of the
** second section, that is a single capital letter `B' in the row.
**
** file3 is the output derivative data file. 
** If file3 is omitted, the standard output is used instead.
** The values in a same row refer to the same vertex of the triangulation.
** The first column is the x-derivative, the second one is the y-derivative,
** the third one is 0, the fourth one is 0 and the last one is 0 too.
** The format of the output file is handled by the constants WIDTH and PREC
** (#DEFINE constants) that are the width of the output field, 
** and the digits of precision.
**        
** The program reads at most MAX_NUM_NODI (#DEFINE constant) input points.
** If more points are given, a warning message will be printed.
**
** The program uses a temporary swap file called 'blocks.tmp' to store
** the data during the computations. 
** One may safely remove it after the use. 
** 
** --------------------------------------------------------------------
**
** SUBROUTINES AND FUNCTIONS
**
** 
**
** int global_index(int N, int i, int j)
**
** This function requires as input parameters: 'N' the number of nodes of an
** open cell or of a closed one, 'j' the index of the triangle in the cell 
** and 'i' the local index of the unknown parameter among 
** q1x,q1y,q2x,q2y,q3x,q3y
** relative to the triangle (a value in the range from 0 to 5).
** This function computes the global index of this unknown parameter
** (a value in the range from 0 to 2N-1) relative to the cell.
**
** Locally, for each triangle, the parameters are sorted following
** the vertices of the triangle and the sequence  d/dx,d/dy.
**
** Globally, for the cell, the parameters are sorted following
** following the nodes of the cell, according to their index relative
** to the cell itself, and the previous sequence.
**
** Inputs
**       N: the number of nodes of an open or closed cell;
**       i: the local index, a value in the range from 0 to 5;
**       j: the index of the triangle in the cell.
**
**
** Output
**      global_index:  the global index corresponding to the local index 'i',
**                     a value in the range from 0 to 2N-1.
** ----------------------------------------------------------------------
*/

#include<stdio.h>
#include<stdlib.h>
#include<math.h>
#define MAX_NUM_NODI 100000
#define WIDTH 20   /* Format width of real numbers written to files */
#define PREC 10    /* Decimal precision width of real numbers written to files */

double derivecptpwsa_v(),compute_d_j_pwsa(),compute_d_j_pwsa_null(),
       derivecptpwsb_v(),compute_d_j_pwsb(),compute_d_j_pwsb_null(),
       build_der(),build_calc(),build_calc_null(),uu(),vv(),
       triangle_max_angle(),compute_for_matrix(),compute_for_vectconst();
void free_ind_node_list();
int global_index(),circoscrivi();

/* Declarations for global variables */
double z1,z2,z3;
double u1,v1,u2,v2,u3,v3,u4,v4,t1,t2,t3;
double u12,v12,u23,v23,u31,v31,u41,v41,u42,v42,u43,v43;

  /* The value 31 in the following declarations
  ** is not used by the algorithm */
  int IndTA[6][3][3]={
  {{2,1,0},{13,12,31},{18,31,31}},
  {{4,3,2},{14,13,31},{18,31,31}},
  {{6,5,4},{15,14,31},{18,31,31}},
  {{8,7,6},{16,15,31},{18,31,31}},
  {{10,9,8},{17,16,31},{18,31,31}},
  {{0,11,10},{12,17,31},{18,31,31}}
  };
  int IndTB[12][3][3]={
  {{2,13,24},{14,27,31},{30,31,31}},
  {{2,1,0},{13,12,31},{24,31,31}},
  {{25,15,2},{28,14,31},{30,31,31}},
  {{4,3,2},{16,15,31},{25,31,31}},
  {{6,17,25},{18,28,31},{30,31,31}},
  {{6,5,4},{17,16,31},{25,31,31}},
  {{26,19,6},{29,18,31},{30,31,31}},
  {{8,7,6},{20,19,31},{26,31,31}},
  {{10,21,26},{22,29,31},{30,31,31}},
  {{10,9,8},{21,20,31},{26,31,31}},
  {{24,23,10},{27,22,31},{30,31,31}},
  {{0,11,10},{12,23,31},{24,31,31}}
  };

struct ind_node_list {long int glob_index; struct ind_node_list *next;};

void free_ind_node_list(Ind_Node_List_Pt)
struct ind_node_list *Ind_Node_List_Pt;
{
 struct ind_node_list *tempPt;

 if (Ind_Node_List_Pt==NULL) return;
 while (Ind_Node_List_Pt->next!=NULL)
   {
    tempPt=Ind_Node_List_Pt->next;
    free(Ind_Node_List_Pt);
    Ind_Node_List_Pt=tempPt;
   }
 free(Ind_Node_List_Pt);
}

void main(argc,argv)
int argc;
char *argv[];
{
  struct triangle {long int v1; long int v2; long int v3;};

  size_t size_double;
  FILE *infile, *trifile, *outfile, *bfile;
  long int num_nodo,cont_nodi,temp_index,iv1,iv2,iv3,
	   triangle_number,triangle_counter,nt;
  double x[MAX_NUM_NODI],y[MAX_NUM_NODI],z[MAX_NUM_NODI];
  struct triangle triangle_order[2*MAX_NUM_NODI-5];
  double At,At1,At2,At3;
  int i,j,ip,jp,N_C; /* Counter for the number of nodes in a cell */
  int ip_global,jp_global;
  int ct1,ct2,pwsb_mode; /* 0 A-type  1 B-type */
  int err,open_cella_flag; /* flag for open or closed cell */
  struct ind_node_list *lista_vicini, *vicino_pt, *vicino_pt_temp;
  double cpt0[216],temp_calc;
  /* 216 comes from 6*3*12, that is from
     (unknown derivatives) * (xx xy yy contributes) * (triangles) */
  char base_buffer[20];
  int stop; /* For loops */
  int shift_right;
  /* Number of counterclockwise rotations needed to transform a
  ** triangle in a cell into its canonical form where iv1<iv2, iv1<iv3 */
  double *A, *v, *gg, *DiagL, c;

 infile=NULL; trifile=NULL; outfile=NULL; bfile=NULL;
 A=NULL; v=NULL; gg=NULL; DiagL=NULL;
 lista_vicini=NULL;
 size_double=sizeof(double);
 if (argc==1)
   {
    printf("\nStart the program with three parameters:\n");
    printf("First parameter: data file name.\n");
    printf("Second parameter: triangulation file name.\n");
    printf("Third parameter: derivative file name.\n");
    printf("                 (Omit it for screen output.)\n");
    exit(1);
   }
 /* Reading */
 infile=fopen(argv[1],"r");
 if (infile==NULL)
   {
    printf("Cannot open file: %s.\n",argv[1]);
    exit(1);
   }
 if (argc==2)
   {
    printf("Missing triangulation file.\n");
    exit(1);
   }
 /* data reading */
 printf("Wait for data reading\n"); 
 num_nodo=0;
 while (!feof(infile))
 {
  if (num_nodo<MAX_NUM_NODI)
   {
    fscanf(infile," %lf %lf %lf ",&x[num_nodo],&y[num_nodo],&z[num_nodo]);
      /* spaces are essential part of the input format string */
    num_nodo=num_nodo+1;
   }
   else
   {
    printf("Too many input data.\n");
    exit(1);
   }
 }
 if (num_nodo<3)
    {
     printf("I need at least three input points.\n");
     exit(1);
    }
 fclose(infile);
 /* end of data reading */
 /* opening triangulation file */
 trifile=fopen(argv[2],"r");
 if (trifile==NULL)
   {
    printf("Cannot open file: %s.\n",argv[2]);
    exit(1);
   }
 if (fgetc(trifile)!='C')
   {
    printf("File %s in a no correct format.\n",argv[2]);
    exit(1);
   }
 else fgetc(trifile); /* carriage return */
 /* opening output file */
 if (argc==3) outfile=stdout; else outfile=fopen(argv[3],"w");
 if (outfile==NULL)
   {
    printf("Cannot open file: %s.\n",argv[3]);
    exit(1);
   }
 /* HANDLING THE PARTICULAR CASE OF ONLY ONE TRIANGLE */
 if (num_nodo==3)
 {
  fclose(trifile);
  temp_calc=x[0]*(y[1]-y[2])+x[1]*(y[2]-y[0])+x[2]*(y[0]-y[1]);
  u1=-(y[0]*(z[1]-z[2])+y[1]*(z[2]-z[0])+y[2]*(z[0]-z[1]))/temp_calc;
  v1=(x[0]*(z[1]-z[2])+x[1]*(z[2]-z[0])+x[2]*(z[0]-z[1]))/temp_calc;
  /* Derivatives of the plane where the triangle lies */
  fprintf(outfile,"%*.*f %*.*f %*.*f %*.*f %*.*f\n",
	  WIDTH,PREC,u1,WIDTH,PREC,v1,
	  WIDTH,PREC,0.,WIDTH,PREC,0.,WIDTH,PREC,0.);
  fprintf(outfile,"%*.*f %*.*f %*.*f %*.*f %*.*f\n",
	  WIDTH,PREC,u1,WIDTH,PREC,v1,
	  WIDTH,PREC,0.,WIDTH,PREC,0.,WIDTH,PREC,0.);
  fprintf(outfile,"%*.*f %*.*f %*.*f %*.*f %*.*f\n",
	  WIDTH,PREC,u1,WIDTH,PREC,v1,
	  WIDTH,PREC,0.,WIDTH,PREC,0.,WIDTH,PREC,0.);
  fclose(outfile);
  return; /* regular exit from main function */
 }
 /* OPENING TEMPORARY SWAP FILE */
 bfile=fopen("blocks.tmp","wb");
 if (bfile==NULL) 
   {
    printf("Cannot open temporary file for writing.\n");
    exit(1);
   }
 /* MAIN LOOP OF COMPUTATIONS OVER TRIANGLES AND THEIR STORAGE */
 printf("Beginning computations for the data of each triangle.\n");
 fscanf(trifile,"%ld ",&iv1);
 triangle_number=0;
 /* Counter for the current triangle. 
    At the end it shall become the number of triangles */
 do
   {
    fscanf(trifile,"%ld %ld ",&iv2,&iv3);
    while(iv3!=-1)
     {
      if ((iv1<iv2)&&(iv1<iv3))
	{/* New triangle: beginning computations and assembling phase */
	 u1=x[iv1]; u2=x[iv2]; u3=x[iv3];
	 v1=y[iv1]; v2=y[iv2]; v3=y[iv3];
	 z1=z[iv1]; z2=z[iv2]; z3=z[iv3];
	 u12=(u1+u2)/2.; v12=(v1+v2)/2.;
	 u23=(u2+u3)/2.; v23=(v2+v3)/2.;
	 u31=(u3+u1)/2.; v31=(v3+v1)/2.;
	 /* selection: 0 uses circumcenter, 1 uses barycenter */
	 if (triangle_max_angle(u1,v1,u2,v2,u3,v3)>=75.)
	   pwsb_mode=1; else pwsb_mode=0;
	 if (pwsb_mode==0)  
           /* I do not need to check for the alignment (err==1)*/
	   circoscrivi(u1,v1,u2,v2,u3,v3,&u4,&v4,NULL);
	  else  /* pwsb_mode==1 */
	  {
	   u4=(u1+u2+u3)/3.; v4=(v1+v2+v3)/3.;
	   t1=-((u3-u2)*(u2-u4)+(v3-v2)*(v2-v4))/
	       (pow(u3-u2,2.)+pow(v3-v2,2.));
	   t2=-((u1-u3)*(u3-u4)+(v1-v3)*(v3-v4))/
	       (pow(u1-u3,2.)+pow(v1-v3,2.));
	   t3=-((u2-u1)*(u1-u4)+(v2-v1)*(v1-v4))/
	       (pow(u2-u1,2.)+pow(v2-v1,2.));
	   u41=(u31+u12)/2.; v41=(v31+v12)/2.;
	   u42=(u12+u23)/2.; v42=(v12+v23)/2.;
	   u43=(u23+u31)/2.; v43=(v23+v31)/2.;
	  }
	 if (pwsb_mode==0)
	 {
	  At1=fabs((u1*(v2-v4)+u2*(v4-v1)+u4*(v1-v2))/2.);
	  At2=fabs((u2*(v3-v4)+u3*(v4-v2)+u4*(v2-v3))/2.);
	  At3=fabs(-(u1*(v3-v4)+u3*(v4-v1)+u4*(v1-v3))/2.);
	  /* Really there are six areas: At1/2 At1/2 At2/2 At2/2 At3/2 At3/2 */
	 }
	 else At=fabs((u2*v3-u3*v2+u3*v1-u1*v3+u1*v2-u2*v1)/2.);
	 /* Computations and updating storage */
	 /* A matrix 
	 ** The computation is splitted in two parts in order to perform
         ** the first one only 6 times instead of 21 times as we should do
         ** using only the second part.
         */
	 for (ip=0;ip<=5;ip++)
	 {
	   for (ct1=0;ct1<6*(1+pwsb_mode);ct1++)
	      for (ct2=1;ct2<=3;ct2++)
		 cpt0[ip+6*(3*ct1+(ct2-1))]=
		    build_der(pwsb_mode,ip,ct2,ct1);
	 } /* end for over ip */
	 for (ip=0;ip<=5;++ip)
	    for (jp=0;jp<=ip;++jp)
	     {
	      /* Determination */
	      /* (We use the area instead of its middle 
                  because there is a 2 factor) */ 
	      temp_calc=
	       compute_for_matrix(pwsb_mode,At,At1,At2,At3,ip,jp,cpt0);
	      fwrite((char *)&temp_calc,size_double,1,bfile);
	     }
	 /* gg vector */
	 for (ip=0;ip<=5;++ip)
	 {
	  for (ct1=0;ct1<6*(1+pwsb_mode);ct1++)
	     for (ct2=1;ct2<=3;ct2++)
		cpt0[3*ct1+ct2-1]=
		  build_calc(pwsb_mode,ip,ct2,ct1);
	  temp_calc=
	   compute_for_vectconst(pwsb_mode,At,At1,At2,At3,cpt0);
	  fwrite((char *)&temp_calc,size_double,1,bfile);
	 } /* end for over ip */
	 /* c constant */
	   for (ct1=0;ct1<6*(1+pwsb_mode);ct1++)
	      for (ct2=1;ct2<=3;ct2++)
		 cpt0[3*ct1+ct2-1]=build_calc_null(pwsb_mode,ct2,ct1);
	 temp_calc=
	  compute_for_vectconst(pwsb_mode,At,At1,At2,At3,cpt0);
	 fwrite((char *)&temp_calc,size_double,1,bfile);
	 triangle_order[triangle_number].v1=iv1;
	 triangle_order[triangle_number].v2=iv2;
	 triangle_order[triangle_number].v3=iv3;
	 if (triangle_number%25==24)
	   printf("Computed data for %ld triangles.\n",
		  triangle_number+1); triangle_number++;
	} /* End of data computations for current triangle */
      iv2=iv3;
      fscanf(trifile,"%ld ",&iv3);
     }  /* End of the reading of the row in the triangulation file */
    fscanf(trifile,"%s ",base_buffer);
    iv1=atol(base_buffer);
   }  /* End of reading triangulation file at the loop exit */
 while (base_buffer[0]!='B');
 /* There are triangle_number trianlges labeled from 0 to triangle_number-1 */
 fclose(bfile);
 fseek(trifile,2L,SEEK_SET); /* Pointing to the first node of the first cell */
 printf("Beginning computations for each node.\n");
 /* MAIN LOOP OVER CELLS */
 bfile=fopen("blocks.tmp","rb");
 if (bfile==NULL)
 {
  printf("Cannot open temporary file for writing.\n");
  exit(1);
 }
 for (cont_nodi=0;cont_nodi<num_nodo;cont_nodi++)
 {
  N_C=0; iv1=cont_nodi;
  open_cella_flag=1;  /* I suppose the cell to be an open one */
  /* Skipping base node */
  fscanf(trifile," %ld ",&temp_index); N_C++;
  free_ind_node_list(lista_vicini);
  /* Creating first node */
  fscanf(trifile," %ld ",&temp_index); N_C++; /*there is at least 1 neigh. node*/ 
  lista_vicini=malloc(sizeof(struct ind_node_list));
  lista_vicini->glob_index=temp_index;
  vicino_pt=lista_vicini;
  /* I shall create new locations pointed by pt->next */
  /* preparing the loop for the reading of other neighbouring nodes */
  fscanf(trifile," %ld ",&temp_index); /* N_C increases later */
  while (temp_index!=-1)
  {
   N_C++;
   vicino_pt->next=malloc(sizeof(struct ind_node_list));
   vicino_pt->next->glob_index=temp_index;
   vicino_pt->next->next=NULL;
   vicino_pt=vicino_pt->next;
   fscanf(trifile," %ld ",&temp_index);
  }  /* End of while relative to the reading of neighbouring nodes */
  if (lista_vicini->glob_index==vicino_pt->glob_index)
    {
     open_cella_flag=0; /* The cell is a closed one */
     N_C--; 
     /* N_C does not consider the repetition of the first neighbouring node */
    }
  /* If the cell is a closed one, the reading of the first neighbouring node
  ** has been done in order to optimize later the computation of
  ** global indexes */
  /* Initialization of dynamical memory */
  A=malloc(size_double*4*N_C*N_C);
  if (A==NULL) {printf("Not enough memory\n"); exit(1); };
  gg=malloc(size_double*2*N_C);
  if (gg==NULL) {printf("Not enough memory\n"); exit(1); };
  v=malloc(size_double*2*N_C);
  if (v==NULL) {printf("Not enough memory\n"); exit(1); };
  DiagL=malloc(size_double*2*N_C);
  if (DiagL==NULL) {printf("Not enough memory\n"); exit(1); };
  /* Data initialization */
  c=0.;
  for (i=0;i<2*N_C;i++)
   {
    for (j=0;j<=i;++j) A[i*2*N_C+j]=0.;
    gg[i]=0.;
   }
  /* Beginning computations for the triangles of every cell */
  vicino_pt=lista_vicini;
  /* initializing iv2 to optimize the for loop with a false iv3 value */
  iv3=vicino_pt->glob_index;
  vicino_pt=vicino_pt->next;
  /* N_C is the number of elements of the cell, open or closed one */
  for (nt=1;nt<=N_C-1-open_cella_flag;nt++)
  {
   iv2=iv3;
   iv3=vicino_pt->glob_index;
   /* Now if nt=N_C-1 and the cell is a closed one iv3 is the correct value
   ** by means of the repetition done when reading the cell indexes */
   /* Transforming the triangle into its canonical 
      form where iv1<iv2, iv2<iv3 */
   shift_right=0;
   if (iv1>iv2 || iv1>iv3)
   {
    temp_index=iv3; iv3=iv2; iv2=iv1; iv1=temp_index;
    shift_right++;
   }
   /* Completing the test */
   if (iv1>iv2 || iv1>iv3)
   {
    temp_index=iv3; iv3=iv2; iv2=iv1; iv1=temp_index;
    shift_right++;
   }
   /* Searching for the index of the current triangle */
   triangle_counter=0;
   stop=0;
   do
   {
   if (triangle_order[triangle_counter].v1!=iv1) triangle_counter++;
    else
     if (triangle_order[triangle_counter].v2!=iv2) triangle_counter++;
      else
       if (triangle_order[triangle_counter].v3!=iv3) triangle_counter++;
	else stop=1;
   }
   while (stop==0 && triangle_counter<triangle_number);
   if (triangle_counter==triangle_number)
    {
     printf("Cannot find the triangle in the temporary swap file.\n");
     exit(1);
    }
   /* triangle_counter is the index of the current triangle 
      into triangle_order */
   /* Transforming again the triangle into its original form without
      changing the value of shift_right */ 
   if (shift_right==1) /* shift_left */
   {
    temp_index=iv1; iv1=iv2; iv2=iv3; iv3=temp_index;
   }
   else
   if (shift_right==2) /* new shift_right (-2=1 mod 3) */
   {
    temp_index=iv3; iv3=iv2; iv2=iv1; iv1=temp_index;
   }
   /* seeking the pointer in the temporary swap file */
   fseek(bfile,size_double*triangle_counter*28L,SEEK_SET);
   /* 
   ** There are 28 stored values per triangle, 21 ones
   ** for A matrix (6*7/2), 6 ones for vector gg and only one
   ** for the constant c.
   */
   /* Building linear system */
   /* Computation of A */
   for (ip=0;ip<=5;++ip)
   {
    ip_global=global_index(N_C,(ip+4*shift_right)%6,nt);
    for (jp=0;jp<=ip;++jp)
    {
     jp_global=global_index(N_C,(jp+4*shift_right)%6,nt);
     fread((char *)&temp_calc,size_double,1,bfile);
     if (jp_global<=ip_global)
       A[ip_global*2*N_C+jp_global]+=temp_calc;
      else
       A[jp_global*2*N_C+ip_global]+=temp_calc;
    }
   }
   /* First phase for gg */
   for (ip=0;ip<=5;++ip)
   {
    ip_global=global_index(N_C,(ip+4*shift_right)%6,nt);
    fread((char *)&temp_calc,size_double,1,bfile);
    gg[ip_global]+=temp_calc;
   }
   /* Computation of c */
   fread((char *)&temp_calc,size_double,1,bfile);
   c+=temp_calc;
   /* Second phase for gg */
   vicino_pt_temp=lista_vicini;
   for (i=1;i<N_C;++i)
   {
    if(vicino_pt_temp->glob_index!=iv2 && vicino_pt_temp->glob_index!=iv3)
    /* avoiding the test over iv1 since it has the same value as cont_nodi */ 
     {
      gg[2*i]+=temp_calc;
      gg[2*i+1]+=temp_calc;
     }
     vicino_pt_temp=vicino_pt_temp->next;
     /* The structure agrees with the loop by means of the previous reading */
   }
   vicino_pt=vicino_pt->next;
  }
  /* transforming phi (computed in gg) to -g (computed in gg too) */
  for (i=0;i<2*N_C;++i) gg[i]=-gg[i]+A[i*2*N_C+i]/2.+c;
  /* getting v as the solution of the linear system Av=gg 
     (remember that -g=gg) */
  err=cholesky(A,v,gg,2*N_C,1,2*N_C,1,DiagL);
  if (err==1) 
  {
   printf("Singular matrix or badly conditioned one for the node %ld.\n",
	   cont_nodi);
   printf("Given a false solution made up of all zeros \n");
   for (i=0;i<2*N_C;++i) v[i]=0.;
  }
  fprintf(outfile,"%*.*f %*.*f %*.*f %*.*f %*.*f\n",
	  WIDTH,PREC,v[0],WIDTH,PREC,v[1],
	  WIDTH,PREC,0.,WIDTH,PREC,0.,WIDTH,PREC,0.);
  free(A); free(gg); free(v); free(DiagL);
  if (cont_nodi%25==24)
    printf("Completed %ld nodes\n",(cont_nodi+1));
 } /* end of main loop */
 fclose(bfile);
 fclose(trifile);
 fclose(outfile);
}

int global_index(N,i,j)
int N,i,j;
{
 switch(i/2)
 {
  /* return statement allows me to avoid break statements 
     in the various cases */
  case 0: /* Relative to the first vertex of the triangle */
	  return (i%2);
  case 1: /* Relative to the second vertex of the triangle */
	  return(2*j+i%2);
  case 2: /* Relative to the third vertex of the triangle */
	  return(2*(j%(N-1)+1)+i%2);
 }
}
