/*
** LINSIST.C
**
** AUTHORS: 
**
**   Maria MORANDI CECCHI               Stefano DE MARCHI,            
**   University of Padua, Italy         University of Udine, Italy
**   email: mcecchi@math.unipd.it       email: demarchi@dimi.uniud.it
**   
**                       Damiano FASOLI
**                       Verona Software Srl, Italy 
**                       email: vrsoft@sis.it
**
**
**
** REVISION DATE: May, 1999  
**
** MODULES CALLED: None 
**
** ------------------------------------------------------------------------
**
** SUBROUTINES AND FUNCTIONS DESCRIPTION:
**
** >>>>>>>>>>>>>>>>>>>
**
** int gsspvt (double* puntA, double* puntx, double* puntb,
**             int matclm,int genvetclm,int n,int s,int inc[])
** 
** This function solves the linear system Ax=b using the Gauss elimination 
** method together with a total pivot strategy.
** Really it solves s linear systems since A is
** a n*n matrix, the vector of unknowns 'x' is a n*s matrix and the vector
** of known values 'b' is a n*s matrix too.
** Rows and columns are numbered from 0 to n-1 or from 0 to s-1.
** The macros a(i,j), b(i,j) and x(i,j) allow this function to handle
** the elements placed in the i-th row and in the j-th column of the
** corresponding matrices A, b and x.
**
** Inputs: 
**         the pointers to the matrices
**           'A', 'x' and 'b' (that are puntA, puntx and puntb); 
**         the vector 'inc' used to
**           swap rows and columns with n or more elements;
**         'matclm' is the exact number of columns used by the
**           matrix A since it is necessary to know where a new row starts
**           (matrices are visited by rows in C language)
**         'genvetclm' is the exact number of columns used by
**           the matrices 'x' and 'b' since it is necessary to know 
**           where a new row starts (matrices are visited by rows in C language)
**
** 
** Function result:
**         if 'gsspvt==0', x is the solution;
**         if 'gsspvt==1' the matrix A is singular one or ill-conditioned
**              (the magnitude of the pivot element is smaller than eps);
**         if 'gsspvt==2', n<1 or s<1 or n>matclm or s>genvetclm. 
** 
** ATTENTION: The initial values of the matrices A and b are lost.
**
** >>>>>>>>>>>>>>>>>>>
**
** int cycletridiag (int n,double alfa[],double beta[],double gamma[],
**                   double* puntx,double* puntb,int s,int genvetclm)
** 
** This function solves a tridiagonal circulant system of dimension n*n with
** matrix having the main diagonal given by the vector 'beta', 
** the diagonal above the main the vector 'gamma'
** (in the subrange 0...n-2 ) and the diagonal below the
** main the vector 'alfa' (in the subrange 1...n-1). 
** It is assumed that:
**    gamma(n-1) is the element placed on the last row and in the first column,
**    alfa(0) is the element placed on the first row and in the last column.
**
** In order to have a real advantage we need n>=4 (when n==3 we use the 
** gaussian elimination with total pivoting). However we need
** n>2 and a matrix with a diagonally dominant or simply with values
** not too small.
**
** The macros b(i,j) and x(i,j) allow this function to handle
** the elements placed in the i-th row and in the j-th column of the
** corresponding matrices b and x.
**
** Inputs:
**        'alpha','beta','gamma': the diagonals;
**        'puntx' and 'puntb' refer to matrices with 's' columns used over
**         'genvetclm' 'genvetclm' the exact number of columns used.
**

** Function result:
**        'cycletridiag==0', all is OK;
**        'cycletridiag==1' the matrix is ill-conditioned or it has a 
**                          singular value;
**        'cycletridiag==2', n<3.
**
** PAY ATTENTION: The initial values of alfa,beta,gamma and b 
**                (pointed by puntb) are lost.
**
** >>>>>>>>>>>>>>>>>>>
**
** int cholesky (double* puntA,double* puntx,double* puntb,
**               int matclm,int genvetclm,int n,int s,double DiagL[])
**
** This function uses the Cholesky method to solve the linear system
** Ax=b where A is a n*n matrix (pointed by 'puntA'), the vector of 
** unknowns 'x' and the vector of known terms, 'b', are n*s 
** matrices (pointed by 'puntx' and 'puntb').
** Note: The index of row and column starts from zero.
**
** The matrix A must be symmetric positive definite.
**
** The macros b(i,j) and x(i,j) allow this function to handle
** the elements placed in the i-th row and in the j-th column of the
** corresponding matrices b and x.
**
** 'Matclm' and 'genvetclm' are the exact numbers of columns of the matrices
** in A,b and x. 
**
** 'DiagL' is a pointer to a double precision vector with at least n elements
** used for internal computations. 
**
** Function result:
**       'Cholesky==0', all is OK;
**       'Cholesky==1' there are problems 
**              (usually A is not a positive definite matrix).
**
**  ATTENTION: The submatrix of A built by the terms above the main diagonal 
**             is lost as well as the values of the matrix b.
** -------------------------------------------------------------------------
*/
#include <math.h>
#define EPS_GAUSS 1.e-16
#define EPS_CHOLESKY 1.e-16

#define a(i,j) *(puntA+(i)*matclm+(j))      
/* macro used by gsspvt */

#define x(i,j) *(puntx+(i)*genvetclm+(j))   
/* macro used by gsspvt,cycletridiag*/

#define b(i,j) *(puntb+(i)*genvetclm+(j))   
/* macro used by gsspvt,cycletridiag*/

int gsspvt (puntA,puntx,puntb,matclm,genvetclm,n,s,inc)
  int n,s,matclm,genvetclm;
  double *puntA;
  double *puntb, *puntx;
  int inc[];
{
      double eps=EPS_GAUSS;

      int i,j,r;
      int k,pivotrow,pivotclm;
      double temp,modpivot;

/* ** initialization ** */
      if ((n < 1) || (s < 1) || (n > matclm) || (s > genvetclm))
       return 2;
      k=0;
      for (i=0;i<n;++i) inc[i]=i;
/* ** Beginning of elimination ** */
Start:
      modpivot=0.;
      pivotrow=-1;
      pivotclm=-1;
/* no pivot */
      for (i=k;i<n;++i)
	for (j=k;j<n;++j)
	{
	 if ( fabs( a(i,j) ) > modpivot)
	 {
	  modpivot=fabs(a(i,j));
	  pivotrow=i;
	  pivotclm=j;
	 }
	}
	if (modpivot < eps)
	 return 1;
	/* The magnitude of the pivot element is smaller than eps and 
           this states that the matrix is badly conditioned or that it is
           a singular one. */

/* ** Swaps ** */
      if ( (pivotrow==-1) && (pivotclm==-1) )
       return 1;
      if (pivotrow != k)
      {
       for (j=0;j<n;++j)
       {
	temp=a(k,j);
	a(k,j)=a(pivotrow,j);
	a(pivotrow,j)=temp;
       }
       for(r=0;r<s;++r)
       {
	temp=b(k,r);
	b(k,r)=b(pivotrow,r);
	b(pivotrow,r)=temp;
       }
      }
      if (pivotclm != k)
      {
       for(i=0;i<n;++i)
       {
	temp=a(i,k);
	a(i,k)=a(i,pivotclm);
	a(i,pivotclm)=temp;
       }
       temp=inc[k];
       inc[k]=inc[pivotclm];
       inc[pivotclm]=temp;
      }
/* ** elimination ** */

      for(i=k+1;i<n;++i)
      {
       temp= a(i,k)/a(k,k);
       for(j=k+1;j<n;++j)
       {
	a(i,j)=a(i,j)-temp*a(k,j);
       }
       for(r=0;r<s;++r)
       {
	b(i,r)=b(i,r)-temp*b(k,r);
       }
      }
    /*  When k=n-2 I should increase k only in order to solve the block
        of the element placed in the position (n-1),(n-1); however
        in this case in the else statement of the following test I check 
        for the singular case  
    */
	if (k < (n-2))
	{
	  k=k+1;
	  goto Start;
	}
	else
	{
	if (fabs( a(n-1,n-1) ) < eps)
	return 1;
	}

/* ** System solution ** */
      for(r=0;r<s;++r)
      {
       for(i=n-1;i>=0;--i)
       {
	temp=0.;
	for(j=i+1;j<n;++j)
	{
	 temp=temp+a(i,j)*x(inc[j],r);
	}
	x(inc[i],r)=(b(i,r)-temp)/a(i,i);
       }
      }
      return 0;
}

int cycletridiag (n,alfa,beta,gamma,puntx,puntb,s,genvetclm)
  int n,s,genvetclm;
  double alfa[],beta[],gamma[];
  double *puntb, *puntx;
{
 double temp,Matrice[3][3];
 int i,j,inc[3],err;
 double eps=EPS_GAUSS;

 if (n<3) return 2;
 if (n==3) goto Gausslabel;
 /* effettivo utilizzo */
 for (i=0;i<n;++i) if (fabs(beta[i])<eps) return 1;
 for (i=1;i<=n-3;++i)
 {
  temp=alfa[i]/beta[i-1];
  beta[i]=beta[i]-temp*gamma[i-1];
  for (j=0;j<s;++j) b(i,j)=b(i,j)-temp*b(i-1,j);
  alfa[i]=-temp*alfa[i-1];  
  /* I consider the last column as the current alfa */
  temp=gamma[n-1]/beta[i-1];
  beta[n-1]=beta[n-1]-temp*alfa[i-1];
  for (j=0;j<s;++j) b(n-1,j)=b(n-1,j)-temp*b(i-1,j);
  gamma[n-1]=-temp*gamma[i-1];
  /* Now the current gamma(n-1) is in the i-th column beginning from 0 */
 }
 Gausslabel:
 Matrice[0][0]=beta[n-3];  Matrice[0][1]=gamma[n-3]; Matrice[0][2]=alfa[n-3];
 Matrice[1][0]=alfa[n-2];  Matrice[1][1]=beta [n-2]; Matrice[1][2]=gamma[n-2];
 Matrice[2][0]=gamma[n-1]; Matrice[2][1]=alfa[n-1];  Matrice[2][2]=beta[n-1];
 err=gsspvt(Matrice,&x(n-3,0),&b(n-3,0),3,genvetclm,3,s,inc);
 if (err==1) return 1;
 for (i=n-4;i>=0;--i)
  for (j=0;j<s;++j)
   x(i,j)=( b(i,j)-alfa[i]*x(n-1,j)-gamma[i]*x(i+1,j) )/beta[i];
return 0;
}

int cholesky (puntA,puntx,puntb,matclm,genvetclm,n,s,DiagL)
int n,s,matclm,genvetclm;
double *puntA;
double *puntb,*puntx;
double DiagL[];

{
int i,j,k,r;
double temp;

/*
** I use A=L*L' where L' is the transposed matrix of A. L' is stored 
** in the upper part of A (not used) apart from the diagonal. 
*/
/* Cholesky factorisation */
for (i=0;i<n;++i)
  {
   for (j=0;j<i;++j)
      {
       temp=0.;
       for (k=0;k<j;++k) temp+=a(k,i)*a(k,j);
       a(j,i)=1./DiagL[j]*(a(i,j)-temp);
      }
    temp=0.;
    for (k=0;k<i;++k) temp+=pow(a(k,i),2.);
    if ((DiagL[i]=a(i,i)-temp)<EPS_CHOLESKY) return 1;
    DiagL[i]=sqrt(DiagL[i]);
  }
/* Solution of the system Ly=b writing the solution in x from b */
for (r=0;r<s;++r)
 for (i=0;i<n;++i)
  {
   temp=0.;
   for (k=0;k<i;++k) temp+=a(k,i)*x(k,r);
   x(i,r)=1./DiagL[i]*(b(i,r)-temp);
  }
/* Solution of the system L'x=y writing the solution in x from x=y */
for (r=0;r<s;++r)
 for (i=n-1;i>=0;--i)
  {
   temp=0.;
   for (k=i+1;k<n;++k) temp+=a(i,k)*x(k,r);
   x(i,r)=1./DiagL[i]*(x(i,r)-temp);
  }
return 0;
}


