/*
** LABSUP3D.C
**
** AUTHORS: 
**
**   Maria MORANDI CECCHI               Stefano DE MARCHI,            
**   University of Padua, Italy         University of Udine, Italy
**   email: mcecchi@math.unipd.it       email: demarchi@dimi.uniud.it
**   
**                       Damiano FASOLI
**                       Verona Software Srl, Italy 
**                       email: vrsoft@sis.it
**
**
**
** REVISION DATE: May, 1999  
**
** MODULES CALLED: LABSUP3D.H
**                 SUPERF.C
**                 GRAPHSUP.C
**                 MATRIXIO.C
**                 LINSIST.C
**
**
** ------------------------------------------------------------------------
**
**
**    LABSUP3D is the acronym that stands for 
**
**    ``LAboratory for Bivariate Surfaces and Patches 3D''.
** 
**                                                                       
**   LABSUP3D must be compiled in a C-development environment         
**   using 'OpenGL' graphics libraries or 'Mesa' ones (with          
**   'GLUT' libraries.)                                               
**   It can be executed in any environment provided with the run-time 
**   subsystems for these libraries. See Makefile for details on the  
**   matter.                                                          
**   The program has been tested with a                               
**   X11-environment supporting GLX extensions using both OpenGL and  
**   Mesa graphics libraries.                                         
**                                                                    
**   -- OpenGL and Silicon Graphics are registered trademarks         
**      of Silicon Graphics, Inc.                                      
**   -- MESA is covered by a GNU LIBRARY GENERAL PUBLIC LICENSE.
**
**  NOTICE: for more details on its compilation see the Makefile and 
**          the file README.1ST.
** ------------------------------------------------------------------------
**
** The aim of this program is the construction of C1-surfaces interpolating
** given data points with their first or second order derivatives 
** over a given triangulation, using piecewise polynomial Bezier patches. 
** Really they can be also Q18, Clough-Tocher or Powell-Sabin polynomial 
** elements to guarantee C1-continuity using the corresponding 
** interpolation methods.
**
** The program allows:
** 
** 1) to see and zoom the underlying triangulation;
**
** 2) to see the C1-surface drawn in the graphic window with a specified 
**    point of view;
**
** 3) to rotate the surface varying the viewing direction in the graphic 
**    window using mouse events;
**
** 4) to zoom parts of the surface displayed in the graphic window 
**    ("selected zoom") or to perform a "central zoom" with a specified user
**    ratio towards the surface or in the opposite direction;
**
** 5) to change colors for the background of the graphic window, for lines and 
**    for the surface itself and to switch 'on' or 'off' hidden surface removal
**    by means of Z-buffer (depth-buffer);
**
** 6) to change at any time the "detail" of the surface drawn computing 
**    more or less points and the interpolation method used;
**
** 7) to view at any time (or to write to an output file specified by the user) 
**    the interpolation points, with or without their derivatives,
**    and the net of Bezier control points for the surface;
**    it is also possible to multiply all these
**    values by a constant and to see graphic effects immediately;
**
** 8) to define a subtriangulation by means of a clipping "square" in the
**    triangulation window in order to work only with the corresponding
**    sub-surface instead of working with the whole surface;
**
** 9) to compute a n-degree parametric Bezier patch interpolating (n+1)(n+2)/2
**    points with their corresponding parameters supplied directly by the user
**    or in a file.
**
** Input data points for the parametric Bezier patch may be specified  
** directly by the user or given in a file. 
** If they are given directly  by the user the program first asks for their
** parameters, which may be automatically chosen following a regular
** subdivision of the domain (which is a triangle) into lines where
** barycentric coordinates are constant, and then asks for the points. 
** Otherwise, if they are given in a file, it is necessary to use
** one row per point, following this format: 
** 
**    u-parameter, v-parameter, w-parameter, x-point, y-point, z-point.
**
** Input data points for C1-surface must be given in a file name.
** The three coordinates of the points may be separated by any 
** combination of spaces, tabs and carriage returns, 
** even though it is recommended to use a row for each point.
**
** Triangulation data file must be in the format generated by
** the program 'do_trian'. 
**
** Derivative data values must be given in a file name.
** The values in a same row refer to the same vertex of the triangulation.
** The first column is the x-derivative, the second the y-derivative,
** the third is the xx-derivative (or 0 if not used), 
** the fourth is the xy-derivative (or 0 if not used) 
** and the last one is the yy-derivative (or 0 if not used).
**
** The viewing parameters are:
**
** 1) 'numln': 
**         describes the number of inner lines where barycentric coordinates
**         are constant that, together with boundary lines of the triangle,
**         define the points where the interpolated function shall be computed.
**         The bigger 'numln' is the higher 'detail' of the drawing is .
**
** 2) The direction of the normal to the viewing plane ('vpn') by means 
**    of the degrees for the latitude and for the longitude. 
**    To be clearer, let us consider the smallest cube that contains all the net 
**    of Bezier control point in R^3: the 'minmaxbox' box.
**    Let us also consider the sphere circumscribed to 'minmaxbox'.
**    Then the two angles define a point on the surface of the sphere. 
**    The normal to the viewing plane is then defined as the line joining 
**    the center of the sphere, the 'vrp' point (view reference point),
**    to this point (towards the user's eye).
**
** 3) The 'vuv' vector (viewing up vector), that is the vector whose 
**    projection onto the viewing plane gives the y-coordinate of graphic
**     window.
**
**
** -------------------------------------------------------------------------
** SUBROUTINES AND FUNCTIONS DESCRIPTION:
**
** >>>>>>>>>>>>>>>
**
** void writemenu()
**
** Writes LABSUP3D menu to the terminal window.
** Successive choices must be done on the `graphic window' instead, 
** once it becomes in 'focus state'.
**
** >>>>>>>>>>>>>>>
**
** void settridisegno()
**
** This function sets the parameters needed to call 'tridisegno'
** with draw_flag==1 or draw_flag==3, 
** in order to draw a Bezier patch or a C1 surface made up of
** piecewise Bezier patches using one of the interpolation methods 
** supported by the program.
**
** This function requires:
** 1) The number of lines for each barycentric coordinate except the ones for the
**    boundary, i.e. the parameter 'numln'
**    which describes the 'detail' of drawing and the
**    number of points computed on each patch.
** 2) The normal direction to the viewing plane (vpn) specifying the 
**    degrees for the latitude and for the longitude.
**    These parameters set the direction for the point of view.
**
** >>>>>>>>>>>>>>>
** 
** void settriang()
**
** Sets all the parameters needed to call 'tridisegno' with draw_flag==2
** in order to draw the triangulation.
**
** >>>>>>>>>>>>>>>
**
** void tridisegno()
**
** Main loop function.
**
** if draw_flag==0, it simply exits.
** if draw_flag==1, it draws a Bezier patch.
** if draw_flag==2, it draws the triangulation.
** if draw_flag==3, it draws the C1 surface.
**
** The parameters for drawing are set in the function 'settridisegno()'.
** If before drawing, draw_flag==3, the function may again compute the view points
** if the user changes the detail of visualization that is 'numln', or simply if
** the user chooses a different interpolation method.
**
** >>>>>>>>>>>>>>>>
**
** void tribezier()
**
** Computes control points for the interpolating Bezier patch.
** 
** >>>>>>>>>>>>>>>>
**
** void tricloughtocher()
**
** Computes the control points net for the C1-surface interpolating 
** given input points with their given first order partial derivatives over 
** a triangulation using Clough-Tocher elements.
** If a subtriangulation is active it works on it instead.
** 
** >>>>>>>>>>>>>>>>
**
** void tripowellsabin()
** 
** Computes control points net for the C1-surface interpolating given input points
** with their given first order partial derivatives over a triangulation
** using Powell-Sabin elements of A-Type and of B-Type .
** If a subtriangulation is active it works on it instead.
**
** >>>>>>>>>>>>>>>>
**
** void triq18() 
**
** Computes control points net for the C1-surface interpolating given input points
** with their given first and second order partial derivatives over a
** triangulation using Q18 elements.
** If a subtriangulation is active it works on it instead.
**
** >>>>>>>>>>>>>>>>
**
** void triinputitptb()
**
** Allows the user to introduce input points to be interpolated with a 
** suitable  Bezier patch. The degree of the patch depends on the number of 
** points introduced.
**
** >>>>>>>>>>>>>>>>
**
** void trimodificaitptb()
**
** Allows the user to vary some of the input points introduced in the
** function 'triinputitptb()'.
**
** >>>>>>>>>>>>>>>>
**
** void trioutputitptb()
** 
** Shows the input points introduced by the function 'triinputitptb()' or modified by
** the function 'trimodificaitptb()'.
**
** >>>>>>>>>>>>>>>>
** 
** void trioutputcptb() 
**
** Shows the control points of the Bezier patch computed by 'tribezier()' 
** interpolating the input points introduced with 'triinputitptb()' or 
** modified by 'trimodificaitptb()'.
**
** >>>>>>>>>>>>>>>>
**
** void triinputitptt()
**
** Requires to specify three file names:
**
** 1) Input data
** 2) Triangulation 
** 3) Derivative data
**
** Then the program is ready to compute the control points net for a specified
** interpolation method. 
**
** >>>>>>>>>>>>>>>>
**
** void trimagnify()
**
** Globals used: itpttpresenza, cptbpresenza, cpttpresenza,
**               Punti_controllo, numpt, numero_triangoli  
**
** Multiplies the z-coordinate of input data points, the derivatives and 
** the computed control points by a factor introduced by the user. 
** If necessary updates the drawing.
**
** >>>>>>>>>>>>>>>>>
**
** void trioutputitptt()
**
** Allows to see the points and/or derivatives introduced by the function
** 'triinputitptt()'.
**
** >>>>>>>>>>>>>>>>>>
**
** void trioutputcptt()
** 
** Shows the control points net for the data introduced by the function 
** 'triinputitptt()' over which an interpolation method has been applied.
**
** >>>>>>>>>>>>>>>>>>
**
** void minandmax(double a,double b,double c,double* min,double* max)
**
** Returns in 'min' and 'max' the minimum and the maximum value among a,b,c .
**
** >>>>>>>>>>>>>>>>>>
**
** double piano(double xnew,double ynew,double x1,double y1,double z1,
**              double x2,double y2,double z2,double x3,double y3,double z3)
**
** Computes the function z(xnew,ynew) over the plane passing through (x1,y1,z1),
** (x2,y2,z2), (x3,y3,z3). If the points are nearly aligned (denominator
** smaller than EPS_ALLIN) the function stops the program. 
**
** >>>>>>>>>>>>>>>>>>
**
** int circoscrivi(double x1,double y1,double x2,double y2,double x3,double y3,
**                 double* xc,double* yc,double r)
** 
** Computes the circle passing through the points (x1,y1), (x2,y2), (x3,y3).
** 
** The function returns
**     i)  the center (xc,yc) and the radius 'r' of the circle  
**         if the function result is 'circoscrivi==0';
**    ii)  if 'r==NULL' the computation of the radius is not performed;
**   iii)  if 'circoscrivi==1' the three points are aligned or nearly aligned ones
**          (there will be a denominator smaller than EPS_ALLIN) 
**
** >>>>>>>>>>>>>>>>>>>
**
** double triangle_max_angle(double x1,double y1,double x2,double y2,
**                           double x3,double y3)
**
** Computes the greatest angle (in degrees) among the three ones
** of the triangle with vertices (x1,y1), (x2,y2) and (x3,y3).
**
** >>>>>>>>>>>>>>>>>>>
**
** void options()
**
** Sets program options.
**
** 1) Width field for real numbers.
** 2) Number of decimal digits.
** 3) Zoom factor for central zooms.
** 4) Color settings for lines, fills and 
**    background with RGB triples in the range [0,1].
**
** >>>>>>>>>>>>>>>>>>>
**
** void read_triang()
**
** The function uses the globals:
**
**         'Triangolazione', 
**         'Triangolazione_Glob' ,
**         'numero_triangoli', 
**         'numero_triangoli_glob' ,
**         'nome_tri_file'.
**
** It reads the triangulation from the file 'nome_tri_file' following the format
** of the program 'do_trian' and stores loaded triangles in the list 
** 'Triangolazione_Glob'. 
** It sets the number of triangles into 'numero_triangoli_glob'. 
** Then it sets the pointer 'Triangolazione' to 'Triangolazione_Glob' and assigns
** 'numero_triangoli_glob' to 'numero_triangoli'.
**
** >>>>>>>>>>>>>>>>>>>
**
** void free_triang_list(struct triang_list* Triang_List_Pt)
**
** Cleans recursively the memory used by the list of triangles 
** pointed by 'Triang_List_Pt'.
**
** >>>>>>>>>>>>>>>>>>>
**
** void free_edge_list(struct edge_list* Edge_List_Pt)
**
** Cleans recursively the memory used by the list of edges 
** pointed by 'Edge_List_Pt'. 
**
** >>>>>>>>>>>>>>>>>>>>
**
** void free_cptt_list(struct cptt_list* Cptt_List_Pt)
**
** Cleans recursively the memory used by the list of control points 
** pointed by 'Cptt_List_Pt'.
**
** >>>>>>>>>>>>>>>>>>>>
**
** void set_ucalcpar_vcalcpar(int numln)
**
** Sets the globals 'ucalcpar' and 'vcalcpar' following a regular 
** subdivision of the standard triangle (0,0), (1,0), (0,1) into 'numln'
** inner lines using (u,v)-barycentric coordinates of their intersections 
** (including the outer border) for setting the values of the arrays 
** 'ucalcpar' and 'vcalcpar'.
**
** >>>>>>>>>>>>>>>>>>>>
**
** void output_file_itptb()
**
** The same as 'trioutputitptb()' except a user-specified
** filename instead 'stdout'. 
**
** >>>>>>>>>>>>>>>>>>>>
**
** void output_file_cptb()
**
** The same as 'trioutputcptb()' except a user-specified
** filename instead 'stdout' 
**
** >>>>>>>>>>>>>>>>>>>>
**
** void output_file_itptt()
**
** The same as 'trioutputitptt()' except a user-specified
** filename instead 'stdout'.
**
** >>>>>>>>>>>>>>>>>>>>
**
** void output_file_cptt()
**
** The same as 'trioutputcptt()' except a user-specified
** filename instead 'stdout'. 
**
** >>>>>>>>>>>>>>>>>>>>
**
** void output_file_calcpt()
**
** Shows computed points for the piecewise Bezier patches building up 
** the C1 surface.
** The points are computed by the function 'compute_view_points(...)'
**
** >>>>>>>>>>>>>>>>>>>>
**
** void printbezt3d(double[][MAXDIM] calcpt)
**
** Used by the function 'output_file_calcpt()'.
**
** >>>>>>>>>>>>>>>>>>>>>
**
** void selectsubtriang()
**
** Globals used: 
**          Triangolazione_Loc, Triangolazione_Glob, 
**          minbox_t_loc[], maxbox_t_loc[], 
**          minbox_t_glob, maxbox_t_glob, 
**          tri_left, tri_right, tri_down, 
**          tri_up, numero_triangoli_loc oltre a x[] e y[].
**
** Globals CHANGED: Triangolazione_Loc, numero_triangoli_loc, 
**                  minbox_t_loc[], maxbox_t_loc[].
**
** First, the function requires to define a selected region as a clipping 'square' 
** for the triangulation window.
** Then, it creates a subtriangulation 'Triangolazione_Loc'
** ('numero_triangoli_loc' is the number of its triangles and 'minbox_t_loc[]' and
**  'maxbox_t_loc[]' are its 'square' bounds) visiting the list of
** triangles 'Triangolazione_Glob' and accepting all the triangles with at
** least one vertex inside the selected region.
** The caller function if necessary sets 'Triangolazione' to 'Triangolazione_Loc',
** 'numero_triangoli' to 'numero_triangoli_loc', 'minbox_t' to 'minbox_t_loc' and
** 'maxbox_t' to 'maxbox_t_loc'.
** 
** >>>>>>>>>>>>>>>>>>>>>>
**
** void printselectionborder(struct edge_list* SelectionBorder, char[] NameFile)
**
** Stores in the file 'NameFile' the edges which define the 
** border of the current subtriangulation. 
** These edges are the ones of the triangles
** of the current subtriangulation for which the opposite 
** vertex is the only one inside the open square defining 
** the subtriangulation (as a clipping 'square' 
** in the triangulation graphic window).
**
** >>>>>>>>>>>>>>>>>>>>>>>
**
** void compute_view_points(double[][MAXDIM] calcpt)
**
** Computes the points for the piecewise Bezier patches building up the C1 surface.
** The points are computed for each Bezier patch using 'numln' inner lines per patch
** (using the same barycentric coordinates)
** besides the ones of the boundary of the patch: their intersections are the
** points where to evaluate the interpolation function (computed points).
** The computation of points is done with reference to the interpolation method
** chosen by the user over the active subtriangulation.
**
** >>>>>>>>>>>>>>>>>>>>>>
** 
*/

#include "labsup3d.h"
#include "superf.c"
#include "matrixio.c"
#include "linsist.c"
#include "graphsup.c"

void main()
{
char scelta;

/* Initializing triangulation variables */
Triangolazione=NULL;
Triangolazione_Glob=NULL;
Triangolazione_Loc=NULL;
Inviluppo_convesso=NULL;
Punti_controllo=NULL;
output_ris=stdout;
printf("Resize graphic window dimensions? (y/n).\n");
scanf(" %c",&scelta);
if (scelta=='y')
{
 printf("Width in pixels: ");
 scanf(" %d",&WINWIDTH);
 printf("Height in pixels: ");
 scanf(" %d",&WINHEIGHT);
 if (WINWIDTH<100 || WINHEIGHT<100 || WINWIDTH>800 || WINHEIGHT>800)
 {
  printf("No valid dimensions.\n");
  exit(1);
 }
}
glinitmygraph();

/* Declaring events that shall be supported */ 
#ifdef LABSUP3D_AUX
	auxReshapeFunc(glmyreshape);
	auxExposeFunc(glmyreshape);
	auxKeyFunc(AUX_a,tribezier);
	auxKeyFunc(AUX_b,tricloughtocher);
	auxKeyFunc(AUX_c,tripowellsabin);
	auxKeyFunc(AUX_d,triq18);
	auxKeyFunc(AUX_1,triinputitptb);
	auxKeyFunc(AUX_2,trimodificaitptb);
	auxKeyFunc(AUX_3,trioutputitptb);
	auxKeyFunc(AUX_4,trioutputcptb);
	auxKeyFunc(AUX_5,triinputitptt);
	auxKeyFunc(AUX_6,trimagnify);
	auxKeyFunc(AUX_7,trioutputitptt);
	auxKeyFunc(AUX_8,trioutputcptt);
	auxKeyFunc(AUX_p,output_file_itptb);
	auxKeyFunc(AUX_q,output_file_cptb);
	auxKeyFunc(AUX_r,output_file_itptt);
	auxKeyFunc(AUX_s,output_file_cptt);
	auxKeyFunc(AUX_t,output_file_calcpt);
	auxKeyFunc(AUX_g,settridisegno);
	auxKeyFunc(AUX_v,settriang);
	auxKeyFunc(AUX_z,glswitchdepth);
	auxKeyFunc(AUX_l,glswitchlight);
	auxKeyFunc(AUX_0,glswitchbox);
	auxKeyFunc(AUX_o,options);
	auxKeyFunc(AUX_ESCAPE,auxQuit);
	auxKeyFunc(AUX_UP,increase_zoom);
	auxKeyFunc(AUX_DOWN,decrease_zoom);	
	auxMouseDownFunc(glmouseaction);
	writemenu();
	auxMainLoop(tridisegno);
#endif

#ifdef LABSUP3D_GLUT
	glutReshapeFunc(glmyreshape);
	glutDisplayFunc(tridisegno);
	glutKeyboardFunc(KeyboardFuncCallBack_glut);
	glutSpecialFunc(SpecialFuncCallBack_glut);
	glutMouseFunc(MouseFuncCallBack_glut);
	writemenu();
	glutMainLoop();
#endif
}

#ifdef LABSUP3D_GLUT

	void MouseFuncCallBack_glut(int button, int state, int x, int y)
	{
	  if (state==GLUT_DOWN) 
	   if (glmouseaction(x,y,button)==GL_TRUE)
		 glutPostRedisplay();
	}

	void KeyboardFuncCallBack_glut(unsigned char key, int x, int y)
	{
	  if (key==27) exit(0);
	  else
	  {
	   switch(key) 
	   {
		case 'a': tribezier(); break;
		case 'b': tricloughtocher(); break;
		case 'c': tripowellsabin(); break;
		case 'd': triq18(); break;
		case '1': triinputitptb(); break;
		case '2': trimodificaitptb(); break;
		case '3': trioutputitptb(); break;
		case '4': trioutputcptb(); break;
		case '5': triinputitptt(); break;
		case '6': trimagnify(); break;
		case '7': trioutputitptt(); break;
		case '8': trioutputcptt(); break;
		case 'p': output_file_itptb(); break;
		case 'q': output_file_cptb(); break;
		case 'r': output_file_itptt(); break;
		case 's': output_file_cptt(); break;
		case 't': output_file_calcpt(); break;
		case 'g': settridisegno(); break;
		case 'v': settriang(); break;
		case 'z': glswitchdepth(); break;
		case 'l': glswitchlight(); break;
		case '0': glswitchbox(); break;
		case 'o': options(); break; 
	   }
	   glutPostRedisplay();
	  }
	}

	void SpecialFuncCallBack_glut(int key, int x, int y)
	{
	 switch(key) 
	   {
		 case GLUT_KEY_UP: increase_zoom(); break;
		 case GLUT_KEY_DOWN: decrease_zoom(); break;
		 default: return;
	   }
	   glutPostRedisplay();
	}

#endif

void writemenu()
  { 
   printf("\n\n*** LABSUP3D MENU ***\n\n");
   printf("-    a    -  Global Bezier method (Only one triangle)\n");
   printf("-    b    -  C1 surface: Clough-Tocher with linear normal derivatives\n");
   printf("-    c    -  C1 surface: Powell-Sabin\n");
   printf("-    d    -  C1 surface: Q18\n");
   printf("-    1    -  Input points for global Bezier method\n");  
   printf("-    2    -  Modify interpolation points for global Bezier method\n");
   printf("-    3    -  View interpolation points for global Bezier method\n");
   printf("-    4    -  View control points for global Bezier method\n");
   printf("-    5    -  Input points, triangulation and derivatives for C1 surfaces\n");
   printf("-    6    -  Multiply input z values and derivatives by a magnifying factor\n");
   printf("-    7    -  View points/derivatives\n");
   printf("-    8    -  View control points for C1 surfaces\n");
   printf("- p q r s -  Output to files for menu items 3 4 7 8 respectively\n");
   printf("-    t    -  Output of the points computed for C1 surfaces\n"); 
   printf("-    g    -  Set graphic parameters and view surface\n");
   printf("-    v    -  View triangulation\n");
   printf("-  z l 0  -  On/Off switches for depth-buffer,light and box respectively\n");
   printf("-    o    -  Output options format and choice of colors\n");
   printf("-   ESC   -  Quit program\n");
   printf("\n MAKE YOUR CHOICES FROM GRAPHIC WINDOW.\n");
   printf(" Mouse left button performs rotation in surface mode and prints\n");
   printf(" the coordinates of a selected point in triangulation mode.\n");       
   printf(" Up and down arrows perform central zoom in surface mode.\n");
   printf(" Mouse right button (twice pressed) performs selected zoom in any mode.\n");
   printf(" Mouse middle button cancels selected zooms in surface mode and allows\n");
   printf(" to work with a subtriangulation in triangulation mode.\n\n");
  }

void settridisegno()
{
 double lambdavpn,phivpn,temp;
 double vuv_temp[3];
 int i;
 static int numln_saved=-1; 

 if (cptbpresenza==0 && cpttpresenza==0) return;
 numln=-1; /* entering while loop */
 while (numln<0 || numpt>MAXCALC)
   {
    printf("Number of lines for each barycentric coordinate\n");
    printf("apart from boundary ones: ");
    scanf(" %d",&numln);
    numpt=(numln+2)*(numln+3)/2;
    if (numln>=0 && numpt<=MAXCALC) break; /* va bene */
   }
 viewoptions:
 printf("Set the normal direction to the view plane (vpn) specifying\n");
 printf("in this order the degrees for the latitude and the longitude.\n");
 scanf(" %lf %lf",&lambdavpn,&phivpn);
 lambdavpn=lambdavpn/180.*PI; phivpn=phivpn/180.*PI;
 vpn[0]=cos(lambdavpn)*cos(phivpn);
 vpn[1]=cos(lambdavpn)*sin(phivpn);
 vpn[2]=sin(lambdavpn);
 printf("Set the vector whose projection onto the view plane yields\n");
 printf("the direction of y-axis in the view plane (vuv)\n");
 printf("(For a function such as z=f(x,y), use '0 0 1' in most cases).\n");  
 err=getmat(1,3,vuv_temp,3);
 vuv[0]=(GLdouble)vuv_temp[0]; 
 vuv[1]=(GLdouble)vuv_temp[1];
 vuv[2]=(GLdouble)vuv_temp[2]; 
 temp=fabs(vpn[1]*vuv[2]-vpn[2]*vuv[1])+fabs(vpn[2]*vuv[0]-vpn[0]*vuv[2])
      +fabs(vpn[0]*vuv[1]-vpn[1]*vuv[0]);
 if (temp<EPS_ALLIN)
 {
  printf("vpn and vuv vectors cannot be parallel\n");
  printf("\nHit a button\n");
  fflush(stdin);
  getchar();
 }
 if (temp<EPS_ALLIN) goto viewoptions;
 
 if (cptbpresenza!=0) /* the choice is exclusive with cpttpresenza!=0 */
 {
  /* minmaxbox already computed by tribezier */
  /* Setting vrp to the center of the minmaxbox */
  for (i=0;i<dim;++i) vrp[i]=(minbox[i]+maxbox[i])/2.;
  parsize=sqrt( pow((maxbox[0]-minbox[0]),2)+pow((maxbox[1]-minbox[1]),2)+
		pow((maxbox[2]-minbox[2]),2))/2.;      
  glsetmyoptions(parsize,vrp,vpn,vuv,0);
  if ((draw_flag!=1) || numln!=numln_saved) 
  {          /* Avoiding remaking computations if already done */
   /* Beginning computations */
   if (calcpt!=NULL) {free(calcpt); calcpt=NULL;}
   calcpt=malloc(numpt*3*sizeof(double));
   if (calcpt==NULL)
     {printf("Not enough memory\n"); exit(1);}
   printf("\nWait for computation of points.\n\n");
   /* computation of parameters */
   set_ucalcpar_vcalcpar(numln);
   /* The list of generalized binomial 'generalfactlist' has been computed before the call
      to 'triitpbez' function */
   err=trihornbez(n,dim,cpt,generalfactlist,ucalcpar,vcalcpar,numpt,calcpt);
   if (err==1) {printf("Error: Wrong parameters.\n"); exit(1);}
  }
 }
 if (cpttpresenza!=0) /* the choice is exclusive with cptbpresenza!=0 */
 {
  /* Global 'minmaxbox' has already been computed 
     during input stage and when making computations */
  /* Setting vrp to the center of the minmaxbox */
  for (i=0;i<3;++i) vrp[i]=(minbox_t[i]+maxbox_t[i])/2.;
  parsize=sqrt( pow((maxbox_t[0]-minbox_t[0]),2)+
		pow((maxbox_t[1]-minbox_t[1]),2)+
		pow((maxbox_t[2]-minbox_t[2]),2))/2.;      
  glsetmyoptions(parsize,vrp,vpn,vuv,0); 
  if ((draw_flag!=3) || numln!=numln_saved)
  {              /* Avoiding remaking computations if already done */
   if (calcpt!=NULL) {free(calcpt); calcpt=NULL;}
   /* Dynamical allocation regardless the number of sub-triangles */
   if (cpttpresenza==1) 
     calcpt=malloc(numpt*numero_triangoli*3*3*sizeof(double));
   if (cpttpresenza==2)
     calcpt=malloc(numpt*numero_triangoli*3*12*sizeof(double));
   if (cpttpresenza==3)
     calcpt=malloc(numpt*numero_triangoli*3*1*sizeof(double));
   if (calcpt==NULL)
     {printf("Not enough memory\n"); exit(1);}
   /* computations are done over global 'calcpt'  
   ** using global 'ucalcpar' and 'vcalcpar' for each macro-micro triangle */
   /* setting ucalcpar and vcalcpar */
   set_ucalcpar_vcalcpar(numln);
   /* setting binomial lists for the various methods */
   /* lista_ass_t has already been updated when making computations */
   if (cpttpresenza==1) gengeneralfact(3,generalfactlist,lista_ass_t);
   if (cpttpresenza==2) gengeneralfact(2,generalfactlist,lista_ass_t);
   if (cpttpresenza==3) gengeneralfact(5,generalfactlist,lista_ass_t);  
   compute_view_points(calcpt);
  } 
 }  
 if (cptbpresenza!=0) draw_flag=1;
 if (cpttpresenza!=0) draw_flag=3;
 numln_saved=numln; 
 /* To watch for numln changements. The value is not lost since it is declared as static */
 writemenu();
}

void settriang()
{ 
 double corr;
  
 if (itpttpresenza==0 || cptbpresenza==1) return; 
 /* If necessary disables the depth-buffer */
 if (depth_flag==1)
   {
    glDisable(GL_DEPTH_TEST);
    depth_flag=0;
   }
 /* setting for the window */
 glLoadIdentity();
 if ((maxbox_t[0]-minbox_t[0])>=(maxbox_t[1]-minbox_t[1])) /* Fitting to width */
  {
   corr=(maxbox_t[0]-minbox_t[0])/10.;
   tri_left=minbox_t[0]-corr;
   tri_right=maxbox_t[0]+corr;
   tri_down=(minbox_t[1]+maxbox_t[1]-maxbox_t[0]+minbox_t[0])/2.-corr;
   tri_up=(minbox_t[1]+maxbox_t[1]+maxbox_t[0]-minbox_t[0])/2.+corr;
  }
 else /* Fitting to height */
  {
   corr=(maxbox_t[1]-minbox_t[1])/10.;
   tri_left=(minbox_t[0]+maxbox_t[0]-maxbox_t[1]+minbox_t[1])/2.-corr;
   tri_right=(minbox_t[0]+maxbox_t[0]+maxbox_t[1]-minbox_t[1])/2.+corr;
   tri_down=minbox_t[1]-corr;
   tri_up=maxbox_t[1]+corr;
  }
 gluOrtho2D(tri_left,tri_right,tri_down,tri_up);
 draw_flag=2;
 writemenu();
}

void tridisegno()
{
 char xstorename_buffer[NAME_FILE_BUFFER_LEN];
 int i; 

 if (draw_flag==0)
   {
	/* Only for glut */
	#ifdef LABSUP3D_GLUT
		setmyglcolor(); 
		LABSUP3D_SWAPBUFFERS;
	#endif
	/* In any case */
    return;
   }
 strcpy(xstorename_buffer,MainName);   
 if (draw_flag==3)  /* Case of C1 surfaces */
 {
  /* remaking computation of points if the user chooses a different method with the
     same value for 'numln' */
  if ((cpttpresenza==1) && (generalfactlist[1]!=3)) 
   {
    if (calcpt!=NULL) {free(calcpt); calcpt=NULL;}
    calcpt=malloc(numpt*numero_triangoli*3*3*sizeof(double));
    if (calcpt==NULL) {printf("Not enough memory\n"); exit(1);}
    gengeneralfact(3,generalfactlist,lista_ass_t);
    compute_view_points(calcpt);
    /* Setting vrp to the center of 'minmaxbox' box */
    for (i=0;i<dim;++i) vrp[i]=(minbox_t[i]+maxbox_t[i])/2.;
    parsize=sqrt( pow((maxbox_t[0]-minbox_t[0]),2)+pow((maxbox_t[1]-minbox_t[1]),2)+
		  pow((maxbox_t[2]-minbox_t[2]),2))/2.;      
    glsetmyoptions(parsize,vrp,vpn,vuv,0);
   } 
  else if ((cpttpresenza==2) && (generalfactlist[1]!=2)) 
   {
    if (calcpt!=NULL) {free(calcpt); calcpt=NULL;}
    calcpt=malloc(numpt*numero_triangoli*3*12*sizeof(double));
    if (calcpt==NULL) {printf("Not enough memory\n"); exit(1);}
    gengeneralfact(2,generalfactlist,lista_ass_t);
    compute_view_points(calcpt);
    /* Setting vrp to the center of 'minmaxbox' box */
    for (i=0;i<dim;++i) vrp[i]=(minbox_t[i]+maxbox_t[i])/2.;
    parsize=sqrt( pow((maxbox_t[0]-minbox_t[0]),2)+pow((maxbox_t[1]-minbox_t[1]),2)+
		  pow((maxbox_t[2]-minbox_t[2]),2))/2.;      
    glsetmyoptions(parsize,vrp,vpn,vuv,0);
   }
  else if ((cpttpresenza==3) && (generalfactlist[1]!=5))   
  {
   if (calcpt!=NULL) {free(calcpt); calcpt=NULL;}
   calcpt=malloc(numpt*numero_triangoli*3*1*sizeof(double));
   if (calcpt==NULL) {printf("Not enough memory\n"); exit(1);}
   gengeneralfact(5,generalfactlist,lista_ass_t); 
   compute_view_points(calcpt);
   /* Setting vrp to the center of 'minmaxbox' box */
   for (i=0;i<dim;++i) vrp[i]=(minbox_t[i]+maxbox_t[i])/2.;
   parsize=sqrt( pow((maxbox_t[0]-minbox_t[0]),2)+pow((maxbox_t[1]-minbox_t[1]),2)+
 		 pow((maxbox_t[2]-minbox_t[2]),2))/2.;      
   glsetmyoptions(parsize,vrp,vpn,vuv,0);
  }
  switch(cpttpresenza)
   {
    case 1: strcat(xstorename_buffer," Clough-Tocher"); break;
    case 2: strcat(xstorename_buffer," Powell-Sabin"); break;
    case 3: strcat(xstorename_buffer," Q18"); break;
   }
 } /* End case when draw_flag==3 */
 if (draw_flag==2) strcat(xstorename_buffer," Triangulation");

 #ifdef LABSUP3D_AUX
	XStoreName(MainDisplay,MainWindow,xstorename_buffer);
 #endif
 
#ifdef LABSUP3D_GLUT
	glutSetWindowTitle(xstorename_buffer);
 #endif
 
 if (draw_flag==1) glvbez3d(calcpt);
 if (draw_flag==2) glvtriang();
 if (draw_flag==3) glvbezt3d(calcpt);
}
 
void tribezier()
   {
    double ptnew[MAXORD][MAXDIM];

    if (itptbpresenza==0) return;
    printf("Wait for computation\n");
    copymatrix(l+1,dim,itpt,ptnew,MAXDIM);
    gengeneralfact(n,generalfactlist,lista_ass);
    err=triitpbez(n,ptnew,cpt,u,v,dim,lista_ass,generalfactlist);
    if (err==1) exit(0);
    if (err==2)
    {
     printf("Singular or bad matrix\n");
     printf("Hit a button\n");
     fflush(stdin);
     getchar();
     writemenu();
    }
    else
    {
     /* minmaxbox bezier */
     err=minmaxbox(l,dim,cpt,minbox,maxbox,MINAMPBOX);
     cptbpresenza=1;
     cpttpresenza=0;
    }
    writemenu();
   }

void tricloughtocher()
{
 double x4,y4,t1,t2,t3;
 int s,ss,first_time_while_flag=1;
 long int iv1,iv2,iv3; 
 struct cptt_list *Cptt_Pt;
 struct triang_list *Triang_Pt;

 if (itpttpresenza==0) return;
  /* Setting the list of indices */
 for (s=0;s<10;++s) ass(s,3,&(lista_ass_t[s]));
 printf("Wait for computation.\n");
 free_cptt_list(Punti_controllo); Punti_controllo=NULL;
 Triang_Pt=Triangolazione;
 /* Creating dummy cell and initializing local counter */
 Punti_controllo=malloc(sizeof(struct cptt_list));
 if (Punti_controllo==NULL) {printf("Not enough memory.\n"); exit(1);}
 Cptt_Pt=Punti_controllo;
 while (Triang_Pt!=NULL)
   {
    iv1=Triang_Pt->triangolo.v1;
    iv2=Triang_Pt->triangolo.v2;
    iv3=Triang_Pt->triangolo.v3;
    x4=(x[iv1]+x[iv2]+x[iv3])/3.;
    y4=(y[iv1]+y[iv2]+y[iv3])/3.;
    t1=((x4-x[iv1])*(x[iv2]-x[iv1])+(y4-y[iv1])*(y[iv2]-y[iv1]))/
       (pow(x[iv2]-x[iv1],2)+pow(y[iv2]-y[iv1],2));
    t2=((x4-x[iv2])*(x[iv3]-x[iv2])+(y4-y[iv2])*(y[iv3]-y[iv2]))/
       (pow(x[iv3]-x[iv2],2)+pow(y[iv3]-y[iv2],2));   
    t3=((x4-x[iv3])*(x[iv1]-x[iv3])+(y4-y[iv3])*(y[iv1]-y[iv3]))/
       (pow(x[iv1]-x[iv3],2)+pow(y[iv1]-y[iv3],2));
   /* creating a new cell pointed by Cptt_Pt->next*/
    Cptt_Pt->next=malloc(sizeof(struct cptt_list));
    if (Cptt_Pt->next==NULL) {printf("Not enough memory.\n"); exit(1);}
    Cptt_Pt->next->cptt.itpmethod=1;
    /* array of points. 0 INDEX POINT IS NOT USED */
    Cptt_Pt->next->cptt.points[1] =z[iv1];
    Cptt_Pt->next->cptt.points[2] =z[iv2];
    Cptt_Pt->next->cptt.points[3] =z[iv3];  
    Cptt_Pt->next->cptt.points[4] =((x[iv2]-x[iv1])*zx[iv1]+
				    (y[iv2]-y[iv1])*zy[iv1] )/3.+z[iv1];  
    Cptt_Pt->next->cptt.points[5] =((x4-x[iv1])*zx[iv1]+
				    (y4-y[iv1])*zy[iv1] )/3.+z[iv1];  
    Cptt_Pt->next->cptt.points[6] =((x[iv3]-x[iv1])*zx[iv1]+
				    (y[iv3]-y[iv1])*zy[iv1] )/3.+z[iv1];  
    Cptt_Pt->next->cptt.points[7] =((x[iv3]-x[iv2])*zx[iv2]+
				    (y[iv3]-y[iv2])*zy[iv2] )/3.+z[iv2];  
    Cptt_Pt->next->cptt.points[8] =((x4-x[iv2])*zx[iv2]+
				    (y4-y[iv2])*zy[iv2] )/3.+z[iv2];  
    Cptt_Pt->next->cptt.points[9] =((x[iv1]-x[iv2])*zx[iv2]+
				    (y[iv1]-y[iv2])*zy[iv2] )/3.+z[iv2];  
    Cptt_Pt->next->cptt.points[10]=((x[iv1]-x[iv3])*zx[iv3]+
				    (y[iv1]-y[iv3])*zy[iv3] )/3.+z[iv3];  
    Cptt_Pt->next->cptt.points[11]=((x4-x[iv3])*zx[iv3]+
				    (y4-y[iv3])*zy[iv3] )/3.+z[iv3];  
    Cptt_Pt->next->cptt.points[12]=((x[iv2]-x[iv3])*zx[iv3]+
				    (y[iv2]-y[iv3])*zy[iv3] )/3.+z[iv3];  
    Cptt_Pt->next->cptt.points[13]=(Cptt_Pt->next->cptt.points[5]+
				    Cptt_Pt->next->cptt.points[8]+
				    (t1-1)*Cptt_Pt->next->cptt.points[1]+
				    (2-3*t1)*Cptt_Pt->next->cptt.points[4]+
				    (3*t1-1)*Cptt_Pt->next->cptt.points[9]-
				    t1*Cptt_Pt->next->cptt.points[2])/2.;  
    Cptt_Pt->next->cptt.points[14]=(Cptt_Pt->next->cptt.points[8]+
				    Cptt_Pt->next->cptt.points[11]+
				    (t2-1)*Cptt_Pt->next->cptt.points[2]+
				    (2-3*t2)*Cptt_Pt->next->cptt.points[7]+
				    (3*t2-1)*Cptt_Pt->next->cptt.points[12]-
				    t2*Cptt_Pt->next->cptt.points[3])/2.;  
    Cptt_Pt->next->cptt.points[15]=(Cptt_Pt->next->cptt.points[11]+
				    Cptt_Pt->next->cptt.points[5]+
				    (t3-1)*Cptt_Pt->next->cptt.points[3]+
				    (2-3*t3)*Cptt_Pt->next->cptt.points[10]+
				    (3*t3-1)*Cptt_Pt->next->cptt.points[6]-
				    t3*Cptt_Pt->next->cptt.points[1])/2.;  
    Cptt_Pt->next->cptt.points[16]=(Cptt_Pt->next->cptt.points[15]+
				    Cptt_Pt->next->cptt.points[5]+
				    Cptt_Pt->next->cptt.points[13] )/3.;  
    Cptt_Pt->next->cptt.points[17]=(Cptt_Pt->next->cptt.points[13]+
				    Cptt_Pt->next->cptt.points[8]+
				    Cptt_Pt->next->cptt.points[14] )/3.;  
    Cptt_Pt->next->cptt.points[18]=(Cptt_Pt->next->cptt.points[14]+
				    Cptt_Pt->next->cptt.points[11]+
				    Cptt_Pt->next->cptt.points[15] )/3.;  
    Cptt_Pt->next->cptt.points[19]=(Cptt_Pt->next->cptt.points[18]+
				    Cptt_Pt->next->cptt.points[16]+
				    Cptt_Pt->next->cptt.points[17] )/3.;
    /* end array of points */
    /* computation for the z-component of global 'minmaxbox' box */
    if (first_time_while_flag==1) /* initializing globale 'minmaxbox' box */
      {
       minbox_t[2]=z[iv1]; maxbox_t[2]=z[iv1];
       first_time_while_flag=0;
      }
    for (ss=1;ss<=19;++ss) /* to do even if initializing */
      {
       if (Cptt_Pt->next->cptt.points[ss]<minbox_t[2])
		  minbox_t[2]=Cptt_Pt->next->cptt.points[ss];
       else if (Cptt_Pt->next->cptt.points[ss]>maxbox_t[2])
		  maxbox_t[2]=Cptt_Pt->next->cptt.points[ss];
      }
    /* analogy with the function minmaxbox */
    if ( (maxbox_t[2]-minbox_t[2]) < MINAMPBOX)
    {
     minbox_t[2]=minbox_t[2]-MINAMPBOX/2;
     maxbox_t[2]=maxbox_t[2]+MINAMPBOX/2;
    }
    Cptt_Pt=Cptt_Pt->next;
    Triang_Pt=Triang_Pt->next;
   } /* end of main 'while' loop */
 /* closing the tail for the list of control points */
 Cptt_Pt->next=NULL;
 /* deleting the dummy cell */
 Cptt_Pt=Punti_controllo->next;
 free(Punti_controllo);
 Punti_controllo=Cptt_Pt;
 /* for menu setting */
 cptbpresenza=0;
 cpttpresenza=1;
 writemenu();
}

void tripowellsabin()
{
 /* it uses globals x y z zx zy Triangolazione Punti_Controllo
 ** lista_ass_t
 */
 int pwsb_mode; /* Subdivision into 6 (0) or 12 triagles (1) */
 double x4,y4,x12,y12,x23,y23,x31,y31,x41,y41,x42,y42,x43,y43,t1,t2,t3;
 int s,ss,first_time_while_flag=1;
 long int iv1,iv2,iv3;
 struct triang_list *Triang_Pt;
 struct cptt_list *Cptt_Pt;

 if (itpttpresenza==0) return;
 /* Setting the list of indices */
 for (s=0;s<6;++s) ass(s,2,&(lista_ass_t[s]));
 printf("Wait for computation.\n");
 free_cptt_list(Punti_controllo); Punti_controllo=NULL;
 Triang_Pt=Triangolazione;
 /* Creating dummy cell and initializing local counter */
 Punti_controllo=malloc(sizeof(struct cptt_list));
 if (Punti_controllo==NULL) {printf("Not enough memory.\n"); exit(1);}
 Cptt_Pt=Punti_controllo;
 while (Triang_Pt!=NULL)
   {
    iv1=Triang_Pt->triangolo.v1;
    iv2=Triang_Pt->triangolo.v2;
    iv3=Triang_Pt->triangolo.v3;
    x12=(x[iv1]+x[iv2])/2.; y12=(y[iv1]+y[iv2])/2.;
    x23=(x[iv2]+x[iv3])/2.; y23=(y[iv2]+y[iv3])/2.;
    x31=(x[iv3]+x[iv1])/2.; y31=(y[iv3]+y[iv1])/2.;
    /* selection: 0 uses circumcenter, 1 uses barycenter */
    if (triangle_max_angle(x[iv1],y[iv1],x[iv2],y[iv2],x[iv3],y[iv3])>=75.)
      pwsb_mode=1; else pwsb_mode=0;
    if (pwsb_mode==0)  /*Don't need to check for the alignment (err==1)*/
     circoscrivi(x[iv1],y[iv1],x[iv2],y[iv2],x[iv3],y[iv3],&x4,&y4,NULL);
     else  /* pwsb_mode==1 */
     {
      x4=(x[iv1]+x[iv2]+x[iv3])/3.; y4=(y[iv1]+y[iv2]+y[iv3])/3.;
      t1=-((x[iv3]-x[iv2])*(x[iv2]-x4)+(y[iv3]-y[iv2])*(y[iv2]-y4))/
	  (pow(x[iv3]-x[iv2],2.)+pow(y[iv3]-y[iv2],2.));
      t2=-((x[iv1]-x[iv3])*(x[iv3]-x4)+(y[iv1]-y[iv3])*(y[iv3]-y4))/
	  (pow(x[iv1]-x[iv3],2.)+pow(y[iv1]-y[iv3],2.));
      t3=-((x[iv2]-x[iv1])*(x[iv1]-x4)+(y[iv2]-y[iv1])*(y[iv1]-y4))/
	  (pow(x[iv2]-x[iv1],2.)+pow(y[iv2]-y[iv1],2.));
      x41=(x31+x12)/2.; y41=(y31+y12)/2.;
      x42=(x12+x23)/2.; y42=(y12+y23)/2.;
      x43=(x23+x31)/2.; y43=(y23+y31)/2.;
     }
    /* creating a new cell pointed by 'Cptt_Pt->next' */
    Cptt_Pt->next=malloc(sizeof(struct cptt_list));
    if (Cptt_Pt->next==NULL) {printf("Not enough memory.\n"); exit(1);}
    Cptt_Pt->next->cptt.itpmethod=(1-pwsb_mode)*2+pwsb_mode*4;
    /* array of points using 0-index point too, besides pwsb_mode */
    Cptt_Pt->next->cptt.points[0]=z[iv1];   /* Common part */
    Cptt_Pt->next->cptt.points[4]=z[iv2];
    Cptt_Pt->next->cptt.points[8]=z[iv3];
    Cptt_Pt->next->cptt.points[11]=
     z[iv1]+(zx[iv1]*(x[iv3]-x[iv1])+zy[iv1]*(y[iv3]-y[iv1]))/4.;
    Cptt_Pt->next->cptt.points[1]=
     z[iv1]+(zx[iv1]*(x[iv2]-x[iv1])+zy[iv1]*(y[iv2]-y[iv1]))/4.;
    Cptt_Pt->next->cptt.points[3]=
     z[iv2]+(zx[iv2]*(x[iv1]-x[iv2])+zy[iv2]*(y[iv1]-y[iv2]))/4.;
    Cptt_Pt->next->cptt.points[5]=
     z[iv2]+(zx[iv2]*(x[iv3]-x[iv2])+zy[iv2]*(y[iv3]-y[iv2]))/4.;
    Cptt_Pt->next->cptt.points[7]=
     z[iv3]+(zx[iv3]*(x[iv2]-x[iv3])+zy[iv3]*(y[iv2]-y[iv3]))/4.;
    Cptt_Pt->next->cptt.points[9]=
     z[iv3]+(zx[iv3]*(x[iv1]-x[iv3])+zy[iv3]*(y[iv1]-y[iv3]))/4.;
    if (pwsb_mode==0) /* Only A-type subdivision */
    {
     Cptt_Pt->next->cptt.points[12]=
      z[iv1]+(zx[iv1]*(x4-x[iv1])+zy[iv1]*(y4-y[iv1]))/2.;
     Cptt_Pt->next->cptt.points[14]=
      z[iv2]+(zx[iv2]*(x4-x[iv2])+zy[iv2]*(y4-y[iv2]))/2.;
     Cptt_Pt->next->cptt.points[16]=
      z[iv3]+(zx[iv3]*(x4-x[iv3])+zy[iv3]*(y4-y[iv3]))/2.;
     Cptt_Pt->next->cptt.points[2]=
      (Cptt_Pt->next->cptt.points[1]+Cptt_Pt->next->cptt.points[3])/2.;
     Cptt_Pt->next->cptt.points[6]=
      (Cptt_Pt->next->cptt.points[5]+Cptt_Pt->next->cptt.points[7])/2.;
     Cptt_Pt->next->cptt.points[10]=
      (Cptt_Pt->next->cptt.points[9]+Cptt_Pt->next->cptt.points[11])/2.;
     Cptt_Pt->next->cptt.points[13]=
      (Cptt_Pt->next->cptt.points[12]+Cptt_Pt->next->cptt.points[14])/2.;
     Cptt_Pt->next->cptt.points[15]=
      (Cptt_Pt->next->cptt.points[14]+Cptt_Pt->next->cptt.points[16])/2.;
     Cptt_Pt->next->cptt.points[17]=
      (Cptt_Pt->next->cptt.points[16]+Cptt_Pt->next->cptt.points[12])/2.;
     Cptt_Pt->next->cptt.points[18]=
      piano(x4,y4,
	    (x4+x12)/2.,(y4+y12)/2.,Cptt_Pt->next->cptt.points[13],
	    (x4+x23)/2.,(y4+y23)/2.,Cptt_Pt->next->cptt.points[15],
	    (x4+x31)/2.,(y4+y31)/2.,Cptt_Pt->next->cptt.points[17]);
    }
    else  /* pwsb_mode==1: Only B-type subdivision */
    {
     Cptt_Pt->next->cptt.points[12]=
      z[iv1]+(zx[iv1]*(x41-x[iv1])+zy[iv1]*(y41-y[iv1]))/2.;
     Cptt_Pt->next->cptt.points[16]=
      z[iv2]+(zx[iv2]*(x42-x[iv2])+zy[iv2]*(y42-y[iv2]))/2.;
     Cptt_Pt->next->cptt.points[20]=
      z[iv3]+(zx[iv3]*(x43-x[iv3])+zy[iv3]*(y43-y[iv3]))/2.;
     Cptt_Pt->next->cptt.points[18]=
      2./3.*(Cptt_Pt->next->cptt.points[16]+Cptt_Pt->next->cptt.points[20])+
      (-2./3.+t1)*z[iv2]+(1./3.-t1)*z[iv3]+
      (2*t1-1)*(Cptt_Pt->next->cptt.points[7]-Cptt_Pt->next->cptt.points[5]);
     Cptt_Pt->next->cptt.points[22]=
      2./3.*(Cptt_Pt->next->cptt.points[20]+Cptt_Pt->next->cptt.points[12])+
      (-2./3.+t2)*z[iv3]+(1./3.-t2)*z[iv1]+
      (2*t2-1)*(Cptt_Pt->next->cptt.points[11]-Cptt_Pt->next->cptt.points[9]);
     Cptt_Pt->next->cptt.points[14]=
      2./3.*(Cptt_Pt->next->cptt.points[12]+Cptt_Pt->next->cptt.points[16])+
      (-2./3.+t3)*z[iv1]+(1./3.-t3)*z[iv2]+
      (2*t3-1)*(Cptt_Pt->next->cptt.points[3]-Cptt_Pt->next->cptt.points[1]);
     Cptt_Pt->next->cptt.points[2]=
      Cptt_Pt->next->cptt.points[1]/2.+Cptt_Pt->next->cptt.points[3]/2.;
     Cptt_Pt->next->cptt.points[6]=
      Cptt_Pt->next->cptt.points[5]/2.+Cptt_Pt->next->cptt.points[7]/2.;
     Cptt_Pt->next->cptt.points[10]=
      Cptt_Pt->next->cptt.points[9]/2.+Cptt_Pt->next->cptt.points[11]/2.;
     Cptt_Pt->next->cptt.points[13]=
      Cptt_Pt->next->cptt.points[14]*3./4.+Cptt_Pt->next->cptt.points[1]/4.;
     Cptt_Pt->next->cptt.points[15]=
      Cptt_Pt->next->cptt.points[14]*3./4.+Cptt_Pt->next->cptt.points[3]/4.;
     Cptt_Pt->next->cptt.points[17]=
      Cptt_Pt->next->cptt.points[18]*3./4.+Cptt_Pt->next->cptt.points[5]/4.;
     Cptt_Pt->next->cptt.points[19]=
      Cptt_Pt->next->cptt.points[18]*3./4.+Cptt_Pt->next->cptt.points[7]/4.;
     Cptt_Pt->next->cptt.points[21]=
      Cptt_Pt->next->cptt.points[22]*3./4.+Cptt_Pt->next->cptt.points[9]/4.;
     Cptt_Pt->next->cptt.points[23]=
      Cptt_Pt->next->cptt.points[22]*3./4.+Cptt_Pt->next->cptt.points[11]/4.;
     Cptt_Pt->next->cptt.points[27]=
      Cptt_Pt->next->cptt.points[22]/2.+Cptt_Pt->next->cptt.points[14]/2.;
     Cptt_Pt->next->cptt.points[28]=
      Cptt_Pt->next->cptt.points[14]/2.+Cptt_Pt->next->cptt.points[18]/2.;
     Cptt_Pt->next->cptt.points[29]=
      Cptt_Pt->next->cptt.points[18]/2.+Cptt_Pt->next->cptt.points[22]/2.;
     Cptt_Pt->next->cptt.points[24]=
      piano(x41,y41,
	    (x4+x41)/2.,(y4+y41)/2.,Cptt_Pt->next->cptt.points[27],
	    (x41+x12)/2.,(y41+y12)/2.,Cptt_Pt->next->cptt.points[13],
	    (x41+x31)/2.,(y41+y31)/2.,Cptt_Pt->next->cptt.points[23]);
     Cptt_Pt->next->cptt.points[25]=
      piano(x42,y42,
	    (x4+x42)/2.,(y4+y42)/2.,Cptt_Pt->next->cptt.points[28],
	    (x42+x12)/2.,(y42+y12)/2.,Cptt_Pt->next->cptt.points[15],
	    (x42+x23)/2.,(y42+y23)/2.,Cptt_Pt->next->cptt.points[17]);
     Cptt_Pt->next->cptt.points[26]=
      piano(x43,y43,
	    (x4+x43)/2.,(y4+y43)/2.,Cptt_Pt->next->cptt.points[29],
	    (x43+x23)/2.,(y43+y23)/2.,Cptt_Pt->next->cptt.points[19],
	    (x43+x31)/2.,(y43+y31)/2.,Cptt_Pt->next->cptt.points[21]);
     Cptt_Pt->next->cptt.points[30]=
      piano(x4,y4,
	    (x4+x41)/2.,(y4+y41)/2.,Cptt_Pt->next->cptt.points[27],
	    (x4+x42)/2.,(y4+y42)/2.,Cptt_Pt->next->cptt.points[28],
	    (x4+x43)/2.,(y4+y43)/2.,Cptt_Pt->next->cptt.points[29]);
    }
    /* end array of points */
    /* computing z-component for global 'minmaxbox' box */
    if (first_time_while_flag==1) /* initializing global 'minmaxbox' box */
      {
       minbox_t[2]=z[iv1]; maxbox_t[2]=z[iv1];
       first_time_while_flag=0;
      }
    for (ss=0;ss<=18+pwsb_mode*12;++ss)   /* Both cases */
      /* to do even if initializing */
      {
       if (Cptt_Pt->next->cptt.points[ss]<minbox_t[2])
		  minbox_t[2]=Cptt_Pt->next->cptt.points[ss];
       else if (Cptt_Pt->next->cptt.points[ss]>maxbox_t[2])
		  maxbox_t[2]=Cptt_Pt->next->cptt.points[ss];
      }
    /* analogy with function 'minmaxbox' */
    if ( (maxbox_t[2]-minbox_t[2]) < MINAMPBOX)
    {
     minbox_t[2]=minbox_t[2]-MINAMPBOX/2;
     maxbox_t[2]=maxbox_t[2]+MINAMPBOX/2;
    }
    Cptt_Pt=Cptt_Pt->next;
    Triang_Pt=Triang_Pt->next;
   } /* end of main 'while' loop */
 /* closing the tail of the list of control points */
 Cptt_Pt->next=NULL;
 /* deleting dummy cell */
 Cptt_Pt=Punti_controllo->next;
 free(Punti_controllo);
 Punti_controllo=Cptt_Pt;
 /* for menu setting */
 cptbpresenza=0;
 cpttpresenza=2;
 writemenu();
}


void triq18()
{
 /* using globals x y z zx zy zxx zxy zyy Triangolazione Punti_Controllo
 ** lista_ass_t
 */
 struct triang_list *Triang_Pt;
 struct cptt_list *Cptt_Pt;
 double q1,q1x,q1y,q1xx,q1xy,q1yy,q2,q2x,q2y,q2xx,q2xy,q2yy,
	q3,q3x,q3y,q3xx,q3xy,q3yy;
 double e1x,e1y,e2x,e2y,e3x,e3y,t1,t2,t3;
 double c01,c11,c31,c41,c02,c12,c32,c42,c03,c13,c33,c43;
 long int iv1,iv2,iv3;
 int s,ss;
 int first_time_while_flag=1; /*to initialize global 'minmaxbox' box */

 if (itpttpresenza==0) return;

 /* setting the list of indices */
 for (s=0;s<21;++s) ass(s,5,&(lista_ass_t[s]));
 printf("Wait for computation.\n");
 free_cptt_list(Punti_controllo); Punti_controllo=NULL;
 Triang_Pt=Triangolazione;
 /* Creating dummy cell and initializing local counter */
 Punti_controllo=malloc(sizeof(struct cptt_list));
 if (Punti_controllo==NULL) {printf("Not enough memory.\n"); exit(1);}
 Cptt_Pt=Punti_controllo;
 while (Triang_Pt!=NULL)
   {
    iv1=Triang_Pt->triangolo.v1;
    iv2=Triang_Pt->triangolo.v2;
    iv3=Triang_Pt->triangolo.v3;
    e1x=x[iv3]-x[iv2]; e1y=y[iv3]-y[iv2];
    e2x=x[iv1]-x[iv3]; e2y=y[iv1]-y[iv3];
    e3x=x[iv2]-x[iv1]; e3y=y[iv2]-y[iv1];
    q1=z[iv1]; q1x=zx[iv1]; q1y=zy[iv1];
    q1xx=zxx[iv1]; q1xy=zxy[iv1]; q1yy=zyy[iv1];
    q2=z[iv2]; q2x=zx[iv2]; q2y=zy[iv2];
    q2xx=zxx[iv2]; q2xy=zxy[iv2]; q2yy=zyy[iv2];
    q3=z[iv3]; q3x=zx[iv3]; q3y=zy[iv3];
    q3xx=zxx[iv3]; q3xy=zxy[iv3]; q3yy=zyy[iv3];
    t1=-(e1x*e3x+e1y*e3y)/(pow(e1x,2.)+pow(e1y,2.));
    t2=-(e2x*e1x+e2y*e1y)/(pow(e2x,2.)+pow(e2y,2.));
    t3=-(e3x*e2x+e3y*e2y)/(pow(e3x,2.)+pow(e3y,2.));
    /* creo nuova cella accessibile con Cptt_Pt->next*/
    Cptt_Pt->next=malloc(sizeof(struct cptt_list));
    if (Cptt_Pt->next==NULL) {printf("Not enough memory.\n"); exit(1);} 
    Cptt_Pt->next->cptt.itpmethod=3;
    /* array dei punti */
    Cptt_Pt->next->cptt.points[20]=q1;
    Cptt_Pt->next->cptt.points[5]=q2;
    Cptt_Pt->next->cptt.points[0]=q3;
    Cptt_Pt->next->cptt.points[19]= (e3x*q1x+e3y*q1y)/5.+q1;
    Cptt_Pt->next->cptt.points[18]=-(e2x*q1x+e2y*q1y)/5.+q1;
    Cptt_Pt->next->cptt.points[10]=-(e3x*q2x+e3y*q2y)/5.+q2;
    Cptt_Pt->next->cptt.points[4]= +(e1x*q2x+e1y*q2y)/5.+q2;
    Cptt_Pt->next->cptt.points[1]= -(e1x*q3x+e1y*q3y)/5.+q3;
    Cptt_Pt->next->cptt.points[6]= +(e2x*q3x+e2y*q3y)/5.+q3;
    Cptt_Pt->next->cptt.points[17]=
	   (pow(e3x,2)*q1xx+2*e3x*e3y*q1xy+pow(e3y,2)*q1yy)/20.
	   +2*(Cptt_Pt->next->cptt.points[19])-q1;
    Cptt_Pt->next->cptt.points[14]=
	   (pow(e3x,2)*q2xx+2*e3x*e3y*q2xy+pow(e3y,2)*q2yy)/20.
	   +2*(Cptt_Pt->next->cptt.points[10])-q2;
    Cptt_Pt->next->cptt.points[3]=
	   (pow(e1x,2)*q2xx+2*e1x*e1y*q2xy+pow(e1y,2)*q2yy)/20.
	   +2*(Cptt_Pt->next->cptt.points[4])-q2;
    Cptt_Pt->next->cptt.points[2]=
	   (pow(e1x,2)*q3xx+2*e1x*e1y*q3xy+pow(e1y,2)*q3yy)/20.
	   +2*(Cptt_Pt->next->cptt.points[1])-q3;
    Cptt_Pt->next->cptt.points[11]=
	   (pow(e2x,2)*q3xx+2*e2x*e2y*q3xy+pow(e2y,2)*q3yy)/20.
	   +2*(Cptt_Pt->next->cptt.points[6])-q3;
    Cptt_Pt->next->cptt.points[15]=
	   (pow(e2x,2)*q1xx+2*e2x*e2y*q1xy+pow(e2y,2)*q1yy)/20.
	   +2*(Cptt_Pt->next->cptt.points[18])-q1;
    Cptt_Pt->next->cptt.points[16]=
	   -(e3x*e2x*q1xx+(e3x*e2y+e3y*e2x)*q1xy+e3y*e2y*q1yy)/20.
	   +(Cptt_Pt->next->cptt.points[19])
	   +(Cptt_Pt->next->cptt.points[18])-q1;
    Cptt_Pt->next->cptt.points[9]=
	   -(e3x*e1x*q2xx+(e3x*e1y+e3y*e1x)*q2xy+e3y*e1y*q2yy)/20.
	   +(Cptt_Pt->next->cptt.points[10])
	   +(Cptt_Pt->next->cptt.points[4])-q2;
    Cptt_Pt->next->cptt.points[7]=
	   -(e1x*e2x*q3xx+(e1x*e2y+e1y*e2x)*q3xy+e1y*e2y*q3yy)/20.
	   +(Cptt_Pt->next->cptt.points[1])
	   +(Cptt_Pt->next->cptt.points[6])-q3;
    c01=Cptt_Pt->next->cptt.points[10]
	-(1-t1)*q2-t1*Cptt_Pt->next->cptt.points[4];
    c11=Cptt_Pt->next->cptt.points[9]
	-(1-t1)*Cptt_Pt->next->cptt.points[4]
	-t1*Cptt_Pt->next->cptt.points[3];
    c31=Cptt_Pt->next->cptt.points[7]
	-(1-t1)*Cptt_Pt->next->cptt.points[2]
	-t1*Cptt_Pt->next->cptt.points[1];
    c41=Cptt_Pt->next->cptt.points[6]
	-(1-t1)*Cptt_Pt->next->cptt.points[1]-t1*q3;
    c02=-(1-t2)*q3-t2*Cptt_Pt->next->cptt.points[6]
	+Cptt_Pt->next->cptt.points[1];
    c12=-(1-t2)*Cptt_Pt->next->cptt.points[6]
	-t2*Cptt_Pt->next->cptt.points[11]+Cptt_Pt->next->cptt.points[7];
    c32=-(1-t2)*Cptt_Pt->next->cptt.points[15]
	-t2*Cptt_Pt->next->cptt.points[18]+Cptt_Pt->next->cptt.points[16];
    c42=-(1-t2)*Cptt_Pt->next->cptt.points[18]
	-t2*q1+Cptt_Pt->next->cptt.points[19];
    c03=-t3*Cptt_Pt->next->cptt.points[19]+Cptt_Pt->next->cptt.points[18]
	-(1-t3)*q1;
    c13=-t3*Cptt_Pt->next->cptt.points[17]+Cptt_Pt->next->cptt.points[16]
	-(1-t3)*Cptt_Pt->next->cptt.points[19];
    c33=-t3*Cptt_Pt->next->cptt.points[10]+Cptt_Pt->next->cptt.points[9]
	-(1-t3)*Cptt_Pt->next->cptt.points[14];
    c43=-t3*q2+Cptt_Pt->next->cptt.points[4]
	-(1-t3)*Cptt_Pt->next->cptt.points[10];
    Cptt_Pt->next->cptt.points[8]=
	   (-c01+4*c11+4*c31-c41)/6.
	   +(1-t1)*Cptt_Pt->next->cptt.points[3]
	   +t1*    Cptt_Pt->next->cptt.points[2];
    Cptt_Pt->next->cptt.points[12]=
	   (-c02+4*c12+4*c32-c42)/6.
	   +(1-t2)*Cptt_Pt->next->cptt.points[11]
	   +t2*    Cptt_Pt->next->cptt.points[15];
    Cptt_Pt->next->cptt.points[13]=
	   (-c03+4*c13+4*c33-c43)/6.
	   +(1-t3)*Cptt_Pt->next->cptt.points[17]
	   +t3*    Cptt_Pt->next->cptt.points[14];
    /* end array of points */
    /* computing z-component for global 'minmaxbox' box */
    if (first_time_while_flag==1) /* initializing global 'minmaxbox' box */
      {
       minbox_t[2]=q1; maxbox_t[2]=q1;
       first_time_while_flag=0;
      }
    for (ss=0;ss<21;++ss) /* to do even if initializing */
      {
       if (Cptt_Pt->next->cptt.points[ss]<minbox_t[2])
		  minbox_t[2]=Cptt_Pt->next->cptt.points[ss];
       else if (Cptt_Pt->next->cptt.points[ss]>maxbox_t[2])
		  maxbox_t[2]=Cptt_Pt->next->cptt.points[ss];
      }
    /* Analogy with 'minmaxbox' function */
    if ( (maxbox_t[2]-minbox_t[2]) < MINAMPBOX)
    {
     minbox_t[2]=minbox_t[2]-MINAMPBOX/2;
     maxbox_t[2]=maxbox_t[2]+MINAMPBOX/2;
    }
    Cptt_Pt=Cptt_Pt->next;
    Triang_Pt=Triang_Pt->next;
   }
 /* closing the tail for the list of control points */
 Cptt_Pt->next=NULL;
 /* deleting dummy cell */
 Cptt_Pt=Punti_controllo->next;
 free(Punti_controllo);
 Punti_controllo=Cptt_Pt;
 /* for menu setting */
 cptbpresenza=0;
 cpttpresenza=3;
 writemenu();
}

void triinputitptb()
/* Using globals numero_triangoli n l err dim */
 {
  int i,j,s;
  double passoloc,temp;
  char scelta;
  char nome_file[NAME_FILE_BUFFER_LEN];  
  FILE *input_file;
  
  strcpy(MainName,"LABSUP3D: Global Bezier ");
  printf("Do I read from file 3D points? (y/n)\n");
  fflush(stdin);
  scanf(" %c",&scelta);
  if (scelta=='y')
    {
     printf("Input file name: \n");
     scanf(" %s",nome_file);
     input_file=fopen(nome_file,"r");
     if (input_file==NULL)
       {
	printf("Cannot open file: %s \n",nome_file);
	exit(1);
       }
     strcat(MainName,nome_file);
     dim=3;
     l=0; /* counter for the number of points read */
     while (!feof(input_file))
       {
	fscanf(input_file," %lf %lf %lf %lf %lf %lf ",&u[l],&v[l],&temp,
	       &(itpt[l][0]),&(itpt[l][1]),&(itpt[l][2]) ); 
	/* w barycentric coordinate is not considered */
	l++;
       }
     fclose(input_file);
     /* TEST SU l */
     if (l>MAXORD || l<2) return;
     n=1;
     while ( l>(n+1)*(n+2)/2 ) n=n+1;
     if (l<(n+1)*(n+2)/2)
     {
      printf("There must be 1/2*(n+1)*(n+2) interpolation points for n>0\n");
      printf("\nHit a button\n");
      fflush(stdin);
      getchar();
      writemenu();
      return;
     }
     l=l-1; /* Numeration from 0 in order to have exactly 'l' points */
     for (s=0;s<=l;++s)
      ass(s,n,&lista_ass[s]);
    } /* end input file case */
  else /* direct input */
    {
     printf("Number of interpolation points:\n");
     scanf("%d",&l);
     if (l>MAXORD || l<2) return;
     n=1;
     while ( l>(n+1)*(n+2)/2 ) n=n+1;
     if (l<(n+1)*(n+2)/2)
     {
      printf("There must be 1/2*(n+1)*(n+2) interpolation points for n>0\n");
      printf("\nHit a button\n");
      fflush(stdin);
      getchar();
      writemenu();
      return;
     }
     l=l-1; /* numeration from 0 in order to have exactly 'l' points */
     for (s=0;s<=l;++s)
      ass(s,n,&lista_ass[s]);
     dim=3;
     printf("\n-  u  - Parametrization with uniform u-v-w");
     printf("\n        over standard triangle with lexicographic order");
     printf("\n-other- Parametrization with user u-v-w \n");
     scanf(" %c",&scelta);
     if (scelta=='u')
     {
       passoloc=1./(double) n;
       s=0; temp=0.;
       for (i=0;i<=n;++i)
       {
	for (j=0;j<=n-i;++j)
	{
	 u[s]=temp;
	 if (j==0) v[s]=0.; else v[s]=v[s-1]+passoloc;
	 s++;
	}
	temp=temp+passoloc;
       }
     }
     else
     {
      printf("\nGive %d u parameters for interpolation points: \n",l+1);
      err=getmat(1,l+1,u,MAXORD);
      if (err==1) exit(0);
      printf("\nGive %d v parameters for interpolation points: \n",l+1);
      err=getmat(1,l+1,v,MAXORD);
      if (err==1) exit(0);
     }
     printf("\nBarycentric coordinates: u\n\n");
     err=putmat(1,l+1,u,MAXORD,amp,prec);
     printf("Barycentric coordinates v:\n\n");
     err=putmat(1,l+1,v,MAXORD,amp,prec);
     printf("Barycentric coordinates w: w\n\n");
     for (s=0;s<=l;++s)
      printf("%*.*f",amp,prec,1-u[s]-v[s]);
     printf("\n\n");
     printf("Give %d %d-dimensional interpolation points:\n",l+1,dim);
     err=getmat(l+1,dim,itpt,MAXDIM);
     printf("\n");
    }/* end of 'else' for direct input */
  /* Freeing memory even for C1 surfaces data */
  free_triang_list(Triangolazione_Glob); Triangolazione_Glob=NULL;
  free_triang_list(Triangolazione_Loc); Triangolazione_Loc=NULL;
  free_edge_list(Inviluppo_convesso); Inviluppo_convesso=NULL;
  free_cptt_list(Punti_controllo); Punti_controllo=NULL;
  if (calcpt!=NULL) {free(calcpt); calcpt=NULL;}
  /* End freeing memory */
  itptbpresenza=1;
  cptbpresenza=0; 
  cpttpresenza=0;
  draw_flag=0;
  numero_triangoli=1; 
  writemenu();
 }
 

void trimodificaitptb()
 {
  int i;

  if (itptbpresenza==0) return;
  do
  {
   printf("Give the number of the point you want modify: ");
   scanf(" %d",&i);
  }
  while (i<0 || i>l);
  printf("\nCurrent value:\n");
  err=putmat(1,dim,&itpt[i][0],MAXDIM,amp,prec);
  printf("New value:\n");
  err=getmat(1,dim,&itpt[i][0],MAXDIM);
  if (err==1) exit(0);
  err=putmat(1,dim,&itpt[i][0],MAXDIM,amp,prec);
  printf("Current value for u parameter: %*.*f\n",amp,prec,u[i]);
  printf("Current value for v parameter: %*.*f\n",amp,prec,v[i]);
  printf("\nNew value for u parameter: ");
  scanf(" %lf",&u[i]);
  printf("\n%*.*f\n",amp,prec,u[i]);
  printf("\nNew value for v parameter: ");
  scanf(" %lf",&v[i]);
  printf("\n%*.*f\n",amp,prec,v[i]);
  strcpy(MainName,"LABSUP3D: Global Bezier ");
  cptbpresenza=0;
  writemenu();
 }


void trioutputitptb()
  {
   int s,j;

   if (itptbpresenza==0) return;
   fprintf(output_ris,"Interpolation points:\n\n");
   fprintf(output_ris,"Number: %d\n",l+1);
   fprintf(output_ris,"Values and u v w parameters:\n\n");
   for(s=0;s<=l;++s)
   {
    fprintf(output_ris,"N.%5d ",s);
    for (j=0;j<dim;++j)
     fprintf(output_ris,"%*.*f",amp,prec,itpt[s][j]);
     fprintf(output_ris,"  %*.*f%*.*f%*.*f",amp,prec,u[s],amp,prec,v[s],
				amp,prec,1-u[s]-v[s]);
    fprintf(output_ris,"\n");
   }
   writemenu();
  }


void trioutputcptb() 
  {
   int s,j;
   
   if (cptbpresenza==0) return;
   fprintf(output_ris,"Control Bezier points:\n\n");
   fprintf(output_ris,"Number: %d\n",l+1);
   fprintf(output_ris,"Values:\n");
   for(s=0;s<=l;++s)
   {
    fprintf(output_ris,"N.%3d%3d%3d ",
	    lista_ass[s].i,lista_ass[s].j,lista_ass[s].k);
    for (j=0;j<dim;++j)
     fprintf(output_ris,"%*.*f",amp,prec,cpt[s][j]);
    fprintf(output_ris,"\n");
   }
   writemenu();
  }

void triinputitptt()
{
 char nome_dat_file[NAME_FILE_BUFFER_LEN], nome_der_file[NAME_FILE_BUFFER_LEN];
 FILE *datfile, *derfile; 
 long int i;

 printf("Input data file: \n");
 scanf(" %s",nome_dat_file);
 datfile=fopen(nome_dat_file,"r");
 if (datfile==NULL)
   {
    printf("Cannot open file: %s \n",nome_dat_file);
    exit(1);
   }
 /* Lettura dati */
 if (!feof(datfile))
   {
    fscanf(datfile," %lf %lf %lf ",&x[0],&y[0],&z[0]); /* spazi essenziali */
    minbox_t_glob[0]=x[0]; maxbox_t_glob[0]=x[0];
    minbox_t_glob[1]=y[0]; maxbox_t_glob[1]=y[0];
   }
 else
   {
    printf("Data file in a no correct format.\n");
    exit(1);
   }
 i=1;
 while (!feof(datfile))
   {
    if (i==MAX_NUM_NODI)
      {
       printf("Too many input data.\n");
       exit(1);
      }
    fscanf(datfile," %lf %lf %lf ",&x[i],&y[i],&z[i]);
    if (x[i]<minbox_t_glob[0]) minbox_t_glob[0]=x[i]; 
      else if (x[i]>maxbox_t_glob[0]) maxbox_t_glob[0]=x[i];
    if (y[i]<minbox_t_glob[1]) minbox_t_glob[1]=y[i]; 
      else if (y[i]>maxbox_t_glob[1]) maxbox_t_glob[1]=y[i];   
    i++;
   }
 minbox_t[0]=minbox_t_glob[0]; maxbox_t[0]=maxbox_t_glob[0];
 minbox_t[1]=minbox_t_glob[1]; maxbox_t[1]=maxbox_t_glob[1];
 fclose(datfile);
 numero_punti=i; /* points are numbered from 0 */
 /* security for having at least three input points for 
    the global variable 'numero_punti' */
 if (numero_punti<3)
   {
    printf("I need at least three input points.\n");
    exit(1);
   } 
 /* Freeing memory */
 free_triang_list(Triangolazione_Glob); Triangolazione_Glob=NULL;
 free_triang_list(Triangolazione_Loc); Triangolazione_Loc=NULL;
 free_edge_list(Inviluppo_convesso); Inviluppo_convesso=NULL;
 free_cptt_list(Punti_controllo); Punti_controllo=NULL;
 if (calcpt!=NULL) {free(calcpt); calcpt=NULL;}
 /* End freeing memory */
 printf("Triangulation file name: \n");
 scanf(" %s",nome_tri_file);
 read_triang(); 
 printf("Derivative file name: \n");
 scanf(" %s",nome_der_file);
 printf("Wait for reading. \n");
 derfile=fopen(nome_der_file,"r");
 if (derfile==NULL)
 {
  printf("Cannot open file: %s \n",nome_der_file);
  exit(1);
 }
 /* Data reading */
 for (i=0; i<numero_punti; i++)
   {
    fscanf(derfile," %lf %lf %lf %lf %lf ",
	   &zx[i],&zy[i],&zxx[i],&zxy[i],&zyy[i]);
   }
 fclose(derfile);
 strcpy(MainName,"LABSUP3D: ");
 strcat(MainName,nome_dat_file);
 strcat(MainName," ");
 strcat(MainName,nome_tri_file);
 strcat(MainName," ");
 strcat(MainName,nome_der_file);   
 itpttpresenza=1;
 cptbpresenza=0;
 cpttpresenza=0;
 draw_flag=0;
 writemenu();
}

void trimagnify()
{
 double magnify_factor;
 long int i;
 int j,ntriang;
 struct cptt_list *Cptt_Pt;

 if (itpttpresenza==0 || itptbpresenza==1 || cptbpresenza==1) return; 
 printf("Magnifying factor for z values and derivatives: \n");
 scanf(" %lf",&magnify_factor);
 printf("Wait for computations\n");
 /* Loop */
 for (i=0; i<numero_punti; i++)
 {
  z[i]=z[i]*magnify_factor;
  zx[i]=zx[i]*magnify_factor;
  zy[i]=zy[i]*magnify_factor;
  zxx[i]=zxx[i]*magnify_factor;
  zxy[i]=zxy[i]*magnify_factor;
  zyy[i]=zyy[i]*magnify_factor;
 }
 if (cpttpresenza!=0) /* this value varies depending on the method used */
 {
  Cptt_Pt=Punti_controllo;
  while (Cptt_Pt!=NULL)
  {
   switch(Cptt_Pt->cptt.itpmethod)
   {
    case 1: /* Clough-Tocher */
     for (j=1;j<=19;++j) Cptt_Pt->cptt.points[j]*=magnify_factor;
     ntriang=3;
     break;
    case 2: /* Powell-Sabin - Suddivisione A */
     for (j=0;j<19;++j) Cptt_Pt->cptt.points[j]*=magnify_factor;
     ntriang=12; /* Dynamical declaration with the greatest value, 12 */
     break;
    case 3: /* Q18 */
     for (j=0;j<21;++j) Cptt_Pt->cptt.points[j]*=magnify_factor;
     ntriang=1;     
     break;
    case 4: /* Powell-Sabin - B Subdivision */ 
     for (j=0;j<31;++j) Cptt_Pt->cptt.points[j]*=magnify_factor;
     ntriang=12;
     break;  
   }
   Cptt_Pt=Cptt_Pt->next;
  }
  minbox_t[2]=minbox_t[2]*magnify_factor;
  maxbox_t[2]=maxbox_t[2]*magnify_factor;
  /* Analogy with 'minmaxbox' function */
  if ( (maxbox_t[2]-minbox_t[2]) < MINAMPBOX)
  {
   minbox_t[2]=minbox_t[2]-MINAMPBOX/2;
   maxbox_t[2]=maxbox_t[2]+MINAMPBOX/2;
  }
  /* Setting vrp to the center of 'minmaxbox' box */
  for (i=0;i<dim;++i) vrp[i]=(minbox_t[i]+maxbox_t[i])/2.;
  parsize=sqrt( pow((maxbox_t[0]-minbox_t[0]),2)+pow((maxbox_t[1]-minbox_t[1]),2)+
		pow((maxbox_t[2]-minbox_t[2]),2))/2.;      
  if (draw_flag==3) glsetmyoptions(parsize,vrp,vpn,vuv,0);
  /* For example the triangulation (draw_flag==2) could be viewed */
  /* calcpt if not null refers to the points currently computed */ 
  if (calcpt!=NULL)
    for (i=0;i<numpt*numero_triangoli*ntriang;++i) calcpt[3*i+2]*=magnify_factor;
 }
 writemenu();
}

void trioutputitptt()
{
 long int i;
 char scelta;

 if (itpttpresenza==0 || cptbpresenza==1) return;
 printf("-    1- Only points\n");
 printf("-    2- Only derivatives\n");
 printf("-no 1 nor 2- Points and derivatives\n");
 fflush(stdin);
 scelta=getchar();
 if (scelta!='2')
 { 
  fprintf(output_ris,"\nInterpolation points:\n\n");
  fprintf(output_ris,"Number: %d\n",numero_punti);
   for(i=0;i<numero_punti;++i)
   {
     fprintf(output_ris,"%5ld %*.*f %*.*f% *.*f\n",
	     i,amp,prec,x[i],amp,prec,y[i],amp,prec,z[i]);
   }
 }
 if (scelta!='1')
 {
  fprintf(output_ris,"\nDerivative estimation given:\n\n");
  for(i=0;i<numero_punti;++i)
  {
    fprintf(output_ris,"%5ld %*.*f %*.*f% *.*f %*.*f %*.*f\n",
	    i,amp,prec,zx[i],amp,prec,zy[i],
	    amp,prec,zxx[i],amp,prec,zxy[i],amp,prec,zyy[i]);
  }
 }
 writemenu();
}

void trioutputcptt()
{
 struct triang_list *Triang_Pt;
 struct cptt_list *Cptt_Pt;
 long int iv1,iv2,iv3;
 int indexes[10]; 
 /*For 3-degree polynomials and therefore for 2-degree ones (micro-splits)*/
 int s;
 double x4,y4,x12,y12,x23,y23,x31,y31,x41,y41,x42,y42,x43,y43;

 if (cpttpresenza==0 || cptbpresenza==1) return;

 /* Results depend from the method and from the triangle chosen */
 Triang_Pt=Triangolazione;
 Cptt_Pt=Punti_controllo;
 while (Triang_Pt!=NULL)
  {
   iv1=Triang_Pt->triangolo.v1;
   iv2=Triang_Pt->triangolo.v2;
   iv3=Triang_Pt->triangolo.v3;
   fprintf(output_ris,"\nTriangle of vertices %5ld %5ld %5ld:\n",iv1,iv2,iv3);
   switch(Cptt_Pt->cptt.itpmethod)
    {
    case 1: /* Clough-Tocher */
	    x4=(x[iv1]+x[iv2]+x[iv3])/3.;
	    y4=(y[iv1]+y[iv2]+y[iv3])/3.;
	    fprintf(output_ris,
		    "Method: Clough-Tocher with linear normal derivatives\n\n");
	    fprintf(output_ris,
		    "Micro-triangle V4=BARICENTRO V1 V2\n");
	    indexes[0]=2;
	    indexes[1]=9;
	    indexes[2]=4;
	    indexes[3]=1;
	    indexes[4]=8;
	    indexes[5]=13;
	    indexes[6]=5;
	    indexes[7]=17;
	    indexes[8]=16;
	    indexes[9]=19;
	    for(s=0;s<10;++s)
	    {
	     fprintf(output_ris,"N.%3d%3d%3d ",
		     lista_ass_t[s].i,
		     lista_ass_t[s].j,
		     lista_ass_t[s].k);
	     fprintf(output_ris,"%*.*f %*.*f %*.*f\n",
		     amp,prec,x4    *lista_ass_t[s].i/3.+
			      x[iv1]*lista_ass_t[s].j/3.+
			      x[iv2]*lista_ass_t[s].k/3.,
		     amp,prec,y4    *lista_ass_t[s].i/3.+
			      y[iv1]*lista_ass_t[s].j/3.+
			      y[iv2]*lista_ass_t[s].k/3.,
		     amp,prec,Cptt_Pt->cptt.points[indexes[s]]);
	    }
	    fprintf(output_ris,
		    "Micro-triangle V4=BARICENTRO V2 V3\n");
	    indexes[0]=3;
	    indexes[1]=12;
	    indexes[2]=7;
	    indexes[3]=2;
	    indexes[4]=11;
	    indexes[5]=14;
	    indexes[6]=8;
	    indexes[7]=18;
	    indexes[8]=17;
	    indexes[9]=19;
	    for(s=0;s<10;++s)
	    {
	     fprintf(output_ris,"N.%3d%3d%3d ",
		     lista_ass_t[s].i,
		     lista_ass_t[s].j,
		     lista_ass_t[s].k);
	     fprintf(output_ris,"%*.*f %*.*f %*.*f\n",
		     amp,prec,x4    *lista_ass_t[s].i/3.+
			      x[iv2]*lista_ass_t[s].j/3.+
			      x[iv3]*lista_ass_t[s].k/3.,
		     amp,prec,y4    *lista_ass_t[s].i/3.+
			      y[iv2]*lista_ass_t[s].j/3.+
			      y[iv3]*lista_ass_t[s].k/3.,
		     amp,prec,Cptt_Pt->cptt.points[indexes[s]]);
	    }
	    fprintf(output_ris,
		    "Micro-triangle V4=BARICENTRO V3 V1\n");
	    indexes[0]=1;
	    indexes[1]=6;
	    indexes[2]=10;
	    indexes[3]=3;
	    indexes[4]=5;
	    indexes[5]=15;
	    indexes[6]=11;
	    indexes[7]=16;
	    indexes[8]=18;
	    indexes[9]=19;
	    for(s=0;s<10;++s)
	    {
	     fprintf(output_ris,"N.%3d%3d%3d ",
		     lista_ass_t[s].i,
		     lista_ass_t[s].j,
		     lista_ass_t[s].k);
	     fprintf(output_ris,"%*.*f %*.*f %*.*f\n",
		     amp,prec,x4    *lista_ass_t[s].i/3.+
			      x[iv3]*lista_ass_t[s].j/3.+
			      x[iv1]*lista_ass_t[s].k/3.,
		     amp,prec,y4    *lista_ass_t[s].i/3.+
			      y[iv3]*lista_ass_t[s].j/3.+
			      y[iv1]*lista_ass_t[s].k/3.,
		     amp,prec,Cptt_Pt->cptt.points[indexes[s]]);
	    }
	    break;
    case 2: /* Powell-Sabin - A Subdivision */
	    circoscrivi(x[iv1],y[iv1],x[iv2],y[iv2],x[iv3],y[iv3],
			&x4,&y4,NULL);
	    x12=(x[iv1]+x[iv2])/2.; y12=(y[iv1]+y[iv2])/2.;
	    x23=(x[iv2]+x[iv3])/2.; y23=(y[iv2]+y[iv3])/2.;
	    x31=(x[iv3]+x[iv1])/2.; y31=(y[iv3]+y[iv1])/2.;
	    fprintf(output_ris,"Method: Powell-Sabin (6 triangles) \n\n");
	    fprintf(output_ris,
		    "Micro-triangle V4=CIRCUMCENTER V1 V12\n");
	    indexes[0]=2;
	    indexes[1]=1;
	    indexes[2]=0;
	    indexes[3]=13;
	    indexes[4]=12;
	    indexes[5]=18;
	    for(s=0;s<6;++s)
	    {
	     fprintf(output_ris,"N.%3d%3d%3d ",
		     lista_ass_t[s].i,
		     lista_ass_t[s].j,
		     lista_ass_t[s].k);
	     fprintf(output_ris,"%*.*f %*.*f %*.*f\n",
		     amp,prec,x4    *lista_ass_t[s].i/2.+
			      x[iv1]*lista_ass_t[s].j/2.+
			      x12   *lista_ass_t[s].k/2.,
		     amp,prec,y4    *lista_ass_t[s].i/2.+
			      y[iv1]*lista_ass_t[s].j/2.+
			      y12   *lista_ass_t[s].k/2.,
		     amp,prec,Cptt_Pt->cptt.points[indexes[s]]);
	    }
	    fprintf(output_ris,
		    "Micro-triangle V4=CIRCUMCENTER V12 V2\n");
	    indexes[0]=4;
	    indexes[1]=3;
	    indexes[2]=2;
	    indexes[3]=14;
	    indexes[4]=13;
	    indexes[5]=18;
	    for(s=0;s<6;++s)
	    {
	     fprintf(output_ris,"N.%3d%3d%3d ",
		     lista_ass_t[s].i,
		     lista_ass_t[s].j,
		     lista_ass_t[s].k);
	     fprintf(output_ris,"%*.*f %*.*f %*.*f\n",
		     amp,prec,x4    *lista_ass_t[s].i/2.+
			      x12   *lista_ass_t[s].j/2.+
			      x[iv2]*lista_ass_t[s].k/2.,
		     amp,prec,y4    *lista_ass_t[s].i/2.+
			      y12   *lista_ass_t[s].j/2.+
			      y[iv2]*lista_ass_t[s].k/2.,
		     amp,prec,Cptt_Pt->cptt.points[indexes[s]]);
	    }
	    fprintf(output_ris,
		    "Micro-triangle V4=CIRCUMCENTER V2 V23\n");
	    indexes[0]=6;
	    indexes[1]=5;
	    indexes[2]=4;
	    indexes[3]=15;
	    indexes[4]=14;
	    indexes[5]=18;
	    for(s=0;s<6;++s)
	    {
	     fprintf(output_ris,"N.%3d%3d%3d ",
		     lista_ass_t[s].i,
		     lista_ass_t[s].j,
		     lista_ass_t[s].k);
	     fprintf(output_ris,"%*.*f %*.*f %*.*f\n",
		     amp,prec,x4    *lista_ass_t[s].i/2.+
			      x[iv2]*lista_ass_t[s].j/2.+
			      x23   *lista_ass_t[s].k/2.,
		     amp,prec,y4    *lista_ass_t[s].i/2.+
			      y[iv2]*lista_ass_t[s].j/2.+
			      y23   *lista_ass_t[s].k/2.,
		     amp,prec,Cptt_Pt->cptt.points[indexes[s]]);
	    }
	    fprintf(output_ris,
		    "Micro-triangle V4=CIRCUMCENTER V23 V3\n");
	    indexes[0]=8;
	    indexes[1]=7;
	    indexes[2]=6;
	    indexes[3]=16;
	    indexes[4]=15;
	    indexes[5]=18;
	    for(s=0;s<6;++s)
	    {
	     fprintf(output_ris,"N.%3d%3d%3d ",
		     lista_ass_t[s].i,
		     lista_ass_t[s].j,
		     lista_ass_t[s].k);
	     fprintf(output_ris,"%*.*f %*.*f %*.*f\n",
		     amp,prec,x4    *lista_ass_t[s].i/2.+
			      x23   *lista_ass_t[s].j/2.+
			      x[iv3]*lista_ass_t[s].k/2.,
		     amp,prec,y4    *lista_ass_t[s].i/2.+
			      y23   *lista_ass_t[s].j/2.+
			      y[iv3]*lista_ass_t[s].k/2.,
		     amp,prec,Cptt_Pt->cptt.points[indexes[s]]);
	    }
	    fprintf(output_ris,
		    "Micro-triangle V4=CIRCUMCENTER V3 V31\n");
	    indexes[0]=10;
	    indexes[1]=9;
	    indexes[2]=8;
	    indexes[3]=17;
	    indexes[4]=16;
	    indexes[5]=18;
	    for(s=0;s<6;++s)
	    {
	     fprintf(output_ris,"N.%3d%3d%3d ",
		     lista_ass_t[s].i,
		     lista_ass_t[s].j,
		     lista_ass_t[s].k);
	     fprintf(output_ris,"%*.*f %*.*f %*.*f\n",
		     amp,prec,x4    *lista_ass_t[s].i/2.+
			      x[iv3]*lista_ass_t[s].j/2.+
			      x31   *lista_ass_t[s].k/2.,
		     amp,prec,y4    *lista_ass_t[s].i/2.+
			      y[iv3]*lista_ass_t[s].j/2.+
			      y31   *lista_ass_t[s].k/2.,
		     amp,prec,Cptt_Pt->cptt.points[indexes[s]]);
	    }
	    fprintf(output_ris,
		    "Micro-triangle V4=CIRCUMCENTER V31 V1\n");
	    indexes[0]=0;
	    indexes[1]=11;
	    indexes[2]=10;
	    indexes[3]=12;
	    indexes[4]=17;
	    indexes[5]=18;
	    for(s=0;s<6;++s)
	    {
	     fprintf(output_ris,"N.%3d%3d%3d ",
		     lista_ass_t[s].i,
		     lista_ass_t[s].j,
		     lista_ass_t[s].k);
	     fprintf(output_ris,"%*.*f %*.*f %*.*f\n",
		     amp,prec,x4    *lista_ass_t[s].i/2.+
			      x31   *lista_ass_t[s].j/2.+
			      x[iv1]*lista_ass_t[s].k/2.,
		     amp,prec,y4    *lista_ass_t[s].i/2.+
			      y31   *lista_ass_t[s].j/2.+
			      y[iv1]*lista_ass_t[s].k/2.,
		     amp,prec,Cptt_Pt->cptt.points[indexes[s]]);
	    }
	    break;
    case 3: /* Q18 */
	    fprintf(output_ris,"Method: Q18\n\n");
	    for(s=0;s<21;++s)
	    {
	     fprintf(output_ris,"N.%3d%3d%3d ",
		     lista_ass_t[s].i,lista_ass_t[s].j,lista_ass_t[s].k);
	     fprintf(output_ris,"%*.*f %*.*f %*.*f\n",
		     amp,prec,x[iv1]*lista_ass_t[s].i/5.+
			      x[iv2]*lista_ass_t[s].j/5.+
			      x[iv3]*lista_ass_t[s].k/5.,
		     amp,prec,y[iv1]*lista_ass_t[s].i/5.+
			      y[iv2]*lista_ass_t[s].j/5.+
			      y[iv3]*lista_ass_t[s].k/5.,
		     amp,prec,Cptt_Pt->cptt.points[s]);
	    }
	    break;
    case 4: /* Powell-Sabin - Suddivisione B */
	    x4=(x[iv1]+x[iv2]+x[iv3])/3.; y4=(y[iv1]+y[iv2]+y[iv3])/3.;
	    x12=(x[iv1]+x[iv2])/2.; y12=(y[iv1]+y[iv2])/2.;
	    x23=(x[iv2]+x[iv3])/2.; y23=(y[iv2]+y[iv3])/2.;
	    x31=(x[iv3]+x[iv1])/2.; y31=(y[iv3]+y[iv1])/2.;
	    x41=(x31+x12)/2.; y41=(y31+y12)/2.;
	    x42=(x12+x23)/2.; y42=(y12+y23)/2.;
	    x43=(x23+x31)/2.; y43=(y23+y31)/2.;
	    fprintf(output_ris,"Method: Powell-Sabin (12 triangles) \n\n");
	    fprintf(output_ris,
		    "Micro-triangle V4=BARICENTRO V41 V12\n");
	    indexes[0]=2;
	    indexes[1]=13;
	    indexes[2]=24;
	    indexes[3]=14;
	    indexes[4]=27;
	    indexes[5]=30;
	    for(s=0;s<6;++s)
	    {
	     fprintf(output_ris,"N.%3d%3d%3d ",
		     lista_ass_t[s].i,
		     lista_ass_t[s].j,
		     lista_ass_t[s].k);
	     fprintf(output_ris,"%*.*f %*.*f %*.*f\n",
		     amp,prec,x4    *lista_ass_t[s].i/2.+
			      x41   *lista_ass_t[s].j/2.+
			      x12   *lista_ass_t[s].k/2.,
		     amp,prec,y4    *lista_ass_t[s].i/2.+
			      y41   *lista_ass_t[s].j/2.+
			      y12   *lista_ass_t[s].k/2.,
		     amp,prec,Cptt_Pt->cptt.points[indexes[s]]);
	    }
	    fprintf(output_ris,
		    "Micro-triangle V41 V1 V12\n");
	    indexes[0]=2;
	    indexes[1]=1;
	    indexes[2]=0;
	    indexes[3]=13;
	    indexes[4]=12;
	    indexes[5]=24;
	    for(s=0;s<6;++s)
	    {
	     fprintf(output_ris,"N.%3d%3d%3d ",
		     lista_ass_t[s].i,
		     lista_ass_t[s].j,
		     lista_ass_t[s].k);
	     fprintf(output_ris,"%*.*f %*.*f %*.*f\n",
		     amp,prec,x41   *lista_ass_t[s].i/2.+
			      x[iv1]*lista_ass_t[s].j/2.+
			      x12   *lista_ass_t[s].k/2.,
		     amp,prec,y41   *lista_ass_t[s].i/2.+
			      y[iv1]*lista_ass_t[s].j/2.+
			      y12   *lista_ass_t[s].k/2.,
		     amp,prec,Cptt_Pt->cptt.points[indexes[s]]);
	    }
	    fprintf(output_ris,
		    "Micro-triangle V4=BARICENTRO V12 V42\n");
	    indexes[0]=25;
	    indexes[1]=15;
	    indexes[2]=2;
	    indexes[3]=28;
	    indexes[4]=14;
	    indexes[5]=30;
	    for(s=0;s<6;++s)
	    {
	     fprintf(output_ris,"N.%3d%3d%3d ",
		     lista_ass_t[s].i,
		     lista_ass_t[s].j,
		     lista_ass_t[s].k);
	     fprintf(output_ris,"%*.*f %*.*f %*.*f\n",
		     amp,prec,x4    *lista_ass_t[s].i/2.+
			      x12   *lista_ass_t[s].j/2.+
			      x42   *lista_ass_t[s].k/2.,
		     amp,prec,y4    *lista_ass_t[s].i/2.+
			      y12   *lista_ass_t[s].j/2.+
			      y42   *lista_ass_t[s].k/2.,
		     amp,prec,Cptt_Pt->cptt.points[indexes[s]]);
	    }
	    fprintf(output_ris,
		    "Micro-triangle V42 V12 V2\n");
	    indexes[0]=4;
	    indexes[1]=3;
	    indexes[2]=2;
	    indexes[3]=16;
	    indexes[4]=15;
	    indexes[5]=25;
	    for(s=0;s<6;++s)
	    {
	     fprintf(output_ris,"N.%3d%3d%3d ",
		     lista_ass_t[s].i,
		     lista_ass_t[s].j,
		     lista_ass_t[s].k);
	     fprintf(output_ris,"%*.*f %*.*f %*.*f\n",
		     amp,prec,x42   *lista_ass_t[s].i/2.+
			      x12   *lista_ass_t[s].j/2.+
			      x[iv2]*lista_ass_t[s].k/2.,
		     amp,prec,y42   *lista_ass_t[s].i/2.+
			      y12   *lista_ass_t[s].j/2.+
			      y[iv2]*lista_ass_t[s].k/2.,
		     amp,prec,Cptt_Pt->cptt.points[indexes[s]]);
	    }
	    fprintf(output_ris,
		    "Micro-triangle V4=BARICENTRO V42 V23\n");
	    indexes[0]=6;
	    indexes[1]=17;
	    indexes[2]=25;
	    indexes[3]=18;
	    indexes[4]=28;
	    indexes[5]=30;
	    for(s=0;s<6;++s)
	    {
	     fprintf(output_ris,"N.%3d%3d%3d ",
		     lista_ass_t[s].i,
		     lista_ass_t[s].j,
		     lista_ass_t[s].k);
	     fprintf(output_ris,"%*.*f %*.*f %*.*f\n",
		     amp,prec,x4    *lista_ass_t[s].i/2.+
			      x42   *lista_ass_t[s].j/2.+
			      x23   *lista_ass_t[s].k/2.,
		     amp,prec,y4    *lista_ass_t[s].i/2.+
			      y42   *lista_ass_t[s].j/2.+
			      y23   *lista_ass_t[s].k/2.,
		     amp,prec,Cptt_Pt->cptt.points[indexes[s]]);
	    }
	    fprintf(output_ris,
		    "Micro-triangle V42 V2 V23\n");
	    indexes[0]=6;
	    indexes[1]=5;
	    indexes[2]=4;
	    indexes[3]=17;
	    indexes[4]=16;
	    indexes[5]=25;
	    for(s=0;s<6;++s)
	    {
	     fprintf(output_ris,"N.%3d%3d%3d ",
		     lista_ass_t[s].i,
		     lista_ass_t[s].j,
		     lista_ass_t[s].k);
	     fprintf(output_ris,"%*.*f %*.*f %*.*f\n",
		     amp,prec,x42   *lista_ass_t[s].i/2.+
			      x[iv2]*lista_ass_t[s].j/2.+
			      x23   *lista_ass_t[s].k/2.,
		     amp,prec,y42   *lista_ass_t[s].i/2.+
			      y[iv2]*lista_ass_t[s].j/2.+
			      y23   *lista_ass_t[s].k/2.,
		     amp,prec,Cptt_Pt->cptt.points[indexes[s]]);
	    }
	    fprintf(output_ris,
		    "Micro-triangle V4=BARICENTRO V23 V43\n");
	    indexes[0]=26;
	    indexes[1]=19;
	    indexes[2]=6;
	    indexes[3]=29;
	    indexes[4]=18;
	    indexes[5]=30;
	    for(s=0;s<6;++s)
	    {
	     fprintf(output_ris,"N.%3d%3d%3d ",
		     lista_ass_t[s].i,
		     lista_ass_t[s].j,
		     lista_ass_t[s].k);
	     fprintf(output_ris,"%*.*f %*.*f %*.*f\n",
		     amp,prec,x4    *lista_ass_t[s].i/2.+
			      x23   *lista_ass_t[s].j/2.+
			      x43   *lista_ass_t[s].k/2.,
		     amp,prec,y4    *lista_ass_t[s].i/2.+
			      y23   *lista_ass_t[s].j/2.+
			      y43   *lista_ass_t[s].k/2.,
		     amp,prec,Cptt_Pt->cptt.points[indexes[s]]);
	    }
	    fprintf(output_ris,
		    "Micro-triangle V43 V23 V3\n");
	    indexes[0]=8;
	    indexes[1]=7;
	    indexes[2]=6;
	    indexes[3]=20;
	    indexes[4]=19;
	    indexes[5]=26;
	    for(s=0;s<6;++s)
	    {
	     fprintf(output_ris,"N.%3d%3d%3d ",
		     lista_ass_t[s].i,
		     lista_ass_t[s].j,
		     lista_ass_t[s].k);
	     fprintf(output_ris,"%*.*f %*.*f %*.*f\n",
		     amp,prec,x43   *lista_ass_t[s].i/2.+
			      x23   *lista_ass_t[s].j/2.+
			      x[iv3]*lista_ass_t[s].k/2.,
		     amp,prec,y43   *lista_ass_t[s].i/2.+
			      y23   *lista_ass_t[s].j/2.+
			      y[iv3]*lista_ass_t[s].k/2.,
		     amp,prec,Cptt_Pt->cptt.points[indexes[s]]);
	    }
	    fprintf(output_ris,
		    "Micro-triangle V4=BARICENTRO V43 V31\n");
	    indexes[0]=10;
	    indexes[1]=21;
	    indexes[2]=26;
	    indexes[3]=22;
	    indexes[4]=29;
	    indexes[5]=30;
	    for(s=0;s<6;++s)
	    {
	     fprintf(output_ris,"N.%3d%3d%3d ",
		     lista_ass_t[s].i,
		     lista_ass_t[s].j,
		     lista_ass_t[s].k);
	     fprintf(output_ris,"%*.*f %*.*f %*.*f\n",
		     amp,prec,x4    *lista_ass_t[s].i/2.+
			      x43   *lista_ass_t[s].j/2.+
			      x31   *lista_ass_t[s].k/2.,
		     amp,prec,y4    *lista_ass_t[s].i/2.+
			      y43   *lista_ass_t[s].j/2.+
			      y31   *lista_ass_t[s].k/2.,
		     amp,prec,Cptt_Pt->cptt.points[indexes[s]]);
	    }
	    fprintf(output_ris,
		    "Micro-triangle V43 V3 V31\n");
	    indexes[0]=10;
	    indexes[1]=9;
	    indexes[2]=8;
	    indexes[3]=21;
	    indexes[4]=20;
	    indexes[5]=26;
	    for(s=0;s<6;++s)
	    {
	     fprintf(output_ris,"N.%3d%3d%3d ",
		     lista_ass_t[s].i,
		     lista_ass_t[s].j,
		     lista_ass_t[s].k);
	     fprintf(output_ris,"%*.*f %*.*f %*.*f\n",
		     amp,prec,x43   *lista_ass_t[s].i/2.+
			      x[iv3]*lista_ass_t[s].j/2.+
			      x31   *lista_ass_t[s].k/2.,
		     amp,prec,y43   *lista_ass_t[s].i/2.+
			      y[iv3]*lista_ass_t[s].j/2.+
			      y31   *lista_ass_t[s].k/2.,
		     amp,prec,Cptt_Pt->cptt.points[indexes[s]]);
	    }
	    fprintf(output_ris,
		    "Micro-triangle V4=BARICENTRO V31 V41\n");
	    indexes[0]=24;
	    indexes[1]=23;
	    indexes[2]=10;
	    indexes[3]=27;
	    indexes[4]=22;
	    indexes[5]=30;
	    for(s=0;s<6;++s)
	    {
	     fprintf(output_ris,"N.%3d%3d%3d ",
		     lista_ass_t[s].i,
		     lista_ass_t[s].j,
		     lista_ass_t[s].k);
	     fprintf(output_ris,"%*.*f %*.*f %*.*f\n",
		     amp,prec,x4    *lista_ass_t[s].i/2.+
			      x31   *lista_ass_t[s].j/2.+
			      x41   *lista_ass_t[s].k/2.,
		     amp,prec,y4    *lista_ass_t[s].i/2.+
			      y31   *lista_ass_t[s].j/2.+
			      y41   *lista_ass_t[s].k/2.,
		     amp,prec,Cptt_Pt->cptt.points[indexes[s]]);
	    }
	    fprintf(output_ris,
		    "Micro-triangle V41 V31 V1\n");
	    indexes[0]=0;
	    indexes[1]=11;
	    indexes[2]=10;
	    indexes[3]=12;
	    indexes[4]=23;
	    indexes[5]=24;
	    for(s=0;s<6;++s)
	    {
	     fprintf(output_ris,"N.%3d%3d%3d ",
		     lista_ass_t[s].i,
		     lista_ass_t[s].j,
		     lista_ass_t[s].k);
	     fprintf(output_ris,"%*.*f %*.*f %*.*f\n",
		     amp,prec,x41   *lista_ass_t[s].i/2.+
			      x31   *lista_ass_t[s].j/2.+
			      x[iv1]*lista_ass_t[s].k/2.,
		     amp,prec,y41   *lista_ass_t[s].i/2.+
			      y31   *lista_ass_t[s].j/2.+
			      y[iv1]*lista_ass_t[s].k/2.,
		     amp,prec,Cptt_Pt->cptt.points[indexes[s]]);
	    }
	    break;
    }  /* fine switch */
   Triang_Pt=Triang_Pt->next;
   Cptt_Pt=Cptt_Pt->next;
  }
 writemenu();
}

void minandmax(a,b,c,min,max)
double a,b,c,*min,*max;
{
 *min=a; *max=a;
 if (b<*min) *min=b; else if (b>*max) *max=b;
 if (c<*min) *min=c; else if (c>*max) *max=c;
}

double piano(xnew,ynew,x1,y1,z1,x2,y2,z2,x3,y3,z3)
double xnew,ynew,x1,y1,z1,x2,y2,z2,x3,y3,z3;
{
 double ret_val;

 ret_val=x1*(y2-y3)+x2*(y3-y1)+x3*(y1-y2);
 if (fabs(ret_val)<EPS_ALLIN)
 {
  printf("Cannot find the plane passing through three nearby aligned points.\n");
  exit(1);
 }
 ret_val=-(xnew*(y1*(z2-z3)+y2*(z3-z1)+y3*(z1-z2))
	   -ynew*(x1*(z2-z3)+x2*(z3-z1)+x3*(z1-z2))
	   +x1*(y3*z2-y2*z3)+x2*(y1*z3-y3*z1)+x3*(y2*z1-y1*z2))
	  /ret_val;
 return ret_val;
}

int circoscrivi(x1,y1,x2,y2,x3,y3,xc,yc,r)
double x1,y1,x2,y2,x3,y3,*xc,*yc,*r;
{
double Denom;

Denom=(2*(x1*(y2-y3)+x2*(y3-y1)+x3*(y1-y2)));
if (fabs(Denom)<EPS_ALLIN) return(1);
*xc=(pow(x1,2.)*(y2-y3)
     +pow(x2,2.)*(y3-y1)
     +(y1-y2)*(pow(x3,2.)
	       +(y2-y3)*(y1-y3)) )
    /Denom;
*yc=-(pow(x1,2.)*(x2-x3)
      -x1*(pow(x2,2.)
      -pow(x3,2.)
      +(y2-y3)*(y2+y3))
      +pow(x2,2.)*x3
      -x2*(pow(x3,2.)
	   +(y3-y1)*(y1+y3))
      +x3*(y2-y1)*(y1+y2) )
     /Denom;
if (r!=NULL) *r=sqrt(pow((*xc-x1),2.)+pow((*yc-y1),2.));
return(0);
}

double triangle_max_angle(x1,y1,x2,y2,x3,y3)
double x1,y1,x2,y2,x3,y3;
{
 double t1,t2,t3;

 t1=acos(1./(sqrt(pow(x2-x1,2.)+pow(y2-y1,2.))*
	     sqrt(pow(x3-x1,2.)+pow(y3-y1,2.)))*
	 ((x2-x1)*(x3-x1)+(y2-y1)*(y3-y1)));
 t2=acos(1./(sqrt(pow(x3-x2,2.)+pow(y3-y2,2.))*
	     sqrt(pow(x1-x2,2.)+pow(y1-y2,2.)))*
	 ((x3-x2)*(x1-x2)+(y3-y2)*(y1-y2)));
 t3=acos(1./(sqrt(pow(x1-x3,2.)+pow(y1-y3,2.))*
	     sqrt(pow(x2-x3,2.)+pow(y2-y3,2.)))*
	 ((x1-x3)*(x2-x3)+(y1-y3)*(y2-y3)));
 if (t2>t1) t1=t2;
 if (t3>t1) t1=t3;
 return(t1/PI*180.);
}

void options()
{
 char scelta;
 
 printf("Width field for real numbers output: \n");
 scanf(" %d",&amp);
 printf("Number of decimal digits: \n");
 scanf(" %d",&prec);
 printf("Zoom factor for central zooms (Previous value %*.*f): \n",
        WIDTH,PREC,zoomscalefactor);  /* float-type variable */
 scanf(" %f",&zoomscalefactor);   
 printf("Color setting (y/n)\n");
 scanf(" %c",&scelta);
 if (scelta=='y')
 {
  printf("Line rgb color: red (0-1): \n");
  scanf(" %f",&rgb_red_line);
  printf("Line rgb color: green (0-1): \n");
  scanf(" %f",&rgb_green_line);
  printf("Line rgb color: blue   (0-1): \n");
  scanf(" %f",&rgb_blue_line);
  printf("Fill rgb color: red (0-1): \n");
  scanf(" %f",&rgb_red_fill);
  printf("Fill rgb color: green (0-1): \n");
  scanf(" %f",&rgb_green_fill);
  printf("Fill rgb color: blue (0-1): \n");
  scanf(" %f",&rgb_blue_fill);
  printf("Background rgb color: red (0-1) \n");
  scanf(" %f",&rgb_red_sfondo);
  printf("Background rgb color: green (0-1) \n");
  scanf(" %f",&rgb_green_sfondo);
  printf("Background rgb color: blue (0-1) \n");
  scanf(" %f",&rgb_blue_sfondo);
 }
 writemenu();
}

void read_triang()
{
 struct triang_list *Triang_Pt, *Triang_Pt_Saved;
 struct edge_list *Edge_Pt, *Edge_Pt_Saved;
 FILE *trifile;
 long int i_base,i_v1,i_v2;
 char base_buffer[20]; /* large enough for indices or a single letter */

 trifile=fopen(nome_tri_file,"r");
 if (trifile==NULL)
   {
    printf("Cannot open file: %s\n",nome_tri_file);
    exit(1);
   }
 /* BEGINNING TRIANGULATION READING */
 if (fgetc(trifile)!='C')
   {
    printf("File %s in a no correct format.\n",nome_tri_file);
    exit(1);
   }
 else fgetc(trifile); /* reading carriage return */
 /* there is always the first triangle */
 /* creating dummy triangle-cell and saving it */
 Triang_Pt=malloc(sizeof(struct triang_list));
 if (Triang_Pt==NULL) {printf("Not enough memory.\n"); exit(1);}
 Triang_Pt_Saved=Triang_Pt;
 /* creating dummy edge-cell and saving it */
 Edge_Pt=malloc(sizeof(struct triang_list));
 if (Edge_Pt==NULL) {printf("Not enough memory.\n"); exit(1);}
 Edge_Pt_Saved=Edge_Pt; 
 numero_triangoli_glob=0;
 fscanf(trifile,"%ld ",&i_base);
 do
   {
    fscanf(trifile,"%ld %ld ",&i_v1,&i_v2);
    while(i_v2!=-1)
     {
      if ((i_base<i_v1)&&(i_base<i_v2))
	{
	 /* adding new triangle */
	 Triang_Pt->next=malloc(sizeof(struct triang_list));
	 if (Triang_Pt->next==NULL) {printf("Not enough memory.\n");
				     exit(1);}
         numero_triangoli_glob=numero_triangoli_glob+1;
	 Triang_Pt->next->triangolo.v1=i_base;
	 Triang_Pt->next->triangolo.v2=i_v1;
	 Triang_Pt->next->triangolo.v3=i_v2;
	 Triang_Pt=Triang_Pt->next;
	}
      i_v1=i_v2;
      fscanf(trifile,"%ld ",&i_v2);
     }
    fscanf(trifile,"%s ",base_buffer);
    i_base=atol(base_buffer);
   }
 while (strcmp(base_buffer,"B")!=0);
 /* END TRIANGULATION READING */
 fclose(trifile);
 /* Closing lists while */
 /* the boundaries for all elements are still undefined (They will not be used) */
 Triang_Pt->next=NULL;
 Edge_Pt->next=NULL;
 /* deleting dummy cells and returning the result to globals variables */
 Triangolazione_Glob=Triang_Pt_Saved->next;
 Triangolazione=Triangolazione_Glob;
 numero_triangoli=numero_triangoli_glob;
 free(Triang_Pt_Saved);
 Inviluppo_convesso=Edge_Pt_Saved->next;
 free(Edge_Pt_Saved);
}

void free_triang_list(Triang_List_Pt)
struct triang_list *Triang_List_Pt;
{
 struct triang_list *tempPt;
 
 if (Triang_List_Pt==NULL) return;
 while (Triang_List_Pt->next!=NULL)
   {
    tempPt=Triang_List_Pt->next;
    free(Triang_List_Pt);
    Triang_List_Pt=tempPt;
   }
 free(Triang_List_Pt);
}

void free_edge_list(Edge_List_Pt)
struct edge_list *Edge_List_Pt;
{
 struct edge_list *tempPt;
 
 if (Edge_List_Pt==NULL) return;
 while (Edge_List_Pt->next!=NULL)
   {
    tempPt=Edge_List_Pt->next;
    free(Edge_List_Pt);
    Edge_List_Pt=tempPt;
   }
 free(Edge_List_Pt);
}

void free_cptt_list(Cptt_List_Pt)
struct cptt_list *Cptt_List_Pt;
{
 struct cptt_list *tempPt;
 
 if (Cptt_List_Pt==NULL) return;
 while (Cptt_List_Pt->next!=NULL)
   {
    tempPt=Cptt_List_Pt->next;
    free(Cptt_List_Pt);
    Cptt_List_Pt=tempPt;
   }
 free(Cptt_List_Pt);
}

void set_ucalcpar_vcalcpar(numln)
int numln;
{
 double passoloc=1./(numln+1);
 int i,j,s;
   
 s=0; ucalcpar[0]=0.; vcalcpar[0]=0;
 for (i=0;i<=numln;++i)
  {
   for (j=1;j<=numln+1-i;++j)
    {
     ucalcpar[s+j]=ucalcpar[s];
     vcalcpar[s+j]=vcalcpar[s+j-1]+passoloc;
    }
   ucalcpar[s+numln-i+2]=ucalcpar[s]+passoloc;
   s=s+numln-i+2;
   vcalcpar[s]=0.;
  }
 s=s+1; ucalcpar[s]=1.; vcalcpar[s]=0.;            
} 

void output_file_itptb()
{
 if (itptbpresenza==0) return;
 printf("\nFile name: \n");
 scanf(" %s",output_ris_nome);
 output_ris=fopen(output_ris_nome,"w");
 if (output_ris==NULL)
   {
    printf("Error in file name: %s\n",output_ris_nome);
    exit(0);
   }
 trioutputitptb();
 fclose(output_ris);
 output_ris=stdout;
}

void output_file_cptb()
{
 if (cptbpresenza==0) return;
 printf("\nFile name: \n");
 scanf(" %s",output_ris_nome);
 output_ris=fopen(output_ris_nome,"w");
 if (output_ris==NULL)
   {
    printf("Error in file name: %s\n",output_ris_nome);
    exit(0);
   }
 trioutputcptb();
 fclose(output_ris);
 output_ris=stdout;
}

void output_file_itptt()
{
 if (itpttpresenza==0) return;
 printf("\nFile name: \n");
 scanf(" %s",output_ris_nome);
 output_ris=fopen(output_ris_nome,"w");
 if (output_ris==NULL)
   {
    printf("Error in file name: %s\n",output_ris_nome);
    exit(0);
   }
 trioutputitptt();
 fclose(output_ris);
 output_ris=stdout;
}

void output_file_cptt()
{
 if (cpttpresenza==0) return;
 printf("\nFile name: \n");
 scanf(" %s",output_ris_nome);
 output_ris=fopen(output_ris_nome,"w");
 if (output_ris==NULL)
   {
    printf("Error in file name: %s\n",output_ris_nome);
    exit(0);
   }
 trioutputcptt();
 fclose(output_ris);
 output_ris=stdout;
}

void output_file_calcpt()
{
 if (cpttpresenza==0) return;
 if (calcpt==NULL) return;
 printbezt3d(calcpt);
 writemenu();
}

void printbezt3d(calcpt)
double calcpt[][MAXDIM];
{
 FILE *file;
 char NameFile[70];
 struct cptt_list *Cptt_Pt;
 struct triang_list *Triang_Pt;
 long int CurrentTriangleNumber;
 int i;

  if (cpttpresenza==0) return;
  if (calcpt==NULL) return;
  printf("Name of the file where you want to store computed points\n");
  scanf(" %s",&NameFile);
  file=fopen(NameFile,"w");
  if (file==NULL)
  {
   printf("Cannot open file %s\n",NameFile);
   exit(1);
  }  
  Cptt_Pt=Punti_controllo;
  Triang_Pt=Triangolazione;
  CurrentTriangleNumber=0;
  while (Cptt_Pt!=NULL)
  {
   CurrentTriangleNumber=CurrentTriangleNumber+1;
   switch(Cptt_Pt->cptt.itpmethod)
   {
    case 1: /* Clough-Tocher with linear normal derivatives  */
     {
       for (i=0;i<3;++i)
          fputmat(file,numpt,3,
                  &calcpt[numpt*((CurrentTriangleNumber-1)*3+i)],
                  3,WIDTH,PREC);
       break;
     }
    case 2: /* Powell-Sabin - A Subdivision */
     {
      /* Considering again 12 triangles to
         index the array 'calcpt' since declaration has been done
         for that case (the worst one) */
       for (i=0;i<6;++i)
          fputmat(file,numpt,3,
                  &calcpt[numpt*((CurrentTriangleNumber-1)*12+i)],
                  3,WIDTH,PREC);
      break;
     }
    case 3: /* Q18 */
     {
       fputmat(file,numpt,3,
               &calcpt[numpt*((CurrentTriangleNumber-1)*1+0)],
               3,WIDTH,PREC);
      break;
     }
    case 4: /* Powell-Sabin - B Subdivision */
     {
       for (i=0;i<12;++i)
         fputmat(file,numpt,3,
                 &calcpt[numpt*((CurrentTriangleNumber-1)*12+i)],
                 3,WIDTH,PREC);
      break;
     }
   } /* end switch */
   Cptt_Pt=Cptt_Pt->next;
   Triang_Pt=Triang_Pt->next;
  }   /* fine while */
 fclose(file);
}

void selectsubtriang()
{
 double select_left, select_right, select_down, select_up;
 struct triang_list *Triang_Pt, *SubTriang_Pt, *SubTriang_Pt_Saved;
 struct edge_list *SelectionBorderPt, *SelectionBorderPt_Saved;
 int accept,accept_triangle,accept_border;
 double temp,temp1,temp2;
 long int iv1,iv2,iv3;
 char ch, NameFile[70];

 /* Selection of current window as a subwindow of that currently shown */
 accept=0;
 while (!accept)
 { 
  printf("Set horizontal bounds between given limits to define a clipping window\n");
  scanf(" %lf %lf",&select_left,&select_right);
  printf("Set vertical bounds between given limits to define a clipping window\n");
  scanf(" %lf %lf",&select_down,&select_up); 
  if (tri_left<=select_left && select_left<=select_right && select_right<=tri_right)
    if (tri_down<=select_down && select_down<=select_up && select_up<=tri_up)
      accept=1;
  if (!accept) 
   {
    printf("\nHorizontal and/or vertical bounds for the clipping window\n");
    printf("must be ordered in increasing way and must lie between the limits\n");
    printf("of the current triangulation window, that are now\n");
    printf("%*.*f %*.*f horizontally and\n",WIDTH,PREC,tri_left,WIDTH,PREC,tri_right);
    printf("%*.*f %*.*f vertically.\n\n",WIDTH,PREC,tri_down,WIDTH,PREC,tri_up);       
   }
 }
 free_triang_list(Triangolazione_Loc); Triangolazione_Loc=NULL;
 /* CREATION OF A SUBTRIANGULATION */
 /* creating dummy triangle-cell and saving it */
 SubTriang_Pt=malloc(sizeof(struct triang_list));
 if (SubTriang_Pt==NULL) {printf("Not enough memory.\n"); exit(1);}
 SubTriang_Pt_Saved=SubTriang_Pt;
 /* CREATION OF SELECTIONBORDER */
 /* creating dummy edge-cell and saving it */
 SelectionBorderPt=malloc(sizeof(struct edge_list));
 if (SelectionBorderPt==NULL) {printf("Not enough memory.\n"); exit(1);}
 SelectionBorderPt_Saved=SelectionBorderPt;
 printf("\nPress 'y' to store in a file the edges of the triangles\n");
 printf("of the current subtriangulation for which the opposite vertex\n");
 printf("is the only one inside the open square defining the subtriangulation.\n");
 printf("(They include new border edges.)\n\n");
 scanf(" %c",&ch);
 if (ch=='y') 
 {
 printf("Name of the file: ");
 scanf(" %s",&NameFile);
 } 
 /* I must initialize local 'minmaxbox' box with largest values 
    so that every accepted triangle may correct it */ 
 minbox_t_loc[0]=maxbox_t_glob[0]; maxbox_t_loc[0]=minbox_t_glob[0];
 minbox_t_loc[1]=maxbox_t_glob[1]; maxbox_t_loc[1]=minbox_t_glob[1]; 
 Triang_Pt=Triangolazione_Glob;
 numero_triangoli_loc=0; 
 while (Triang_Pt!=NULL)
 { 
  iv1=Triang_Pt->triangolo.v1;
  iv2=Triang_Pt->triangolo.v2;
  iv3=Triang_Pt->triangolo.v3;
  /* Accepting triangles with at least one vertex inside the open selected region */
  
  /* For triangles having only one 'good' vertex, 
     I accept the opposite edge in the border */
  accept_triangle=0; accept_border=0;
  if ((temp=x[iv1])>select_left && temp<select_right)
    if ((temp=y[iv1])>select_down && temp<select_up) 
      {accept_triangle++; accept_border=1;}
  if ((temp=x[iv2])>select_left && temp<select_right)
    if ((temp=y[iv2])>select_down && temp<select_up) 
      {accept_triangle++; accept_border=2;}
  if ((temp=x[iv3])>select_left && temp<select_right)
    if ((temp=y[iv3])>select_down && temp<select_up) 
      {accept_triangle++; accept_border=3;}         
  if (accept_triangle>0) /* The triangle is part of subtriangulation */
  {
   /* adding new triangle */
   SubTriang_Pt->next=malloc(sizeof(struct triang_list));
   if (SubTriang_Pt->next==NULL) {printf("Not enough memory.\n"); exit(1);}
   numero_triangoli_loc=numero_triangoli_loc+1;
   SubTriang_Pt->next->triangolo.v1=iv1;
   SubTriang_Pt->next->triangolo.v2=iv2;
   SubTriang_Pt->next->triangolo.v3=iv3;
   minandmax(x[iv1],x[iv2],x[iv3],&temp1,&temp2);
   if (temp1<minbox_t_loc[0]) minbox_t_loc[0]=temp1;
   if (temp2>maxbox_t_loc[0]) maxbox_t_loc[0]=temp2;
   minandmax(y[iv1],y[iv2],y[iv3],&temp1,&temp2);
   if (temp1<minbox_t_loc[1]) minbox_t_loc[1]=temp1;
   if (temp2>maxbox_t_loc[1]) maxbox_t_loc[1]=temp2;
   SubTriang_Pt=SubTriang_Pt->next;
  }
  if (accept_triangle==1)
  { /* Using 'accept_border' to accept the edge opposite to the vertex inside
       the selection region */
   /* adding new border */  
   SelectionBorderPt->next=malloc(sizeof(struct edge_list));
   if (SelectionBorderPt==NULL) {printf("Not enough memory\n"); exit(1);}
   switch(accept_border)
   { 
    /* For each case, I store the border so that the third vertex of the triangle is
    ** in the right region with respect to the oriented constraint (edge).
    ** Considering that 'iv1', 'iv2' and 'iv3' vertices are disposed in a 
    ** counterclockwise direction there is no need to test for the reading from the
    ** triangulation file (triangles belonging to the subtriangulation are disposed
    ** in the same way as those belonging to the main triangulation) */
    case 1:
     SelectionBorderPt->next->bordo.v1=iv3;
     SelectionBorderPt->next->bordo.v2=iv2;
     break;
    case 2:
     SelectionBorderPt->next->bordo.v1=iv1;
     SelectionBorderPt->next->bordo.v2=iv3;
     break;
    case 3:
     SelectionBorderPt->next->bordo.v1=iv2;
     SelectionBorderPt->next->bordo.v2=iv1;
     break;
   } /* switch end */
   SelectionBorderPt=SelectionBorderPt->next;
  } /* end adding border edge */
  Triang_Pt=Triang_Pt->next;
 } /* end while */
 /* Closing lists while pointers to neighbours are still undefined 
    (they will not be used) */
 SubTriang_Pt->next=NULL;
 SelectionBorderPt->next=NULL;
 /* deleting dummy cells and returning result to globals variables */
 Triangolazione_Loc=SubTriang_Pt_Saved->next;
 free(SubTriang_Pt_Saved);
 /* eventual printing to file of the border of selection */
 if (ch=='y') printselectionborder(SelectionBorderPt_Saved->next,NameFile);
 /* freeing memory used by the border of the selected region */
 free_edge_list(SelectionBorderPt_Saved);
 /* Don't assign SelectionBorderPt_Saved=NULL since the variable is local */
}

void printselectionborder(SelectionBorder,NameFile)
struct edge_list *SelectionBorder;
char NameFile[];
{
 FILE *file;
 struct edge_list *SelectionBorderPt;
  
 /* securities handled by the caller function */
 file=fopen(NameFile,"w");
 if (file==NULL)
 {
  printf("Cannot open file %s\n",NameFile);
  exit(1);
 }
 SelectionBorderPt=SelectionBorder;
 while (SelectionBorderPt!=NULL)
 {
  fprintf(file,"%ld %ld\n",SelectionBorderPt->bordo.v1,SelectionBorderPt->bordo.v2);
  SelectionBorderPt=SelectionBorderPt->next;
 } 
 fclose(file);
}

void compute_view_points(calcpt)
double calcpt[][MAXDIM];
{
 long int iv1,iv2,iv3,CurrentTriangleNumber;
 struct cptt_list *Cptt_Pt;
 struct triang_list *Triang_Pt;
 int s;
 double cpt_loc[MAX_CPT_NUMBER_C1][MAXDIM];
 int indexes[10]; 
 /* For 3-degree polynomials and therefore for 2-degree ones di grado (micro-splits) */
 double x4,y4,x12,y12,x23,y23,x31,y31,x41,y41,x42,y42,x43,y43;

  /* globals used: x y ucalcpar vcalcpar Triangolazione Punti_Controllo
  ** besides numpt numln and dim */
  /* Computations are done per micro-patch over contiguous parts of 'calcpt' */

  /* initializing pointers */
  Cptt_Pt=Punti_controllo;
  Triang_Pt=Triangolazione;
  CurrentTriangleNumber=0;
  while (Cptt_Pt!=NULL)
  {
   iv1=Triang_Pt->triangolo.v1;
   iv2=Triang_Pt->triangolo.v2;
   iv3=Triang_Pt->triangolo.v3;
   CurrentTriangleNumber=CurrentTriangleNumber+1;
   /* scelta metodo */
   switch(Cptt_Pt->cptt.itpmethod)
   {
    case 1: /* Clough-Tocher with linear normal derivatives */
     {
      /* lista_ass_t already computed properly when making computations */
      x4=(x[iv1]+x[iv2]+x[iv3])/3.;
      y4=(y[iv1]+y[iv2]+y[iv3])/3.;
	    /* Micro-triangle V4=BA V1 V2 */
	    indexes[0]=2;
	    indexes[1]=9;
	    indexes[2]=4;
	    indexes[3]=1;
	    indexes[4]=8;
	    indexes[5]=13;
	    indexes[6]=5;
	    indexes[7]=17;
	    indexes[8]=16;
	    indexes[9]=19;
	    for(s=0;s<10;++s)
	    {
	     cpt_loc[s][0]= x4    *lista_ass_t[s].i/3.+
			    x[iv1]*lista_ass_t[s].j/3.+
			    x[iv2]*lista_ass_t[s].k/3.;
	     cpt_loc[s][1]= y4    *lista_ass_t[s].i/3.+
			    y[iv1]*lista_ass_t[s].j/3.+
			    y[iv2]*lista_ass_t[s].k/3.;
	     cpt_loc[s][2]=Cptt_Pt->cptt.points[indexes[s]];
	    }
      /* generalfactlist already computed properly by settridisegno */
      err=trihornbez(3,1,cpt_loc,generalfactlist,ucalcpar,vcalcpar,
                     numpt,&calcpt[numpt*((CurrentTriangleNumber-1)*3+0)]);
	    /* Micro-triangle V4=BARYCENTER V2 V3 */
	    indexes[0]=3;
	    indexes[1]=12;
	    indexes[2]=7;
	    indexes[3]=2;
	    indexes[4]=11;
	    indexes[5]=14;
	    indexes[6]=8;
	    indexes[7]=18;
	    indexes[8]=17;
	    indexes[9]=19;
	    for(s=0;s<10;++s)
	    {
	     cpt_loc[s][0]= x4    *lista_ass_t[s].i/3.+
			    x[iv2]*lista_ass_t[s].j/3.+
			    x[iv3]*lista_ass_t[s].k/3.;
	     cpt_loc[s][1]= y4    *lista_ass_t[s].i/3.+
			    y[iv2]*lista_ass_t[s].j/3.+
			    y[iv3]*lista_ass_t[s].k/3.;
	     cpt_loc[s][2]=Cptt_Pt->cptt.points[indexes[s]];
	    }
      /* generalfactlist already computed properly by settridisegno */
      err=trihornbez(3,1,cpt_loc,generalfactlist,ucalcpar,vcalcpar,
                     numpt,&calcpt[numpt*((CurrentTriangleNumber-1)*3+1)]);
	    /* Micro-triangle V4=BARYCENTER V3 V1 */
	    indexes[0]=1;
	    indexes[1]=6;
	    indexes[2]=10;
	    indexes[3]=3;
	    indexes[4]=5;
	    indexes[5]=15;
	    indexes[6]=11;
	    indexes[7]=16;
	    indexes[8]=18;
	    indexes[9]=19;
	    for(s=0;s<10;++s)
	    {
	     cpt_loc[s][0]= x4    *lista_ass_t[s].i/3.+
			    x[iv3]*lista_ass_t[s].j/3.+
			    x[iv1]*lista_ass_t[s].k/3.;
	     cpt_loc[s][1]= y4    *lista_ass_t[s].i/3.+
			    y[iv3]*lista_ass_t[s].j/3.+
			    y[iv1]*lista_ass_t[s].k/3.;
	     cpt_loc[s][2]=Cptt_Pt->cptt.points[indexes[s]];
	    }
      /* generalfactlist already computed properly by settridisegno */
      err=trihornbez(3,1,cpt_loc,generalfactlist,ucalcpar,vcalcpar,
                     numpt,&calcpt[numpt*((CurrentTriangleNumber-1)*3+2)]);
      break;
     }
    case 2: /* Powell-Sabin - A Subdivision */
     {
      /* Considering again 12 triangles to
         index the array 'calcpt' since declaration has been done
         for that case (the worst one) */
      /* lista_ass_t already computed properly when making computations */
      circoscrivi(x[iv1],y[iv1],x[iv2],y[iv2],x[iv3],y[iv3],
		  &x4,&y4,NULL);
      x12=(x[iv1]+x[iv2])/2.; y12=(y[iv1]+y[iv2])/2.;
      x23=(x[iv2]+x[iv3])/2.; y23=(y[iv2]+y[iv3])/2.;
      x31=(x[iv3]+x[iv1])/2.; y31=(y[iv3]+y[iv1])/2.;
	    /* Micro-triangle V4=CIRCUMCENTER V1 V12 */
	    indexes[0]=2;
	    indexes[1]=1;
	    indexes[2]=0;
	    indexes[3]=13;
	    indexes[4]=12;
	    indexes[5]=18;
	    for(s=0;s<6;++s)
	    {
	     cpt_loc[s][0]= x4    *lista_ass_t[s].i/2.+
			    x[iv1]*lista_ass_t[s].j/2.+
			    x12   *lista_ass_t[s].k/2.;
	     cpt_loc[s][1]= y4    *lista_ass_t[s].i/2.+
			    y[iv1]*lista_ass_t[s].j/2.+
			    y12   *lista_ass_t[s].k/2.;
	     cpt_loc[s][2]=Cptt_Pt->cptt.points[indexes[s]];
	    }
      /* generalfactlist already computed properly by settridisegno */
      err=trihornbez(2,1,cpt_loc,generalfactlist,ucalcpar,vcalcpar,
                     numpt,&calcpt[numpt*((CurrentTriangleNumber-1)*12+0)]);
	    /* Micro-triangle V4=CIRCUMCENTER V12 V2 */
	    indexes[0]=4;
	    indexes[1]=3;
	    indexes[2]=2;
	    indexes[3]=14;
	    indexes[4]=13;
	    indexes[5]=18;
	    for(s=0;s<6;++s)
	    {
	     cpt_loc[s][0]= x4    *lista_ass_t[s].i/2.+
			    x12   *lista_ass_t[s].j/2.+
			    x[iv2]*lista_ass_t[s].k/2.;
	     cpt_loc[s][1]= y4    *lista_ass_t[s].i/2.+
			    y12   *lista_ass_t[s].j/2.+
			    y[iv2]*lista_ass_t[s].k/2.;
	     cpt_loc[s][2]=Cptt_Pt->cptt.points[indexes[s]];
	    }
      /* generalfactlist already computed properly by settridisegno */
      err=trihornbez(2,1,cpt_loc,generalfactlist,ucalcpar,vcalcpar,
                     numpt,&calcpt[numpt*((CurrentTriangleNumber-1)*12+1)]);
	    /* Micro-triangle V4=CIRCUMCENTER V2 V23 */
	    indexes[0]=6;
	    indexes[1]=5;
	    indexes[2]=4;
	    indexes[3]=15;
	    indexes[4]=14;
	    indexes[5]=18;
	    for(s=0;s<6;++s)
	    {
	     cpt_loc[s][0]= x4    *lista_ass_t[s].i/2.+
			    x[iv2]*lista_ass_t[s].j/2.+
			    x23   *lista_ass_t[s].k/2.;
	     cpt_loc[s][1]= y4    *lista_ass_t[s].i/2.+
			    y[iv2]*lista_ass_t[s].j/2.+
			    y23   *lista_ass_t[s].k/2.;
	     cpt_loc[s][2]=Cptt_Pt->cptt.points[indexes[s]];
	    }
      /* generalfactlist already computed properly by settridisegno */
      err=trihornbez(2,1,cpt_loc,generalfactlist,ucalcpar,vcalcpar,
                     numpt,&calcpt[numpt*((CurrentTriangleNumber-1)*12+2)]);
	    /* Micro-triangle V4=CIRCUMCENTER V23 V3 */
	    indexes[0]=8;
	    indexes[1]=7;
	    indexes[2]=6;
	    indexes[3]=16;
	    indexes[4]=15;
	    indexes[5]=18;
	    for(s=0;s<6;++s)
	    {
	     cpt_loc[s][0]= x4    *lista_ass_t[s].i/2.+
			    x23   *lista_ass_t[s].j/2.+
			    x[iv3]*lista_ass_t[s].k/2.;
	     cpt_loc[s][1]= y4    *lista_ass_t[s].i/2.+
			    y23   *lista_ass_t[s].j/2.+
			    y[iv3]*lista_ass_t[s].k/2.;
	     cpt_loc[s][2]=Cptt_Pt->cptt.points[indexes[s]];
	    }
      /* generalfactlist already computed properly by settridisegno */
      err=trihornbez(2,1,cpt_loc,generalfactlist,ucalcpar,vcalcpar,
                     numpt,&calcpt[numpt*((CurrentTriangleNumber-1)*12+3)]);
	    /* Micro-triangle V4=CIRCUMCENTER V3 V31 */
	    indexes[0]=10;
	    indexes[1]=9;
	    indexes[2]=8;
	    indexes[3]=17;
	    indexes[4]=16;
	    indexes[5]=18;
	    for(s=0;s<6;++s)
	    {
	     cpt_loc[s][0]= x4    *lista_ass_t[s].i/2.+
			    x[iv3]*lista_ass_t[s].j/2.+
			    x31   *lista_ass_t[s].k/2.;
	     cpt_loc[s][1]= y4    *lista_ass_t[s].i/2.+
			    y[iv3]*lista_ass_t[s].j/2.+
			    y31   *lista_ass_t[s].k/2.;
	     cpt_loc[s][2]=Cptt_Pt->cptt.points[indexes[s]];
	    }
      /* generalfactlist already computed properly by settridisegno */
      err=trihornbez(2,1,cpt_loc,generalfactlist,ucalcpar,vcalcpar,
                     numpt,&calcpt[numpt*((CurrentTriangleNumber-1)*12+4)]);
	    /* Micro-triangle V4=CIRCUMCENTER V31 V1 */
	    indexes[0]=0;
	    indexes[1]=11;
	    indexes[2]=10;
	    indexes[3]=12;
	    indexes[4]=17;
	    indexes[5]=18;
	    for(s=0;s<6;++s)
	    {
	     cpt_loc[s][0]= x4    *lista_ass_t[s].i/2.+
			    x31   *lista_ass_t[s].j/2.+
			    x[iv1]*lista_ass_t[s].k/2.;
	     cpt_loc[s][1]= y4    *lista_ass_t[s].i/2.+
			    y31   *lista_ass_t[s].j/2.+
			    y[iv1]*lista_ass_t[s].k/2.;
	     cpt_loc[s][2]=Cptt_Pt->cptt.points[indexes[s]];
	    }
      /* generalfactlist already computed properly by settridisegno */
      err=trihornbez(2,1,cpt_loc,generalfactlist,ucalcpar,vcalcpar,
                     numpt,&calcpt[numpt*((CurrentTriangleNumber-1)*12+5)]);
      break;
     }
    case 3: /* Q18 */
     {
      /* lista_ass_t already computed properly when making computations */
      for (s=0;s<21;++s)
	{
	 cpt_loc[s][0]=x[iv1]*lista_ass_t[s].i/5.+
		       x[iv2]*lista_ass_t[s].j/5.+
		       x[iv3]*lista_ass_t[s].k/5.;
	 cpt_loc[s][1]=y[iv1]*lista_ass_t[s].i/5.+
		       y[iv2]*lista_ass_t[s].j/5.+
		       y[iv3]*lista_ass_t[s].k/5.;
	 cpt_loc[s][2]=Cptt_Pt->cptt.points[s];
	}
      /* generalfactlist already computed properly by settridisegno */
      err=trihornbez(5,1,cpt_loc,generalfactlist,ucalcpar,vcalcpar,
                     numpt,&calcpt[numpt*((CurrentTriangleNumber-1)*1+0)]);
      break;
     }
    case 4: /* Powell Sabin - B splitting */
     {
      /* lista_ass_t already computed properly when making computations */
      x4=(x[iv1]+x[iv2]+x[iv3])/3.; y4=(y[iv1]+y[iv2]+y[iv3])/3.;
      x12=(x[iv1]+x[iv2])/2.; y12=(y[iv1]+y[iv2])/2.;
      x23=(x[iv2]+x[iv3])/2.; y23=(y[iv2]+y[iv3])/2.;
      x31=(x[iv3]+x[iv1])/2.; y31=(y[iv3]+y[iv1])/2.;
      x41=(x31+x12)/2.; y41=(y31+y12)/2.;
      x42=(x12+x23)/2.; y42=(y12+y23)/2.;
      x43=(x23+x31)/2.; y43=(y23+y31)/2.;
	    /* Micro-triangle V4=BARYCENTER V41 V12 */
	    indexes[0]=2;
	    indexes[1]=13;
	    indexes[2]=24;
	    indexes[3]=14;
	    indexes[4]=27;
	    indexes[5]=30;
	    for(s=0;s<6;++s)
	    {
	     cpt_loc[s][0]= x4    *lista_ass_t[s].i/2.+
			    x41   *lista_ass_t[s].j/2.+
			    x12   *lista_ass_t[s].k/2.;
	     cpt_loc[s][1]= y4    *lista_ass_t[s].i/2.+
			    y41   *lista_ass_t[s].j/2.+
			    y12   *lista_ass_t[s].k/2.;
	     cpt_loc[s][2]=Cptt_Pt->cptt.points[indexes[s]];
	    }
      /* generalfactlist already computed properly by settridisegno */
      err=trihornbez(2,1,cpt_loc,generalfactlist,ucalcpar,vcalcpar,
                     numpt,&calcpt[numpt*((CurrentTriangleNumber-1)*12+0)]);
	    /* Micro-triangle V41 V1 V12 */
	    indexes[0]=2;
	    indexes[1]=1;
	    indexes[2]=0;
	    indexes[3]=13;
	    indexes[4]=12;
	    indexes[5]=24;
	    for(s=0;s<6;++s)
	    {
	     cpt_loc[s][0]= x41   *lista_ass_t[s].i/2.+
			    x[iv1]*lista_ass_t[s].j/2.+
			    x12   *lista_ass_t[s].k/2.;
	     cpt_loc[s][1]= y41   *lista_ass_t[s].i/2.+
			    y[iv1]*lista_ass_t[s].j/2.+
			    y12   *lista_ass_t[s].k/2.;
	     cpt_loc[s][2]=Cptt_Pt->cptt.points[indexes[s]];
	    }
      /* generalfactlist already computed properly by settridisegno */
      err=trihornbez(2,1,cpt_loc,generalfactlist,ucalcpar,vcalcpar,
                     numpt,&calcpt[numpt*((CurrentTriangleNumber-1)*12+1)]);
	    /* Micro-triangle V4=BARYCENTER V12 V42 */
	    indexes[0]=25;
	    indexes[1]=15;
	    indexes[2]=2;
	    indexes[3]=28;
	    indexes[4]=14;
	    indexes[5]=30;
	    for(s=0;s<6;++s)
	    {
	     cpt_loc[s][0]= x4    *lista_ass_t[s].i/2.+
			    x12   *lista_ass_t[s].j/2.+
			    x42   *lista_ass_t[s].k/2.;
	     cpt_loc[s][1]= y4    *lista_ass_t[s].i/2.+
			    y12   *lista_ass_t[s].j/2.+
			    y42   *lista_ass_t[s].k/2.;
	     cpt_loc[s][2]=Cptt_Pt->cptt.points[indexes[s]];
	    }
      /* generalfactlist already computed properly by settridisegno */
      err=trihornbez(2,1,cpt_loc,generalfactlist,ucalcpar,vcalcpar,
                     numpt,&calcpt[numpt*((CurrentTriangleNumber-1)*12+2)]);
	    /* Micro-triangle V42 V12 V2 */
	    indexes[0]=4;
	    indexes[1]=3;
	    indexes[2]=2;
	    indexes[3]=16;
	    indexes[4]=15;
	    indexes[5]=25;
	    for(s=0;s<6;++s)
	    {
	     cpt_loc[s][0]= x42   *lista_ass_t[s].i/2.+
			    x12   *lista_ass_t[s].j/2.+
			    x[iv2]*lista_ass_t[s].k/2.;
	     cpt_loc[s][1]= y42   *lista_ass_t[s].i/2.+
			    y12   *lista_ass_t[s].j/2.+
			    y[iv2]*lista_ass_t[s].k/2.;
	     cpt_loc[s][2]=Cptt_Pt->cptt.points[indexes[s]];
	    }
      /* generalfactlist already computed properly by settridisegno */
      err=trihornbez(2,1,cpt_loc,generalfactlist,ucalcpar,vcalcpar,
                     numpt,&calcpt[numpt*((CurrentTriangleNumber-1)*12+3)]);
	    /* Micro-triangle V4=BARYCENTER V42 V23 */
	    indexes[0]=6;
	    indexes[1]=17;
	    indexes[2]=25;
	    indexes[3]=18;
	    indexes[4]=28;
	    indexes[5]=30;
	    for(s=0;s<6;++s)
	    {
	     cpt_loc[s][0]= x4    *lista_ass_t[s].i/2.+
			    x42   *lista_ass_t[s].j/2.+
			    x23   *lista_ass_t[s].k/2.;
	     cpt_loc[s][1]= y4    *lista_ass_t[s].i/2.+
			    y42   *lista_ass_t[s].j/2.+
			    y23   *lista_ass_t[s].k/2.;
	     cpt_loc[s][2]=Cptt_Pt->cptt.points[indexes[s]];
	    }
      /* generalfactlist already computed properly by settridisegno */
      err=trihornbez(2,1,cpt_loc,generalfactlist,ucalcpar,vcalcpar,
                     numpt,&calcpt[numpt*((CurrentTriangleNumber-1)*12+4)]);
	    /* Micro-triangle V42 V2 V23 */
	    indexes[0]=6;
	    indexes[1]=5;
	    indexes[2]=4;
	    indexes[3]=17;
	    indexes[4]=16;
	    indexes[5]=25;
	    for(s=0;s<6;++s)
	    {
	     cpt_loc[s][0]= x42   *lista_ass_t[s].i/2.+
			    x[iv2]*lista_ass_t[s].j/2.+
			    x23   *lista_ass_t[s].k/2.;
	     cpt_loc[s][1]= y42   *lista_ass_t[s].i/2.+
			    y[iv2]*lista_ass_t[s].j/2.+
			    y23   *lista_ass_t[s].k/2.;
	     cpt_loc[s][2]=Cptt_Pt->cptt.points[indexes[s]];
	    }
      /* generalfactlist already computed properly by settridisegno */
      err=trihornbez(2,1,cpt_loc,generalfactlist,ucalcpar,vcalcpar,
                     numpt,&calcpt[numpt*((CurrentTriangleNumber-1)*12+5)]);
	    /* Micro-triangle V4=BARYCENTER V23 V43 */
	    indexes[0]=26;
	    indexes[1]=19;
	    indexes[2]=6;
	    indexes[3]=29;
	    indexes[4]=18;
	    indexes[5]=30;
	    for(s=0;s<6;++s)
	    {
	     cpt_loc[s][0]= x4    *lista_ass_t[s].i/2.+
			    x23   *lista_ass_t[s].j/2.+
			    x43   *lista_ass_t[s].k/2.;
	     cpt_loc[s][1]= y4    *lista_ass_t[s].i/2.+
			    y23   *lista_ass_t[s].j/2.+
			    y43   *lista_ass_t[s].k/2.;
	     cpt_loc[s][2]=Cptt_Pt->cptt.points[indexes[s]];
	    }
      /* generalfactlist already computed properly by settridisegno */
      err=trihornbez(2,1,cpt_loc,generalfactlist,ucalcpar,vcalcpar,
                     numpt,&calcpt[numpt*((CurrentTriangleNumber-1)*12+6)]);
	    /* Micro-triangle V43 V23 V3 */
	    indexes[0]=8;
	    indexes[1]=7;
	    indexes[2]=6;
	    indexes[3]=20;
	    indexes[4]=19;
	    indexes[5]=26;
	    for(s=0;s<6;++s)
	    {
	     cpt_loc[s][0]= x43   *lista_ass_t[s].i/2.+
			    x23   *lista_ass_t[s].j/2.+
			    x[iv3]*lista_ass_t[s].k/2.;
	     cpt_loc[s][1]= y43   *lista_ass_t[s].i/2.+
			    y23   *lista_ass_t[s].j/2.+
			    y[iv3]*lista_ass_t[s].k/2.;
	     cpt_loc[s][2]=Cptt_Pt->cptt.points[indexes[s]];
	    }
      /* generalfactlist already computed properly by settridisegno */
      err=trihornbez(2,1,cpt_loc,generalfactlist,ucalcpar,vcalcpar,
                     numpt,&calcpt[numpt*((CurrentTriangleNumber-1)*12+7)]);
	    /* Micro-triangle V4=BARYCENTER V43 V31 */
	    indexes[0]=10;
	    indexes[1]=21;
	    indexes[2]=26;
	    indexes[3]=22;
	    indexes[4]=29;
	    indexes[5]=30;
	    for(s=0;s<6;++s)
	    {
	     cpt_loc[s][0]= x4    *lista_ass_t[s].i/2.+
			    x43   *lista_ass_t[s].j/2.+
			    x31   *lista_ass_t[s].k/2.;
	     cpt_loc[s][1]= y4    *lista_ass_t[s].i/2.+
			    y43   *lista_ass_t[s].j/2.+
			    y31   *lista_ass_t[s].k/2.;
	     cpt_loc[s][2]=Cptt_Pt->cptt.points[indexes[s]];
	    }
      /* generalfactlist already computed properly by settridisegno */
      err=trihornbez(2,1,cpt_loc,generalfactlist,ucalcpar,vcalcpar,
                     numpt,&calcpt[numpt*((CurrentTriangleNumber-1)*12+8)]);
	    /* Micro-triangle V43 V3 V31 */
	    indexes[0]=10;
	    indexes[1]=9;
	    indexes[2]=8;
	    indexes[3]=21;
	    indexes[4]=20;
	    indexes[5]=26;
	    for(s=0;s<6;++s)
	    {
	     cpt_loc[s][0]= x43   *lista_ass_t[s].i/2.+
			    x[iv3]*lista_ass_t[s].j/2.+
			    x31   *lista_ass_t[s].k/2.;
	     cpt_loc[s][1]= y43   *lista_ass_t[s].i/2.+
			    y[iv3]*lista_ass_t[s].j/2.+
			    y31   *lista_ass_t[s].k/2.;
	     cpt_loc[s][2]=Cptt_Pt->cptt.points[indexes[s]];
	    }
      /* generalfactlist already computed properly by settridisegno */
      err=trihornbez(2,1,cpt_loc,generalfactlist,ucalcpar,vcalcpar,
                     numpt,&calcpt[numpt*((CurrentTriangleNumber-1)*12+9)]);
	    /* Micro-triangle V4=BARYCENTER V31 V41 */
	    indexes[0]=24;
	    indexes[1]=23;
	    indexes[2]=10;
	    indexes[3]=27;
	    indexes[4]=22;
	    indexes[5]=30;
	    for(s=0;s<6;++s)
	    {
	     cpt_loc[s][0]= x4    *lista_ass_t[s].i/2.+
			    x31   *lista_ass_t[s].j/2.+
			    x41   *lista_ass_t[s].k/2.;
	     cpt_loc[s][1]= y4    *lista_ass_t[s].i/2.+
			    y31   *lista_ass_t[s].j/2.+
			    y41   *lista_ass_t[s].k/2.;
	     cpt_loc[s][2]=Cptt_Pt->cptt.points[indexes[s]];
	    }
      /* generalfactlist already computed properly by settridisegno */
      err=trihornbez(2,1,cpt_loc,generalfactlist,ucalcpar,vcalcpar,
                     numpt,&calcpt[numpt*((CurrentTriangleNumber-1)*12+10)]);
	    /* Micro-triangle V41 V31 V1 */
	    indexes[0]=0;
	    indexes[1]=11;
	    indexes[2]=10;
	    indexes[3]=12;
	    indexes[4]=23;
	    indexes[5]=24;
	    for(s=0;s<6;++s)
	    {
	     cpt_loc[s][0]= x41   *lista_ass_t[s].i/2.+
			    x31   *lista_ass_t[s].j/2.+
			    x[iv1]*lista_ass_t[s].k/2.;
	     cpt_loc[s][1]= y41   *lista_ass_t[s].i/2.+
			    y31   *lista_ass_t[s].j/2.+
			    y[iv1]*lista_ass_t[s].k/2.;
	     cpt_loc[s][2]=Cptt_Pt->cptt.points[indexes[s]];
	    }
      /* generalfactlist already computed properly by settridisegno */
      err=trihornbez(2,1,cpt_loc,generalfactlist,ucalcpar,vcalcpar,
                     numpt,&calcpt[numpt*((CurrentTriangleNumber-1)*12+11)]);
      break;
     }
   } /* end switch */
   Cptt_Pt=Cptt_Pt->next;
   Triang_Pt=Triang_Pt->next;
  } /* end while */
}








