/*
** FUNZ_PWS.C
**
** AUTHORS: 
**
**   Maria MORANDI CECCHI               Stefano DE MARCHI,            
**   University of Padua, Italy         University of Udine, Italy
**   email: mcecchi@math.unipd.it       email: demarchi@dimi.uniud.it
**   
**                       Damiano FASOLI
**                       Verona Software Srl, Italy 
**                       email: vrsoft@sis.it
**
**
**
** REVISION DATE: May, 1999  
**
**
** MODULES CALLED: NONE
**
**-------------------------------------------------------------------------
** BACKGROUND
**
** Let (xi,yi,qi), i=0,...,N-1 be N points in the space R^3. Let us consider 
** their projections (xi,yi) onto the plane z=0 and let T be a triangulation
** of these points. A triangulation of them is the set of the triangles having
** these points and only them as vertices, so that two distinct triangles 
** intersect at most in a common edge or in a common vertex. 
**
** The aim of this program is the computation of the value of the functional
**
** phi(v)=   sum  [ Integral[ (Qxx^2 + 2*Qxy^2 + Qyy^2  ) dx dy ] ]     
**         t in T      t      
** 
** where Q is a Powell-Sabin polynomial patch, restricted to the triangle t.
** 
** 
** REFERENCES
**
**  M. Morandi Cecchi, S. De Marchi, D. Fasoli,
**      "A Package for Representing C^1 Interpolating Surfaces: 
**       Application to the Lagoon of Venice's Bed."
**       Numer. Algorithms, 1999.
**
**  D. Fasoli,
**      Laurea's Thesis, University of Padua, 1995 (in Italian).
**
**  G. Farin:
**     "Triangular Bernstein-Bezier patches",
**      Comput. Aided Geom. Design,3:83-127, 1986.
**
** -------------------------------------------------------------------------
** 
** SYNOPSIS:  funz_pws file1 file2 file3
**
** 'file1' is the data file made up of 3-dimensional points. 
** The coordinates of the points may be separated by any 
** combination of spaces, tabs and carriage returns, 
** even though we recommend to use a row per point for clearness. 
** 
** 'file2' is the triangulation data file in the format generated by
** the program 'do_trian'. The file has three sections. 
** The first, starting with a single capital letter `C' in the first row, 
** lists in subsequent blocks separated by -1 the index of
** a node and those of its neighbours in a counterclockwise
** order. 
** Note: this program uses only this first section and the heading of the
** second section, that is a single capital letter `B' in the row.
**
** 'file3' contains the derivatives data in the following format. 
** The values in a same row refer to the same vertex of the triangulation.
** The first column is the x-derivative, the second one is the y-derivative,
** the third, the fourth and the fifth ones contain the value 0.
**
** The output is the value of the functional the Powell-Sabin 
** interpolation function
** at the points specified in file1, the triangulation specified in file2
** and the derivatives values specified in file3.
** The format of the output is handled by the two constants WIDTH and PREC
** that are the width of the output field, and the number of decimal digits.      
**
** The program supports at most MAX_NUM_NODI (#DEFINE constant) input points.
** If more points are given, it exits by printing a warning message.
** 
**
** -------------------------------------------------------------------------------
** FUNCTIONS AND SUBROUTINES
**
** >>>>>>>>>>>>>>>>>>>>>>>>>
**
** double piano(double xnew,double ynew,double x1,double y1,double z1,
**              double x2,double y2,double z2,
**              double x3,double y3,double z3)
**
** Inputs: 
**        xnew,ynew: the points where to evalutate the function;
**        x1,y1,z1: coordinates of the first point;
**        x2,y2,z2: coordinates of the second point;
**        x3,y3,z3: coordinates of the third point;
** Output:
**        The value of the function z(xnew,ynew) on the plane passing 
**        through (x1,y1,z1),(x2,y2,z2), (x3,y3,z3) so that  
**        (xnew,ynew,z(xnew,ynew)) lies onto this plane. 
** 
** If the points are aligned or almost aligned
** (denominator smaller than EPS_ALLIN) the function returs and the program
** stops.
**
** >>>>>>>>>>>>>>>>>>>>>>>>
**
** double eval_derivative_from_cpt(int ABflag,int ntype,int ntriang,double d[])
**
**
** GLOBAL VARIABLES: IndTA IndTB
**                   u4,v4 
**                   (circumcenter if ABflag==0, barycenter if ABflag==1)
**                   u12,v12,u23,v23,u31,v31, 
**                   (middle edge points of the macro-triangle V1 V2 V3)
**                   u41,v41,u42,v42,u43,v43 
**                   (if ABflag==1 intersections between lines connecting 
**                    middle edge points of macro-triangle and lines connecting
**                    barycenter with vertices of macro-triangle) 
**                   where Vi=(ui,vi) i=1,2,3,12,23,31,4,41,42,43 are the vertices.
**
** Let Q_ntriang be the quadratic polynomial Powell-Sabin patch over the ntriang-th 
** sub-triangle of a
**
** A-type Powell-Sabin subdivision if ABflag==0 or 
** B-type Powell-Sabin subdivision if ABflag==1,
**
** and let A_ntriang be the area of that sub-triangle.
**
** The sub-triangles are sorted following the index 'ntriang' as follows
**
** V4 V1 V12
** V4 V12 V2
** V4 V2 V23    for a A-type subdivision
** V4 V23 V3
** V4 V3 V31
** V4 V31 V1
**
** or
**
** V4 V41 V12
** V41 V1 V12
** V4 V12 V42
** V42 V12 V2
** V4 V42 V23
** V42 V2 V23   for a B-type subdivision
** V4 V23 V43
** V43 V23 V3
** V4 V43 V31
** V43 V3 V31
** V4 V31 V41
** V41 V31 V1.
**
** 
** Then, apart from the constant (1/2A_ntriang^2), this function computes
**
**    d Q_ntriang    
** (  -----------  )    if ntype==1,      
**       d xx         
**
**    d Q_ntriang    
** (  -----------  )    if ntype==2,      
**       d xy         
**
**    d Q_ntriang    
** (  -----------  )    if ntype==3,      
**       d yy        
**
** where Q is the quadratic polynomial Powell-Sabin patch 
** (having the array d[] as control points) over the whole macro-triangle
** with vertices V1,V2,V3.   
**
** >>>>>>>>>>>>>>>>>>>>>>>>>
** 
** int circoscrivi(double x1,double y1,double x2,double y2,
**                 double x3,double y3,double* xc,double* yc,double* r) 
**
** It computes the circle passing through the points  
** (x1,y1), (x2,y2), (x3,y3).
**
** The circle is identified by three output parameters:
** the center, by its coordinates (xc,yc), and the radius r 
** if the function 'circoscrivi==0'. If 'circoscrivi==1' the three points 
** are aligned or nearby aligned
** (that is the denominator is smaller than EPS_ALLIN).
**
** If r==NULL the computation of the radius is not performed.
**
** >>>>>>>>>>>>>>>>>>>>>>>>>
** 
** double triangle_max_angle(double x1,double y1,double x2,double y2,
**                           double x3,double y3)
**
** This function computes the greatest angle, in degrees, among the three ones
** of the triangle with vertices (x1,y1), (x2,y2) and (x3,y3).
**
** >>>>>>>>>>>>>>>>>>>>>>>>>
**
** double uu(int ABflag,int ntriang,int nvertex)
**
** GLOBAL VARIABLES: u4,v4 
**                   (circumcenter if ABflag==0, barycenter if ABflag==1)
**                   u12,v12,u23,v23,u31,v31, 
**                   (middle edge points of macro-triangle V1 V2 V3)
**                   u41,v41,u42,v42,u43,v43 
**                   (if ABflag==1 intersections between lines connecting 
**                    middle edge points of macro-triangle and lines connecting
**                    barycenter with vertices of macro-triangle) 
**
** This function returns the x-coordinate of the nvertex-th vertex
** (among 1,2,3) of the ntriang-th subtriangle of an
** A-type Powell-Sabin subdivision if ABflag==0 or a
** B-type Powell-Sabin subdivision if ABflag==1.
**
** For an A-type subdivision the subtriangles used by 'ntriang' are ordered
** as follows, from 0 to 5:
** V4 V1 V12
** V4 V12 V2
** V4 V2 V23
** V4 V23 V3
** V4 V3 V31
** V4 V31 V1
**
** where Vi=(ui,vi) i=1,2,3,12,23,31,4 are the vertices.
**
** For a B-type subdivision the subtriangles used by 'ntriang' are ordered
** as follows, from 0 to 11:
** V4 V41 V12
** V41 V1 V12
** V4 V12 V42
** V42 V12 V2
** V4 V42 V23
** V42 V2 V23
** V4 V23 V43
** V43 V23 V3
** V4 V43 V31
** V43 V3 V31
** V4 V31 V41
** V41 V31 V1
**
** where Vi=(ui,vi) i=1,2,3,12,23,31,4,41,42,43 are the vertices.
**
** >>>>>>>>>>>>>>>>>>>>>>>>>>>>>
**
** double vv(int ABflag,int ntriang,int nvertex)
**
** GLOBAL VARIABLES: u4,v4 
**                   (circumcenter if ABflag==0, barycenter if ABflag==1)
**                   u12,v12,u23,v23,u31,v31, 
**                   (middle edge points of macro-triangle V1 V2 V3)
**                   u41,v41,u42,v42,u43,v43 
**                   (if ABflag==1 intersections between lines connecting 
**                    middle edge points of macro-triangle and lines connecting
**                    barycenter with vertices of macro-triangle) 
**
** This function returns the y-coordinate of the nvertex-th vertex
** (among 1,2,3) of the ntriang-th subtriangle of an
** A-type Powell-Sabin subdivision if ABflag==0 or a
** B-type Powell-Sabin subdivision if ABflag==1.
**
** For an A-type subdivision the subtriangles used by 'ntriang' are ordered
** as follows, from 0 to 5:
** V4 V1 V12
** V4 V12 V2
** V4 V2 V23
** V4 V23 V3
** V4 V3 V31
** V4 V31 V1
**
** where Vi=(ui,vi) i=1,2,3,12,23,31,4 are the vertices.
**
** For a B-type subdivision the subtriangles used by 'ntriang' are ordered
** as follows, from 0 to 11:
** V4 V41 V12
** V41 V1 V12
** V4 V12 V42
** V42 V12 V2
** V4 V42 V23
** V42 V2 V23
** V4 V23 V43
** V43 V23 V3
** V4 V43 V31
** V43 V3 V31
** V4 V31 V41
** V41 V31 V1
**
** where Vi=(ui,vi) i=1,2,3,12,23,31,4,41,42,43 are the vertices.
**
** -------------------------------------------------------------------------------
*/

#include<stdio.h>
#include<stdlib.h>
#include<math.h>
#define MAX_NUM_NODI 100000
#define WIDTH 20   /* Format output width for the value of the functional */
#define PREC 10    /* Decimal precision width for the value of the functional */

#define EPS_ALLIN 1.e-13
#define PIGRECO 3.14159265358979323846

double triangle_max_angle(), piano(), uu(), vv(),
       eval_derivative_from_cpt();
int circoscrivi();

/* Declarations for global variables */
double z1,z2,z3;
double u1,v1,u2,v2,u3,v3,t1,t2,t3;
double zu1,zu2,zu3,zv1,zv2,zv3;
double u4,v4,u12,v12,u23,v23,u31,v31,u41,v41,u42,v42,u43,v43;
int IndTA[6][3][3]={
  {{2,1,0},{13,12,31},{18,31,31}},
  {{4,3,2},{14,13,31},{18,31,31}},
  {{6,5,4},{15,14,31},{18,31,31}},
  {{8,7,6},{16,15,31},{18,31,31}},
  {{10,9,8},{17,16,31},{18,31,31}},
  {{0,11,10},{12,17,31},{18,31,31}}
  };
int IndTB[12][3][3]={
  {{2,13,24},{14,27,31},{30,31,31}},
  {{2,1,0},{13,12,31},{24,31,31}},
  {{25,15,2},{28,14,31},{30,31,31}},
  {{4,3,2},{16,15,31},{25,31,31}},
  {{6,17,25},{18,28,31},{30,31,31}},
  {{6,5,4},{17,16,31},{25,31,31}},
  {{26,19,6},{29,18,31},{30,31,31}},
  {{8,7,6},{20,19,31},{26,31,31}},
  {{10,21,26},{22,29,31},{30,31,31}},
  {{10,9,8},{21,20,31},{26,31,31}},
  {{24,23,10},{27,22,31},{30,31,31}},
  {{0,11,10},{12,23,31},{24,31,31}}
  };

void main(argc,argv)
int argc;
char *argv[];
{
  FILE *infile, *trifile, *derfile;
  char base_buffer[20];
  long int num_nodo,iv1,iv2,iv3,nt;
  double x[MAX_NUM_NODI],y[MAX_NUM_NODI],z[MAX_NUM_NODI],
	 zx[MAX_NUM_NODI],zy[MAX_NUM_NODI];
  double d[31],At,At1,At2,At3;
  /* Control points of Powell-Sabin element labeled from 0 to 30
     ( the worst case is the B-type subdivision) */
  double trash;
  int ct1,ct2,pwsb_mode;
  double cpt0[216],temp_calc,temp_calc1,temp_calc2,temp_calc3,phi;

 infile=NULL; trifile=NULL; derfile=NULL;
 if (argc==1)
   {
    printf("\nStart the program with three parameters:\n");
    printf("First parameter: data file name.\n");
    printf("Second parameter: triangulation file name.\n");
    printf("Third parameter: derivative file name.\n");
    exit(1);
   }
 /* Reading */
 infile=fopen(argv[1],"r");
 if (infile==NULL)
   {
    printf("Cannot open file: %s.\n",argv[1]);
    exit(1);
   }
 if (argc==2)
   {
    printf("Missing triangulation file.\n");
    exit(1);
   }
 if (argc==3)
   {
    printf("Missing derivative file.\n");
    exit(1);
   }
 /* opening derivative file */
 derfile=fopen(argv[3],"r");
 if (derfile==NULL)
   {
    printf("Cannot open file: %s.\n",argv[3]);
    exit(1);
   }
 /* data reading */
 printf("Wait for data reading.\n");
 num_nodo=0;
 while (!feof(infile))
 {
  if (num_nodo<MAX_NUM_NODI)
   {
    fscanf(infile," %lf %lf %lf ",&x[num_nodo],&y[num_nodo],&z[num_nodo]);
      /* spaces are essential part of the input format string */
    if (feof(derfile))
    {
     printf("Derivative file %s in a no correct format.\n",argv[3]);
     exit(1);
    }
    fscanf(derfile," %lf %lf %lf %lf %lf ",
	   &zx[num_nodo],&zy[num_nodo],&trash,&trash,&trash);
      /* spaces are essential part of the input format string */
    num_nodo=num_nodo+1;
   }
   else
   {
    printf("Too many input data.\n");
    exit(1);
   }
 }
 if (num_nodo<3)
    {
     printf("I need at least three input points.\n");
     exit(1);
    }
 fclose(infile);
 fclose(derfile);
 /* end of data reading */

 /* opening triangulation file */
 trifile=fopen(argv[2],"r");
 if (trifile==NULL)
   {
    printf("Cannot open file: %s.\n",argv[2]);
    exit(1);
   }
 if (fgetc(trifile)!='C')
   {
    printf("Triangulation file %s in a no correct format.\n",argv[2]);
    exit(1);
   }
 else fgetc(trifile); /* carriage return */

 /* MAIN LOOP OF FUNCTIONAL COMPUTATION OVER TRIANGLES */
 printf("Starting functional computation for each triangle.\n");
 phi=0.;
 fscanf(trifile,"%ld ",&iv1);
 nt=1;
 /* Counter of current triangle */
 do
   {
    fscanf(trifile,"%ld %ld ",&iv2,&iv3);
    while(iv3!=-1)
     {
      if ((iv1<iv2)&&(iv1<iv3))
	{/* New triangle: beginning computations */
	 u1=x[iv1]; u2=x[iv2]; u3=x[iv3];
	 v1=y[iv1]; v2=y[iv2]; v3=y[iv3];
	 z1=z[iv1]; z2=z[iv2]; z3=z[iv3];
	 zu1=zx[iv1]; zu2=zx[iv2]; zu3=zx[iv3];
	 zv1=zy[iv1]; zv2=zy[iv2]; zv3=zy[iv3];
         u12=(u1+u2)/2.; v12=(v1+v2)/2.;
         u23=(u2+u3)/2.; v23=(v2+v3)/2.;
         u31=(u3+u1)/2.; v31=(v3+v1)/2.;
         /* selection: 0 uses circumcenter, 1 uses barycenter */
         if (triangle_max_angle(u1,v1,u2,v2,u3,v3)>=75.)
           pwsb_mode=1; else pwsb_mode=0;
         if (pwsb_mode==0)  /* Now I do not need to check for the alignment  (err==1)*/
           circoscrivi(u1,v1,u2,v2,u3,v3,&u4,&v4,NULL);
          else  /* pwsb_mode==1 */
          { 
           u4=(u1+u2+u3)/3.; v4=(v1+v2+v3)/3.;
           t1=-((u3-u2)*(u2-u4)+(v3-v2)*(v2-v4))/
               (pow(u3-u2,2.)+pow(v3-v2,2.));
           t2=-((u1-u3)*(u3-u4)+(v1-v3)*(v3-v4))/
               (pow(u1-u3,2.)+pow(v1-v3,2.));
           t3=-((u2-u1)*(u1-u4)+(v2-v1)*(v1-v4))/
               (pow(u2-u1,2.)+pow(v2-v1,2.));
           u41=(u31+u12)/2.; v41=(v31+v12)/2.;
           u42=(u12+u23)/2.; v42=(v12+v23)/2.;
           u43=(u23+u31)/2.; v43=(v23+v31)/2.;
          }
	  if (pwsb_mode==0)
	  {
	   At1=fabs((u1*(v2-v4)+u2*(v4-v1)+u4*(v1-v2))/2.);
           At2=fabs((u2*(v3-v4)+u3*(v4-v2)+u4*(v2-v3))/2.);
	   At3=fabs(-(u1*(v3-v4)+u3*(v4-v1)+u4*(v1-v3))/2.);
	   /* Really there are six areas: At1/2 At1/2 At2/2 At2/2 At3/2 At3/2 */
	  }
	  else At=fabs((u2*v3-u3*v2+u3*v1-u1*v3+u1*v2-u2*v1)/2.);
          /* array of points using pwsb_mode */
          d[0]=z1;   /* Common part */
          d[4]=z2;
          d[8]=z3;
          d[11]=z1+(zu1*(u3-u1)+zv1*(v3-v1))/4.;
          d[1]=z1+(zu1*(u2-u1)+zv1*(v2-v1))/4.;
          d[3]=z2+(zu2*(u1-u2)+zv2*(v1-v2))/4.;
          d[5]=z2+(zu2*(u3-u2)+zv2*(v3-v2))/4.;
          d[7]=z3+(zu3*(u2-u3)+zv3*(v2-v3))/4.;
          d[9]=z3+(zu3*(u1-u3)+zv3*(v1-v3))/4.;
          if (pwsb_mode==0) /* Continuing with A-type subdivision */
          {
           d[12]=z1+(zu1*(u4-u1)+zv1*(v4-v1))/2.;
           d[14]=z2+(zu2*(u4-u2)+zv2*(v4-v2))/2.;
           d[16]=z3+(zu3*(u4-u3)+zv3*(v4-v3))/2.;
           d[2]=(d[1]+d[3])/2.;
           d[6]=(d[5]+d[7])/2.;
           d[10]=(d[9]+d[11])/2.;
           d[13]=(d[12]+d[14])/2.;
           d[15]=(d[14]+d[16])/2.;
           d[17]=(d[16]+d[12])/2.;
           d[18]=piano(u4,v4,(u4+u12)/2.,(v4+v12)/2.,d[13],
                             (u4+u23)/2.,(v4+v23)/2.,d[15],
                             (u4+u31)/2.,(v4+v31)/2.,d[17]);
          }
          else  /* pwsb_mode==1  Continuing with B-type subdivision */
          {
           d[12]=z1+(zu1*(u41-u1)+zv1*(v41-v1))/2.;
           d[16]=z2+(zu2*(u42-u2)+zv2*(v42-v2))/2.;
           d[20]=z3+(zu3*(u43-u3)+zv3*(v43-v3))/2.;
           d[18]=2./3.*(d[16]+d[20])+
                        (-2./3.+t1)*z2+(1./3.-t1)*z3+
                        (2*t1-1)*(d[7]-d[5]);
           d[22]=2./3.*(d[20]+d[12])+
                        (-2./3.+t2)*z3+(1./3.-t2)*z1+
                        (2*t2-1)*(d[11]-d[9]);
           d[14]=2./3.*(d[12]+d[16])+
                        (-2./3.+t3)*z1+(1./3.-t3)*z2+
                        (2*t3-1)*(d[3]-d[1]);
           d[2]=d[1]/2.+d[3]/2.;
           d[6]=d[5]/2.+d[7]/2.;
           d[10]=d[9]/2.+d[11]/2.;
           d[13]=d[14]*3./4.+d[1]/4.;
           d[15]=d[14]*3./4.+d[3]/4.;
           d[17]=d[18]*3./4.+d[5]/4.;
           d[19]=d[18]*3./4.+d[7]/4.;
           d[21]=d[22]*3./4.+d[9]/4.;
           d[23]=d[22]*3./4.+d[11]/4.;
           d[27]=d[22]/2.+d[14]/2.;
           d[28]=d[14]/2.+d[18]/2.;
           d[29]=d[18]/2.+d[22]/2.;
           d[24]=piano(u41,v41,(u4+u41)/2.,(v4+v41)/2.,d[27],
                               (u41+u12)/2.,(v41+v12)/2.,d[13],
                               (u41+u31)/2.,(v41+v31)/2.,d[23]);
           d[25]=piano(u42,v42,(u4+u42)/2.,(v4+v42)/2.,d[28],
                               (u42+u12)/2.,(v42+v12)/2.,d[15],
                               (u42+u23)/2.,(v42+v23)/2.,d[17]);
           d[26]=piano(u43,v43,(u4+u43)/2.,(v4+v43)/2.,d[29],
                               (u43+u23)/2.,(v43+v23)/2.,d[19],
                               (u43+u31)/2.,(v43+v31)/2.,d[21]);
           d[30]=piano(u4,v4,(u4+u41)/2.,(v4+v41)/2.,d[27],
                             (u4+u42)/2.,(v4+v42)/2.,d[28],
                             (u4+u43)/2.,(v4+v43)/2.,d[29]);
          }
         /* end of array of control points */
	  /* Computation of second derivatives for the interpolation function
             (their control points), computation of the functional value over
             the triangle together with and updating */
         switch(pwsb_mode)
         {
          case 0:
          temp_calc=0;
          for (ct1=0;ct1<2;ct1++)     
          {    
           /* The constant 1./(4.*pow(a,3.)) comes from
              (1/2a^2)^2 * (2a / ( (0+1)*(0+2)) )
              where a is the area of the micro-triangle
           */
           temp_calc1=(pow(eval_derivative_from_cpt(0,1,ct1,d),2.)+
                       2*pow(eval_derivative_from_cpt(0,2,ct1,d),2.)+
                       pow(eval_derivative_from_cpt(0,3,ct1,d),2.))*
                       1./(4.*pow(At1/2.,3.));
           temp_calc2=(pow(eval_derivative_from_cpt(0,1,ct1+2,d),2.)+
                       2*pow(eval_derivative_from_cpt(0,2,ct1+2,d),2.)+
                       pow(eval_derivative_from_cpt(0,3,ct1+2,d),2.))*
                       1./(4.*pow(At2/2.,3.));
           temp_calc3=(pow(eval_derivative_from_cpt(0,1,ct1+4,d),2.)+
                       2*pow(eval_derivative_from_cpt(0,2,ct1+4,d),2.)+
                       pow(eval_derivative_from_cpt(0,3,ct1+4,d),2.))*
                       1./(4.*pow(At3/2.,3.));
           temp_calc+=temp_calc1+temp_calc2+temp_calc3;
          }  
	 break;
         case 1:
         temp_calc=0;
         for (ct1=0;ct1<6;ct1++)
          {
           /* The constant 1./(4.*pow(a,3.)) comes from
              (1/2a^2)^2 * (2a / ( (0+1)*(0+2)) )
              where a is the area of the micro-triangle
           */
           temp_calc1=(pow(eval_derivative_from_cpt(1,1,2*ct1,d),2.)+
                       2*pow(eval_derivative_from_cpt(1,2,2*ct1,d),2.)+
                       pow(eval_derivative_from_cpt(1,3,2*ct1,d),2.))*
                       1./(4.*pow(At/24.,3.));      
           temp_calc2=(pow(eval_derivative_from_cpt(1,1,2*ct1+1,d),2.)+
                       2*pow(eval_derivative_from_cpt(1,2,2*ct1+1,d),2.)+
                       pow(eval_derivative_from_cpt(1,3,2*ct1+1,d),2.))*
                       1./(4.*pow(At/8.,3.));
           temp_calc+=temp_calc1+temp_calc2;
          }
         break;
         }
         phi=phi+temp_calc;

	 if (nt%100==0)
	   printf("Done %ld triangles.\n",nt); nt++;
	} /* End of current triangle */
      iv2=iv3;
      fscanf(trifile,"%ld ",&iv3);
     }  /* End of the reading of the row in the triangulation file */
    fscanf(trifile,"%s ",base_buffer);
    iv1=atol(base_buffer);
   }  /* End of reading triangulation file at the loop exit */
 while (base_buffer[0]!='B');
 fclose(trifile);
 printf("Functional value: %*.*f\n",WIDTH,PREC,phi);
}

double piano(xnew,ynew,x1,y1,z1,x2,y2,z2,x3,y3,z3)
double xnew,ynew,x1,y1,z1,x2,y2,z2,x3,y3,z3;
{
 double ret_val;

 ret_val=x1*(y2-y3)+x2*(y3-y1)+x3*(y1-y2);
 if (fabs(ret_val)<EPS_ALLIN)
 {
  printf("Cannot find the plane passing for three nearby aligned points.\n");
  exit(1);
 }
 ret_val=-(xnew*(y1*(z2-z3)+y2*(z3-z1)+y3*(z1-z2))
	   -ynew*(x1*(z2-z3)+x2*(z3-z1)+x3*(z1-z2))
	   +x1*(y3*z2-y2*z3)+x2*(y1*z3-y3*z1)+x3*(y2*z1-y1*z2))
	  /ret_val;
 return ret_val;
}

double eval_derivative_from_cpt(ABflag,ntype,ntriang,d)
 int ABflag; /* 0 A-type Powell-Sabin   1 B-type Powell-Sabin */
 int ntype,ntriang;
 double d[]; /* Pointer to z-coordinate of the array of control points used */
{
 double temp_calc,uu1,uu2,uu3,vv1,vv2,vv3;

 switch(ABflag)
 {
  case 0:
  uu1=uu(0,ntriang,1); vv1=vv(0,ntriang,1);
  uu2=uu(0,ntriang,2); vv2=vv(0,ntriang,2);
  uu3=uu(0,ntriang,3); vv3=vv(0,ntriang,3);
  switch(ntype)
  {
   case 1:
   temp_calc=d[IndTA[ntriang][2][0]]*(vv3-vv2)*(vv3-vv2);
   temp_calc+=d[IndTA[ntriang][0][2]]*(vv1-vv3)*(vv1-vv3);
   temp_calc+=d[IndTA[ntriang][0][0]]*(vv2-vv1)*(vv2-vv1);
   temp_calc+=d[IndTA[ntriang][1][1]]*2*(vv3-vv2)*(vv1-vv3);
   temp_calc+=d[IndTA[ntriang][1][0]]*2*(vv3-vv2)*(vv2-vv1);
   temp_calc+=d[IndTA[ntriang][0][1]]*2*(vv1-vv3)*(vv2-vv1);
   break;
   case 2:
   temp_calc=-d[IndTA[ntriang][2][0]]*(vv3-vv2)*(uu3-uu2);
   temp_calc-=d[IndTA[ntriang][0][2]]*(vv1-vv3)*(uu1-uu3);
   temp_calc-=d[IndTA[ntriang][0][0]]*(vv2-vv1)*(uu2-uu1);
   temp_calc-=d[IndTA[ntriang][1][1]]*
	      ((vv3-vv2)*(uu1-uu3)+(vv1-vv3)*(uu3-uu2));
   temp_calc-=d[IndTA[ntriang][1][0]]*
	      ((vv3-vv2)*(uu2-uu1)+(vv2-vv1)*(uu3-uu2));
   temp_calc-=d[IndTA[ntriang][0][1]]*
	      ((vv1-vv3)*(uu2-uu1)+(vv2-vv1)*(uu1-uu3));
   break;
   case 3:
   temp_calc=d[IndTA[ntriang][2][0]]*(uu3-uu2)*(uu3-uu2);
   temp_calc+=d[IndTA[ntriang][0][2]]*(uu1-uu3)*(uu1-uu3);
   temp_calc+=d[IndTA[ntriang][0][0]]*(uu2-uu1)*(uu2-uu1);
   temp_calc+=d[IndTA[ntriang][1][1]]*2*(uu3-uu2)*(uu1-uu3);
   temp_calc+=d[IndTA[ntriang][1][0]]*2*(uu3-uu2)*(uu2-uu1);
   temp_calc+=d[IndTA[ntriang][0][1]]*2*(uu1-uu3)*(uu2-uu1);
   break;
  } /* end switch over ntype */
  break;
  case 1:
  uu1=uu(1,ntriang,1); vv1=vv(1,ntriang,1);
  uu2=uu(1,ntriang,2); vv2=vv(1,ntriang,2);
  uu3=uu(1,ntriang,3); vv3=vv(1,ntriang,3);
  switch(ntype)
  {
   case 1:
   temp_calc=d[IndTB[ntriang][2][0]]*(vv3-vv2)*(vv3-vv2);
   temp_calc+=d[IndTB[ntriang][0][2]]*(vv1-vv3)*(vv1-vv3);
   temp_calc+=d[IndTB[ntriang][0][0]]*(vv2-vv1)*(vv2-vv1);
   temp_calc+=d[IndTB[ntriang][1][1]]*2*(vv3-vv2)*(vv1-vv3);
   temp_calc+=d[IndTB[ntriang][1][0]]*2*(vv3-vv2)*(vv2-vv1);
   temp_calc+=d[IndTB[ntriang][0][1]]*2*(vv1-vv3)*(vv2-vv1);
   break;
   case 2:
   temp_calc=-d[IndTB[ntriang][2][0]]*(vv3-vv2)*(uu3-uu2);
   temp_calc-=d[IndTB[ntriang][0][2]]*(vv1-vv3)*(uu1-uu3);
   temp_calc-=d[IndTB[ntriang][0][0]]*(vv2-vv1)*(uu2-uu1);
   temp_calc-=d[IndTB[ntriang][1][1]]*
	      ((vv3-vv2)*(uu1-uu3)+(vv1-vv3)*(uu3-uu2));
   temp_calc-=d[IndTB[ntriang][1][0]]*
	      ((vv3-vv2)*(uu2-uu1)+(vv2-vv1)*(uu3-uu2));
   temp_calc-=d[IndTB[ntriang][0][1]]*
	      ((vv1-vv3)*(uu2-uu1)+(vv2-vv1)*(uu1-uu3));
   break;
   case 3:
   temp_calc=d[IndTB[ntriang][2][0]]*(uu3-uu2)*(uu3-uu2);
   temp_calc+=d[IndTB[ntriang][0][2]]*(uu1-uu3)*(uu1-uu3);
   temp_calc+=d[IndTB[ntriang][0][0]]*(uu2-uu1)*(uu2-uu1);
   temp_calc+=d[IndTB[ntriang][1][1]]*2*(uu3-uu2)*(uu1-uu3);
   temp_calc+=d[IndTB[ntriang][1][0]]*2*(uu3-uu2)*(uu2-uu1);
   temp_calc+=d[IndTB[ntriang][0][1]]*2*(uu1-uu3)*(uu2-uu1);
   break;
  } /* end switch over ntype */
  break;
 } /* end switch over ABflag */
 return(temp_calc);
}

int circoscrivi(x1,y1,x2,y2,x3,y3,xc,yc,r)
double x1,y1,x2,y2,x3,y3,*xc,*yc,*r;
{
double Denom;

Denom=(2*(x1*(y2-y3)+x2*(y3-y1)+x3*(y1-y2)));
if (fabs(Denom)<EPS_ALLIN) return(1);
*xc=(pow(x1,2.)*(y2-y3)
     +pow(x2,2.)*(y3-y1)
     +(y1-y2)*(pow(x3,2.)
	       +(y2-y3)*(y1-y3)) )
    /Denom;
*yc=-(pow(x1,2.)*(x2-x3)
      -x1*(pow(x2,2.)
      -pow(x3,2.)
      +(y2-y3)*(y2+y3))
      +pow(x2,2.)*x3
      -x2*(pow(x3,2.)
	   +(y3-y1)*(y1+y3))
      +x3*(y2-y1)*(y1+y2) )
     /Denom;
if (r!=NULL) *r=sqrt(pow((*xc-x1),2.)+pow((*yc-y1),2.));
return(0);
}

double triangle_max_angle(x1,y1,x2,y2,x3,y3)
double x1,y1,x2,y2,x3,y3;
{
 double t1,t2,t3;

 t1=acos(1./(sqrt(pow(x2-x1,2.)+pow(y2-y1,2.))*
	     sqrt(pow(x3-x1,2.)+pow(y3-y1,2.)))*
	 ((x2-x1)*(x3-x1)+(y2-y1)*(y3-y1)));
 t2=acos(1./(sqrt(pow(x3-x2,2.)+pow(y3-y2,2.))*
	     sqrt(pow(x1-x2,2.)+pow(y1-y2,2.)))*
	 ((x3-x2)*(x1-x2)+(y3-y2)*(y1-y2)));
 t3=acos(1./(sqrt(pow(x1-x3,2.)+pow(y1-y3,2.))*
	     sqrt(pow(x2-x3,2.)+pow(y2-y3,2.)))*
	 ((x1-x3)*(x2-x3)+(y1-y3)*(y2-y3)));
 if (t2>t1) t1=t2;
 if (t3>t1) t1=t3;
 return(t1/PIGRECO*180.);
}

double uu(ABflag,ntriang,nvertex)
 int ABflag; /* 0 A-type Powell-Sabin    1 B-type Powell-Sabin */
 int ntriang,nvertex;
{
 switch(ABflag)
 {
  case 0: switch(nvertex)
	  {
	   case 1: return(u4);
		   break;
	   case 2: if (ntriang==0) return(u1);
		   else if (ntriang==1) return(u12);
		   else if (ntriang==2) return(u2);
		   else if (ntriang==3) return(u23);
		   else if (ntriang==4) return(u3);
		   else if (ntriang==5) return(u31);
		   break;
	   case 3: if (ntriang==0) return(u12);
		   else if (ntriang==1) return(u2);
		   else if (ntriang==2) return(u23);
		   else if (ntriang==3) return(u3);
		   else if (ntriang==4) return(u31);
		   else if (ntriang==5) return(u1);
		   break;
	  }  /* end switch over nvertex */
	  break;
  case 1: switch(nvertex)
	   {
	    case 1: if (ntriang==0) return(u4);
		    else if (ntriang==1) return(u41);
		    else if (ntriang==2) return(u4);
		    else if (ntriang==3) return(u42);
		    else if (ntriang==4) return(u4);
		    else if (ntriang==5) return(u42);
		    else if (ntriang==6) return(u4);
		    else if (ntriang==7) return(u43);
		    else if (ntriang==8) return(u4);
		    else if (ntriang==9) return(u43);
		    else if (ntriang==10) return(u4);
		    else if (ntriang==11) return(u41);
		    break;
	    case 2: if (ntriang==0) return(u41);
		    else if (ntriang==1) return(u1);
		    else if (ntriang==2) return(u12);
		    else if (ntriang==3) return(u12);
		    else if (ntriang==4) return(u42);
		    else if (ntriang==5) return(u2);
		    else if (ntriang==6) return(u23);
		    else if (ntriang==7) return(u23);
		    else if (ntriang==8) return(u43);
		    else if (ntriang==9) return(u3);
		    else if (ntriang==10) return(u31);
		    else if (ntriang==11) return(u31);
		    break;
	    case 3: if (ntriang==0) return(u12);
		    else if (ntriang==1) return(u12);
		    else if (ntriang==2) return(u42);
		    else if (ntriang==3) return(u2);
		    else if (ntriang==4) return(u23);
		    else if (ntriang==5) return(u23);
		    else if (ntriang==6) return(u43);
		    else if (ntriang==7) return(u3);
		    else if (ntriang==8) return(u31);
		    else if (ntriang==9) return(u31);
		    else if (ntriang==10) return(u41);
		    else if (ntriang==11) return(u1);
		    break;
	   } /* end switch over nvertex */
	   break;
 } /* End switch over ABflag */
}

double vv(ABflag,ntriang,nvertex)
 int ABflag; /* 0 A-type Powell-Sabin   1 B-type Powell-Sabin */
 int ntriang,nvertex;
{
 switch(ABflag)
 {
  case 0: switch(nvertex)
	  {
	   case 1: return(v4);
		   break;
	   case 2: if (ntriang==0) return(v1);
		   else if (ntriang==1) return(v12);
		   else if (ntriang==2) return(v2);
		   else if (ntriang==3) return(v23);
		   else if (ntriang==4) return(v3);
		   else if (ntriang==5) return(v31);
		   break;
	   case 3: if (ntriang==0) return(v12);
		   else if (ntriang==1) return(v2);
		   else if (ntriang==2) return(v23);
		   else if (ntriang==3) return(v3);
		   else if (ntriang==4) return(v31);
		   else if (ntriang==5) return(v1);
		   break;
	  }  /* end switch over nvertex */
	  break;
  case 1: switch(nvertex)
	   {
	    case 1: if (ntriang==0) return(v4);
		    else if (ntriang==1) return(v41);
		    else if (ntriang==2) return(v4);
		    else if (ntriang==3) return(v42);
		    else if (ntriang==4) return(v4);
		    else if (ntriang==5) return(v42);
		    else if (ntriang==6) return(v4);
		    else if (ntriang==7) return(v43);
		    else if (ntriang==8) return(v4);
		    else if (ntriang==9) return(v43);
		    else if (ntriang==10) return(v4);
		    else if (ntriang==11) return(v41);
		    break;
	    case 2: if (ntriang==0) return(v41);
		    else if (ntriang==1) return(v1);
		    else if (ntriang==2) return(v12);
		    else if (ntriang==3) return(v12);
		    else if (ntriang==4) return(v42);
		    else if (ntriang==5) return(v2);
		    else if (ntriang==6) return(v23);
		    else if (ntriang==7) return(v23);
		    else if (ntriang==8) return(v43);
		    else if (ntriang==9) return(v3);
		    else if (ntriang==10) return(v31);
		    else if (ntriang==11) return(v31);
		    break;
	    case 3: if (ntriang==0) return(v12);
		    else if (ntriang==1) return(v12);
		    else if (ntriang==2) return(v42);
		    else if (ntriang==3) return(v2);
		    else if (ntriang==4) return(v23);
		    else if (ntriang==5) return(v23);
		    else if (ntriang==6) return(v43);
		    else if (ntriang==7) return(v3);
		    else if (ntriang==8) return(v31);
		    else if (ntriang==9) return(v31);
		    else if (ntriang==10) return(v41);
		    else if (ntriang==11) return(v1);
		    break;
	   } /* End switch over nvertex */
	   break;
 } /* End switch over ABflag */
}

















