/*
** DO_GLDER.C
**
** AUTHORS: 
**
**   Maria MORANDI CECCHI               Stefano DE MARCHI,            
**   University of Padua, Italy         University of Udine, Italy
**   email: mcecchi@math.unipd.it       email: demarchi@dimi.uniud.it
**   
**                       Damiano FASOLI
**                       Verona Software Srl, Italy 
**                       email: vrsoft@sis.it
**
**
**
** REVISION DATE: May, 1999  
**
**
** MODULES CALLED: DO_DER_A.C
**                 LINSIST.C 
**
** --------------------------------------------------------------------
** BACKGROUND
**
** Let (xi,yi,qi) i=0,...,N-1 be N points in the space R^3. Let us 
** consider their projections (xi,yi) onto the plane z=0.
** Let T be a triangulation of these points, that is a set of triangles 
** having all these points, and only them, as vertices so that two
** distinct triangles intersect at most in a common edge or in a common
** vertex.
** 
** The aim is to interpolate the points (xi,yi,qi) with a C1-function 
** of two variables. 
** The Q18 interpolation scheme builds such a function using a Q18 polynomial
** patch on each triangle, assuming to know the first and second derivatives
** at each vertex of the triangulation T.
** 
** The program computes the parameters which define the Q18 patch
** (in the following we shall refer to them as 
**  qix,qiy,qixx,qixy,qiyy i=0,...,N-1, or more often
** as to the 5N-dimensional vector v) in the
** "smoothest" possible way for a Q18 interpolating function over the whole
** triangulation T. To evaluate the "smoothness" of
** the Q18 interpolating function, it is minimized the functional
**
** phi(v)=   sum  [ Integral[ (Qxx^2 + 2*Qxy^2 + Qyy^2  ) dx dy ] ]     
**         t in T      t      
** 
** with respect to v.
**
**
** METHODOLOGY 
**
**               phi(v)= 1/2 * v' A v + g' v + c 
**
** is minimum when v solves the system
**
**                A v + g == 0.
** 
** The program computes 
**
**               c=phi(0), A, gi=phi(ei)-1/2*aii-c, 
**
** where ei is the i-th vector of the canonical basis of R^(5N), 
** and solves the resulting linear system with the Cholesky factorisation.
** 
**
** REFERENCES:
**
**  M. Morandi Cecchi, S. De Marchi, D. Fasoli,
**     "A Package for Representing C^1 Interpolating Surfaces: 
**      Application to the Lagoon of Venice's Bed."
**      Numer. Algorithms, 1999.
**
**  D. Fasoli,
**      Master's Thesis, University of Padua, 1995 (in Italian). 
**
**  R.E. Barnhill and G. Farin:
**    "C^1 quintic interpolation over triangles: 
**     Two explicit representations"
**     Int. J. Numer. Meth. Eng., 17(12): 1763-1778, 1981.
**
**  P. Alfeld:
**    "Derivative Generation for multivariate 
**     scattered data functional minimization", 
**     Comp. Aided Graph. Design,2: 281-296, 1985.
**
**
** ---------------------------------------------------------------
** SYNOPSIS:  do_glder file1 file2 [file3]
**
** file1 is the data file made up of 3-dimensional points. 
** The coordinates of the points may be separated by any 
** combination of spaces, tabs and carriage returns, 
** even though it is recommended to use a row for each point. 
** 
** file2 is the triangulation data file in the format generated by
** 'do_trian' program. This file has three sections; 
** the first, starting with a single capital letter C in the first row, 
** lists in subsequent blocks separated by -1 the index of
** a node and the indexes of its neighbours in a counterclockwise
** order. 
** The present program uses only this first section 
** and the heading of the second section, 
** which is made of a single capital letter B.
**
** file3 is the output derivative data file. 
** Note that the standard output is used if it is omitted. 
** The values in a same row refer to the same vertex of the 
** triangulation: 
**    the first column is the x-derivative;
**    the second one is the y-derivative;
**    the third one is the xx-derivative;
**    the fourth one is the xy-derivative;
**    the last one is the yy-derivative.
** Output format is handled by the two constants, WIDTH and PREC,
** that are the width of the output field, and the number 
** of decimal digits used to print.
**        
** The program supports at most MAX_NUM_NODI (#DEFINE constant) 
** input points.
** If more points are given, it exits with a warning message.
** 
** ------------------------------------------------------------------
** 
** SUBROUTINES AND FUNCTIONS DESCRIPTION:
**
**
** int global_index(int i,int iv1,int iv2,int iv3) 
**
** This function transforms i, a local index in the range 0,...,14 
** relative to the parameters 
**
**    qx(V1), qy(V1), qxx(V1), qxy(V1), qyy(V1),
**    qx(V2), qy(V2), qxx(V2), qxy(V2), qyy(V2) 
**    qx(V3), qy(V3), qxx(V3), qxy(V3), qyy(V3) 
** 
** for the triangle V1,V2,V3 
** into a global index according to
** the vertex among V1,V2,V3 to which it refers to. 
** Global indexes are sorted by the index of the node in 
** the triangulation and by the sequence of derivatives
** qx,qy,qxx,qxy,qyy.  
**
** Output: the global index of the parameter.
**
** --------------------------------------------------------------------------
*/

#include<stdio.h>
#include<stdlib.h>
#include<math.h>
#define MAX_NUM_NODI 100000
#define WIDTH 20 /* Format width of real numbers written to files*/
#define PREC 10  /* Decimal precision width of real numbers written to files */

double derivecptq18_v(),compute_d_rst_Q18(),compute_d_rst_Q18_null(),
       sum_cpt3_prod();
int global_index();

/* Declarations for global variables */
double q1,q2,q3;
double e1x,e1y,e2x,e2y,e3x,e3y,t1,t2,t3;

void main(argc,argv)
int argc;
char *argv[];
{
  FILE *infile, *trifile, *outfile;
  char base_buffer[20];
  int num_nodo,iv1,iv2,iv3;
  double x[MAX_NUM_NODI],y[MAX_NUM_NODI],z[MAX_NUM_NODI];
  double x1,y1,x2,y2,x3,y3,At;
  int i,j,ip_global,jp_global,nt;
  int ip,jp,r,s,err;
  double *A, *v, *gg, c, el_gg_temp, c_temp, temp_calc;
  double *DiagL;
  double cpt3[45][10];

 infile=NULL; trifile=NULL; outfile=NULL;
 A=NULL; v=NULL; gg=NULL; DiagL=NULL;
 if (argc==1)
   {
    printf("\nStart the program with three parameters:\n");
    printf("First paramete: data file name.\n");
    printf("Second parameter: triangulation file name.\n");
    printf("Third parameter: derivative file name.\n");
    printf("Omit third parameter for screen output.\n");
    exit(1);
   }
 infile=fopen(argv[1],"r");
 if (infile==NULL)
   {
    printf("Cannot open file: %s.\n",argv[1]);
    exit(1);
   }
 if (argc==2)
   {
    printf("Missing triangulation file.\n");
    exit(1);
   }
 /* data reading */
 printf("Waiting for data reading.\n");
 num_nodo=0;
 while (!feof(infile))
 {
  if (num_nodo<MAX_NUM_NODI)
   {
    fscanf(infile," %lf %lf %lf ",&x[num_nodo],&y[num_nodo],&z[num_nodo]);
      /* the spaces are essential part of the input format */
    num_nodo=num_nodo+1;
   }
   else
   {
    printf("Too many input data.\n");
    exit(1);
   }
 }
 if (num_nodo<3)
    {
     printf("I need at least three input points.\n");
     exit(1);
    }
 fclose(infile);
 /* end of data reading */
 /* opening triangulation file */
 trifile=fopen(argv[2],"r");
 if (trifile==NULL)
   {
    printf("Cannot open file: %s.\n",argv[2]);
    exit(1);
   }
 if (fgetc(trifile)!='C')
   {
    printf("File %s in a no correct format.\n",argv[2]);
    exit(1);
   }
 else fgetc(trifile); /* reading carriage return  */
 /* apro file output */
 if (argc==3) outfile=stdout; else outfile=fopen(argv[3],"w");
 if (outfile==NULL)
   {
    printf("Cannot open file: %s.\n",argv[3]);
    exit(1);
   }
  /* Creation of dynamical structures used by the program */
  A=malloc(sizeof(double)*25*num_nodo*num_nodo);
  if (A==NULL) {printf("Not enough memory\n"); exit(1); };
  gg=malloc(sizeof(double)*5*num_nodo);
  if (gg==NULL) {printf("Not enough memory\n"); exit(1); };
  v=malloc(sizeof(double)*5*num_nodo);
  if (v==NULL) {printf("Not enough memory\n"); exit(1); };
  DiagL=malloc(sizeof(double)*5*num_nodo);
  if (DiagL==NULL) {printf("Not enough memory\n"); exit(1); };
  /* Data initialization */
  c=0.;
  for (i=0;i<5*num_nodo;i++)
   {
    for (j=0;j<=i;++j) A[i*5*num_nodo+j]=0.;
    gg[i]=0.;
   }
 /* Main cycle over triangles  */
 fscanf(trifile,"%d ",&iv1);
 nt=1; /* Counter of the triangle being assembled */
 do
   {
    fscanf(trifile,"%d %d ",&iv2,&iv3);
    while(iv3!=-1)
     {
      if ((iv1<iv2)&&(iv1<iv3)) /* Each triangle must be considered once */
	{
	 /* New triangle: Begininning computations and assembling phase */
         x1=x[iv1]; x2=x[iv2]; x3=x[iv3];
	 y1=y[iv1]; y2=y[iv2]; y3=y[iv3];
	 q1=z[iv1]; q2=z[iv2]; q3=z[iv3];
	 e1x=x3-x2; e1y=y3-y2; e2x=x1-x3; e2y=y1-y3; e3x=x2-x1; e3y=y2-y1;
	 At=fabs((x2*y3-x3*y2+x3*y1-x1*y3+x1*y2-x2*y1)/2.); /* Area */
	 t1=-(e1x*e3x+e1y*e3y)/(pow(e1x,2.)+pow(e1y,2.));
	 t2=-(e2x*e1x+e2y*e1y)/(pow(e2x,2.)+pow(e2y,2.));
	 t3=-(e3x*e2x+e3y*e2y)/(pow(e3x,2.)+pow(e3y,2.));
	 /* computations and updating assembling phase */
	 /* Updating costant c */
	 c_temp=0.;
	 i=0; for (r=0;r<=3;++r) for (s=0;s<=3-r;++s)
	 {
	  temp_calc=compute_d_rst_Q18_null(r+2,s)*e1y*e1y;
	  temp_calc+=compute_d_rst_Q18_null(r,s+2)*e2y*e2y;
	  temp_calc+=compute_d_rst_Q18_null(r,s)*e3y*e3y;
	  temp_calc+=compute_d_rst_Q18_null(r+1,s+1)*2*e1y*e2y;
	  temp_calc+=compute_d_rst_Q18_null(r+1,s)*2*e1y*e3y;
	  temp_calc+=compute_d_rst_Q18_null(r,s+1)*2*e2y*e3y;
	  cpt3[0][i]=temp_calc;

	  temp_calc=-compute_d_rst_Q18_null(r+2,s)*e1y*e1x;
	  temp_calc-=compute_d_rst_Q18_null(r,s+2)*e2y*e2x;
	  temp_calc-=compute_d_rst_Q18_null(r,s)*e3y*e3x;
	  temp_calc-=compute_d_rst_Q18_null(r+1,s+1)*(e1y*e2x+e2y*e1x);
	  temp_calc-=compute_d_rst_Q18_null(r+1,s)*(e1y*e3x+e3y*e1x);
	  temp_calc-=compute_d_rst_Q18_null(r,s+1)*(e2y*e3x+e3y*e2x);
	  cpt3[1][i]=temp_calc;

	  temp_calc=compute_d_rst_Q18_null(r+2,s)*e1x*e1x;
	  temp_calc+=compute_d_rst_Q18_null(r,s+2)*e2x*e2x;
	  temp_calc+=compute_d_rst_Q18_null(r,s)*e3x*e3x;
	  temp_calc+=compute_d_rst_Q18_null(r+1,s+1)*2*e1x*e2x;
	  temp_calc+=compute_d_rst_Q18_null(r+1,s)*2*e1x*e3x;
	  temp_calc+=compute_d_rst_Q18_null(r,s+1)*2*e2x*e3x;
	  cpt3[2][i]=temp_calc;
	  i++;
	 }
	 c_temp=sum_cpt3_prod(cpt3[0],cpt3[0])+
		2.*sum_cpt3_prod(cpt3[1],cpt3[1])+
		sum_cpt3_prod(cpt3[2],cpt3[2]);
	 c_temp=c_temp*25./(28.*pow(At,3));
	 /* The constant 25./(28.*pow(At,3)) comes from
            (5./At^2)^2 * (2*At/((6+1)(6+2))) */
         c=c+c_temp;
	 /* Updating vector gg */
	 /*   First phase */
	 for (i=0;i<5*num_nodo;++i)
	 {
	  if(iv1!=i/5 && iv2!=i/5 && iv3!=i/5)
	  {
	   gg[i]+=c_temp;
	  }
	 }
	 /*   Second phase */
	 for (ip=0;ip<=14;++ip)
	 {
	  el_gg_temp=0.;
	  i=0; for (r=0;r<=3;++r) for (s=0;s<=3-r;++s)
	  {
	   temp_calc=compute_d_rst_Q18(r+2,s,ip)*e1y*e1y;
	   temp_calc+=compute_d_rst_Q18(r,s+2,ip)*e2y*e2y;
	   temp_calc+=compute_d_rst_Q18(r,s,ip)*e3y*e3y;
	   temp_calc+=compute_d_rst_Q18(r+1,s+1,ip)*2*e1y*e2y;
	   temp_calc+=compute_d_rst_Q18(r+1,s,ip)*2*e1y*e3y;
	   temp_calc+=compute_d_rst_Q18(r,s+1,ip)*2*e2y*e3y;
	   cpt3[0][i]=temp_calc;

	   temp_calc=-compute_d_rst_Q18(r+2,s,ip)*e1y*e1x;
	   temp_calc-=compute_d_rst_Q18(r,s+2,ip)*e2y*e2x;
	   temp_calc-=compute_d_rst_Q18(r,s,ip)*e3y*e3x;
	   temp_calc-=compute_d_rst_Q18(r+1,s+1,ip)*(e1y*e2x+e2y*e1x);
	   temp_calc-=compute_d_rst_Q18(r+1,s,ip)*(e1y*e3x+e3y*e1x);
	   temp_calc-=compute_d_rst_Q18(r,s+1,ip)*(e2y*e3x+e3y*e2x);
	   cpt3[1][i]=temp_calc;

	   temp_calc=compute_d_rst_Q18(r+2,s,ip)*e1x*e1x;
	   temp_calc+=compute_d_rst_Q18(r,s+2,ip)*e2x*e2x;
	   temp_calc+=compute_d_rst_Q18(r,s,ip)*e3x*e3x;
	   temp_calc+=compute_d_rst_Q18(r+1,s+1,ip)*2*e1x*e2x;
	   temp_calc+=compute_d_rst_Q18(r+1,s,ip)*2*e1x*e3x;
	   temp_calc+=compute_d_rst_Q18(r,s+1,ip)*2*e2x*e3x;
	   cpt3[2][i]=temp_calc;
	   i++;
	  }
	  el_gg_temp=sum_cpt3_prod(cpt3[0],cpt3[0])+
		     2.*sum_cpt3_prod(cpt3[1],cpt3[1])+
		     sum_cpt3_prod(cpt3[2],cpt3[2]);
	  el_gg_temp=el_gg_temp*25./(28.*pow(At,3));
	 /* The constant 25./(28.*pow(At,3)) comes from
            (5./At^2)^2 * (2*At/((6+1)(6+2))) */
	  gg[global_index(ip,iv1,iv2,iv3)]+=el_gg_temp;
	 } /* End ip for*/
	 /* A matrix
         ** The computation is splitted in two parts in order to perform
         ** the first one only 15 times instead of 120 times as we should do
         ** using only the second part.          
         */
	 for (ip=0;ip<=14;ip++)
	 {
	  i=0; for (r=0;r<=3;++r) for (s=0;s<=3-r;++s)
	  {
	   /* Updating A matrix with the xx integrals */
	   temp_calc=derivecptq18_v(r+2,s,ip)*e1y*e1y;
	   temp_calc+=derivecptq18_v(r,s+2,ip)*e2y*e2y;
	   temp_calc+=derivecptq18_v(r,s,ip)*e3y*e3y;
	   temp_calc+=derivecptq18_v(r+1,s+1,ip)*2*e1y*e2y;
	   temp_calc+=derivecptq18_v(r+1,s,ip)*2*e1y*e3y;
	   temp_calc+=derivecptq18_v(r,s+1,ip)*2*e2y*e3y;
	   cpt3[ip][i]=temp_calc;

	   /* Updating A matrix with the xy integrals */
	   temp_calc=-derivecptq18_v(r+2,s,ip)*e1y*e1x;
	   temp_calc-=derivecptq18_v(r,s+2,ip)*e2y*e2x;
	   temp_calc-=derivecptq18_v(r,s,ip)*e3y*e3x;
	   temp_calc-=derivecptq18_v(r+1,s+1,ip)*(e1y*e2x+e2y*e1x);
	   temp_calc-=derivecptq18_v(r+1,s,ip)*(e1y*e3x+e3y*e1x);
	   temp_calc-=derivecptq18_v(r,s+1,ip)*(e2y*e3x+e3y*e2x);
	   cpt3[ip+15][i]=temp_calc;

	   /* Updating A matrix with the yy integrals */
	   temp_calc=derivecptq18_v(r+2,s,ip)*e1x*e1x;
	   temp_calc+=derivecptq18_v(r,s+2,ip)*e2x*e2x;
	   temp_calc+=derivecptq18_v(r,s,ip)*e3x*e3x;
	   temp_calc+=derivecptq18_v(r+1,s+1,ip)*2*e1x*e2x;
	   temp_calc+=derivecptq18_v(r+1,s,ip)*2*e1x*e3x;
	   temp_calc+=derivecptq18_v(r,s+1,ip)*2*e2x*e3x;
	   cpt3[ip+30][i]=temp_calc;
	   i++;
	  }
	 } /* end ip for*/
	 for (ip=0;ip<15;ip++)
	 {
	  ip_global=global_index(ip,iv1,iv2,iv3);
	  for (jp=0;jp<=ip;jp++)
	  {
	   jp_global=global_index(jp,iv1,iv2,iv3);
	   temp_calc=sum_cpt3_prod(cpt3[ip],cpt3[jp])+
		     2.*sum_cpt3_prod(cpt3[ip+15],cpt3[jp+15])+
		     sum_cpt3_prod(cpt3[ip+30],cpt3[jp+30]);
	   temp_calc=temp_calc*50./(28.*pow(At,3));
	   /* The constant 25./(28.*pow(At,3)) comes from
              (5./At^2)^2 * (2*At/((6+1)(6+2))) */
	   if(ip_global>=jp_global)
	     A[ip_global*5*num_nodo+jp_global]+=temp_calc;
	    else
	     A[jp_global*5*num_nodo+ip_global]+=temp_calc;
	  }  /* fine for su jp */
	 } /* fine for su ip */
	 if (nt%5==0)
	   printf("Assembled %d triangles.\n",nt); nt++;
	} /* End of computations and assembling phase for the triangle */
      iv2=iv3;
      fscanf(trifile,"%d ",&iv3);
     }  /* End of the reading of the current row in the triangulation file */
    fscanf(trifile,"%s ",base_buffer);
    iv1=atol(base_buffer);
   }  /* End of the reading of the triangulation file when the cycle exits */
 while (base_buffer[0]!='B');
 fclose(trifile);
 /* transforming phi (computed in gg) to -g (computed in gg too) */
 for (i=0;i<5*num_nodo;++i) gg[i]=-gg[i]+A[i*5*num_nodo+i]/2.+c;
 /* Getting v as the solution of the linear system Av=gg 
    (remember that -g=gg) */
 printf("Solving system.\n");
 err=cholesky(A,v,gg,5*num_nodo,1,5*num_nodo,1,DiagL);
 if (err==1)
 {
  printf("Singular or bad matrix.\n");
  exit(1);
 }
 printf("Output copying.\n");
 for (i=0;i<num_nodo;++i)
 {
  fprintf(outfile,"%*.*f %*.*f %*.*f %*.*f %*.*f\n",
	  WIDTH,PREC,v[5*i],WIDTH,PREC,v[5*i+1],
	  WIDTH,PREC,v[5*i+2],WIDTH,PREC,v[5*i+3],WIDTH,PREC,v[5*i+4]);
 }
 fclose(outfile);
}

int global_index(i,iv1,iv2,iv3)
int i;
int iv1,iv2,iv3;
{
     if (i/5==0) return 5*iv1+i%5;
else if (i/5==1) return 5*iv2+i%5;
else if (i/5==2) return 5*iv3+i%5;
else
 {
  printf("Conversion of indexes is not possible.\n");
  exit(1);
 }
}











