/*
** CLT_DER2.C
**
** AUTHORS: 
**
**   Maria MORANDI CECCHI               Stefano DE MARCHI,            
**   University of Padua, Italy         University of Udine, Italy
**   email: mcecchi@math.unipd.it       email: demarchi@dimi.uniud.it
**   
**                       Damiano FASOLI
**                       Verona Software Srl, Italy 
**                       email: vrsoft@sis.it
**
**
** REVISION DATE: May, 1999
**
** MODULES CALLED: CLT_D_A.C
**                 LINSIST.C 
**
** ------------------------------------------------------------------------ 
**
** BACKGROUND
**
** Let (xi,yi,qi) i=0,...,N-1 be N points in R^3. Let us consider their
** projections (xi,yi) onto the plane z=0 and let T be a triangulation
** of these points. A triangulation of them is a set of triangles having
** for vertices all these points (and only these ones) so that two
** distinct triangles intersect at most in a common edge or in a common
** vertex.
** 
** The aim is to build the bivariate C1-function that interpolates
** at the points (xi,yi,qi).
** 
** The Clough-Tocher (CT) interpolation method builds up such a function 
** using a Clough-Tocher polynomial patch at each triangle, assuming
** that the first and second partial derivatives are known
** at each vertex of the triangulation T.
** The CT-method, splits a triangle in three sub-triangles (using the barycenter)
** over which it builds three cubic polynomial patches.
** The program simply computes such parameters (in the following
** we shall refer to them as to qix,qiy i=0,...,N-1 or more often
** as to the 2N-dimensional vector v).
** For every fixed i in the range from 0 to N-1, it computes
** these parameters in the "smoothest" possible way for a Clough-Tocher 
** interpolating function over the triangulation TLoc(i). 
** TLoc(i) is the set of all triangles having the 
** i-th point, Vi, as a common vertex (that is the cell centered in Vi).
**
** It then considers only the values qix,qiy where i is now fixed.    
** The criteria considered to evaluate the "smoothness" of the 
** CT-interpolating function is the minimization of the functional
**
** phi(vLoc(i))=   sum  [ Integral[ (Qxx^2 + 2*Qxy^2 + Qyy^2  ) dx dy ] ]     
**              t in TLoc(i)  t      
** 
** with respect to vLoc(i), the same of v when T becomes TLoc(i).
**
** Then, the estimate is improved using the following formula:
**
**       g_i = (1-q) * g_{i,i} + q * R(d_{i,j},g_{i,j}),   0 <= q <= 1,
**
** where g_i :  is the actual estimation of the gradient at the i-th point P_i;
**       g_{i,i}, g_{i,j} :  are the actual estimation of the gradient 
**                           at the i-th, j-th point P_j (j=1,...,k)
**                           in the neighboor of the i-th one;
**       k : is the number of neighbouring nodes;
**       d_{i,j} :  are the Euclidean distances between P_i and P_j;
**        
** and 
**                         g_{i,1}*a*E_1/d_{i,1}^c+...+g_{i,k}*a*E_k/d_{i,k}^c
**   R(d_{i,j},g_{i,j})=  -----------------------------------------------------
**                              a*E_1/ d_{i,1}^c+ ... +a*E_k/d_{i,k}^c
**
**  with E_j=exp(-b*d_{i,j}^c); a,b,c are real numbers.
**
**  Choosing a=1, b=0 and c=2 one gets formula (8) of the paper.
**  The program requires in input the values q,a,b,c, as well.
**  We suggest to use q=0.5.
**
**
** METHODOLOGY 
**
** For every fixed i in the range from 0 to N-1,
**
**      phi(vLoc(i))= 1/2 * vLoc(i)' A vLoc(i) + g' vLoc(i) + c 
**
** is minimum when vLoc(i) solves A vLoc(i) + g == 0.
**
** Let M be the number of nodes in the cell TLoc(i).
** The program computes
** 
**      c=phi(0), A, gj=phi(ej)-1/2*ajj-c, 
**
** where ej is the j-th vector of the canonical basis of R^(2M), 
** and solves the resulting linear system with the Cholesky factorisation. 
**
** Then, it considers only the parameters 
** relative to Vi, the center of TLoc(i).
**
**  
** REFERENCES
**
**  R. J. Renka and A. K. Cline:
**     "A triangle-based C^1 interpolation method"
**      Rocky Mountain J. Math, 14(1):223-237, 1984.
**
**  P. Alfeld:
**    "Derivative Generation for multivariate 
**     scattered data functional minimization", 
**     Comp. Aided Graph. Design,2:281-296, 1985.
**
**  G. Farin:
**     "Triangular Bernstein-Bezier patches",
**      Comput. Aided Geom. Design,3:83-127, 1986.
**
**  D. Fasoli,
**      Laurea's Thesis, University of Padua, 1995 (in Italian). 
**
**  S. De Marchi,
**      "On Computing Derivatives for C^1 Interpolating 
**       Schemes: an Optimization"
**       Computing,60:29-53, 1998.
**
** -----------------------------------------------------------------------
** SYNOPSIS: clt_der2 file1 file2 [file3]
**
** file1 is the data file made up of the 3-dimensional points. 
** The coordinates of the points may be separated by any 
** combination of spaces, tabs and carriage returns. 
** We recommend to use a row for each point, for clearness. 
** 
** file2 is the triangulation data file in the format generated by
** the program 'do_trian'. The file has three sections. 
** The first, starting with a single capital letter `C' in the first row, 
** lists in subsequent blocks separated by -1 the index of
** a node and those of its neighbours in a counterclockwise
** order. 
** Note: this program uses only this first section and the heading of the
** second section, that is a single capital letter `B' in the row.
**
** file3 is the output derivative data file. 
** If file3 is omitted, the standard output is used instead.
** 
** The values in a row refer to the same vertex of the triangulation.
** The first column is the x-derivative, the second is the y-derivative,
** the third is 0, the fourth is 0 and the last one is 0, too.
**
** The output format is handled by the constants WIDTH and PREC in #DEFINE,
** that are the width of the output field and the number of decimal digits,
** respectively.
**        
** The program supports at most MAX_NUM_NODI (#DEFINE constant) input points.
** If more points are given, it exits with a warning message.
**
** The program uses a temporary swap file called blocks.tmp to store
** data during computations. You may safely remove it after the use. 
**
** ------------------------------------------------------------------------------
**
** SUBROUTINES AND FUNCTIONS
**
** 
** int global_index(int N, int i, int j)
**
** Inputs:
**       N, the number of nodes of an open cell or of a closed one, 
**       j, the index of the triangle in the cell 
**       i, the local index of the unknown parameter among 
**          q1x,q1y,q2x,q2y,q3x,q3y
**          relative to the triangle (a value in the range from 0 to 5).
**
** Output: 
**       The function computes the global index of the unknown parameters
**       (a value in the range from 0 to 2N-1) with respect to the cell.
**       Locally, for each triangle, the parameters are sorted following
**       the vertices of the triangle and the sequence d/dx,d/dy.
**       Globally, for the cell, the parameters are sorted following 
**       the nodes of the cell, according to their index with respect 
**       to the cell itself, and the above sequence.
**
** -----------------------------------------------------------------------------
*/

#include<stdio.h>
#include<stdlib.h>
#include<math.h>
#define MAX_NUM_NODI 80000
#define WIDTH 20   /*  Format width of real numbers written to files */
#define PREC 10    /* Decimal precision width of real numbers written to files */

double derivecptclt_v(),compute_d_j_clt(),compute_d_j_clt_null(),
       sum_cpt1_prod();
void free_ind_node_list();
int global_index();

/* Declarations for global variables */
double q1,q2,q3;
double u1,v1,u2,v2,u3,v3,t1,t2,t3;

struct ind_node_list {long int glob_index; struct ind_node_list *next;};

void free_ind_node_list(Ind_Node_List_Pt)
struct ind_node_list *Ind_Node_List_Pt;
{
 struct ind_node_list *tempPt;

 if (Ind_Node_List_Pt==NULL) return;
 while (Ind_Node_List_Pt->next!=NULL)
   {
    tempPt=Ind_Node_List_Pt->next;
    free(Ind_Node_List_Pt);
    Ind_Node_List_Pt=tempPt;
   }
 free(Ind_Node_List_Pt);
}

void main(argc,argv)
int argc;
char *argv[];
{

/* Parameters to be  read in input */ 
  double q;
  double alpha,beta,gamma;
/*                                 */
  struct triangle {long int v1; long int v2; long int v3;};

  size_t size_double;
  FILE *infile, *trifile, *outfile, *bfile;
  long int num_nodo,cont_nodi,temp_index,iv1,iv2,iv3,
	   triangle_number,triangle_counter,nt;
  double x[MAX_NUM_NODI],y[MAX_NUM_NODI],z[MAX_NUM_NODI];
  double zx[MAX_NUM_NODI],zx_ext[MAX_NUM_NODI],zy[MAX_NUM_NODI],zy_ext[MAX_NUM_NODI];
  double sum_neigh_weights[MAX_NUM_NODI];
  struct triangle triangle_order[2*MAX_NUM_NODI-5];
  double u4,v4,At;
  int i,j,ip,jp,N_C; /*  Counter for the number of nodes in a cell  */
  int ip_global,jp_global;
  int r,s,err,open_cella_flag; /* flag for an open or closed cell */
  struct ind_node_list *lista_vicini, *vicino_pt, *vicino_pt_temp;
  double cpt1[54][3],temp_calc;
  char base_buffer[20];
  int stop; /* For cycles */
  int shift_right;
  /*  Number of the counterclockwise rotations needed to transform a 
      triangle in a cell in its canonical form where  iv1<iv2, iv1<iv3 */
  double *A, *v, *gg, *DiagL, c;
  int IndT1[4][4]={{2,9,4,1},{8,13,5,0},{17,16,0,0},{19,0,0,0}};
  int IndT2[4][4]={{3,12,7,2},{11,14,8,0},{18,17,0,0},{19,0,0,0}};
  int IndT3[4][4]={{1,6,10,3},{5,15,11,0},{16,18,0,0},{19,0,0,0}};

 /* Read the free parameters q, alpha, beta and gamma */
 printf("Give me q (between 0 and 1) \n");  
 scanf(" %lf",&q);
 printf("With reference to the formula alpha*EXP(-beta d^2)/d^gamma\n");
 printf("Give me the parameter alpha: ");
 scanf(" %lf",&alpha);
 printf("Give me the parameter beta: ");
 scanf(" %lf",&beta);
 printf("Give me the parameter gamma: ");
 scanf(" %lf",&gamma);
 infile=NULL; trifile=NULL; outfile=NULL; bfile=NULL;
 A=NULL; v=NULL; gg=NULL; DiagL=NULL;
 lista_vicini=NULL;
 size_double=sizeof(double);
 if (argc==1)
   {
    printf("\nStart the program with three parameters:\n");
    printf("First parameter: data file name.\n");
    printf("Second parameter: triangulation file name.\n");
    printf("Third parameter: derivative file name.\n");
    printf("                 (Omit it for output on standard output.)\n");
    exit(1);
   }
 /* Reading  */
 infile=fopen(argv[1],"r");
 if (infile==NULL)
   {
    printf("Cannot open file: %s.\n",argv[1]);
    exit(1);
   }
 if (argc==2)
   {
    printf("Missing triangulation file.\n");
    exit(1);
   }
 /* data reading */
 printf("Wait for data reading\n"); 
 num_nodo=0;
 while (!feof(infile))
 {
  if (num_nodo<MAX_NUM_NODI)
   {
    fscanf(infile," %lf %lf %lf ",&x[num_nodo],&y[num_nodo],&z[num_nodo]);
      /* spaces are essential part of the input format string */
    num_nodo=num_nodo+1;
   }
   else
   {
    printf("Too many input data.\n");
    exit(1);
   }
 }
 if (num_nodo<3)
    {
     printf("Need at least three input points.\n");
     exit(1);
    }
 fclose(infile);
 /* end of data reading */
 /* opening triangulation file */
 trifile=fopen(argv[2],"r");
 if (trifile==NULL)
   {
    printf("Cannot open file: %s.\n",argv[2]);
    exit(1);
   }
 if (fgetc(trifile)!='C')
   {
    printf("File %s has no correct format.\n",argv[2]);
    exit(1);
   }
 else fgetc(trifile); /* reading carriage return */
 /* opening output file  */
 if (argc==3) outfile=stdout; else outfile=fopen(argv[3],"w");
 if (outfile==NULL)
   {
    printf("Cannot open file: %s.\n",argv[3]);
    exit(1);
   }
 /* HANDLING THE PARTICULAR CASE OF ONLY ONE TRIANGLE */
 if (num_nodo==3)
 {
  fclose(trifile);
  temp_calc=x[0]*(y[1]-y[2])+x[1]*(y[2]-y[0])+x[2]*(y[0]-y[1]);
  u1=-(y[0]*(z[1]-z[2])+y[1]*(z[2]-z[0])+y[2]*(z[0]-z[1]))/temp_calc;
  v1=(x[0]*(z[1]-z[2])+x[1]*(z[2]-z[0])+x[2]*(z[0]-z[1]))/temp_calc;
  /* Derivatives of the plane where the triangle lies  */
  fprintf(outfile,"%*.*f %*.*f %*.*f %*.*f %*.*f\n",
	  WIDTH,PREC,u1,WIDTH,PREC,v1,
	  WIDTH,PREC,0.,WIDTH,PREC,0.,WIDTH,PREC,0.);
  fprintf(outfile,"%*.*f %*.*f %*.*f %*.*f %*.*f\n",
	  WIDTH,PREC,u1,WIDTH,PREC,v1,
	  WIDTH,PREC,0.,WIDTH,PREC,0.,WIDTH,PREC,0.);
  fprintf(outfile,"%*.*f %*.*f %*.*f %*.*f %*.*f\n",
	  WIDTH,PREC,u1,WIDTH,PREC,v1,
	  WIDTH,PREC,0.,WIDTH,PREC,0.,WIDTH,PREC,0.);
  fclose(outfile);
  return; /* regular exit from main function */
 }
 /* OPENING TEMPORARY SWAP FILE */
 bfile=fopen("blocks.tmp","wb");
 if (bfile==NULL) 
   {
    printf("Cannot open temporary file for writing.\n");
    exit(1);
   }
 /*  MAIN CYCLE, COMPUTATIONS OVER TRIANGLES, AND THEIR STORAGE  */
 printf("Starting computations for the data of each triangle.\n");
 fscanf(trifile,"%ld ",&iv1);
 triangle_number=0;
 /*  Counter for the current triangle. 
    At the end it will become the number of triangles */
 do
   {
    fscanf(trifile,"%ld %ld ",&iv2,&iv3);
    while(iv3!=-1)
     {
      if ((iv1<iv2)&&(iv1<iv3))
	{/* New triangle: beginning computations and assembling phase */
	 u1=x[iv1]; u2=x[iv2]; u3=x[iv3];
	 v1=y[iv1]; v2=y[iv2]; v3=y[iv3];
	 q1=z[iv1]; q2=z[iv2]; q3=z[iv3];
	 u4=(u1+u2+u3)/3.; v4=(v1+v2+v3)/3.;
	 At=fabs((u2*v3-u3*v2+u3*v1-u1*v3+u1*v2-u2*v1)/2.);
	 t1= ((u4-u1)*(u2-u1)+(v4-v1)*(v2-v1))/
	     (pow((u2-u1),2.)+pow((v2-v1),2.));
	 t2= ((u4-u2)*(u3-u2)+(v4-v2)*(v3-v2))/
	     (pow((u3-u2),2.)+pow((v3-v2),2.));
	 t3= ((u4-u3)*(u1-u3)+(v4-v3)*(v1-v3))/
	     (pow((u1-u3),2.)+pow((v1-v3),2.));
	 /* Computations and updating of the storage */
	 /* A matrix 
         ** The computation is splitted in two parts in order to perform
         ** the first one only 6 times instead of 21 times as we should do
         ** using only the second part.
          */
	 for (ip=0;ip<=5;ip++)
	 {
	  i=0; for (r=0;r<=1;++r) for (s=0;s<=1-r;++s)
	  {
	   /* First triangle V4 V1 V2 */
	   temp_calc=derivecptclt_v(IndT1[r+2][s],ip)*(v2-v1)*(v2-v1);
	   temp_calc+=derivecptclt_v(IndT1[r][s+2],ip)*(v4-v2)*(v4-v2);
	   temp_calc+=derivecptclt_v(IndT1[r][s],ip)*(v1-v4)*(v1-v4);
	   temp_calc+=derivecptclt_v(IndT1[r+1][s+1],ip)*2*(v2-v1)*(v4-v2);
	   temp_calc+=derivecptclt_v(IndT1[r+1][s],ip)*2*(v2-v1)*(v1-v4);
	   temp_calc+=derivecptclt_v(IndT1[r][s+1],ip)*2*(v4-v2)*(v1-v4);
	   cpt1[ip][i]=temp_calc;

	   temp_calc=-derivecptclt_v(IndT1[r+2][s],ip)*(v2-v1)*(u2-u1);
	   temp_calc-=derivecptclt_v(IndT1[r][s+2],ip)*(v4-v2)*(u4-u2);
	   temp_calc-=derivecptclt_v(IndT1[r][s],ip)*(v1-v4)*(u1-u4);
	   temp_calc-=derivecptclt_v(IndT1[r+1][s+1],ip)*
		      ((v2-v1)*(u4-u2)+(v4-v2)*(u2-u1));
	   temp_calc-=derivecptclt_v(IndT1[r+1][s],ip)*
		      ((v2-v1)*(u1-u4)+(v1-v4)*(u2-u1));
	   temp_calc-=derivecptclt_v(IndT1[r][s+1],ip)*
		      ((v4-v2)*(u1-u4)+(v1-v4)*(u4-u2));
	   cpt1[ip+6][i]=temp_calc;

	   temp_calc=derivecptclt_v(IndT1[r+2][s],ip)*(u2-u1)*(u2-u1);
	   temp_calc+=derivecptclt_v(IndT1[r][s+2],ip)*(u4-u2)*(u4-u2);
	   temp_calc+=derivecptclt_v(IndT1[r][s],ip)*(u1-u4)*(u1-u4);
	   temp_calc+=derivecptclt_v(IndT1[r+1][s+1],ip)*2*(u2-u1)*(u4-u2);
	   temp_calc+=derivecptclt_v(IndT1[r+1][s],ip)*2*(u2-u1)*(u1-u4);
	   temp_calc+=derivecptclt_v(IndT1[r][s+1],ip)*2*(u4-u2)*(u1-u4);
	   cpt1[ip+12][i]=temp_calc;

	   /* Second triangle V4 V2 V3 */
	   temp_calc=derivecptclt_v(IndT2[r+2][s],ip)*(v3-v2)*(v3-v2);
	   temp_calc+=derivecptclt_v(IndT2[r][s+2],ip)*(v4-v3)*(v4-v3);
	   temp_calc+=derivecptclt_v(IndT2[r][s],ip)*(v2-v4)*(v2-v4);
	   temp_calc+=derivecptclt_v(IndT2[r+1][s+1],ip)*2*(v3-v2)*(v4-v3);
	   temp_calc+=derivecptclt_v(IndT2[r+1][s],ip)*2*(v3-v2)*(v2-v4);
	   temp_calc+=derivecptclt_v(IndT2[r][s+1],ip)*2*(v4-v3)*(v2-v4);
	   cpt1[ip+18][i]=temp_calc;

	   temp_calc=-derivecptclt_v(IndT2[r+2][s],ip)*(v3-v2)*(u3-u2);
	   temp_calc-=derivecptclt_v(IndT2[r][s+2],ip)*(v4-v3)*(u4-u3);
	   temp_calc-=derivecptclt_v(IndT2[r][s],ip)*(v2-v4)*(u2-u4);
	   temp_calc-=derivecptclt_v(IndT2[r+1][s+1],ip)*
		      ((v3-v2)*(u4-u3)+(v4-v3)*(u3-u2));
	   temp_calc-=derivecptclt_v(IndT2[r+1][s],ip)*
		      ((v3-v2)*(u2-u4)+(v2-v4)*(u3-u2));
	   temp_calc-=derivecptclt_v(IndT2[r][s+1],ip)*
		      ((v4-v3)*(u2-u4)+(v2-v4)*(u4-u3));
	   cpt1[ip+24][i]=temp_calc;

	   temp_calc=derivecptclt_v(IndT2[r+2][s],ip)*(u3-u2)*(u3-u2);
	   temp_calc+=derivecptclt_v(IndT2[r][s+2],ip)*(u4-u3)*(u4-u3);
	   temp_calc+=derivecptclt_v(IndT2[r][s],ip)*(u2-u4)*(u2-u4);
	   temp_calc+=derivecptclt_v(IndT2[r+1][s+1],ip)*2*(u3-u2)*(u4-u3);
	   temp_calc+=derivecptclt_v(IndT2[r+1][s],ip)*2*(u3-u2)*(u2-u4);
	   temp_calc+=derivecptclt_v(IndT2[r][s+1],ip)*2*(u4-u3)*(u2-u4);
	   cpt1[ip+30][i]=temp_calc;

	   /* Third triangle V4 V3 V1 */
	   temp_calc=derivecptclt_v(IndT3[r+2][s],ip)*(v1-v3)*(v1-v3);
	   temp_calc+=derivecptclt_v(IndT3[r][s+2],ip)*(v4-v1)*(v4-v1);
	   temp_calc+=derivecptclt_v(IndT3[r][s],ip)*(v3-v4)*(v3-v4);
	   temp_calc+=derivecptclt_v(IndT3[r+1][s+1],ip)*2*(v1-v3)*(v4-v1);
	   temp_calc+=derivecptclt_v(IndT3[r+1][s],ip)*2*(v1-v3)*(v3-v4);
	   temp_calc+=derivecptclt_v(IndT3[r][s+1],ip)*2*(v4-v1)*(v3-v4);
	   cpt1[ip+36][i]=temp_calc;

	   temp_calc=-derivecptclt_v(IndT3[r+2][s],ip)*(v1-v3)*(u1-u3);
	   temp_calc-=derivecptclt_v(IndT3[r][s+2],ip)*(v4-v1)*(u4-u1);
	   temp_calc-=derivecptclt_v(IndT3[r][s],ip)*(v3-v4)*(u3-u4);
	   temp_calc-=derivecptclt_v(IndT3[r+1][s+1],ip)*
		      ((v1-v3)*(u4-u1)+(v4-v1)*(u1-u3));
	   temp_calc-=derivecptclt_v(IndT3[r+1][s],ip)*
		      ((v1-v3)*(u3-u4)+(v3-v4)*(u1-u3));
	   temp_calc-=derivecptclt_v(IndT3[r][s+1],ip)*
		      ((v4-v1)*(u3-u4)+(v3-v4)*(u4-u1));
	   cpt1[ip+42][i]=temp_calc;

	   temp_calc=derivecptclt_v(IndT3[r+2][s],ip)*(u1-u3)*(u1-u3);
	   temp_calc+=derivecptclt_v(IndT3[r][s+2],ip)*(u4-u1)*(u4-u1);
	   temp_calc+=derivecptclt_v(IndT3[r][s],ip)*(u3-u4)*(u3-u4);
	   temp_calc+=derivecptclt_v(IndT3[r+1][s+1],ip)*2*(u1-u3)*(u4-u1);
	   temp_calc+=derivecptclt_v(IndT3[r+1][s],ip)*2*(u1-u3)*(u3-u4);
	   temp_calc+=derivecptclt_v(IndT3[r][s+1],ip)*2*(u4-u1)*(u3-u4);
	   cpt1[ip+48][i]=temp_calc;
	   i++;
	  } /* End for s (and for r) */
	 } /* End for ip */
	 for (ip=0;ip<=5;++ip)
	    for (jp=0;jp<=ip;++jp)
	     {
	      temp_calc=(sum_cpt1_prod(cpt1[ip],cpt1[jp])+
			 2.*sum_cpt1_prod(cpt1[ip+6],cpt1[jp+6])+
			 sum_cpt1_prod(cpt1[ip+12],cpt1[jp+12]));
	      temp_calc+=(sum_cpt1_prod(cpt1[ip+18],cpt1[jp+18])+
			  2.*sum_cpt1_prod(cpt1[ip+24],cpt1[jp+24])+
			  sum_cpt1_prod(cpt1[ip+30],cpt1[jp+30]));
	      temp_calc+=(sum_cpt1_prod(cpt1[ip+36],cpt1[jp+36])+
			  2.*sum_cpt1_prod(cpt1[ip+42],cpt1[jp+42])+
			  sum_cpt1_prod(cpt1[ip+48],cpt1[jp+48]));
              /* The constant 3./(8. *pow(At/3.,3.)
                 comes from (6/4 a^2)^2 * (2a / (2+1)(2+2) ) 
                 where a=At/3 
               */
	      temp_calc=temp_calc*6./(8. *pow(At/3.,3.));
	      fwrite((char *)&temp_calc,size_double,1,bfile);
	     }
	 /* gg vector */
	 for (ip=0;ip<=5;++ip)
	 {
	  i=0; for (r=0;r<=1;++r) for (s=0;s<=1-r;++s)
	  {
	   /* First triangle V4 V1 V2 */
	   temp_calc=compute_d_j_clt(IndT1[r+2][s],ip)*(v2-v1)*(v2-v1);
	   temp_calc+=compute_d_j_clt(IndT1[r][s+2],ip)*(v4-v2)*(v4-v2);
	   temp_calc+=compute_d_j_clt(IndT1[r][s],ip)*(v1-v4)*(v1-v4);
	   temp_calc+=compute_d_j_clt(IndT1[r+1][s+1],ip)*2*(v2-v1)*(v4-v2);
	   temp_calc+=compute_d_j_clt(IndT1[r+1][s],ip)*2*(v2-v1)*(v1-v4);
	   temp_calc+=compute_d_j_clt(IndT1[r][s+1],ip)*2*(v4-v2)*(v1-v4);
	   cpt1[0][i]=temp_calc;

	   temp_calc=-compute_d_j_clt(IndT1[r+2][s],ip)*(v2-v1)*(u2-u1);
	   temp_calc-=compute_d_j_clt(IndT1[r][s+2],ip)*(v4-v2)*(u4-u2);
	   temp_calc-=compute_d_j_clt(IndT1[r][s],ip)*(v1-v4)*(u1-u4);
	   temp_calc-=compute_d_j_clt(IndT1[r+1][s+1],ip)*
		      ((v2-v1)*(u4-u2)+(v4-v2)*(u2-u1));
	   temp_calc-=compute_d_j_clt(IndT1[r+1][s],ip)*
		      ((v2-v1)*(u1-u4)+(v1-v4)*(u2-u1));
	   temp_calc-=compute_d_j_clt(IndT1[r][s+1],ip)*
		      ((v4-v2)*(u1-u4)+(v1-v4)*(u4-u2));
	   cpt1[1][i]=temp_calc;

	   temp_calc=compute_d_j_clt(IndT1[r+2][s],ip)*(u2-u1)*(u2-u1);
	   temp_calc+=compute_d_j_clt(IndT1[r][s+2],ip)*(u4-u2)*(u4-u2);
	   temp_calc+=compute_d_j_clt(IndT1[r][s],ip)*(u1-u4)*(u1-u4);
	   temp_calc+=compute_d_j_clt(IndT1[r+1][s+1],ip)*2*(u2-u1)*(u4-u2);
	   temp_calc+=compute_d_j_clt(IndT1[r+1][s],ip)*2*(u2-u1)*(u1-u4);
	   temp_calc+=compute_d_j_clt(IndT1[r][s+1],ip)*2*(u4-u2)*(u1-u4);
	   cpt1[2][i]=temp_calc;

	   /* Second triangle V4 V2 V3 */
	   temp_calc=compute_d_j_clt(IndT2[r+2][s],ip)*(v3-v2)*(v3-v2);
	   temp_calc+=compute_d_j_clt(IndT2[r][s+2],ip)*(v4-v3)*(v4-v3);
	   temp_calc+=compute_d_j_clt(IndT2[r][s],ip)*(v2-v4)*(v2-v4);
	   temp_calc+=compute_d_j_clt(IndT2[r+1][s+1],ip)*2*(v3-v2)*(v4-v3);
	   temp_calc+=compute_d_j_clt(IndT2[r+1][s],ip)*2*(v3-v2)*(v2-v4);
	   temp_calc+=compute_d_j_clt(IndT2[r][s+1],ip)*2*(v4-v3)*(v2-v4);
	   cpt1[3][i]=temp_calc;

	   temp_calc=-compute_d_j_clt(IndT2[r+2][s],ip)*(v3-v2)*(u3-u2);
	   temp_calc-=compute_d_j_clt(IndT2[r][s+2],ip)*(v4-v3)*(u4-u3);
	   temp_calc-=compute_d_j_clt(IndT2[r][s],ip)*(v2-v4)*(u2-u4);
	   temp_calc-=compute_d_j_clt(IndT2[r+1][s+1],ip)*
		      ((v3-v2)*(u4-u3)+(v4-v3)*(u3-u2));
	   temp_calc-=compute_d_j_clt(IndT2[r+1][s],ip)*
		      ((v3-v2)*(u2-u4)+(v2-v4)*(u3-u2));
	   temp_calc-=compute_d_j_clt(IndT2[r][s+1],ip)*
		      ((v4-v3)*(u2-u4)+(v2-v4)*(u4-u3));
	   cpt1[4][i]=temp_calc;

	   temp_calc=compute_d_j_clt(IndT2[r+2][s],ip)*(u3-u2)*(u3-u2);
	   temp_calc+=compute_d_j_clt(IndT2[r][s+2],ip)*(u4-u3)*(u4-u3);
	   temp_calc+=compute_d_j_clt(IndT2[r][s],ip)*(u2-u4)*(u2-u4);
	   temp_calc+=compute_d_j_clt(IndT2[r+1][s+1],ip)*2*(u3-u2)*(u4-u3);
	   temp_calc+=compute_d_j_clt(IndT2[r+1][s],ip)*2*(u3-u2)*(u2-u4);
	   temp_calc+=compute_d_j_clt(IndT2[r][s+1],ip)*2*(u4-u3)*(u2-u4);
	   cpt1[5][i]=temp_calc;

	   /* Third triangle V4 V3 V1 */
	   temp_calc=compute_d_j_clt(IndT3[r+2][s],ip)*(v1-v3)*(v1-v3);
	   temp_calc+=compute_d_j_clt(IndT3[r][s+2],ip)*(v4-v1)*(v4-v1);
	   temp_calc+=compute_d_j_clt(IndT3[r][s],ip)*(v3-v4)*(v3-v4);
	   temp_calc+=compute_d_j_clt(IndT3[r+1][s+1],ip)*2*(v1-v3)*(v4-v1);
	   temp_calc+=compute_d_j_clt(IndT3[r+1][s],ip)*2*(v1-v3)*(v3-v4);
	   temp_calc+=compute_d_j_clt(IndT3[r][s+1],ip)*2*(v4-v1)*(v3-v4);
	   cpt1[6][i]=temp_calc;

	   temp_calc=-compute_d_j_clt(IndT3[r+2][s],ip)*(v1-v3)*(u1-u3);
	   temp_calc-=compute_d_j_clt(IndT3[r][s+2],ip)*(v4-v1)*(u4-u1);
	   temp_calc-=compute_d_j_clt(IndT3[r][s],ip)*(v3-v4)*(u3-u4);
	   temp_calc-=compute_d_j_clt(IndT3[r+1][s+1],ip)*
		      ((v1-v3)*(u4-u1)+(v4-v1)*(u1-u3));
	   temp_calc-=compute_d_j_clt(IndT3[r+1][s],ip)*
		      ((v1-v3)*(u3-u4)+(v3-v4)*(u1-u3));
	   temp_calc-=compute_d_j_clt(IndT3[r][s+1],ip)*
		      ((v4-v1)*(u3-u4)+(v3-v4)*(u4-u1));
	   cpt1[7][i]=temp_calc;

	   temp_calc=compute_d_j_clt(IndT3[r+2][s],ip)*(u1-u3)*(u1-u3);
	   temp_calc+=compute_d_j_clt(IndT3[r][s+2],ip)*(u4-u1)*(u4-u1);
	   temp_calc+=compute_d_j_clt(IndT3[r][s],ip)*(u3-u4)*(u3-u4);
	   temp_calc+=compute_d_j_clt(IndT3[r+1][s+1],ip)*2*(u1-u3)*(u4-u1);
	   temp_calc+=compute_d_j_clt(IndT3[r+1][s],ip)*2*(u1-u3)*(u3-u4);
	   temp_calc+=compute_d_j_clt(IndT3[r][s+1],ip)*2*(u4-u1)*(u3-u4);
	   cpt1[8][i]=temp_calc;
	   i++;
	  } /* End for s (and for r) */
	  temp_calc=(sum_cpt1_prod(cpt1[0],cpt1[0])+
		     2.*sum_cpt1_prod(cpt1[1],cpt1[1])+
		     sum_cpt1_prod(cpt1[2],cpt1[2]));
	  temp_calc+=(sum_cpt1_prod(cpt1[3],cpt1[3])+
		      2.*sum_cpt1_prod(cpt1[4],cpt1[4])+
		      sum_cpt1_prod(cpt1[5],cpt1[5]));
	  temp_calc+=(sum_cpt1_prod(cpt1[6],cpt1[6])+
		      2.*sum_cpt1_prod(cpt1[7],cpt1[7])+
		      sum_cpt1_prod(cpt1[8],cpt1[8]));
          /* The constant 3./(8. *pow(At/3.,3.)
             comes from (6/4 a^2)^2 * (2a / (2+1)(2+2) ) 
             where a=At/3 */
	  temp_calc=temp_calc*3./(8. *pow(At/3.,3.));
	  fwrite((char *)&temp_calc,size_double,1,bfile);
	 } /* End for ip */
	 /* c constant */
	 i=0; for (r=0;r<=1;++r) for (s=0;s<=1-r;++s)
	 {
	  /* First triangle V4 V1 V2 */
	  temp_calc=compute_d_j_clt_null(IndT1[r+2][s])*(v2-v1)*(v2-v1);
	  temp_calc+=compute_d_j_clt_null(IndT1[r][s+2])*(v4-v2)*(v4-v2);
	  temp_calc+=compute_d_j_clt_null(IndT1[r][s])*(v1-v4)*(v1-v4);
	  temp_calc+=compute_d_j_clt_null(IndT1[r+1][s+1])*2*(v2-v1)*(v4-v2);
	  temp_calc+=compute_d_j_clt_null(IndT1[r+1][s])*2*(v2-v1)*(v1-v4);
	  temp_calc+=compute_d_j_clt_null(IndT1[r][s+1])*2*(v4-v2)*(v1-v4);
	  cpt1[0][i]=temp_calc;

	  temp_calc=-compute_d_j_clt_null(IndT1[r+2][s])*(v2-v1)*(u2-u1);
	  temp_calc-=compute_d_j_clt_null(IndT1[r][s+2])*(v4-v2)*(u4-u2);
	  temp_calc-=compute_d_j_clt_null(IndT1[r][s])*(v1-v4)*(u1-u4);
	  temp_calc-=compute_d_j_clt_null(IndT1[r+1][s+1])*
		     ((v2-v1)*(u4-u2)+(v4-v2)*(u2-u1));
	  temp_calc-=compute_d_j_clt_null(IndT1[r+1][s])*
		     ((v2-v1)*(u1-u4)+(v1-v4)*(u2-u1));
	  temp_calc-=compute_d_j_clt_null(IndT1[r][s+1])*
		     ((v4-v2)*(u1-u4)+(v1-v4)*(u4-u2));
	  cpt1[1][i]=temp_calc;

	  temp_calc=compute_d_j_clt_null(IndT1[r+2][s])*(u2-u1)*(u2-u1);
	  temp_calc+=compute_d_j_clt_null(IndT1[r][s+2])*(u4-u2)*(u4-u2);
	  temp_calc+=compute_d_j_clt_null(IndT1[r][s])*(u1-u4)*(u1-u4);
	  temp_calc+=compute_d_j_clt_null(IndT1[r+1][s+1])*2*(u2-u1)*(u4-u2);
	  temp_calc+=compute_d_j_clt_null(IndT1[r+1][s])*2*(u2-u1)*(u1-u4);
	  temp_calc+=compute_d_j_clt_null(IndT1[r][s+1])*2*(u4-u2)*(u1-u4);
	  cpt1[2][i]=temp_calc;

	  /* Second triangle V4 V2 V3 */
	  temp_calc=compute_d_j_clt_null(IndT2[r+2][s])*(v3-v2)*(v3-v2);
	  temp_calc+=compute_d_j_clt_null(IndT2[r][s+2])*(v4-v3)*(v4-v3);
	  temp_calc+=compute_d_j_clt_null(IndT2[r][s])*(v2-v4)*(v2-v4);
	  temp_calc+=compute_d_j_clt_null(IndT2[r+1][s+1])*2*(v3-v2)*(v4-v3);
	  temp_calc+=compute_d_j_clt_null(IndT2[r+1][s])*2*(v3-v2)*(v2-v4);
	  temp_calc+=compute_d_j_clt_null(IndT2[r][s+1])*2*(v4-v3)*(v2-v4);
	  cpt1[3][i]=temp_calc;

	  temp_calc=-compute_d_j_clt_null(IndT2[r+2][s])*(v3-v2)*(u3-u2);
	  temp_calc-=compute_d_j_clt_null(IndT2[r][s+2])*(v4-v3)*(u4-u3);
	  temp_calc-=compute_d_j_clt_null(IndT2[r][s])*(v2-v4)*(u2-u4);
	  temp_calc-=compute_d_j_clt_null(IndT2[r+1][s+1])*
		     ((v3-v2)*(u4-u3)+(v4-v3)*(u3-u2));
	  temp_calc-=compute_d_j_clt_null(IndT2[r+1][s])*
		     ((v3-v2)*(u2-u4)+(v2-v4)*(u3-u2));
	  temp_calc-=compute_d_j_clt_null(IndT2[r][s+1])*
		     ((v4-v3)*(u2-u4)+(v2-v4)*(u4-u3));
	  cpt1[4][i]=temp_calc;

	  temp_calc=compute_d_j_clt_null(IndT2[r+2][s])*(u3-u2)*(u3-u2);
	  temp_calc+=compute_d_j_clt_null(IndT2[r][s+2])*(u4-u3)*(u4-u3);
	  temp_calc+=compute_d_j_clt_null(IndT2[r][s])*(u2-u4)*(u2-u4);
	  temp_calc+=compute_d_j_clt_null(IndT2[r+1][s+1])*2*(u3-u2)*(u4-u3);
	  temp_calc+=compute_d_j_clt_null(IndT2[r+1][s])*2*(u3-u2)*(u2-u4);
	  temp_calc+=compute_d_j_clt_null(IndT2[r][s+1])*2*(u4-u3)*(u2-u4);
	  cpt1[5][i]=temp_calc;

	  /* Third triangle V4 V3 V1 */
	  temp_calc=compute_d_j_clt_null(IndT3[r+2][s])*(v1-v3)*(v1-v3);
	  temp_calc+=compute_d_j_clt_null(IndT3[r][s+2])*(v4-v1)*(v4-v1);
	  temp_calc+=compute_d_j_clt_null(IndT3[r][s])*(v3-v4)*(v3-v4);
	  temp_calc+=compute_d_j_clt_null(IndT3[r+1][s+1])*2*(v1-v3)*(v4-v1);
	  temp_calc+=compute_d_j_clt_null(IndT3[r+1][s])*2*(v1-v3)*(v3-v4);
	  temp_calc+=compute_d_j_clt_null(IndT3[r][s+1])*2*(v4-v1)*(v3-v4);
	  cpt1[6][i]=temp_calc;

	  temp_calc=-compute_d_j_clt_null(IndT3[r+2][s])*(v1-v3)*(u1-u3);
	  temp_calc-=compute_d_j_clt_null(IndT3[r][s+2])*(v4-v1)*(u4-u1);
	  temp_calc-=compute_d_j_clt_null(IndT3[r][s])*(v3-v4)*(u3-u4);
	  temp_calc-=compute_d_j_clt_null(IndT3[r+1][s+1])*
		     ((v1-v3)*(u4-u1)+(v4-v1)*(u1-u3));
	  temp_calc-=compute_d_j_clt_null(IndT3[r+1][s])*
		     ((v1-v3)*(u3-u4)+(v3-v4)*(u1-u3));
	  temp_calc-=compute_d_j_clt_null(IndT3[r][s+1])*
		     ((v4-v1)*(u3-u4)+(v3-v4)*(u4-u1));
	  cpt1[7][i]=temp_calc;

	  temp_calc=compute_d_j_clt_null(IndT3[r+2][s])*(u1-u3)*(u1-u3);
	  temp_calc+=compute_d_j_clt_null(IndT3[r][s+2])*(u4-u1)*(u4-u1);
	  temp_calc+=compute_d_j_clt_null(IndT3[r][s])*(u3-u4)*(u3-u4);
	  temp_calc+=compute_d_j_clt_null(IndT3[r+1][s+1])*2*(u1-u3)*(u4-u1);
	  temp_calc+=compute_d_j_clt_null(IndT3[r+1][s])*2*(u1-u3)*(u3-u4);
	  temp_calc+=compute_d_j_clt_null(IndT3[r][s+1])*2*(u4-u1)*(u3-u4);
	  cpt1[8][i]=temp_calc;
	  i++;
	 }  /* End for s (and for r) */
	 temp_calc=(sum_cpt1_prod(cpt1[0],cpt1[0])+
		    2.*sum_cpt1_prod(cpt1[1],cpt1[1])+
		    sum_cpt1_prod(cpt1[2],cpt1[2]));
	 temp_calc+=(sum_cpt1_prod(cpt1[3],cpt1[3])+
		     2.*sum_cpt1_prod(cpt1[4],cpt1[4])+
		     sum_cpt1_prod(cpt1[5],cpt1[5]));
	 temp_calc+=(sum_cpt1_prod(cpt1[6],cpt1[6])+
		     2.*sum_cpt1_prod(cpt1[7],cpt1[7])+
		     sum_cpt1_prod(cpt1[8],cpt1[8]));
         /* The constant 3./(8. *pow(At/3.,3.)
            comes from (6/4 a^2)^2 * (2a / (2+1)(2+2) ) 
            where a=At/3 */
	 temp_calc=temp_calc*3./(8. *pow(At/3.,3.));
	 fwrite((char *)&temp_calc,size_double,1,bfile);
	 triangle_order[triangle_number].v1=iv1;
	 triangle_order[triangle_number].v2=iv2;
	 triangle_order[triangle_number].v3=iv3;
	 if (triangle_number%25==24)
	   printf("Computed data for %ld triangles.\n",
		  triangle_number+1); triangle_number++;
	} /* End of data computations for the current triangle */
      iv2=iv3;
      fscanf(trifile,"%ld ",&iv3);
     }  /* End of the reading of the row in the triangulation file */
    fscanf(trifile,"%s ",base_buffer);
    iv1=atol(base_buffer);
   }  /* End of reading triangulation file at the exit of the cycle */
 while (base_buffer[0]!='B');
 /* There are triangle_number triangles numbered from 0 to triangle_number-1 */
 fclose(bfile);
 fseek(trifile,2L,SEEK_SET);   /* Pointing to the first node of the first cell */
 printf("Starting computations for each node.\n");
 /**/
 for (cont_nodi=0;cont_nodi<num_nodo;cont_nodi++)
 {
  zx_ext[cont_nodi]=0.; zy_ext[cont_nodi]=0.;
  sum_neigh_weights[cont_nodi]=0.;
 } 
 /* MAIN CYCLE OVER CELLS  */
 bfile=fopen("blocks.tmp","rb");
 if (bfile==NULL)
 {
  printf("Cannot open temporary file for reading.\n");
  exit(1);
 }
 for (cont_nodi=0;cont_nodi<num_nodo;cont_nodi++)
 {
  N_C=0; iv1=cont_nodi;
  open_cella_flag=1;  /* Suppose the cell is an open one */
  /* Skipping base node */
  fscanf(trifile," %ld ",&temp_index); N_C++;
  free_ind_node_list(lista_vicini);
  /* Creating first node */
  fscanf(trifile," %ld ",&temp_index); N_C++; 
 /* there exists at least 1 neighbouring node */
  lista_vicini=malloc(sizeof(struct ind_node_list));
  lista_vicini->glob_index=temp_index;
  vicino_pt=lista_vicini;
  /* I shall create the new locations pointed by pt->next */
  /* Preparing the cycle for the reading of the other neighbouring nodes*/
  fscanf(trifile," %ld ",&temp_index); /* N_C will be increased later */
  while (temp_index!=-1)
  {
   N_C++;
   vicino_pt->next=malloc(sizeof(struct ind_node_list));
   vicino_pt->next->glob_index=temp_index;
   vicino_pt->next->next=NULL;
   vicino_pt=vicino_pt->next;
   fscanf(trifile," %ld ",&temp_index);
  }  /* end while relative to neighbouring nodes reading */
  if (lista_vicini->glob_index==vicino_pt->glob_index)
    {
     open_cella_flag=0; /* The cell is closed */
     N_C--; /* N_C does not consider the repetition of the first neigh. node */
    }
  /*  If the cell is a closed one, the reading of the repetition of the first
  ** neighbouring node has been done to optimize later the computation for
  ** global indexes */
  /* Initialization of dynamic memory  */
  A=malloc(size_double*4*N_C*N_C);
  if (A==NULL) {printf("Not enough memory\n"); exit(1); };
  gg=malloc(size_double*2*N_C);
  if (gg==NULL) {printf("Not enough memory\n"); exit(1); };
  v=malloc(size_double*2*N_C);
  if (v==NULL) {printf("Not enough memory\n"); exit(1); };
  DiagL=malloc(size_double*2*N_C);
  if (DiagL==NULL) {printf("Not enough memory\n"); exit(1); };
  /* Data initialization  */
  c=0.;
  for (i=0;i<2*N_C;i++)
   {
    for (j=0;j<=i;++j) A[i*2*N_C+j]=0.;
    gg[i]=0.;
   }
  /* Beginning computations for each triangle of every cell */
  vicino_pt=lista_vicini;
  /* initializing iv2 to optimize the for cycle with a false iv3 index */
  iv3=vicino_pt->glob_index;
  vicino_pt=vicino_pt->next;
  /* N_C is the number of elements of the cell, open or closed */
  for (nt=1;nt<=N_C-1-open_cella_flag;nt++)
  {
   iv2=iv3;
   iv3=vicino_pt->glob_index;
   /* Now if nt=N_C-1 and the cell is a closed one, iv3 is the correct value
   ** by means of the repetition done when reading indexes for the cell */
   /* Conversion of the triangle in its canonical form where iv1<iv2, iv1<iv3 */
   shift_right=0;
   if (iv1>iv2 || iv1>iv3)
   {
    temp_index=iv3; iv3=iv2; iv2=iv1; iv1=temp_index;
    shift_right++;
   }
   /* Completing the test */
   if (iv1>iv2 || iv1>iv3)
   {
    temp_index=iv3; iv3=iv2; iv2=iv1; iv1=temp_index;
    shift_right++;
   }
   /* Searching for the index of the current triangle */
   triangle_counter=0;
   stop=0;
   do
   {
   if (triangle_order[triangle_counter].v1!=iv1) triangle_counter++;
    else
     if (triangle_order[triangle_counter].v2!=iv2) triangle_counter++;
      else
       if (triangle_order[triangle_counter].v3!=iv3) triangle_counter++;
	else stop=1;
   }
   while (stop==0 && triangle_counter<triangle_number);
   if (triangle_counter==triangle_number)
    {
     printf("Cannot find the triangle in the temporary file.\n");
     exit(1);
    }
   /* triangle_counter is the index of the current triangle into triangle_order */
   /* Transforming the triangle into its original form again without changing
   ** the value of shift_right */
   if (shift_right==1) /* making a shift_left */
   {
    temp_index=iv1; iv1=iv2; iv2=iv3; iv3=temp_index;
   }
   else
   if (shift_right==2) /* making a new shift_right (-2=1 mod 3) */
   {
    temp_index=iv3; iv3=iv2; iv2=iv1; iv1=temp_index;
   }
   /* seeking the pointer in the temporary swap file */
   fseek(bfile,size_double*triangle_counter*28L,SEEK_SET);
   /* 
   ** There are 28 stored values per triangle, 21 ones
   ** for A matrix (6*7/2), 6 ones for vector gg and only one
   ** for the constant c.
   */
   /* Building linear system */
   /* Computation of the matrix A */
   for (ip=0;ip<=5;++ip)
   {
    ip_global=global_index(N_C,(ip+4*shift_right)%6,nt);
    for (jp=0;jp<=ip;++jp)
    {
     jp_global=global_index(N_C,(jp+4*shift_right)%6,nt);
     fread((char *)&temp_calc,size_double,1,bfile);
     if (jp_global<=ip_global)
       A[ip_global*2*N_C+jp_global]+=temp_calc;
      else
       A[jp_global*2*N_C+ip_global]+=temp_calc;
    }
   }
   /* FIRST step for gg */
   for (ip=0;ip<=5;++ip)
   {
    ip_global=global_index(N_C,(ip+4*shift_right)%6,nt);
    fread((char *)&temp_calc,size_double,1,bfile);
    gg[ip_global]+=temp_calc;
   }
   /* Compute c */
   fread((char *)&temp_calc,size_double,1,bfile);
   c+=temp_calc;
   /* SECOND step for gg */
   vicino_pt_temp=lista_vicini;
   for (i=1;i<N_C;++i)
   {
    if(vicino_pt_temp->glob_index!=iv2 && vicino_pt_temp->glob_index!=iv3)
    /* skipping the test over iv1 if it has the same value as cont_nodi */
     {
      gg[2*i]+=temp_calc;
      gg[2*i+1]+=temp_calc;
     }
     vicino_pt_temp=vicino_pt_temp->next;
     /* This structure agrees with the loop by means of the previous reading */
   }
   vicino_pt=vicino_pt->next;
  } /* fine for su nt */
  /* transforming phi (computed in gg) to -g (computeted in gg too) */
  for (i=0;i<2*N_C;++i) gg[i]=-gg[i]+A[i*2*N_C+i]/2.+c;
  /*  v is the solution of the linear system Av=gg 
  ** (remember that -g=gg) */
  err=cholesky(A,v,gg,2*N_C,1,2*N_C,1,DiagL);
  if (err==1) 
  {
   printf("Singular or ill-conditioned matrix for the node %ld.\n",
	   cont_nodi);
   printf("Give fictitious solution with all zeros \n");
   for (i=0;i<2*N_C;++i) v[i]=0.;
  }
  /* Update the central contribution */
  zx[cont_nodi]=v[0]; zy[cont_nodi]=v[1];
  /* Update the external contributions and the partial sum
     of the inverse squared distances of the neighboors */ 
  vicino_pt_temp=lista_vicini;
  for (i=1;i<N_C;++i)
  {
   temp_calc=alpha*exp(-beta*
                       (pow(x[cont_nodi]-x[vicino_pt_temp->glob_index],2.)+
                        pow(y[cont_nodi]-y[vicino_pt_temp->glob_index],2.)))/
             pow((pow(x[cont_nodi]-x[vicino_pt_temp->glob_index],2.)+
                  pow(y[cont_nodi]-y[vicino_pt_temp->glob_index],2.)),gamma/2.);  
   sum_neigh_weights[cont_nodi]+=temp_calc;
   zx_ext[vicino_pt_temp->glob_index]+=v[2*i]*temp_calc;
   zy_ext[vicino_pt_temp->glob_index]+=v[2*i+1]*temp_calc;
   vicino_pt_temp=vicino_pt_temp->next;
   }     
  free(A); free(gg); free(v); free(DiagL);
  if (cont_nodi%25==24)
    printf("Completed %ld nodes \n",(cont_nodi+1));
 } /* End of main loop */
 fclose(bfile);
 fclose(trifile);
 for (cont_nodi=0;cont_nodi<num_nodo;cont_nodi++)
 {
  zx_ext[cont_nodi]=zx_ext[cont_nodi]/sum_neigh_weights[cont_nodi];
  zy_ext[cont_nodi]=zy_ext[cont_nodi]/sum_neigh_weights[cont_nodi];
  zx[cont_nodi]=(1.-q)*zx[cont_nodi]+q*zx_ext[cont_nodi];
  zy[cont_nodi]=(1.-q)*zy[cont_nodi]+q*zy_ext[cont_nodi]; 
  fprintf(outfile,"%*.*f %*.*f %*.*f %*.*f %*.*f\n",
	  WIDTH,PREC,zx[cont_nodi],WIDTH,PREC,zy[cont_nodi],
	  WIDTH,PREC,0.,WIDTH,PREC,0.,WIDTH,PREC,0.); 
 }
 fclose(outfile);
}

int global_index(N,i,j)
int N,i,j;
{
 switch(i/2)
 {
  /* return a statement that allows to avoid 'break' statements 
     in various cases
  /* Forget breaks */
  case 0: /* Relative to the first vertex of the triangle */
	  return (i%2);
  case 1: /* Relative to the second vertex of the triangle */
	  return(2*j+i%2);
  case 2: /* Relative to the third vertex of the triangle*/
	  return(2*(j%(N-1)+1)+i%2);
 }
}
