/*****************************************************************/
/*  win_util.c                                                   */
/*  by Rick Phillips                                             */
/*****************************************************************/

/* Include files */
#include <stdio.h>
#include "general.h"
#include "win_util.h"

/* Global variables */
WSADATA WSAData;
static int init = 0;



/****************************************************************************************
    WinSockInit
	
    Purpose:
		Initialize the WinSock routines.  Call this routine before making any other 
		WinSock calls.

    Argument List:
		void

    Returns:
		TRUE on success, FALSE on failure.
****************************************************************************************/
int WinSockInit(void)
{
	/* Return success if WinSock has already been initialized */
    if (init++)
		return(TRUE);
	
	/* Initialize the WinSock library */
    if (WSAStartup(0x0101, &WSAData)) {
		init = 0;
		fprintf(STDERR"WinSockInit: WSAStartup failed.\n");
		WSACleanup();
		return(FALSE);
    }

	/* Success! */
    return(TRUE);
}



/****************************************************************************************
    WinSockFinish
	
    Purpose:
		Release the WinSock library.  Call this routine when you are finished with all
		WinSock calls.

    Argument List:
		void

    Returns:
		TRUE on success, FALSE on failure.
****************************************************************************************/
int WinSockFinish(void)
{
	/* Return success if WinSock had already been released or doesn't need to be released yet */
    if (init == 0 || --init > 0)
		return(TRUE);

	/* Release WinSock */
	if (WSACleanup()) {
		init = 0;
		fprintf(STDERR"WinSockFinish: WSACleanup failed.\n");
		return(FALSE);
    }
    
	/* Success! */
    return(TRUE);
}



/****************************************************************************************
    SendData
	
    Purpose:
		Send data on an open socket.  This is a synchronous routine with a timeout.
		Control isn't returned until all the data has been sent or the timeout value is 
		exceeded.

    Argument List:
		I	theSocket		socket to use.
		I	buffer			pointer to the buffer containing the data to be sent.
		I	bufferSize		number of bytes to send.

    Returns:
		Success			Actual number of bytes sent (will be equal to bufferSize).
		Timeout			Actual number of bytes sent (will be less than bufferSize, but >= 0).
		Socket Error	SOCKET_ERROR.  Use WSAGetLastError() to get the actual error.
****************************************************************************************/
long SendData(SOCKET theSocket, void *buffer, long bufferSize)
{
    long			status = 1;
    long			bytesSent = 0;
    long			byteCount;
    fd_set			fdSet, fdMasterSet;
    struct timeval	timeOut;
    
    /* Initialize the timeout value */
    timeOut.tv_sec = SOCKET_TIMEOUT_SECONDS;
    timeOut.tv_usec = 0;

    /* Initialize the master fd set */
    FD_ZERO(&fdMasterSet);
    FD_SET(theSocket, &fdMasterSet);

    /* Loop sending chunks of data until all has been sent */
    while (bytesSent < bufferSize) {
    
    	/* Reset the fd set */
    	fdSet = fdMasterSet;
    	
    	/* Wait until the socket is ready or until we time out */
    	status = select(theSocket+1, NULL, &fdSet, NULL, &timeOut);
    	
    	switch (status) {
			case 1:		/* Socket is ready */
			    /* Write a chunk of data */
				if ((byteCount = send(theSocket, ((char far *)buffer)+bytesSent, bufferSize-bytesSent, 0)) == SOCKET_ERROR) {
					fprintf(STDERR"SendData failed (socket error on send)\n");
					return(SOCKET_ERROR);
				}
			    /* Update the counter */
			    bytesSent += byteCount;
			    break;
		
			case 0:		/* Socket timed out */
				fprintf(STDERR"SendData failed (time out)\n");
				return(bytesSent);
				break;

			case SOCKET_ERROR:
				fprintf(STDERR"SendData failed (socket error on select)\n");
		    	return(SOCKET_ERROR);
		    	break;

			default:	/* Unknown error */
				fprintf(STDERR"SendData failed (unknown error)\n");
		    	return(SOCKET_ERROR);
		    	break;
		}
    }

    return(bytesSent);
}



/****************************************************************************************
    ReceiveData
	
    Purpose:
		Receive data on an open socket.  This is a synchronous routine with a timeout.
		Control isn't returned until all the data has been received or the timeout value is 
		exceeded.

    Argument List:
		I	theSocket		socket to use.
		I	buffer			pointer to the buffer to copy the received data into.
		I	bufferSize		number of bytes to receive.

    Returns:
		Success			Actual number of bytes received (will be equal to bufferSize).
		Timeout			Actual number of bytes received (will be less than bufferSize, but >= 0).
		Socket Error	SOCKET_ERROR.  Use WSAGetLastError() to get the actual error.
****************************************************************************************/
long ReceiveData(SOCKET theSocket, void *buffer, long bufferSize)
{
    int				status = 1;
    long			bytesReceived = 0;
    long			byteCount;
    fd_set 			fdSet, fdMasterSet;
    struct timeval 	timeOut;

    /* Initialize the timeout value */
    timeOut.tv_sec = SOCKET_TIMEOUT_SECONDS;
    timeOut.tv_usec = 0;

    /* Initialize the master fd set */
    FD_ZERO(&fdMasterSet);
    FD_SET(theSocket, &fdMasterSet);

    /* Loop receiving chunks of data until all has been received or waitSecs is exceeded */
    while ((bytesReceived < bufferSize) && (status > 0)) {

    	/* Reset the fd set */
    	fdSet = fdMasterSet;
    	
    	/* Wait until the socket is ready or until we time out */
    	status = select(theSocket+1, &fdSet, NULL, NULL, &timeOut);
    	
    	switch (status) {
    		case 1:		/* Socket is ready */
			    /* Read a chunk of data */
				if ((byteCount = recv(theSocket, (char far *)buffer+bytesReceived, bufferSize-bytesReceived, 0)) == SOCKET_ERROR) {
					fprintf(STDERR"ReceiveData failed (socket error on recv)\n");
					return(SOCKET_ERROR);
				}
			    /* Update the counter */
			    bytesReceived += byteCount;
			    break;
			    
			case 0:		/* Socket timed out */
				fprintf(STDERR"ReceiveData failed (time out)\n");
				return(bytesReceived);
				break;

			case SOCKET_ERROR:
				fprintf(STDERR"ReceiveData failed (socket error on select)\n");
		    	return(SOCKET_ERROR);
		    	break;

			default:	/* Unknown error */
				fprintf(STDERR"ReceiveData failed (unknown error)\n");
		    	return(SOCKET_ERROR);
		    	break;
		}
    }

    return(bytesReceived);
}



/****************************************************************************************
    bcopy
	
    Purpose:
		Copy memory.

    Argument List:
		I	s		pointer to source.
		O	d		pointer to destination.
		I	len		number of bytes to copy.

    Returns:
    	void
****************************************************************************************/
void bcopy(const void *s, void *d, int len)
{
	for(; len > 0; len--)
		*((char *)d)++ = *((char *)s)++;
}



/****************************************************************************************
    bzero
	
    Purpose:
		Initialize memory to zero.

    Argument List:
		O	t		pointer to target.
		I	len		number of bytes to initialize.

    Returns:
    	void
****************************************************************************************/
void bzero(void *t, int len)
{
	for(; len > 0; len--)
		*((char *)t)++ = (char) 0;
}



/****************************************************************************************
    bcmp
	
    Purpose:
		Compare memory.

    Argument List:
		I	m1		pointer to first memory block.
		I	m2		pointer to second memory block.
		I	len		number of bytes to compare.

    Returns:
    	void
****************************************************************************************/
int bcmp(const void *m1, const void *m2, int len)
{
	for(; len > 0; len--, ((char *)m1)++, ((char *)m2)++)
		if (*((char *)m1) != *((char *)m2))
			return 1;
	return 0;
}



