ClearAll["NSDemo*"]

NSDemoClearScreen[] := Do[Print["\n"], {40}]

NSDemoPause[] := Module[{s},
   s = InputString["Type 'quit' to exit from this demo or hit Enter to continue: "];
   Return[s == "quit"]
]

NSDemoPrintMatrix[m_?MatrixQ] :=
( Print["      {"] ;
  Map[Print["        ", #]&, m];
  Print["      }"]
)
  

NSDemoUsage[] := Module[{},
  Print["To see the usage type\n"];
  Print["   NetSolve[]\n"];
	
  Print["and you will see a list NetSolve functions and hints for their usage\n"];

  NetSolve[];  
  Print["\n"];  
  If[NSDemoPause[], Return[True], Null];

  Print["NetSolve function can be divided into following categories:"];
  Print["  - problem solving:"];
  Print["      NetSolve[ProblemName[arguments, ...]]"];
  Print["  - nonblocking (NB) problem solving:"];
  Print["      NetSolveNB[ProblemName[arguments,...]"];
  Print["      NetSolveProbe[request]"];
  Print["      NetSolveWait[request]"];
  Print["  - getting/setting an agent:"];
  Print["      NetSolveGetAgent[]"];
  Print["      NetSolveSetAgent[Agent]"];
  Print["  - error handling"];
  Print["      NetSolveError[]"];
  Print["      NetSolveErrorMsg[rc]"];
  Print["  - info functions:"];
  Print["      NetSolve[\"?problems\"]"];
  Print["      NetSolve[\"?servers\"]"];
  Print["      NetSolve[\"?FuncName[]\"]\n"];
  
  Return[NSDemoPause[]]

]

NSDemoInfo[] := Module[{s},
  Print["The list of available problems can be obtained by the function\n"];
  Print["  NetSolve[\"?problems\"]"];
  
  Print["\nPlease hit Enter to see a list of the problems"];
  InputString["available from agent " <> NetSolveGetAgent[] <> ": "];
  
  Print["\n"];
  NetSolve["?problems"];
  Print["\n"];
  
  If[NSDemoPause[], Return[True], Null];

  Print["The same way you could get a list of the available server by the function\n"];
  Print["  NetSolve[\"?servers\"]"];
  
  Print["\nPlease hit Enter to see a list of the servers"];
  InputString["connected to your agent: "];
  
  Print["\n"];
  NetSolve["?servers"];
  Print["\n"];

  Print["You probably would need information about how to invoke a problem."];
  Print["NetSolve problems are like Mathematica's function so you have to"];
  Print["put ending '[]'. The format is \n"];
  
  Print["  NetSolve[\"?FuncName[]\"]"];
  Print["\n  where FuncName is the problem name\n"];
  
  Print["\nFor example, we can get help on iqsort problem by typing\n"];
  
  Print["  NetSolve[\"?iqsort[]\"]\n"];
  
  InputString["Please hit Enter to see the result: "];
  
  Print["\n-----------------------"];
  Print["\n"];
  NetSolve["?iqsort[]"];
  Print["\n-----------------------\n"];
  
  Print["There is a lot of information about the types of every"];
  Print["argument (object in NetSolve's terms) the problem have\n"];

  Return[ NSDemoPause[]]
]

NSDemoSolve[] := Module[{v,A,B,C,D},
  v = Table[Ceiling[100*Random[]], {i, 10}];
  
  Print["\nNow, lets try to call iqsort[] problem. It is a very simple problem"];
  Print["All it does is sorting the integer vector passed as input\n"];
  
  Print["For example lets have the vector\n"];
  Print["  v = " <> ToString[v] <> ",\n"];
  Print["which has been generated this way:"];
  Print["  v = Table[Ceiling[100*Random[]], {i, 10}]\n"];
  
  Print["To sort it using NetSolve we have to call:\n"];
  Print["  NetSolve[iqsort[v]]\n"];
  
  Print["The result is a sorted vector:\n"];
  
  v = NetSolve[iqsort[v]];
  
  Print[ToString[v]];
  
  Print["\n"];

  If[NSDemoPause[], Return[True], Null];

  Print["Now, lets call a little bit more complex problem. This is dgemm which"];
  Print["performs the following:\n"];
  
  Print[" a A . B + b C\n"];
  
  A = Partition[Range[9],3];
  B = Transpose[A]*2;
  C = -IdentityMatrix[3];
  
  Print["Lets have A,B,C as follows:\n"];
  Print["A = " <> ToString[A]];
  Print["B = " <> ToString[B]];
  Print["C = " <> ToString[C]];
  
  Print["\nThey were generated this way:"];
  Print["  A = Partition[Range[9],3]"];
  Print["  B = Transpose[A]*2"];
  Print["  C = -IdentityMatrix[3]\n"];
  
  Print["We call dgemm[] for example like this:\n"];
  Print["  NetSolve[dgemm[\"N\",\"N\", 2, A, B, 3, C]]"];
  
  C = NetSolve[dgemm["N","N",2, A, B, 3, C]];

  Print["\nThe result is: " <> ToString[C] <> "\n"];

  If[NSDemoPause[], Return[True], Null];

  Print["Lets try another example using complex types. Lets multiply"];
  Print["the matrices C and D which are complex.\n"];
  
  C = A - B I;
  D = B + A I;
  
  
  Print["We have generated C and D generated\n"];
  Print["  C = A - B I"];
  Print["  D = B + A I\n"];
  
  Print["  C = "];
  NSDemoPrintMatrix[C];
  Print["  D = "];
  NSDemoPrintMatrix[D];
  
  Print["\nLets call NetSolve now\n"];
  Print["  NetSolve[zmatmul[C,D]]\n"];
  
  Print["The result is: "];
  C = NetSolve[zmatmul[C,D]];
  NSDemoPrintMatrix[C];
  Print["\n"];

  If[NSDemoPause[], Return[True], Null];

  Print["Now lets solve the following small linear system of equations"];
  Print["  A . x = B\n"];
  
  Print["  where A and B are matrices\n"];
  
  Print["We generate A to be a random matrix\n"];
  
  Print["  A = Table[Random[], {i,3}, {j,3}]\n"];
  
  Print["Lets B[[i]] = Sum[j*A[[i,j]], {j,1,3}]\n"];
  
  Print["So that the result should be close to {1.0, 2.0, 3.0}\n"];

  A = Table[Random[], {i,3}, {j,3}];
  B = Table[ Sum[j*A[[i,j]], {j,1,3}], {i,3}];
  
  
  Print["\n  A ="];
  NSDemoPrintMatrix[A];
  Print["  B = ", B];
  Print["\nLets call NetSolve[linsol[A, B]] now\n"];
  
  C = NetSolve[linsol[A,B]];
  
  Print["The result is: ", C, "\n"];

  Return[NSDemoPause[]];
]

NSDemoSolveNB[] := Module[{request},
 
  Print["\n"];
  Print["When you use blocking calls you can do nothing while you are"];
  Print["waiting for the result. When you want to overlap the calculation"];
  Print["made by NetSolve with other kind of useful work you can use"];
  Print["nonblocking (NB) calls. Then you call NetSolveNB[] instead of"];
  Print["NetSolve[]. The format of the function is the same,but you receive"];
  Print["an integer, which is the request number instead of the result itself"];
  Print["When you call nonblocking function you receive the execution back"];
  Print["almost immediately. Once you have the request back, you have three"];
  Print["possibilities:\n"];
  Print["  1. To do some useful work"];
  Print["  2. To check if the request is completed"];
  Print["  3. To block and to wait for the result\n"];
  
  Print["Lets demonstrate this with an example. Suppose we would like"];
  Print["to sort a vector using nonblocking call.\n"];
  Print["  request = NetSolveNB[iqsort[{9,8,7,6,5,4,3,2,1}]]"];
  
  request = NetSolveNB[iqsort[{9,8,7,6,5,4,3,2,1}]];
  
  Print["  request = ", request,"\n"];
  
  Print["Lets try NetSolveProbe[request] now\n"];
  Print["The result is: ", NetSolveProbe[request]];
  
  Print["\nWe can block and wait for the result"];
  Print["  ", NetSolveWait[request], "\n"];
  
  Return[]
]

NSDemoChangingAgent[] := Module[{},
  Return[ NSDemoPause[]]
]
   
NSDemo[] := Module[{},
   Print["This demo function written in Mathematica shows most of"] ;
   Print["the features NetSolve offers to you.\n"];
   
   If[ StringQ[NetSolveGetAgent[] == False ],
       (Print["NetSolve is not installed. Please follow the debscribed"];
       Print["above procedure and run the demo again"];
       Return[]),
       If[ NetSolveGetAgent[] == "", 
           (Print["You have not setup an agent. Please do this as described"];
	    Print["above and run this demo again.\n"];
	    Return[]),
	   Null
	 ]
     ];
  
   Print["Now you are ready to use NetSolve\n"];
   
   If[NSDemoPause[], Return[], Null];
   
   
   If[NSDemoUsage[], Return[], Null];
   If[NSDemoInfo[], Return[], Null];
   If[NSDemoSolve[], Return[], Null];
   If[NSDemoSolveNB[], Return[], Null];
   
   Print["Done..."];
   Null
]

NSDemo[]
ClearAll["NSDemo*"]
