/******************************************************************/
/*                                                                */
/*    Testing the IBP routines                                    */
/*                                                                */
/*    Author: Dorian Arnold                                       */
/*                                                                */
/******************************************************************/


#include "netsolve.h"
#include <stdio.h>
#include <assert.h>
#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>

#ifndef TRUE
#define TRUE 1
#define FALSE 0
#endif

#define Mciswspace(C) ( (((C) > 8) && ((C) < 14)) || ((C) == 32) )


/***************************************************
  The following Auxiliary functions were stolen from
  the ATLAS project:  http://icl.cs.utk.edu/atlas

  IsYes(), GetString

****************************************************/
int IsYes(char def, char *spc, char *expstr)
{
   char ch, ln[256];
   fprintf(stdout, "%s%s [%c]: ", spc, expstr, def);
   if (fgets(ln, 256, stdin) == NULL) ch=def;
   else if (ln[0] == '\0' || ln[0] == '\n') ch=def;
   else ch = ln[0];
   return( ((ch == 'y') || (ch == 'Y')) );
}

void GetString(FILE *fpin, char *Default, char *spc, char *expstr, 
               int len, char *str0)
/*
 * Get a string of length len, not including NULL terminator; pads
 * any extra len with NULLs
 */
{
   char str[512], *sp;
   int i;

   assert(len+1 <= 512);
   if (expstr)
   {
      if (Default) fprintf(stdout, "%sEnter %s [%s]: ", spc, expstr, Default);
      else fprintf(stdout, "%sEnter %s:", spc, expstr);
   }
   sp = fgets(str, 512, fpin);
   if ( (sp == NULL) || (str[0] == '\0') || (str[0] == '\n') )
   {
      if (Default) strcpy(str0, Default);
      else str0[0] = '\0';
      return;
   }
   str[len] = '\0';
   for (i=0; (str0[i] = str[i]); i++);
   if (i) i--;
   while (Mciswspace(str0[i])) i--;
   while (++i < len) str0[i] = '\0';
   str0[i] = '\0';
}

void my_pause()
{
  if (!IsYes('y', "", "Are you ready to continue?")) exit(-1);
}

DSI_FILE * renew_dsi_file(DSI_FILE * oldfile,
                          char *hostname, int flag, int permissions, int size,
                          dsi_type storage_system)
{
  DSI_FILE * newfile;
  if(ns_dsi_close(oldfile)){
    ns_perror("ns_dsi_close()");
    return NULL;
  }

  newfile = ns_dsi_open(hostname, flag, permissions, size, storage_system);

  if(newfile == NULL){
    ns_perror("ns_dsi_open()");
    return NULL;
  }

  return newfile;
}

#define DIMENSION 100
#define DSI_SIZE 1024*1024
#define DSI_MODE O_CREAT|O_RDWR 
#define DSI_PERM 744
#define DSI_TYPE IBP

int main()
{
  DSI_FILE *dsi_file;
  DSI_OBJECT *dsi_object;
  char ibp_server_name[256]="ibp.cs.utk.edu";
  int interactive, multiple_dsi_files;
  int *int_vector_in=NULL, *int_matrix_in=NULL;
  int *int_vector_out=NULL, *int_matrix_out=NULL;
  float *float_vector_in=NULL, *float_matrix_in=NULL;
  float *float_vector_out=NULL, *float_matrix_out=NULL;
  double *double_vector_in=NULL, *double_matrix_in=NULL;
  double *double_vector_out=NULL, *double_matrix_out=NULL;
  char *char_vector_in=NULL, *char_matrix_in=NULL;
  char *char_vector_out=NULL, *char_matrix_out=NULL;
  int i, j, status;

  fprintf(stdout, "\nThis program tests the NetSolve/IBP integration.\n");

  if(IsYes('n', "", "Do you wish for the testing to be interactive?"))
    interactive = TRUE;
  else
    interactive = FALSE;

  GetString(stdin, ibp_server_name, "   ", 
                   "Host name for the IBP server", 255, ibp_server_name);

  if(IsYes('n', "", "Do you wish to use a single DSI file for all remote objects?"))
    multiple_dsi_files = FALSE;
  else
    multiple_dsi_files = TRUE;

  /* open at least the first dsi_file */
  dsi_file = ns_dsi_open(ibp_server_name, DSI_MODE, DSI_PERM, DSI_SIZE, DSI_TYPE);
  if(dsi_file == NULL){
    ns_perror("ns_dsi_open()");
    exit(-1);
  }

  fprintf(stdout, "\nWe will now test the \"netsl()\" call with IBP objects\n");

  /*======== TESTING INTEGER VECTORS ===============*/
  if( (!interactive) ||
      IsYes('y', "   ", "Execute DSI Integer Vector Test?")){
    fprintf(stdout, "\n   Testing netsl() with DSI Vector of INTS ...");

    int_vector_in = (int *)malloc(DIMENSION * sizeof(int));
    int_vector_out = (int *)malloc(DIMENSION * sizeof(int));
    for(i=0; i<DIMENSION; i++)
      int_vector_in[i] = i;

    dsi_object = ns_dsi_write_vector(dsi_file, int_vector_in, DIMENSION, NETSOLVE_I);
    if(dsi_object == NULL){
      ns_perror("ns_dsi_write_vector()");
      if(interactive && !IsYes('y', "", "Failure Occurred! Do you wish to continue?"))
          abort();
    }
    else{
      status = netsl("int_vector_add5()", DIMENSION, dsi_object, int_vector_out);

      if(status < 0){
        netslerr(status);
        if(interactive && !IsYes('y', "", "Failure Occurred! Do you wish to continue?"))
          abort();
      }
      else{
        int success=1;
        for(i=0; i<DIMENSION; i++){
          if( int_vector_out[i] != (int_vector_in[i] + 5) ){
            success = 0;
            break;
          }
        }
        if(success){
          fprintf(stdout, "passed");
        }
        else{
          fprintf(stdout, "failure (incorrect output results).\n");
          if(interactive && !IsYes('y', "", "Failure Occurred! Do you wish to continue?"))
            abort();
        }
      }
    }
    free(int_vector_in); free(int_vector_out);
  }

  /*======== TESTING FLOAT VECTORS ===============*/
  if( (!interactive) ||
      IsYes('y', "   ", "Execute DSI Float Vector Test?")){
    fprintf(stdout, "\n   Testing netsl() with DSI Vector of FLOATS ...");
  
    if(multiple_dsi_files){
      dsi_file = renew_dsi_file(dsi_file, ibp_server_name,
			        DSI_MODE, DSI_PERM, DSI_SIZE, DSI_TYPE);
      if(!dsi_file){
        fprintf(stderr, "Call to open new DSI file failed. Aborting.\n");
        abort();
      }
    }

    float_vector_in = (float *)malloc(DIMENSION * sizeof(float));
    float_vector_out = (float *)malloc(DIMENSION * sizeof(float));
    for(i=0; i<DIMENSION; i++)
      float_vector_in[i] = i;

    dsi_object = ns_dsi_write_vector(dsi_file, float_vector_in, DIMENSION, NETSOLVE_S);
    if(dsi_object == NULL){
      ns_perror("ns_dsi_write_vector()");
      if(interactive && !IsYes('y', "", "Failure Occurred! Do you wish to continue?"))
        abort();
    }
    else{
      status = netsl("float_vector_add5()", DIMENSION, dsi_object, float_vector_out);

      if(status < 0){
        netslerr(status);
        if(interactive && !IsYes('y', "", "Failure Occurred! Do you wish to continue?"))
            abort();
      }
      else{
        int success=1;
        for(i=0; i<DIMENSION; i++){
          if( float_vector_out[i] != (float_vector_in[i] + 5) ){
            success = 0;
            break;
          }
        }
        if(success){
          fprintf(stdout, "passed");
        }
        else{
          fprintf(stdout, "failure (incorrect output results).\n");
          if(interactive && !IsYes('y', "", "Failure Occurred! Do you wish to continue?"))
            exit(-1);
        }
      }
    }
    free(float_vector_in); free(float_vector_out);
  }

  /*======== TESTING DOUBLE VECTORS ===============*/
  if( (!interactive) ||
      IsYes('y', "   ", "Execute DSI Double Vector Test?")){
    fprintf(stdout, "\n   Testing netsl() with DSI Vector of DOUBLES ...");

    if(multiple_dsi_files){
      dsi_file = renew_dsi_file(dsi_file, ibp_server_name,
			        DSI_MODE, DSI_PERM, DSI_SIZE, DSI_TYPE);
      if(!dsi_file){
        fprintf(stderr, "Call to open new DSI file failed. Aborting.\n");
        abort();
      }
    }

    double_vector_in = (double *)malloc(DIMENSION * sizeof(double));
    double_vector_out = (double *)malloc(DIMENSION * sizeof(double));
    for(i=0; i<DIMENSION; i++)
      double_vector_in[i] = 7.0;

    dsi_object = ns_dsi_write_vector(dsi_file, double_vector_in, DIMENSION, NETSOLVE_D);
    if(dsi_object == NULL){
      ns_perror("ns_dsi_write_vector()");
      if(interactive && !IsYes('y', "", "Failure Occurred! Do you wish to continue?"))
        abort();
    }
    else{
      status = netsl("double_vector_add5()", DIMENSION, dsi_object, double_vector_out);

      if(status < 0){
        netslerr(status);
        if(interactive && !IsYes('y', "", "Failure Occurred! Do you wish to continue?"))
          abort();
      }
      else{
        int success=1;
        for(i=0; i<DIMENSION; i++){
          if( double_vector_out[i] != (double_vector_in[i] + 5) ){
            success = 0;
            break;
          }
        }
        if(success){
          fprintf(stdout, "passed");
        }
        else{
          fprintf(stdout, "failure (incorrect output results).\n");
          if(interactive && !IsYes('y', "", "Failure Occurred! Do you wish to continue?"))
            abort();
        }
      }
    }
    free(double_vector_in); free(double_vector_out);
  }

  /*======== TESTING CHAR VECTORS ===============*/
  if( (!interactive) ||
      IsYes('y', "   ", "Execute DSI Character Vector Test?")){
    fprintf(stdout, "\n   Testing netsl() with DSI Vector of CHARS ...");

    if(multiple_dsi_files){
      dsi_file = renew_dsi_file(dsi_file, ibp_server_name,
			        DSI_MODE, DSI_PERM, DSI_SIZE, DSI_TYPE);
      if(!dsi_file){
        fprintf(stderr, "Call to open new DSI file failed. Aborting.\n");
        abort();
      }
    }

    char_vector_in = (char *)malloc(DIMENSION * sizeof(char));
    char_vector_out = (char *)malloc(DIMENSION * sizeof(char));
    for(i=0; i<DIMENSION; i++)
      char_vector_in[i] = i%256;

    dsi_object = ns_dsi_write_vector(dsi_file, char_vector_in, DIMENSION, NETSOLVE_CHAR);
    if(dsi_object == NULL){
      ns_perror("ns_dsi_write_vector()");
      if(interactive && !IsYes('y', "", "Failure Occurred! Do you wish to continue?"))
        abort();
    }
    else{
      status = netsl("char_vector_add5()", DIMENSION, dsi_object, char_vector_out);

      if(status < 0){
        netslerr(status);
        if(interactive && !IsYes('y', "", "Failure Occurred! Do you wish to continue?"))
          abort();
      }
      else{
        int success=1;
        for(i=0; i<DIMENSION; i++){
          if( char_vector_out[i] != (char_vector_in[i] + 5) ){
            success = 0;
            break;
          }
        }
        if(success){
          fprintf(stdout, "passed");
        }
        else{
          fprintf(stdout, "failure (incorrect output results).\n");
          if(interactive && !IsYes('y', "", "Failure Occurred! Do you wish to continue?"))
            abort();
        }
      }
    }
    free(char_vector_in); free(char_vector_out);
  }

  /*======== TESTING INT MATRIX ===============*/
  if( (!interactive) ||
      IsYes('y', "   ", "Execute DSI Integer Matrix Test?")){
    fprintf(stdout, "\n   Testing netsl() with DSI Matrix of INTS ...");

    if(multiple_dsi_files){
      dsi_file = renew_dsi_file(dsi_file, ibp_server_name,
			        DSI_MODE, DSI_PERM, DSI_SIZE, DSI_TYPE);
      if(!dsi_file){
        fprintf(stderr, "Call to open new DSI file failed. Aborting.\n");
        abort();
      }
    }

    int_matrix_in = (int *)malloc(DIMENSION * DIMENSION * sizeof(int));
    int_matrix_out = (int *)malloc(DIMENSION * DIMENSION * sizeof(int));
    for(i=0; i<DIMENSION; i++)
      for(j=0; j<DIMENSION; j++)
        int_matrix_in[i*DIMENSION + j] = j;

    dsi_object = ns_dsi_write_matrix(dsi_file, int_matrix_in,
                                   DIMENSION, DIMENSION, NETSOLVE_I);
    if(dsi_object == NULL){
      ns_perror("ns_dsi_write_matrix()");
      if(interactive && !IsYes('y', "", "Failure Occurred! Do you wish to continue?"))
        abort();
    }
    else{
      status = netsl("int_matrix_add5()", DIMENSION, DIMENSION, dsi_object, int_matrix_out);

      if(status < 0){
        netslerr(status);
        if(interactive && !IsYes('y', "", "Failure Occurred! Do you wish to continue?"))
          abort();
      }
      else{
        int success=1;
        for(i=0; i<DIMENSION; i++)
          for(j=0; j<DIMENSION; j++){
            if( int_matrix_out[i*DIMENSION+j] != (int_matrix_in[i*DIMENSION+j] + 5) ){
              success = 0;
              break;
            }
          }
          if(success){
            fprintf(stdout, "passed");
          }
        else{
          fprintf(stdout, "failure (incorrect output results).\n");
          if(interactive && !IsYes('y', "", "Failure Occurred! Do you wish to continue?"))
            abort();
        }
      }
    }
    free(int_matrix_in); free(int_matrix_out);
  }

  /*======== TESTING FLOAT MATRIX ===============*/
  if( (!interactive) ||
      IsYes('y', "   ", "Execute DSI Float Matrix Test?")){
    fprintf(stdout, "\n   Testing netsl() with DSI Matrix of FLOATS ...");

    if(multiple_dsi_files){
      dsi_file = renew_dsi_file(dsi_file, ibp_server_name,
			        DSI_MODE, DSI_PERM, DSI_SIZE, DSI_TYPE);
      if(!dsi_file){
        fprintf(stderr, "Call to open new DSI file failed. Aborting.\n");
        abort();
      }
    }

    float_matrix_in = (float *)malloc(DIMENSION * DIMENSION * sizeof(float));
    float_matrix_out = (float *)malloc(DIMENSION * DIMENSION * sizeof(float));
    for(i=0; i<DIMENSION; i++)
      for(j=0; j<DIMENSION; j++)
        float_matrix_in[i*DIMENSION + j] = j;

    dsi_object = ns_dsi_write_matrix(dsi_file, float_matrix_in,
                                   DIMENSION, DIMENSION, NETSOLVE_S);
    if(dsi_object == NULL){
      ns_perror("ns_dsi_write_matrix()");
      if(interactive && !IsYes('y', "", "Failure Occurred! Do you wish to continue?"))
        abort();
    }
    else{
      status = netsl("float_matrix_add5()", DIMENSION, DIMENSION, dsi_object, float_matrix_out);

      if(status < 0){
        netslerr(status);
        if(interactive && !IsYes('y', "", "Failure Occurred! Do you wish to continue?"))
          abort();
      }
      else{
        int success=1;
        for(i=0; i<DIMENSION; i++)
          for(j=0; j<DIMENSION; j++){
            if( float_matrix_out[i*DIMENSION+j] != (float_matrix_in[i*DIMENSION+j] + 5) ){
              success = 0;
              break;
            }
          }
          if(success){
            fprintf(stdout, "passed");
          }
        else{
          fprintf(stdout, "failure (incorrect output results).\n");
          if(interactive && !IsYes('y', "", "Failure Occurred! Do you wish to continue?"))
            abort();
        }
      }
    }
    free(float_matrix_in); free(float_matrix_out);
  }

  /*======== TESTING DOUBLE MATRIX ===============*/
  if( (!interactive) ||
      IsYes('y', "   ", "Execute DSI Double Matrix Test?")){
    fprintf(stdout, "\n   Testing netsl() with DSI Matrix of DOUBLES ...");

    if(multiple_dsi_files){
      dsi_file = renew_dsi_file(dsi_file, ibp_server_name,
			        DSI_MODE, DSI_PERM, DSI_SIZE, DSI_TYPE);
      if(!dsi_file){
        fprintf(stderr, "Call to open new DSI file failed. Aborting.\n");
        abort();
      }
    }

    double_matrix_in = (double *)malloc(DIMENSION * DIMENSION * sizeof(double));
    double_matrix_out = (double *)malloc(DIMENSION * DIMENSION * sizeof(double));
    for(i=0; i<DIMENSION; i++)
      for(j=0; j<DIMENSION; j++)
        double_matrix_in[i*DIMENSION + j] = j;

    dsi_object = ns_dsi_write_matrix(dsi_file, double_matrix_in,
                                   DIMENSION, DIMENSION, NETSOLVE_D);
    if(dsi_object == NULL){
      ns_perror("ns_dsi_write_matrix()");
      if(interactive && !IsYes('y', "", "Failure Occurred! Do you wish to continue?"))
        abort();
    }
    else{
      status = netsl("double_matrix_add5()", DIMENSION, DIMENSION, dsi_object, double_matrix_out);

      if(status < 0){
        netslerr(status);
        if(!IsYes('y', "", "Failure Occurred! Do you wish to continue?"))
          exit(-1);
      }
      else{
        int success=1;
        for(i=0; i<DIMENSION; i++)
          for(j=0; j<DIMENSION; j++){
            if( double_matrix_out[i*DIMENSION+j] != (double_matrix_in[i*DIMENSION+j] + 5) ){
              success = 0;
              break;
            }
          }
          if(success){
            fprintf(stdout, "passed");
          }
          else{
            fprintf(stdout, "failure (incorrect output results).\n");
            if(interactive && !IsYes('y', "", "Failure Occurred! Do you wish to continue?"))
              abort();
	  }
      }
    }
    free(double_matrix_in); free(double_matrix_out);
  }

  /*======== TESTING CHAR MATRIX ===============*/
  if( (!interactive) ||
      IsYes('y', "   ", "Execute DSI Char Matrix Test?")){
    fprintf(stdout, "\n   Testing netsl() with DSI Matrix of CHAR ...");

    if(multiple_dsi_files){
      dsi_file = renew_dsi_file(dsi_file, ibp_server_name,
			        DSI_MODE, DSI_PERM, DSI_SIZE, DSI_TYPE);
      if(!dsi_file){
        fprintf(stderr, "Call to open new DSI file failed. Aborting.\n");
        abort();
      }
    }

    char_matrix_in = (char *)malloc(DIMENSION * DIMENSION * sizeof(char));
    char_matrix_out = (char *)malloc(DIMENSION * DIMENSION * sizeof(char));
    for(i=0; i<DIMENSION; i++)
      for(j=0; j<DIMENSION; j++)
        char_matrix_in[i*DIMENSION + j] = j%256;

    dsi_object = ns_dsi_write_matrix(dsi_file, char_matrix_in,
                                     DIMENSION, DIMENSION, NETSOLVE_CHAR);
    if(dsi_object == NULL){
      ns_perror("ns_dsi_write_matrix()");
      if(interactive && !IsYes('y', "", "Failure Occurred! Do you wish to continue?"))
        abort();
    }
    else{
      status = netsl("char_matrix_add5()", DIMENSION, DIMENSION, dsi_object, char_matrix_out);

      if(status < 0){
        netslerr(status);
        if(interactive && !IsYes('y', "", "Failure Occurred! Do you wish to continue?"))
          abort();
      }
      else{
        int success=1;
        for(i=0; i<DIMENSION; i++)
          for(j=0; j<DIMENSION; j++){
            if( char_matrix_out[i*DIMENSION+j] != (char_matrix_in[i*DIMENSION+j] + 5) ){
              success = 0;
              break;
            }
          }
          if(success){
            fprintf(stdout, "passed");
          }
        else{
          fprintf(stdout, "failure (incorrect output results).\n");
          if(interactive && !IsYes('y', "", "Failure Occurred! Do you wish to continue?"))
            abort();
        }
      }
    }
    free(char_matrix_in); free(char_matrix_out);
  }

  fprintf(stdout, "\nWe will now test the \"netslnb()\" call with IBP objects\n");

  /*======== TESTING DOUBLE MATRIX ===============*/
  if( (!interactive) ||
      IsYes('y', "   ", "Execute DSI Double Matrix Test?")){
    fprintf(stdout, "\n   Testing netslnb() with DSI Matrix of DOUBLES ...");

    if(multiple_dsi_files){
      dsi_file = renew_dsi_file(dsi_file, ibp_server_name,
			        DSI_MODE, DSI_PERM, DSI_SIZE, DSI_TYPE);
      if(!dsi_file){
        fprintf(stderr, "Call to open new DSI file failed. Aborting.\n");
        abort();
      }
    }

    double_matrix_in = (double *)malloc(DIMENSION * DIMENSION * sizeof(double));
    double_matrix_out = (double *)malloc(DIMENSION * DIMENSION * sizeof(double));
    for(i=0; i<DIMENSION; i++)
      for(j=0; j<DIMENSION; j++)
        double_matrix_in[i*DIMENSION + j] = 7;

    dsi_object = ns_dsi_write_matrix(dsi_file, double_matrix_in,
                                     DIMENSION, DIMENSION, NETSOLVE_D);
    if(dsi_object == NULL){
      ns_perror("ns_dsi_write_matrix()");
      if(interactive && !IsYes('y', "", "Failure Occurred! Do you wish to continue?"))
        abort();
    }
    else{
      status = netslnb("double_matrix_add5()", DIMENSION, DIMENSION, dsi_object, double_matrix_out);

      if(status < 0){
        netslerr(status);
        if(interactive && !IsYes('y', "", "Failure Occurred! Do you wish to continue?"))
	  abort();
      }
      else{
        int request = status;
        if (request >= 0){
          status = netslpr(request);
          fprintf(stderr,".");
          while(status == NetSolveNotReady){
            sleep(1);
            fprintf(stderr,".");
            status = netslpr(request);
          }
          fprintf(stderr,"\n");
          if (status == NetSolveOK)
            status = netslwt(request);
        }
      }

      if (status != NetSolveOK){
        netslerr(status);
        if(interactive && !IsYes('y', "", "Failure Occurred! Do you wish to continue?"))
	  abort();
      }
      else{
        int success=1;
        for(i=0; i<DIMENSION; i++)
          for(j=0; j<DIMENSION; j++){
            if( double_matrix_out[i*DIMENSION+j] != (double_matrix_in[i*DIMENSION+j] + 5) ){
              success = 0;
              break;
            }
          }
          if(success){
            fprintf(stdout, "passed");
          }
        else{
          fprintf(stdout, "failure (incorrect output results).\n");
          if(interactive && !IsYes('y', "", "Failure Occurred! Do you wish to continue?"))
            abort();
        }
      }
    }
    free(double_matrix_in); free(double_matrix_out);
  }

  /*======== TESTING "FILE TOO LARGE" ===============*/
  if( interactive && IsYes('n', "   ", "Test for maximum allocation of DSI files?")){
    int size=1024*1024;
    int k;

    while(1){
      k = size/1024;
      fprintf(stdout, "Allocating file of size %d Kbytes ... ", k);

      dsi_file = ns_dsi_open(ibp_server_name, DSI_MODE, DSI_PERM, size, DSI_TYPE);
      if(dsi_file == NULL){
        ns_perror("ns_dsi_open()");
        break;
      }
      else{
	fprintf(stdout, "success\n");
        size *= 2;
      }
    }
  }

  /*======== TESTING "OBJECT TOO LARGE" ===============*/
  if( interactive && IsYes('n', "   ", "Test for allocation of too large object?")){

    dsi_file = ns_dsi_open(ibp_server_name, DSI_MODE, DSI_PERM,
                           DIMENSION, DSI_TYPE);
    if(dsi_file == NULL){
      ns_perror("ns_dsi_open()");
      abort();
    }

    int_vector_in = (int *)malloc(DSI_SIZE * sizeof(int));

    dsi_object = ns_dsi_write_vector(dsi_file, int_vector_in,
				     DSI_SIZE, NETSOLVE_I);
    if(dsi_object == NULL){
      ns_perror("ns_dsi_write_vector()");
      fprintf(stderr, "ns_dsi_write_vector() failed like we expected -- object too big.\n");
    }
    else{
      fprintf(stderr, "ns_dsi_write_vector() should have failed -- object too big.\n");
      int_vector_out = (int *)malloc(DSI_SIZE * sizeof(int));

      status = ns_dsi_read_vector(dsi_object, int_vector_out, DIMENSION, NETSOLVE_I);
      if(status != 0){
        ns_perror("ns_dsi_read_vector()");
      }
      else{
        int success = 1;
	for(i=0; i<DIMENSION; i++){
          if( int_vector_in[i] == int_vector_out[i] ){
            success = 0;
            break;
          }
	}
      }
    }
  }

  return 0;
}
