/*****************************************************************/
/*      server_process_message.c                                 */
/*      Henri Casanova                                           */
/*---------------------------------------------------------------*/
/*  processMessage()                                             */
/*****************************************************************/

#include "core.h"
#include "serverglobal.h"
#include "serverprocessmessage.h"
#include "workloadmanager.h"
#include "pong.h"
#include "generateservice.h"
#include "abort.h"
#include <sys/types.h>
#include <signal.h>
#include <unistd.h>
#include <time.h>

/*
 *  processMessage()
 *
 *  A message has been received. Let's see what it's about
 */
void processMessage(int sock)
{
  NS_Communicator *comm;
  int tag;

  comm = acceptTransaction(sock);
  if (comm == NULL)
  {
    netsolvePerror("");
    return;  
  }
  
  if (recvInt(comm,&tag) == -1)
  {
    netsolvePerror("");
    endTransaction(comm);
    return;
  }

  switch(tag)
  {
    case NS_PROT_ARE_YOU_THERE:
      processAreYouThere(comm);
      endTransaction(comm);
      break;
    case NS_PROT_NEW_AGENT: /* No fork */
      processNewAgent(comm);  
      endTransaction(comm);
      break;
    case NS_PROT_PONG_REQUEST: /* fork */
      if (fork())
      {
        endTransaction(comm);
        return;
      }
      else
      {
        close(global.sock);
        processPongRequest(comm);
        endTransaction(comm);
        exit(0);
      }
      break;
    case NS_PROT_PROBLEM_SOLVE:
      processProblemSolve(comm);
      endTransaction(comm);
      break;
    case NS_PROT_PROBLEM_SOLVE_SEQUENCE:
      processProblemSolve_sequence(comm);
      endTransaction(comm);
      break;
    case NS_PROT_REGISTER_AGAIN:
      processRegisterAgain(comm);
      endTransaction(comm);
      break;
    case NS_PROT_KILL_SERVER:
      processKillServer(comm);
      endTransaction(comm);
      break;
    case NS_PROT_SERVICE_FINISHED:
      processServiceFinished(comm);
      endTransaction(comm);
      break;
    case NS_PROT_KILL_JOB:
      processKillJob(comm);
      endTransaction(comm);
      break;
    default:
      if (sendInt(comm,NS_PROT_PROTOCOL_ERROR) == -1)
        netsolvePerror("");
      endTransaction(comm);
      return;
  }
}

/*
 * processKillJob()
 */
void processKillJob(NS_Communicator *comm)
{
  int pid;
  int status;

  if (recvInt(comm,&pid) == -1)
  {
    netsolvePerror("");
    return;
  }
  if (kill(pid,SIGKILL) != 0)
  {
#ifdef VIEW
    perror("kill");
#endif
    status = -1;
  }
  else 
    status = 1;

  if (status == 1)
  { 
    if (sendInt(comm,NetSolveOK) == -1)
    {
      netsolvePerror("");
      return;
    }
  }
  else
  {
    if (sendInt(comm,NetSolveServerError) == -1)
    {
      netsolvePerror("");
      return;
    }
  }
  return;
}

/*
 * processAreYouThere()  
 */
void processAreYouThere(NS_Communicator *comm)  
{
  if (sendInt(comm,NS_PROT_I_AM_THERE) == -1)
  {
    netsolvePerror("");
    return;
  }
  return;
}

/*  
 * processKillServer()
 */   
void processKillServer(NS_Communicator *comm)
{   
  char *username = getUserName();
  char *s;
    
  if (recvString(comm,&s) == -1)
  {
    netsolvePerror("");
    return;
  }

  if (username == NULL)
  {   
#ifdef VIEW
    fprintf(stderr,"Killable by anyone.\n");
#endif
    username = s;
  }   
      
  if (strcmp(s,username))
  {
    if (sendInt(comm,NS_PROT_NOT_ALLOWED) == -1)
      netsolvePerror("");
    free(username);
    return;
  }
  else
  {  
    if (sendInt(comm,NS_PROT_KILLED) == -1)
      netsolvePerror("");
#ifdef VIEW
    fprintf(stderr,"Murdered\n");
#endif
    free(username);
    endTransaction(comm);
    close(global.sock);
    Abort();
    exit(0);
  } 
}   

/*
 * processRegisterAgain()
 */
void processRegisterAgain(NS_Communicator *comm)
{
  NS_IPaddr_type IPaddr;
  NS_AgentDesc **alist;

  if (recvIPaddr(comm,&IPaddr) == -1)
  {
    netsolvePerror("");
    return;
  }
  alist = lookupAgent(&IPaddr);
  if (*alist == NULL)
  {
    free(alist);
    return;
  }
  registerToNetSolve(*alist,0);
  free(alist);
  return;

}


/*
 * processNewAgent()
 */
void processNewAgent(NS_Communicator *comm)
{
  NS_AgentDesc *ad;

  ad = recvAgentDesc(comm);
  if (ad == NULL)
  {
    netsolvePerror("");
    return;
  }

  addAgent(ad); 

  /* Restart the workload manager */
  (void)kill(global.workload_manager_pid,SIGTERM);
  global.workload_manager_pid = startWorkloadManager();

  return;
}

/*
 * processPongRequest()
 *
 * This routine asks for the server to perform a pong measurement
 * between himself and some other host
 */
void processPongRequest(NS_Communicator *comm)
{
  NS_HostDesc *hd;
  int port;
  int latency,bandwidth;
  time_t date;
  int idate;

  hd = recvHostDesc(comm);
  if (hd == NULL)
  {
    netsolvePerror("");
    return;
  }
  if (recvInt(comm,&port) == -1)
  {
    netsolvePerror("");
    free(hd);
    return; 
  }

  time(&date);
  doPong(hd->hostname,hd->IPaddr,port,&latency,&bandwidth);

  if (sendInt(comm,latency) == -1)
  {
    netsolvePerror("");
    free(hd);
    return;
  }
  if (sendInt(comm,bandwidth) == -1)
  {
    netsolvePerror("");
    free(hd);
    return;
  }
  idate = (int)date;
  if (sendInt(comm,idate) == -1)
  {
    netsolvePerror("");
    free(hd);
    return;
  }
  free(hd);
  return;
}

/*
 * processPongBandwidth()
 */
void processPongBandwidth(NS_Communicator *comm)
{
  char *data;
  int size;

  if (recvInt(comm,&size) == -1)
  {
    netsolvePerror("");
    return;
  }
  data = (char *)calloc(size,sizeof(char));
  if (recvArray(comm,NETSOLVE_CHAR,data,size) == -1)
  {
    netsolvePerror("");
    return;
  }
  if (sendArray(comm,NETSOLVE_CHAR,data,size) == -1)
  {
    netsolvePerror("");
    return;
  }
  return;
}

/*
 * processProblemSolve()
 */
void processProblemSolve(NS_Communicator *comm)
{
  NS_ProblemDesc *pd;
  NS_ProblemDesc **plist;
  int tag,restriction_index;
  NS_IPaddr_type proxy_IPaddr;
  int proxy_port;
  char *agent;

  tag = NS_PROT_OK;
  if (sendInt (comm, tag) == -1)
  {
    netsolvePerror ("");
    return;
  }

  pd = recvProblemDesc(comm);
  if (pd == NULL)
  {
    netsolvePerror("");
    return;
  }
  if ((recvString(comm,&agent) == -1) ||
      (recvIPaddr(comm,&proxy_IPaddr) == -1) ||
      (recvInt(comm,&proxy_port) == -1))
  {
    netsolvePerror("");
    return;
  }
  
  plist = lookupProblem(pd->nickname);
  if (*plist == NULL)
  {
    if (sendInt(comm,NS_PROT_PROBLEM_NOT_FOUND) == -1)
    {
      netsolvePerror("");
      freeProblemDesc(pd);
      return;
    }
    free(plist);
    return;
  }
  if (!compareProblemDesc(pd,*plist))
  {
    if (sendInt(comm,NS_PROT_BAD_SPECIFICATION) == -1)
    {
      netsolvePerror("");
      freeProblemDesc(pd);
      return;
    }
    free(plist);
    freeProblemDesc(pd);
    return;
  }
  freeProblemDesc(pd);
  pd = *plist;
  free(plist);

  /* Workload Threshhold */
  if (global.my_self->workload_threshhold != -1)
  {
    if (global.my_self->workload_threshhold < getMyWorkload())
    {
#ifdef VIEW
      fprintf(stderr,"Job refused: Workload beyond threshhold\n");
#endif
      if (sendInt(comm,NS_PROT_NOT_ALLOWED) == -1)
      {
        netsolvePerror("");
        return;
      }
      return;
    }
  }

  /* Restrictions */
  restriction_index = checkRestriction(comm,global.my_self->restrictions,
                  global.my_self->nb_restrictions);
  if (restriction_index == -1)
  {
    if (ns_errno == NetSolveNotAllowed)
    {
#ifdef VIEW
      fprintf(stderr,"Job refused: Not an allowed client at this time\n");
#endif
      if (sendInt(comm,NS_PROT_NOT_ALLOWED) == -1)
      {
        netsolvePerror("");
        return;
      }
      return;
    }
    if (ns_errno == NetSolveSystemError)
    {
        /* nothing */
    }
  }

  /* At this point, either we had a system Error or
     we found the restirction slot */

  if (sendInt(comm,NS_PROT_ACCEPTED) == -1)
  {
    netsolvePerror("");
    return;
  }

  generateServiceProcess(comm,restriction_index,pd,agent,
                         proxy_IPaddr,proxy_port);
  return;
}

/*
 * processServiceFinished()
 */
void processServiceFinished(NS_Communicator *comm)
{
  int restriction_index;

  if (recvInt(comm,&restriction_index) == -1)
  {
    netsolvePerror("");
    return;
  }
  if (sendInt(comm,666) == -1)
  {
    netsolvePerror("");
    return;
  }

  if (restriction_index == -1)
    return;

  ((global.my_self->restrictions)[restriction_index]->pending)--;
  if ((global.my_self->restrictions)[restriction_index]->pending <= 0)
    (global.my_self->restrictions)[restriction_index]->pending = 0;
  return;
}
