/*****************************************************************/
/*      netsolveupf.c                                            */
/*      Henri Casanova                                           */
/*****************************************************************/

#include "core.h"
#include "netsolveupf.h"
#include "serverglobal.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>

static char *unsafe_calls[] = {
"write", "setrlimit", "getdents", "umount", "munlock", "fstat",
"getpgid", "sigaction", "setfsgid", "chroot", "times", "sigsuspend",
"setpgid", "getpgrp", "break", "putpmsg", "query_module", "pause",
"writev", "rename", "truncate", "profil", "waitpid", "sigreturn",
"setresgid", "readdir", "fsync", "sigaltstack", "lstat", "dup2",
"getpmsg", "clone", "lstat", "getppid", "umount", "close",
"setgid", "bdflush", "vm86old", "statfs", "mount", "sgetmask",
"idle", "sigaction", "wait", "fork", "setfsuid", "settimeofday",
"pwrite", "ssetmask", "sigpending", "oldfstat", "syscall",
"sysinfo", "symlink", "ioctl", "ftruncate", "sched_getparam", "creat",
"lchown", "setresuid", "fcntl", "setsid", "mprotect", "setuid",
"gtty", "lstat", "umask", "iopl", "kill", "vfork",
"uname", "stime", "signal", "truncate", "readv", "getcwd",
"getpriority", "msync", "link", "sched_setparam", "getgid", "select",
"getrusage", "lock", "_llseek", "mmap", "get_kernel_syms", "setgroups",
"ulimit", "munmap", "quotactl", "getrlimit", "brk", "sched",
"personality", "getpid", "vhangup", "ioperm", "mremap", "ptrace",
"dup", "getsid", "getegid", "uselib", "sigprocmask", "getuid",
"init_module", "ipc", "capget", "getresgid", "pipe", "read",
"open", "setdomainname", "setregid", "fstat64", "mpx", "alarm",
"pread", "poll", "flock", "sigsuspend", "fdatasync", "prctl",
"prof", "sysfs", "sethostname", "geteuid", "swapon", "capset",
"vm86", "create_module", "execve", "utime", "reboot", "socketcall",
"fchdir", "getresuid", "sendfile", "time", "setreuid", "select",
"ustat", "mkdir", "rmdir", "adjtimex", "acct", "sched_setscheduler",
"mlockall", "fstatfs", "stat", "sigpending", "mmap2", "chdir",
"swapoff", "sigtimedwait", "sigqueueinfo", "syslog", "fchmod", "nfsservctl",
"oldstat", "readlink", "munlockall", "ftruncate64", "stty", "sync",
"setitimer", "fchown", "access", "gettimeofday", "mknod", "sched_get_priority_min",
"oldolduname", "getgroups", "chmod", "mlock", "unlink", "_sysctl",
"sigprocmask", "rt_sigreturn", "lseek", "setpriority", "ftime", "chown",
"stat64", ""};

/*
 * processUPF()
 */
int processUPF(NS_ProblemDesc *pd,char *wd,char **xname,char **xpath)
{
  int i;
  char *filename;
  int language;
  int status;
  char buffer[256];
  int fd;

  /* generating the .o files */
  for (i=0;i<pd->nb_input_objects;i++)
  {
    if (pd->input_objects[i]->object_type != NETSOLVE_UPF)
      continue;
    /*sprintf(buffer,"upfinput%d",i);*/
    pd->input_objects[i]->attributes.upf_attributes.filename = NULL;
    pd->input_objects[i]->attributes.upf_attributes.funcname = NULL;
    pd->input_objects[i]->attributes.upf_attributes.language = -1;
    sprintf(buffer,"./input%d",i);
    fd = open(buffer,O_RDONLY,0666);
    if (fd < 0)
    {
#if (defined(DEBUG) || defined(VIEW))
      ns_printinfo();
      fprintf(stderr,"File '%s' not found\n",buffer); fflush(stderr);
#endif
      ns_errno = NetSolveFileError;
      return -1;
    }

#if defined(DEBUG)
    ns_printinfo();
    fprintf(stderr, "Reading Object %d from File ... \n", i); fflush(stderr);
#endif
    if (readObjectFromFile(fd,pd->input_objects[i]) == -1)
      return -1;
    close(fd);
    filename = pd->input_objects[i]->attributes.upf_attributes.filename;
    language = pd->input_objects[i]->attributes.upf_attributes.language;
    
#if defined(DEBUG)
    ns_printinfo();
    fprintf(stderr, "Generating UPF for Object %d file ... \n", i); fflush(stderr);
#endif
    status = generateUPFobject(filename,language);
    if (status == -1)
      return -1;
  }

  /* Recompile the numerical-x file */
#if defined(DEBUG)
    ns_printinfo();
    fprintf(stderr, "Recompiling the numerical-x file ... \n");
#endif
  /* Recompile the numerical-x file */
  if (reCompileNumerical(pd,wd) == -1)
    return -1;

#if defined(DEBUG)
    ns_printinfo();
    fprintf(stderr, "Relinking the numerical-x file ... \n");
#endif
  /* Perform the link */
   if (reLink(pd,wd,xname,xpath) == -1)
     return -1;

   fflush(stdout);
   fflush(stderr);
   return 0;
}

/*
 * reCompileNumerical
 */
int reCompileNumerical(NS_ProblemDesc *pd,char *wd)
{
  char command[256];
  char makefilename[256];
  char buffer[256];
  char line[80];
  char temp_string[50];
  char filename[100];
  FILE *f;
  FILE *fin;
  struct stat st;
  int i;
  int language;

  f = fopen("./effectiveupf.h","w");
  if (f == NULL)
  {
#if (defined(DEBUG) || defined(VIEW))
    ns_printinfo();
    fprintf(stderr,"Impossible to create file 'effectiveupf.h'\n");
#endif
    ns_errno = NetSolveFileError;
    return -1;
  }
  for (i=0;i<pd->nb_input_objects;i++)
  { 
    if (pd->input_objects[i]->object_type != NETSOLVE_UPF)
      continue;
    if(pd->input_objects[i]->attributes.upf_attributes.language == UPF_LANG_C){
      fprintf(f,"#define upf%d %s\n",i,
        pd->input_objects[i]->attributes.upf_attributes.funcname);
    }
    else{
#if (defined(F2CADD_) || defined(F2CADD__))
    fprintf(f,"#define upf%d %s_\n",i,
        pd->input_objects[i]->attributes.upf_attributes.funcname);
#endif
#ifdef F2CNOCHANGE
    fprintf(f,"#define upf%d %s\n",i,
        pd->input_objects[i]->attributes.upf_attributes.funcname);
#endif
#ifdef F2CUPCASE
    netsolveCaps(pd->input_objects[i]->attributes.upf_attributes.funcname);
    fprintf(f,"#define upf%d %s\n",i,
        pd->input_objects[i]->attributes.upf_attributes.funcname);
#endif
    }
  }
  fclose(f);

  sprintf(command, "cp %s/src/Makefile.def .",
          global.netsolve_root_path);
  system(command);

  f = fopen("Makefile.def", "r+");
  fprintf(f, "                                                          ");

  fseek(f, 0, SEEK_END);
  fprintf(f, "\ninclude %s/conf/Makefile.inc\n",
          global.netsolve_root_path);
  fclose(f);

  /* do the recompilation without DUMMYUPF flag */
  sprintf(filename,
          "%s/src/Makefile.numerical", global.netsolve_root_path);
  sprintf(temp_string, "numerical-%s.o :", pd->file);
  fin = fopen(filename, "r");
  fgets(line, 80, fin);
  while(!strstr(line, temp_string)){
    fgets(line, 80, fin);
  }
  fgets(line, 80, fin);
  fgets(line, 80, fin);

  sprintf(makefilename,"Makefile.numerical-%s",pd->nickname);
  f = fopen(makefilename,"w");
  if (f == NULL)
  {
#if (defined(DEBUG) || defined(VIEW))
    ns_printinfo();
    fprintf(stderr,"Impossible to create file '%s'\n",makefilename);
#endif
    ns_errno = NetSolveFileError;
    return -1;
  }

  fprintf(f,"include Makefile.def\n\n");
  fprintf(f,"numerical-%s.o:\n",pd->file);
  while(!strstr(line, "-o")){
    if(!strstr(line, "DUMMY")){
      fprintf(f,"%s", line);
    }
    else{
      fprintf(f,"\t\t-I. -c \\\n");

    }
    fgets(line, 80, fin);
  }
  fprintf(f,"%s", line);

  fclose(f);
  fclose(fin);

  sprintf(command,"make -f %s",makefilename);
  system(command);

  /* Has it worked ? */
  sprintf(buffer,"./numerical-%s.o",pd->file);
  if (stat(buffer,&st))
  {
#if (defined(DEBUG) || defined(VIEW))
    ns_printinfo();
    fprintf(stderr,"Compilation failed\n");
#endif
    ns_errno = NetSolveUPFError;
    return -1;
  }

  return 1;
}


/*
 * generateUPFobject()
 */
int generateUPFobject(char *filename, int language)
{
  char command[256];
  char buffer[256];
  char *makefilename;
  struct stat st;
  int safe;

  /* rename the file according to the language */
  switch(language)
  {
    case UPF_LANG_FORTRAN:
      sprintf(command,"mv %s %s.f",filename,filename);
      break;
    case UPF_LANG_C:
      sprintf(command,"mv %s %s.c",filename,filename);
      break;
    default:
      ns_errno = NetSolveInternalError;
      return -1;
  } 

  system(command);

  /* Create a small Makefile */
  makefilename = createUPFMakefile(filename,language);

  if (makefilename == NULL)
    return -1;
  sprintf(buffer,"make -f %s",makefilename);
  system(buffer);
  free(makefilename);

  /* Check that the object file has been generated */
  sprintf(buffer,"./%s.o",filename);
  if (stat(buffer,&st))
  {
#if (defined(DEBUG) || defined(VIEW))
    ns_printinfo();
    fprintf(stderr,"Object file '%s' not generated\n",buffer);
#endif
    ns_errno = NetSolveUPFError;
    return -1; 
  }
 
  /* Check the security */
  safe = checkSecurity(buffer);
  if (safe == -1)
    return -1;

  if (safe == 0)
  {
#if (defined(DEBUG) || defined(VIEW))
    ns_printinfo();
    fprintf(stderr,"UPF appears to be unsafe\n");
#endif
    ns_errno = NetSolveUPFUnsafe;
    return -1;
  }
  return 0;
}

/*
 * createUPFMakefile()
 */
char *createUPFMakefile(char *filename,int language)
{
  char buffer[256];
  char *makefilename;
  FILE *f;

  sprintf(buffer,"./Makefile.%s",filename);
  makefilename = strdup(buffer);

  f = fopen(buffer,"w");
  if (f == NULL)
  {
#if (defined(DEBUG) || defined(VIEW))
    ns_printinfo();
    fprintf(stderr,"Impossible to create file '%s'\n",makefilename);
#endif
    free(makefilename);
    ns_errno = NetSolveFileError;
    return NULL;
  }

  fprintf(f,"default: %s.o\n\n",filename);
  fprintf(f,"%s.o : %s",filename,filename);
  switch(language)
  {
    case UPF_LANG_FORTRAN:
      fprintf(f,".f\n");
      fprintf(f,"\t$(FC) -c %s.f -o %s.o\n",filename,filename);
      break;
    case UPF_LANG_C:
      fprintf(f,".c\n");
      fprintf(f,"\t$(CC) -c %s.c -o %s.o\n",filename,filename);
      break;
    default:
      ns_errno = NetSolveInternalError;
      return NULL;
  }
  fclose(f);
  return makefilename;
}

/*
 * checkSecurity()
 */
int checkSecurity(char *object)
{
  FILE *stream;
  char command[256];
  char line[256];

  sprintf(command,"nm -u %s",object);
  stream = popen(command,"r");
  if (stream == NULL)
  {
    perror("popen");
    ns_errno = NetSolveSystemError;
    return -1;
  }

  while(fgets(line,256,stream) != NULL)
  {
    char *begin = line;
    char *end;

#if (defined(solaris))
    if (strncmp(line,"Undefined symbols from",22) == 0)
      continue;
#endif
#ifdef PLAT_T3E
    if (strncmp(line,"./",2) == 0)
      continue;
#endif
#ifdef irix
    if (strncmp(line,"Undefined",9) == 0)
      continue;
#endif
#ifdef osf
    if (strncmp(line,"Name        ",12) == 0)
      continue;
#endif
    if (strcmp(line,"\n") == 0)
      continue;
 
    while((*begin == '_') || (*begin == ' '))
      begin++;

    end = (char *)((char *)line + strlen(line))-1;
    while ((*end == '\n')||(*end == ' ')||(*end == '_'))
      *(end--)='\0';

#ifdef irix
    begin = end;
    while ((*(begin-1) != '|'))
      begin--;
#endif

    netsolveCaps(begin);
    if (!isAuthorized(begin)){
      return 0;
    }
  }
  pclose(stream);
  return 1;
}

/*
 * isAuthorized()
 */
int isAuthorized(char *s)
{
  int i=0;
  int authorized=1;

  while (unsafe_calls[i][0] != '\0')
  {
    if (!strcmp(s,unsafe_calls[i])){
      authorized = 0;
      break;
    }
    i++;
  }

  if(!authorized){
#if (defined(DEBUG) || defined(VIEW))
  ns_printinfo();
  fprintf(stderr,"'%s' is an unauthorized symbol\n",s);
#endif
  }

  return authorized;
}

/*
 *  reLink()
 */
int reLink(NS_ProblemDesc *pd,char*wd,char **xname,char **xpath)
{
  char buffer[256];
  char makefilename[256];
  char command[256];
  char arch[20];
  FILE *f;
  struct stat st;
  int i;

  sprintf(makefilename,"Makefile.service-%s",pd->nickname);
  f = fopen(makefilename,"w");
  if (f == NULL)
  {
#if (defined(DEBUG) || defined(VIEW))
    ns_printinfo();
    fprintf(stderr,"Impossible to create file '%s'\n",makefilename);
#endif
    ns_errno = NetSolveFileError;
    return -1;
  }
  fprintf(f,"include Makefile.def\n");
  fprintf(f,"include %s/src/Makefile.num_libs\n\n",global.netsolve_root_path);
  
  fprintf(f,"%s/service-%s-upfed : \n",wd,pd->file);
  fprintf(f,"\t cd $(OBJDIR);$(LINKER) $(UPF) \\\n");
  for (i=0;i<pd->nb_input_objects;i++)
  {
    if (pd->input_objects[i]->object_type != NETSOLVE_UPF)
      continue;
    fprintf(f,"\t\t%s/input%d-upf.o\\\n",wd,i);
  }

  fprintf(f,"\t\t%s/numerical-%s.o \\\n",wd, pd->file);
  fprintf(f,"\t\t$(F_FROM_C_LIB) \\\n");
  if (!strcmp(pd->customized,"NONE"))
  {
    fprintf(f,"\t\t$(STANDARDSERVICEOBJ) \\\n");
  }
  else if (!strcmp(pd->customized,"ITER_SOLVE") )
  {
    fprintf(f,"\t\t$(ITERSOLVESERVICEOBJ) \\\n");
  }
  else if (!strcmp(pd->customized,"SCALAPACK_MPI"))
  {
    fprintf(f," SCALAPACK_MPI: TO DO\n");
  }
  else if (!strcmp(pd->customized,"CONDOR"))
  {
    fprintf(f," CONDOR: TO DO\n");
  }
  fprintf(f,"\t\t$(LIBS) $(HBMLIB) $(ARCHDLIB) $(AUTH_LIBS) $(IBPLIB) $(LIB%s) "
            "-o $@ \n", pd->file);

  fclose(f);
  
  /* start the make */
  fprintf(stderr,"Check out %s/%s\n",wd,makefilename);
  sprintf(command,"make -f %s",makefilename);
  system(command);
  /* Has the executable been created */
  sprintf(buffer,"service-%s-upfed",pd->file);
  if (stat(buffer,&st))
  {
    ns_errno = NetSolveUPFError;
    return -1;
  }

  *xname = strdup(buffer);
  *xpath = strdup(wd);
  return 0;
}
