/*****************************************************************/
/*      globusservive.c                                          */
/*      Henri Casanova                                           */
/*---------------------------------------------------------------*/
/*****************************************************************/

#include "core.h"
#include "serverglobal.h"
#include "standardservice.h"
#include "serviceutil.h"
#include <fcntl.h>
#include <unistd.h>
#include "hbm_wrappers.h"
#include <time.h>
#include <signal.h>

#ifdef sunos
int __main() { }
int MAIN_() { }
#endif

/*
 * globusService()
 *
 * argv[0] <proxy IP> <proxy port>
 */
int main(int argc, char **argv)
{
  time_t date1,date2;
  int elapsed;
  NS_ProblemDesc *pd;
  int solved;
  char *wd=".";
  int tag;
  int fd;
  int err;
  int filemode;
  int client_major;
  int stdoutfd;
  int i;
  char buffer[256];
  char *agent_name;
  int mypid = getpid();
  char stdoutfile[256];
  NS_IPaddr_type proxy_IPaddr;
  int proxy_port;
  NS_Socket_type listening_socket;
  NS_Socket_type sock;
  int ID;
  int lifelink_pid;
  int my_port;
  NS_IPaddr_type my_IPaddr;
  NS_Communicator *comm;

  /* Parse the comand line */
  if (argc != 3)
  {
#ifdef VIEW
    fprintf(stderr,"standardService: wrong number of command line args.\n");
#endif
    exit(0);
  }

  proxy_IPaddr = getIPaddr(argv[1]);
  proxy_port = atoi(argv[2]);

  /* Setting a port on which I'll listen for input data */
  listening_socket = bindToFirstAvailablePort(3000,&my_port);
  if (listening_socket == -1)
  {
    netsolvePerror("bindToFirstAvailablePort()");
    exit(0);
  }
  listen(listening_socket,MAX_CONNECTIONS);

  /* Connect back to the proxy */
  comm = contactHost(NULL,proxy_IPaddr,proxy_port,DATA_XDR);
  if (comm == NULL)
  {
    netsolvePerror("contactHost()");
    exit(0);
  }

  /* Send my IP-addr and port */
  my_IPaddr = getMyIPaddr();
  if ((sendIPaddr(comm,&my_IPaddr) == -1)||
      (sendInt(comm,my_port) == -1))
  {
    netsolvePerror("");
    exit(0);
  }

  /* Get the Problem Descriptor from the proxy */
  pd = recvProblemDesc(comm);
  if (pd == NULL)
  {
    netsolvePerror("recvProblemDesc()");
    exit(0);
  }

  /* Get the REAL port for the proxy !! */
  if (recvInt(comm,&proxy_port) == -1)
  {
    netsolvePerror("recvInt()");
    exit(0);
  }

  /* Cut the connection with the proxy */
  endTransaction(comm);
  
  /* Accepting connection from the client */
  if ((sock = acceptConnection(listening_socket)) == -1)
  {
    netsolvePerror("acceptConnection()");
    notifyProxy(proxy_IPaddr,proxy_port,ns_errno,ID);
    exit(0);
  }

  comm = acceptTransaction(sock);
  if (comm == NULL)
  {
    netsolvePerror("acceptTransaction()");
    notifyProxy(proxy_IPaddr,proxy_port,ns_errno,ID);
    exit(0);
  }

  /* Receive the ID and the client_major */
  if ((recvInt(comm,&ID) == -1) ||
      (recvInt(comm,&client_major) == -1))
  {
    netsolvePerror("recvInt()");
    notifyProxy(proxy_IPaddr,proxy_port,ns_errno,ID);
    exit(0);
  }

  /* Receiving the input objects */
  if (recvInputObjects(comm,pd) == -1)
  {
    netsolvePerror("recvInputObjects()");
    notifyProxy(proxy_IPaddr,proxy_port,ns_errno,ID);
    exit(0);
  }

  /* Cutting the transaction to the client */
  endTransaction(comm);

  /* Transpose */
  for (i=0;i<pd->nb_input_objects;i++)
  {
    if (pd->input_objects[i]->object_type == NETSOLVE_MATRIX){
      if (pd->input_objects[i]->attributes.matrix_attributes.major != pd->major)
        transposeMatrix(pd->input_objects[i]);
    }
    if (pd->input_objects[i]->object_type == NETSOLVE_SPARSEMATRIX){
      if (pd->input_objects[i]->attributes.sparsematrix_attributes.major != pd->major)
        transposeSparseMatrix(pd->input_objects[i]);
    }
    else
      continue;
  }
 
  /* Initializes the output objects */
  if (initOutputObjects(pd) == -1)
  {
    netsolvePerror("initOutputObjects()");
    notifyProxy(proxy_IPaddr,proxy_port,ns_errno,ID);
    exit(0);
  }

  /* Redirecting the stdout */
  sprintf(stdoutfile,"%s/netsolve-stdout",wd);
  stdoutfd = redirectStdout(stdoutfile);
  if (stdoutfd == -1)
  {
    notifyProxy(proxy_IPaddr,proxy_port,ns_errno,ID);
    netsolvePerror("redirectStdout()");
    exit(0);
  }
  
  /* Solve the problem */
  time(&date1);
  solved = solve(pd,pd->input_objects,pd->output_objects);
  time(&date2);
  elapsed = (int)(date2-date1);

  notifyProxy(proxy_IPaddr,proxy_port,NetSolveOK,ID);

  /* Listening on the socket */
  if ((sock = acceptConnection(listening_socket)) == -1)
  {
    netsolvePerror("acceptConnection()");
    exit(0);
  }

  comm = acceptTransaction(sock);
  if (comm == NULL)
  {
    netsolvePerror("acceptTransaction()");
    exit(0);
  }

  /* Sending back the Computational time elapsed */
  if (sendInt(comm,elapsed) == -1)
  {
    endTransaction(comm);
    netsolvePerror("sendInt()");
    exit(0);
  }

  /* Sending back the stdout */
  fflush(stdout);
  close(stdoutfd);
  if (sendFileAsString(comm,stdoutfile) == -1)
  {
    endTransaction(comm);
    netsolvePerror("sendFileAsString()");
    exit(0);
  }

  sendInt(comm,solved);

  /* Error */
  if (solved != NS_PROT_SOLVED)
  {
    endTransaction(comm);
    exit(0);
  }

  /* Send back the output */
  /* this takes care of the transpose */
  if (sendOutputObjects(comm,pd,client_major) == -1)
  {
    netsolvePerror("sendOutputObjects()");
    endTransaction(comm);
    exit(0);
  }

  /* Terminating the transaction */
  endTransaction(comm);

#ifdef VIEW
  fprintf(stderr,"Completed\n");
#endif

  /* Cleaning up */
  close(listening_socket);
  exit(0);
}

/*
 * sendOutputObjects()
 */
int sendOutputObjects(NS_Communicator *comm, NS_ProblemDesc *pd, int client_major)
{
  NS_Object **list = pd->output_objects;
  int nb = pd->nb_output_objects;
  int i;

  for (i=0;i<nb;i++)
  {
    switch(list[i]->object_type)
    {
      case NETSOLVE_MATRIX:  /* transpose ? */
        if (pd->major != client_major)
        {
          if (transposeMatrix(list[i]) == -1)
          {
#ifdef VIEW
            fprintf(stderr,"Error while transposing matrix\n");
#endif
            return -1;
          }
        }
        break;
      case NETSOLVE_SPARSEMATRIX:  /* transpose ? */
        if (pd->major != client_major)
        {
          if (transposeSparseMatrix(list[i]) == -1)
          {
#ifdef VIEW
            fprintf(stderr,"Error while transposing sparse matrix\n");
#endif
            return -1;
          }
        }
        break;
      default:  /* Nothing */
        break;
    }  /* end switch */

    if (sendObject(comm,list[i]) == -1)
    {
      netsolvePerror("sendObject()");
      return -1;
    }
  } /* end for */ 
  return 1;
}

/*
 * initOutputObjects()
 */
int initOutputObjects(NS_ProblemDesc *pd)
{
  NS_Object **list = pd->output_objects;
  int nb = pd->nb_output_objects;
  char buffer[256];
  int i;
    
  for (i=0;i<nb;i++)
  {
    /* do special initializations */
    switch(list[i]->object_type)
    {
      case NETSOLVE_FILE: /* Create a file name */
        sprintf(buffer,"fileoutput%d",i);
        list[i]->attributes.file_attributes.filename = strdup(buffer);
        break;
      case NETSOLVE_MATRIX: /* Set the major */
        list[i]->attributes.matrix_attributes.major = pd->major;
        list[i]->attributes.matrix_attributes.l = -1;
        break;
      case NETSOLVE_SPARSEMATRIX: /* Set the major */
        list[i]->attributes.sparsematrix_attributes.major = pd->major;
        break;
      default: /* Do nothing */
        break;
    }
  }
  return 1;
}

/*
 * recvInputObjects()
 */
int recvInputObjects(NS_Communicator *comm, NS_ProblemDesc *pd)
{
  NS_Object **list = pd->input_objects;
  int nb = pd->nb_input_objects;
  char buffer[256];
  int i;

  for (i=0;i<nb;i++)
  {
    /* do initializations */
    switch(list[i]->object_type)
    {
      case NETSOLVE_MATRIX: 
        list[i]->attributes.matrix_attributes.m = -1;
        list[i]->attributes.matrix_attributes.n = -1;
        list[i]->attributes.matrix_attributes.l = -1;
        list[i]->attributes.matrix_attributes.major = -1;
        list[i]->attributes.matrix_attributes.ptr = NULL;
        list[i]->attributes.matrix_attributes.d = NULL;
        break;
      case NETSOLVE_SPARSEMATRIX:
        list[i]->attributes.sparsematrix_attributes.m = -1;
        list[i]->attributes.sparsematrix_attributes.n = -1;
        list[i]->attributes.sparsematrix_attributes.major = -1;
        list[i]->attributes.sparsematrix_attributes.f = -1;
        list[i]->attributes.sparsematrix_attributes.rc_ptr = NULL;
        list[i]->attributes.sparsematrix_attributes.rc_index = NULL;
        list[i]->attributes.sparsematrix_attributes.ptr = NULL;
        list[i]->attributes.sparsematrix_attributes.d = NULL;
        break;
      case NETSOLVE_VECTOR:
        list[i]->attributes.vector_attributes.m = -1;
        list[i]->attributes.vector_attributes.ptr = NULL;
        break;
      case NETSOLVE_SCALAR:
        list[i]->attributes.scalar_attributes.ptr = NULL;
        break;
      case NETSOLVE_STRING:
        list[i]->attributes.string_attributes.ptr = NULL;
        break;
      case NETSOLVE_STRINGLIST:
        list[i]->attributes.stringlist_attributes.strings = NULL;
        list[i]->attributes.stringlist_attributes.m = -1;
        break;
      case NETSOLVE_FILE: /* Create a file name */
        sprintf(buffer,"fileinput%d",i);
        list[i]->attributes.file_attributes.filename = strdup(buffer);
        break;
      case NETSOLVE_PACKEDFILES: /* Create a file name */
        sprintf(buffer,"packedfileinput%d",i);
        list[i]->attributes.packedfiles_attributes.defaultprefix = strdup(buffer);
        list[i]->attributes.packedfiles_attributes.filenames = NULL;
        list[i]->attributes.packedfiles_attributes.m = -1;
        break;
      case NETSOLVE_UPF: /* Create a file name */
#ifdef VIEW
        fprintf(stderr,"There should not be any UPF here !!!\n");
#endif
        return -1;
        break;
      default: /* Nothing to so */
        break;
    }

    /* receive the object from the net */
    if (recvObject(comm,list[i]) == -1)
    {
      netsolvePerror("recvObject()");
      return -1;
    }

    /* Perform special post-processing !! */
    switch(list[i]->object_type)
    {
      case NETSOLVE_MATRIX: /* transpose */
        if (list[i]->attributes.matrix_attributes.major == pd->major)
          break;
        if (transposeMatrix(list[i]) == -1)
        {
#ifdef VIEW
          fprintf(stderr,"Error while transposing matrix\n");
#endif
          return -1;
        }
        break;
      case NETSOLVE_SPARSEMATRIX: /* transpose */
        if (list[i]->attributes.sparsematrix_attributes.major == pd->major)
          break;
        if (transposeSparseMatrix(list[i]) == -1)
        {
#ifdef VIEW
          fprintf(stderr,"Error while transposing matrix\n");
#endif
          return -1;
        }
        break;
      default: /* Nothing to do */
        break;
    }  /* end switch */

  }  /* end for */
  return 1;
}

