#include <stdlib.h>
#ifdef petsc29
#include "petscmat.h"
#include "petscvec.h"
#else
#include "mat.h"
#include "vec.h"
#endif
#include "error_returns.h"
#include "mec.h"
#include "analysis.h"

int read_utah_mat
(MPI_Comm comm,char *mat_name,char *rhs_name,int band,
 int *r_first,int *r_last,int *r_nnz,int *r_total,
 int **r_ptr,int **r_idx,double **r_val,double **r_rhs)
{
  FILE* file;
  int mytid,ntids,first,last,msize,nsize,nnz,my_nnz,nb,*blocks,ierr;

  MPI_Comm_size(comm,&ntids); MPI_Comm_rank(comm,&mytid);

  /* get matrix elements */
  file = fopen(mat_name,"r");
  fscanf(file,"%d %d %d",&msize,&nsize,&nnz);
  if (msize!=nsize) ERR_REPORTii("Matrix not square",msize,nsize);
  *r_total = msize;

  /* processor 1 reads the whole matrix and figures out block structure */
  if (mytid==0) {
    int *ico,*ptr,*idx,n; double *val;
    ALLOCATE(ico,nnz,int,"big pointers");
    ALLOCATE(idx,nnz,int,"big indices");
    ALLOCATE(val,nnz,double,"big values");
    for (n=0; n<nnz; n++) {
      int i,j; double v,z=0.;
      if (feof(file)) ERR_REPORTi("Premature end of matrix file at",n);
      fscanf(file,"%d %d %lf\n",&i,&j,&v);
      ico[n] = i; idx[n] = j; val[n] = v;
    }
    if (!feof(file)) ERR_REPORT("Elements left at end of matrix file");
    ierr = coo_to_crs(ico,&ptr,idx,val,0,msize,nnz); ERR_RETURN(ierr);
    free(ico); *r_ptr = ptr; *r_idx = idx; *r_val = val;
    ierr = find_block_structure
      (comm,ptr,idx,val,msize,&nb,&blocks); ERR_RETURN(ierr);
    free(ptr); free(idx); free(val);
    rewind(file);
    /* lose first line: */ {int m,n,nnz; fscanf(file,"%d %d %d",&m,&n,&nnz);}
  }

  /* broadcast block structure */
  {
    int first_block,last_block;
    if (ntids>0) {
      MPI_Bcast(&nb,1,MPI_INT,0,comm);
      if (mytid!=0) ALLOCATE(blocks,nb+1,int,"blocks");
      MPI_Bcast(blocks,nb+1,MPI_INT,0,comm);
    }
    first_block = (nb*mytid)/ntids; last_block = (nb*(mytid+1))/ntids;
    first = blocks[first_block]; last = blocks[last_block];
    *r_first = first; *r_last = last;
    printf("[%d] I get blocks %d--%d, elements %d--%d\n",
	   mytid,first_block,last_block,first,last);
    free(blocks);
  }

  /* everyone reads the matrix to count how many nonzeros are local */
  {
    int n;
    my_nnz = 0;
    for (n=0; n<nnz; n++) {
      int i,j; double v,z=0.;
      if (feof(file)) ERR_REPORTi("Premature end of matrix file at",n);
      fscanf(file,"%d %d %lf\n",&i,&j,&v);
      if (i>=first && i<last) my_nnz++;
    }
    *r_nnz = my_nnz;
    if (!feof(file)) ERR_REPORT("Elements left at end of matrix file");
  }
  rewind(file);

  /* now actually construct the arrays */
  {
    int *ico,*ptr,*idx,inz,local_size = last-first,n; double *val,*rhs;
    ALLOCATE(ico,my_nnz,int,"pointers");
    ALLOCATE(idx,my_nnz,int,"indices");
    ALLOCATE(val,my_nnz,double,"values");
    /* lose first line: */ {int m,n,nnz; fscanf(file,"%d %d %d",&m,&n,&nnz);}
    inz = 0;
    for (n=0; n<nnz; n++) {
      int i,j; double v,z=0.;
      if (feof(file)) ERR_REPORTi("Premature end of matrix file at",n);
      fscanf(file,"%d %d %lf\n",&i,&j,&v);
      if (i>=first && i<last) {
	ico[inz] = i; idx[inz] = j; val[inz] = v; inz++;
      }
    }
    if (!feof(file)) ERR_REPORT("Elements left at end of matrix file");
    fclose(file);
    ierr = coo_to_crs
      (ico,&ptr,idx,val,first,last-first,my_nnz); ERR_RETURN(ierr);
    free(ico); *r_ptr = ptr; *r_idx = idx; *r_val = val;
  }

  /* get right hand side */
  {
    double *rhs; int i,nrhs,ielt;
    file = fopen(rhs_name,"r");
    fscanf(file,"%d %d",&nrhs,&i);
    if (nrhs!=1) ERR_REPORT("Can only handle one rhs");
    if (msize!=i) ERR_REPORTii("Rhs not same size as matrix",msize,i);
    ALLOCATE(rhs,last-first,double,"rhs");
    ielt = 0;
    for (i=0; i<msize; i++) {
      double v;
      if (feof(file)) ERR_REPORTi("Premature end of rhs file at",i);
      fscanf(file,"%lf",&v);
      if (i>=first && i<last) {
	rhs[i-first] = v;}
    }
    *r_rhs = rhs;
    if (!feof(file)) ERR_REPORT("Elements left at end of rhs file");
    fclose(file);
  }
  return 0;
}
