#include <stdlib.h>
#include <stdio.h>
#include "error_returns.h"
#include "aztec_matrix_auxs.h"

int aztec_nnzero(int *indices,int local_size)
{
  return indices[local_size]-1;
}

int crs_to_aztec(int nrows,int first,int *ptr,int *idx,double *mv)
{
  int shift,row;
  double *diag;

#ifdef DEBUG
  int ierr,len;
  ierr = crs_nnzeros(ptr,nrows,&len); ERR_RETURN(ierr);
#endif

  diag = (double *) malloc(nrows*sizeof(double));

  shift = 1;
  for (row=nrows-1; row>=0; row--) {
    int icol;
    for (icol=ptr[row+1]-1; icol>=ptr[row]; icol--) {
      int col=idx[icol];
      if (col==first+row) {
	diag[row] = mv[icol]; shift++;
      } else {
#ifdef DEBUG
	if (icol+shift>len)
	  ERR_REPORTiii("index too large in row (is/max/row)\n",
			icol+shift,len,row);
#endif
	mv[icol+shift] = mv[icol]; idx[icol+shift] = idx[icol];
      }
    }
  }

  idx[0] = nrows+1;
  for (row=0; row<nrows; row++) {
    mv[row] = diag[row];
    idx[row+1] = idx[row]+(ptr[row+1]-ptr[row])-1;
  }
  mv[nrows]=-47.11;

  free(diag);

  return 0;
}

int crs_to_aztec_i(int nrows,int* idx,int *ptr,int *ind,double *mv)
{
  int shift,row;
  double *diag;

#ifdef DEBUG
  int ierr,len;
  ierr = crs_nnzeros(ptr,nrows,&len); ERR_RETURN(ierr);
#endif

  diag = (double *) malloc(nrows*sizeof(double));

  shift = 1;
  for (row=nrows-1; row>=0; row--) {
    int icol;
    for (icol=ptr[row+1]-1; icol>=ptr[row]; icol--) {
      int col=idx[icol];
      if (col==ind[row]) {
	diag[row] = mv[icol]; shift++;
      } else {
#ifdef DEBUG
	if (icol+shift>len)
	  ERR_REPORTiii("index too large in row (is/max/row)\n",
			icol+shift,len,row);
#endif
	mv[icol+shift] = mv[icol]; idx[icol+shift] = idx[icol];
      }
    }
  }

  idx[0] = nrows+1;
  for (row=0; row<nrows; row++) {
    mv[row] = diag[row];
    idx[row+1] = idx[row]+(ptr[row+1]-ptr[row])-1;
  }
  mv[nrows]=-47.11;

  free(diag);

  return 0;
}

int aztec_to_crs(int nrows,int first,int *ptr,int *idx,double *mv)
{
  double *diag;
  int src,tar,row;

  if (nrows+1!=idx[0]) ERR_REPORTii("#rows mismatch, par=%d, idx[0]=%d",
				    nrows,idx[0]);

  /* we can create the pointers without moving anything around */
  ptr[0] = 0;
  for (row=0; row<nrows; row++)
    ptr[row+1] = ptr[row] + (idx[row+1]-idx[row])
      + 1 /* <= one loc for diag */;

  /* store the diagonal temporarily, so that we have lebensraum
     in the idx and mv array */
  diag = malloc((nrows+1)*sizeof(double));
  for (row=0; row<nrows; row++) diag[row] = mv[row];

  /* now shuffle the whole caboodle */
  src = nrows+1;
  tar = 0;
  for (row=0;row<nrows;row++) {
    /* elements before the diagonal */
    for ( ; idx[src]<row+first 
	    && tar<ptr[row+1]-1 /* <= leave one loc for the diag */ ; ) {
      idx[tar] = idx[src]; mv[tar++] = mv[src++];}
    /* diagonal itself */
    idx[tar] = row; mv[tar++] = diag[row];
    /* elements after the diagonal */
    for ( ; tar<ptr[row+1] ; ) {
      idx[tar] = idx[src]; mv[tar++] = mv[src++];}
  }
  free(diag);
  return 0;
}

int print_aztec_matrix(double *mv,int *idx,int first,int mysize)
{
  int i,j,nnzero;

  if (idx[0]!=mysize+1)
    ERR_REPORTii("Inconsistent mysize/idx[0]\n",mysize,idx[0]);

  printf("AzTec DMRS matrix:\n");
  nnzero = idx[idx[0]-1];
  printf("#nonzeros: %d\n",nnzero);
  printf("index array:");
  for (i=0; i<nnzero; i++) printf(" %d->%d",i,idx[i]); printf("\n");
  for (i=0; i<mysize; i++) {
    printf("row %d+%d=%d; diagonal=%7.3e; offd:",first,i,first+i,mv[i]);
    for (j=idx[i]; j<idx[i+1]; j++)
      printf(" %d=>%7.3e",idx[j],mv[j]);
    printf("\n");
  }
  return 0;
}

