#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include "mpi.h"
#include "sparse_globals.h"
#include "error_returns.h"
#include "az_aztec.h"
#include "./netsolve_aztec.h"
#include "./aztec_impl.h"
#include "aztec_auxs.h"
#include "aztec_matrix_auxs.h"
#include "matrix_auxs.h"
#include "parallel_auxs.h"
#include "driver/iterative.h"

int aztec_iterative_solve
(MPI_Comm comm,
 double *mat_el,double *x_el,
 int first,int last,int nnzero,int *idx,
 double *return_vec_el,
 double rtol,int maxit, int *conv,int *its
 );

int aztec_iterative_driver
(MPI_Comm comm, double *values,int *indices,int *pointers,
 int first,int local_size,
 double *rhs_vector,double *sol_vector,
 iterative_info_block info)
{
  aztec_info_block aztec_info = (aztec_info_block) info;
  double *aztec_values;
  int *aztec_indices, conv,its,nnz,ierr;

  ierr = crs_nnzeros(pointers,local_size,0,&nnz); ERR_RETURN(ierr);

  /* allocate aztec arrays that have that silly extra location */
  ALLOCATE(aztec_values,nnz+1,double,"aztec values");
  memcpy(aztec_values,values,nnz*sizeof(double));
  ALLOCATE(aztec_indices,nnz+1,int,"aztec indices");
  memcpy(aztec_indices,indices,nnz*sizeof(int));
  
  ierr = crs_to_aztec
    (local_size,first,pointers,aztec_indices,aztec_values); ERR_RETURN(ierr);
  
#ifdef DEBUG
  ierr = print_aztec_matrix
    (aztec_values,aztec_indices,first,local_size); ERR_RETURN(ierr);
#endif
  
  ierr = aztec_iterative_solve
    (comm,aztec_values,rhs_vector, first,first+local_size,nnz,
     aztec_indices,sol_vector,
     aztec_info->rtol,aztec_info->maxit,&conv,&its); ERR_RETURN(ierr);
  ierr = iterative_set_return_params(info,conv,its); ERR_RETURN(ierr);

  free(aztec_values);

  return 0;
}

int aztec_iterative_solve
(MPI_Comm comm, double *mat_el,double *x_el,
 int first,int last, int nnzero,int *idx,
 double *return_vec_el, 
 double rtol,int maxit, int *conv,int *its
 )
{
  int proc_config[AZ_PROC_SIZE];
  int *update_index,*extern_index,*data_org,*cpntr;
  int options[AZ_OPTIONS_SIZE];
  int *index=NULL,*bpntr=NULL,*rpntr=NULL;
  int *external,*update,n_update;
#ifdef DEBUG
  int n_external;
#endif
  double params[AZ_PARAMS_SIZE];
  double status[AZ_STATUS_SIZE];
  int mytid,i;

  MPI_Comm_rank(comm,&mytid);
  AZ_set_proc_config(proc_config,comm);
  n_update = last-first;
  ALLOCATE(update,n_update,int,"aztec update array");
  for (i=0; i<n_update; i++) update[i] = first+i;
#ifdef DEBUG
  AZ_find_local_indices(n_update,idx,update,&external,&n_external,
			AZ_MSR_MATRIX,bpntr);
  AZ_check_msr(idx,n_update,n_external,AZ_GLOBAL,proc_config);
#endif
  AZ_transform(proc_config,&external,
	       idx,mat_el,update,
	       &update_index,&extern_index,&data_org,
	       n_update,
	       index,bpntr,rpntr,
	       &cpntr,AZ_MSR_MATRIX);
#ifdef DEBUG
  AZ_check_msr(idx,n_update,n_external,AZ_LOCAL,proc_config);
#endif
  AZ_defaults(options,params);
  options[AZ_conv] = AZ_r0;
  params[AZ_tol] = rtol;
  options[AZ_solver] = AZ_bicgstab;
  options[AZ_precond] = AZ_Jacobi;
  options[AZ_max_iter] = maxit;
  {
    double *invec,*outvec;
    int insize,outsize,i;
    aztec_get_vecsizes(data_org,&insize,&outsize);
    if (n_update!=insize)
      ERR_REPORTii("Something fishy: nupdate/invector size=\n",
		   n_update,insize);
    invec = (double *) malloc(insize*sizeof(double));
    outvec = (double *) malloc(outsize*sizeof(double));
    for (i=0; i<last-first; i++) invec[i]=x_el[i];
    AZ_reorder_vec(invec,data_org,update_index,rpntr);
    AZ_solve(outvec,invec,
	     options,params,
	     index,idx,rpntr,cpntr,bpntr,mat_el,data_org,
	     status,proc_config);
    *its = status[AZ_its];
    *conv = (status[AZ_why]==AZ_normal);
    
#ifdef DEBUG
    if (mytid==0) printf("itmeth out vector unordered:");
    for (i=0; i<outsize; i++) printf("%7.3e ",outvec[i]); printf("\n");
#endif
    AZ_invorder_vec(outvec,data_org,update_index,rpntr,return_vec_el);
#ifdef DEBUG
    if (mytid==0) printf("itmeth out vector inv ordered: ");
    for (i=0; i<last-first; i++) printf("[%d] %e, ",first+i,return_vec_el[i]);
    printf("\n");
#endif
    free(invec); free(outvec);
  }
  
  return 0;
}
  
