/*
 *   $Id: nsmath_gateway.c,v 1.3 2001/07/23 19:56:23 shi Exp $
 * 
 */

#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <stdlib.h>
#include <assert.h>

#include "core.h"
#include "netsolveerror.h"
#include "client.h"
#include "cfortranclient.h"

#include "mathlink.h"
#include "nsmath.h"
#include "ns_utils.h"
#include "nsmath_gateway.h"

extern FILE *nsmath_log;


#define TRUE 1
#define FALSE 0


#if MLVERSION < 3
#define ML_SYMBOL ml_charp
#else 
#define ML_SYMBOL kcharp_ct
#endif
 
char *nsmath_symbols[] = {"", "List", "Complex", NULL};

int flag=0;
#define NSMATH_LIST 1
#define NSMATH_COMPLEX 2


#if MLVERSION < 3
int get_string_from_ml(char **s)
{
   int rc;
   ML_SYMBOL symb;
   
   rc = MLGetString(stdlink, &symb);
   *s = strdup(symb);
   MLDisownString(stdlink, symb);
   
   return rc;
}

int put_string_to_ml(char *s)
{
   return MLPutString(stdlink, s);
}

#else
int get_string_from_ml(char **s)
{
   int rc;
   const unsigned char *symb;
   long len;
   
   rc = MLGetByteString(stdlink, &symb, &len, 1);
   log(8, "get_string_from_ml: len = %ld", len);
   *s = malloc(len+1);
   strncpy(*s, symb, len);
   (*s)[len] = '\0';
   MLDisownByteString(stdlink, symb, len);
   
   return rc;
}

int put_string_to_ml(char *s)
{
   return MLPutByteString(stdlink, s, strlen(s));
}
#endif /* MLVERSION < 3 */

#if MLVERSION < 3

static
int get_symbol_as_id()
{
   char *symb;
   int  i;
   
   MLGetSymbol(stdlink, &symb);
   log(8, "  get_symbol_as_id: symbol read = %s", symb);
   for(i=0; nsmath_symbols[i]; i++)
     if (strcmp(symb, nsmath_symbols[i])==0) {
	MLDisownSymbol(stdlink, symb); 
	return i;
     }
   MLDisownSymbol(stdlink, symb);
   return 0;
}
#else
int get_symbol_as_id()
{
   const unsigned char *symb;
   char *s;
   int  i;
   long len;
   
/*   MLGetByteSymbol(stdlink, &symb, &len, 1); */
    MLGetByteString(stdlink,&symb,&len,1);
   
   s = malloc(len+1);
   strncpy(s, symb, len);
   s[len] = '\0';
     
   log(8, "  get_symbol_as_id: symbol read = %s", s);
   for(i=0; nsmath_symbols[i]; i++)
     if (strcmp(s, nsmath_symbols[i])==0) {
	MLDisownSymbol(stdlink, symb);
	free(s);
	return i;
     }
   MLDisownSymbol(stdlink, symb);
   free(s);
   return 0;
}
#endif

void ignore_ml_input()
{
   MLNewPacket(stdlink);
}

static 
int get_real_from_ml(int type, int i, void **storage, 
		     int alloc_flag, int size, int *ml_rc)
{
   if (type != MLTKREAL && type != MLTKINT) return 1;
   if (alloc_flag)
     *storage = malloc(size);
   if (size == sizeof(float)) {
      if (MLGetFloat(stdlink, (float *) *storage)) {
	 log(8, "  arg #%d of type float = %f", i+1, 
	     *((float *) *storage));
      }
      else {
	 *ml_rc = MLError(stdlink);
	 return 1;
      }
   }
   else {
      if (MLGetDouble(stdlink, (double *) *storage)) {
	 log(8, "  arg #%d of type double = %lf", i+1, 
	     *((double *) *storage));
      }
      else {
	 *ml_rc = MLError(stdlink);
	 return 1;
      }
   }
   return 0;
}

static
int get_complex_from_ml(void **storage, int alloc_flag, int size, int get_both)
{
   int   i_part = 0, type;
   long  l;
   
   type = MLGetType(stdlink);
   log(2, "  get_complex_from_ml()");
      
   switch (type) {
    case MLTKFUNC:
      MLGetArgCount(stdlink, &l);
    case MLTKSYM:
      if (get_symbol_as_id() != NSMATH_COMPLEX) return 1;
      i_part = 1;
      break;
    case MLTKINT:
    case MLTKREAL:
      break;
    default:
      return 1;
   }
   i_part |= get_both;
   if (i_part) log(4, "  reading whole complex");
   else log(4, "  reading only real part");
   if (alloc_flag) {
      *storage = malloc(size);
      memset(*storage, 0 , size);
   }
   if (size == sizeof(scomplex)) 
     MLGetFloat(stdlink, &((scomplex *) *storage)->r);
   else
     MLGetDouble(stdlink, &((dcomplex *) *storage)->r); 
   if (i_part) {
      if (size == sizeof(scomplex)) 
	MLGetFloat(stdlink, &((scomplex *) *storage)->i);
      else
	MLGetDouble(stdlink, &((dcomplex *) *storage)->i);
   }
   return 0;
}

int get_complex_array_from_ml(int *type, int object_type, 
			      void **storage,
			      int alloc_flag, int size,
			      NS_Object *obj)
{
   long dm[3];
   long depth, l;
   int  id, j, k;
   void *ptr;
   
   dm[0] = dm[1] = 1;
   depth = 1;
   if (*type != MLTKFUNC) return 1;
   MLGetArgCount(stdlink, &dm[0]);
   if (get_symbol_as_id() != NSMATH_LIST) return 1;
   for(j=0; j<dm[0]; j++) {
      *type = MLGetNext(stdlink);
      id = 0;
      switch (*type) {
       case MLTKFUNC:
	 MLGetArgCount(stdlink, &l);
	 *type = MLGetNext(stdlink);
	 id = get_symbol_as_id();
	 if (id == NSMATH_LIST) {
	    if (j==0) {
	       dm[1] = l;
	       depth = 2;
	    }
	    else if (depth != 2 || dm[1]!=l) return 1;
	 }
	 if (id != NSMATH_LIST && id != NSMATH_COMPLEX) return 1;
	 break;
       case MLTKREAL:
       case MLTKINT:
	 break;
       default: return 1;
      }
      if (j==0 && alloc_flag) {
	 *storage = malloc(dm[0]*dm[1]*size);
	 memset(*storage, 0, dm[0]*dm[1]*size);
      }
      for(k=0; k<dm[1]; k++) {
	 log(8, "  j=%d, k=%d", j, k);
	 ptr = (char *) *storage + (j*dm[1]+k)*size;
	 get_complex_from_ml(&ptr, 0, size, 
			     k==0 && id == NSMATH_COMPLEX);
      }
   }

   if (object_type == NETSOLVE_MATRIX) {
      obj->attributes.matrix_attributes.major = ROW_MAJOR;
      obj->attributes.matrix_attributes.m = dm[0];
      obj->attributes.matrix_attributes.n = (depth == 2) ? dm[1] : 1;
      obj->attributes.matrix_attributes.l = dm[0];
      obj->attributes.matrix_attributes.d = NULL;
   }
   else 
     obj->attributes.vector_attributes.m = dm[0];
   
   return 0;
}

void put_complex_to_ml(void *storage, int size)
{
   MLPutNext(stdlink, MLTKFUNC);
   MLPutArgCount(stdlink, 2);
   MLPutSymbol(stdlink, "Complex");
   if (size == sizeof(scomplex)) {
      MLPutFloat(stdlink, ((scomplex *) storage)->r);
      MLPutFloat(stdlink, ((scomplex *) storage)->i);
   }
   else {
      MLPutDouble(stdlink, ((dcomplex *) storage)->r);
      MLPutDouble(stdlink, ((dcomplex *) storage)->i);
   }
}

static
int check_and_copy_objects(NS_ProblemDesc *pd)
{
   int 		i, j, k, ni;
   int          type, lang, len, rc, badarg, dummy;
   int		object_type, data_type;
   char         *s;
   char         **heads, **pps;
   long 	l, depth, *dims, dm[3];
   NS_Object 	**in_objs;
   void 	**ptr, *pvoid;
   NS_ScalarAttributes *scalar_attr;
   
   log(8, "check_and_copy_objects()");
   
   in_objs = pd->input_objects;
   ni = pd->nb_input_objects;
   
   for(i=0; i<ni; i++) {
      type = MLGetNext(stdlink);
      log(8, "  argument %d has MLType '%c'", i+1, type);
      if (type == MLTKERROR) {
	 s = strdup(MLErrorMessage(stdlink));
	 MLClearError(stdlink);
	 log_and_printf_msg(2, "Error getting %dth argument from MathLink", i+1);
	 log_and_printf_msg(2, "MLError: %s", s);
	 free((void *) s);
	 ns_errno = NetSolveInternalError;
	 return 0;
      }
      badarg = 0;
      rc = MLEOK;
      
      object_type = in_objs[i]->object_type;
      data_type = in_objs[i]->data_type;
      
      switch (object_type) {
	 
       case NETSOLVE_SCALAR: 
	 scalar_attr = &in_objs[i]->attributes.scalar_attributes;
	 switch (data_type) {
	  case NETSOLVE_B:
	    if (!(badarg = type != MLTKINT)) {
	       scalar_attr->ptr = malloc(sizeof(char));
	       if (MLGetInteger(stdlink, &dummy)) {
		  *((char *) scalar_attr->ptr) = (char) dummy;
		  log(8, "  arg #%d of type byte = %d", i+1, (int) (char) dummy);
	       }
	       else {
		  badarg = 1;
		  rc = MLError(stdlink);
	       }
	    }
	    break;
	    
	  case NETSOLVE_I:
	    if (!(badarg = type != MLTKINT)) {
	       scalar_attr->ptr = malloc(sizeof(int));
	       if (MLGetInteger(stdlink, (int *) scalar_attr->ptr)) {
		  log(8, "  arg #%d of type integer = %d", i+1, 
		      *((int *) scalar_attr->ptr));
	       }
	       else {
		  badarg = 1;
		  rc = MLError(stdlink);
	       }
	    }
	    break;
	    
	  case NETSOLVE_S:
	    badarg = get_real_from_ml(type, i, &scalar_attr->ptr, TRUE,
				      sizeof(float), &rc);
	    break;
	  case NETSOLVE_D:
	    badarg = get_real_from_ml(type, i, &scalar_attr->ptr, TRUE, 
				      sizeof(double), &rc);
	    break;
	    
	  case NETSOLVE_C:
	    badarg = get_complex_from_ml(&scalar_attr->ptr, TRUE, sizeof(scomplex), 0);
	    break;
	    
	  case NETSOLVE_Z:
	    badarg = get_complex_from_ml(&scalar_attr->ptr, TRUE, sizeof(dcomplex), 0);
	    break;
	    
	  case NETSOLVE_CHAR:
	    if (!(badarg = type != MLTKSTR)) {
	       if (get_string_from_ml(&s)) {
		  scalar_attr->ptr = malloc(sizeof(char));
		  *((char *) scalar_attr->ptr) = s[0];
		  log(8, "  arg #%d of type char = '%c'", i+1, s[0]);
		  free(s);
	       }
	       else {
		  badarg = 1;
		  rc = MLError(stdlink);
	       }
	    }
	    break;
	    
	  default:
	    badarg = 1;
	 }
	 break;
	 
       case NETSOLVE_UPF:
	 if (!(badarg = type != MLTKSTR)) {
	    if (get_string_from_ml(&s)) {
	       log(8, "  arg #%d of type file = %s", i+1, s);	       
	       in_objs[i]->attributes.upf_attributes.filename = (void *) strdup(s);
	       len = strlen(s);
	       in_objs[i]->attributes.upf_attributes.language =
		 lang = (len>2 && strcmp(&s[len-2], ".c")==0) 
		   ? UPF_LANG_C
		 : (len>2 && strcmp(&s[len-2], ".f")==0) 
		   ? UPF_LANG_FORTRAN 
		 : -1;
	       if (lang==-1) {
		  badarg = 1;
		  break;
	       }
	       in_objs[i]->attributes.upf_attributes.language = lang;
	       s[len-2] = 0;
	       in_objs[i]->attributes.upf_attributes.funcname = strdup(s);
	       
	       free(s);
	    }
	    else {
	       badarg = 1;
	       rc = MLError(stdlink);
	    }
	 }
	 break;
	 
	 
       case NETSOLVE_FILE:
	 if (!(badarg = type != MLTKSTR)) {
	    if (get_string_from_ml(&s)) {
	       in_objs[i]->attributes.file_attributes.filename = (void *) strdup(s);
	       log(8, "  arg #%d of type file = %s", i+1, s);
	       free(s);
	    }
	    else {
	       badarg = 1;
	       rc = MLError(stdlink);
	    }
	 }
	 break;
      
       case NETSOLVE_STRING:
	 if (!(badarg = type != MLTKSTR)) {
	    if (get_string_from_ml(&s)) {
	       in_objs[i]->attributes.string_attributes.ptr = strdup(s);
	       log(8, "  arg #%d of type string = %s", i+1, s);
	       free(s);
	    }
	    else {
	       badarg = 1;
	       rc = MLError(stdlink);
	    }
	 } 
	 break;
	 
       case NETSOLVE_VECTOR:
       case NETSOLVE_MATRIX:
	 log(8, "  arg #%d is vector/matrix", i+1);
	 badarg = type != MLTKFUNC && data_type != NETSOLVE_CHAR;
	 if (!badarg) {
	    
	    switch (data_type) {
	     case NETSOLVE_B:
	       dm[0] = dm[1] = 1;
	       depth = 1;
	       ptr = (object_type == NETSOLVE_MATRIX) 
		 ? &in_objs[i]->attributes.matrix_attributes.ptr
		 : &in_objs[i]->attributes.vector_attributes.ptr;
	       badarg = type != MLTKFUNC;
	       if (!badarg) {
		  MLGetArgCount(stdlink, &dm[0]);
		  if ((badarg = get_symbol_as_id() != NSMATH_LIST)) break;
		  for(j=0; j<dm[0]; j++) {
		     type = MLGetNext(stdlink);
		     if ((badarg = type != MLTKINT && type != MLTKFUNC)) break;
		     if (type == MLTKFUNC) {
			MLGetArgCount(stdlink, &l);
			if ((badarg = depth == 2 && l!=dm[1])) break;
			dm[1] = l;
			if ((badarg = get_symbol_as_id() != NSMATH_LIST)) break;
			depth = 2;
		     }
		     if (! *ptr) *ptr = malloc(dm[0]*dm[1]*sizeof(char));
		     for(k=0; k<dm[1]; k++) {
			MLGetInteger(stdlink, &dummy);
			*((char *) *ptr + j*dm[1] + k) = (char) dummy;
		     }
		  }
		  if (object_type == NETSOLVE_MATRIX) {
		     in_objs[i]->attributes.matrix_attributes.major = ROW_MAJOR;
		     in_objs[i]->attributes.matrix_attributes.m = dm[0];
		     in_objs[i]->attributes.matrix_attributes.n = (depth == 2)?dm[1]:1;
		     in_objs[i]->attributes.matrix_attributes.l = dm[0];
		     in_objs[i]->attributes.matrix_attributes.d = NULL;
		  }
		  else in_objs[i]->attributes.vector_attributes.m = dm[0];
	       }
	       break;
	       
	     case NETSOLVE_CHAR:
	       dm[0] = dm[1] = 1;
	       depth = 1;
	       
	       switch (type) {
		case MLTKSTR:
		  if ((badarg = object_type == NETSOLVE_MATRIX)) break;
		  ptr = &in_objs[i]->attributes.vector_attributes.ptr;
		  
		  get_string_from_ml(&s);
		  *ptr = strdup(s);
		  in_objs[i]->attributes.vector_attributes.m = strlen(s);
		  log(2, "  arg #%d of type vector of chars = %s", i+1, s);
		  free(s);
		  break;
		  
		case MLTKFUNC:
		  if ((badarg = object_type == NETSOLVE_VECTOR)) break;
		  ptr = &in_objs[i]->attributes.matrix_attributes.ptr;
		  
		  MLGetArgCount(stdlink, &dm[1]);
		  if ((badarg = get_symbol_as_id() != NSMATH_LIST)) break;
		  depth = 2;
		  pps = (char **) malloc(dm[1]*sizeof(char *));
		  for(j=0; j<dm[1]; j++) {
		     type = MLGetNext(stdlink);
		     if ((badarg = type != MLTKSTR)) break;
		     if ((badarg = !get_string_from_ml(&s))) break;
		     pps[j] = strdup(s);
		     free(s);
		     k = strlen(pps[j]);
		     if (dm[0] < k) dm[0] = k;
		  }
		  *ptr = malloc(sizeof(char)*dm[0]*dm[1]);
		  memset(*ptr, 0, sizeof(char)*dm[0]*dm[1]);
		  for(j=0; j<dm[1]; j++) {
		     strncpy((char *) *ptr + j*dm[0], pps[j], dm[0]);
		     free(pps[j]);
		  }
		  free(pps);
		  in_objs[i]->attributes.matrix_attributes.major = ROW_MAJOR;
		  in_objs[i]->attributes.matrix_attributes.m = dm[1];
		  in_objs[i]->attributes.matrix_attributes.n = (depth == 2) ? dm[0]:1;
		  in_objs[i]->attributes.matrix_attributes.l = dm[1];
		  in_objs[i]->attributes.matrix_attributes.d = NULL;
	       }
	       break;
	       
	     case NETSOLVE_I:
	       if (MLGetIntegerArray(stdlink, (int **) &pvoid, &dims, 
				     &heads, &depth)) {
		  badarg = depth<1 || depth>2;
		  if (!badarg) {
		     for(j=0, l=1; j<depth; j++) 
		       l *= dims[j];
		     
		     if (object_type == NETSOLVE_VECTOR) 
		       ptr = &in_objs[i]->attributes.vector_attributes.ptr;
		     else
		       ptr = &in_objs[i]->attributes.matrix_attributes.ptr;
		     
		     *ptr = calloc(l, sizeof(int));
		     memcpy(*ptr, pvoid, l*sizeof(int));
		     
		     if (object_type == NETSOLVE_MATRIX) {
			in_objs[i]->attributes.matrix_attributes.major = ROW_MAJOR;
			in_objs[i]->attributes.matrix_attributes.m = dims[0];
			in_objs[i]->attributes.matrix_attributes.n = (depth == 2) ? dims[1] : 1;
			in_objs[i]->attributes.matrix_attributes.l = dims[0];
			in_objs[i]->attributes.matrix_attributes.d = NULL;
		     }
		     else 
		       in_objs[i]->attributes.vector_attributes.m = dims[0];
		  }
	       }
	       else {
		  badarg = 1;
		  rc = MLError(stdlink);
	       }
	       MLDisownIntegerArray(stdlink, (int *) pvoid, dims, heads, depth);
	       break;
	       
	     case NETSOLVE_S:
	       if (MLGetFloatArray(stdlink,(float **) &pvoid, &dims,
				   &heads, &depth)) {
		  log(5, "  depth = %ld, dims = [%ld, %ld]", depth, dims[0], dims[1]);
		  
		  badarg = depth<1 || depth>2;
		  if (!badarg) {
		     for(j=0, l=1; j<depth; j++) 
		       l *= dims[j];
		     
		     if (object_type == NETSOLVE_VECTOR) 
		       ptr = &in_objs[i]->attributes.vector_attributes.ptr;
		     else
		       ptr = &in_objs[i]->attributes.matrix_attributes.ptr;
		     
		     *ptr = calloc(l, sizeof(float));
		     
		     memcpy(*ptr, pvoid, l*sizeof(float));
		     
		     if (object_type == NETSOLVE_MATRIX) {
			in_objs[i]->attributes.matrix_attributes.major = ROW_MAJOR;
			in_objs[i]->attributes.matrix_attributes.m = dims[0];
			in_objs[i]->attributes.matrix_attributes.n = (depth == 2) ? dims[1] : 1;
			in_objs[i]->attributes.matrix_attributes.l = dims[0];
			in_objs[i]->attributes.matrix_attributes.d = NULL;
		     }
		     else
		       in_objs[i]->attributes.vector_attributes.m = dims[0];
		  }
	       }
	       else {
		  badarg = 1;
		  rc = MLError(stdlink);
	       }
	       MLDisownFloatArray(stdlink, (float *) pvoid, dims, heads, depth);
	       break;
	       
	     case NETSOLVE_D:
	       if (MLGetDoubleArray(stdlink,(double **) &pvoid, &dims, 
				    &heads, &depth)) {
	       log(5, "  depth = %ld, dims = [%ld, %ld]", depth, dims[0], dims[1]);
	       badarg = depth<1 || depth>2;
		  if (!badarg) {
		     for(j=0, l=1; j<depth; j++) 
		       l *= dims[j];
		     
		     if (object_type == NETSOLVE_VECTOR) 
		       ptr = &in_objs[i]->attributes.vector_attributes.ptr;
		     else
		       ptr = &in_objs[i]->attributes.matrix_attributes.ptr;
		     
		     *ptr = calloc(l, sizeof(double));
		     
		     memcpy(*ptr, pvoid, l*sizeof(double));
		     
		     if (object_type == NETSOLVE_MATRIX) {
			in_objs[i]->attributes.matrix_attributes.major = ROW_MAJOR;
			in_objs[i]->attributes.matrix_attributes.m = dims[0];
			in_objs[i]->attributes.matrix_attributes.n = (depth == 2) ? dims[1] : 1;
			in_objs[i]->attributes.matrix_attributes.l = dims[0];
			in_objs[i]->attributes.matrix_attributes.d = NULL;
		     }
		     else 
		       in_objs[i]->attributes.vector_attributes.m = dims[0];
		  }
	       }
	       else {
		  badarg = 1;
		  rc = MLError(stdlink);
	       }
	       MLDisownDoubleArray(stdlink, (double *) pvoid, dims, heads, depth);
	       break;
	       
	     case NETSOLVE_C:
	       if (object_type == NETSOLVE_VECTOR) 
		 ptr = &in_objs[i]->attributes.vector_attributes.ptr;
	       else
		 ptr = &in_objs[i]->attributes.matrix_attributes.ptr;
	       
	       badarg = get_complex_array_from_ml(&type, object_type, ptr, TRUE,
						  sizeof(scomplex),
						  in_objs[i]);
	       break;
	       
	     case NETSOLVE_Z:
	       if (object_type == NETSOLVE_VECTOR) 
		 ptr = &in_objs[i]->attributes.vector_attributes.ptr;
	       else
		 ptr = &in_objs[i]->attributes.matrix_attributes.ptr;
	       
	       badarg = get_complex_array_from_ml(&type, object_type, ptr, TRUE,
						  sizeof(dcomplex),
						  in_objs[i]);
	       break;
	    }
	    if (!badarg) {
	       if (object_type == NETSOLVE_VECTOR) 
		 log(5, "  Object %d is a vector of size %d", 
		     i+1, in_objs[i]->attributes.vector_attributes.m);
	       if (object_type == NETSOLVE_MATRIX)
		 log(5, "  Object %d is a matrix with dimensions (%d,%d)",
		     i+1, in_objs[i]->attributes.matrix_attributes.m,
		     in_objs[i]->attributes.matrix_attributes.n);
/************************************************/
         if(object_type == NETSOLVE_MATRIX) {
            if(in_objs[i]->attributes.matrix_attributes.major!=pd->major)
               if (transposeMatrix(in_objs[i]) == -1)
                    {
#ifdef VIEW
          fprintf(STDERR "Error while transposing matrix\n");
#endif
          return -1;
                    }
 } 
/********************************************/
}
	    
}
	 break;
	 
       default: badarg = 1;
      }
      if (badarg) {
	 ignore_ml_input();
	 if (rc != MLEOK) {
	    s = strdup(MLErrorMessage(stdlink));
	    MLClearError(stdlink);
	    log_and_printf_msg(1, "Error retriving arg %d", i+1);
	    log_and_printf_msg(1, "MLError: %s", s);
	    free((void *) s);
	 }
	 else {
	    log_and_printf_msg(1, "Error: argument %d has invalid type", i+1);
	    log_and_printf_msg(1, "       NetSolve type %d expected, ML type '%c' passed", 
			       data_type, type);
	 }
	 ns_errno = NetSolveBadValues;
	 return 0;
      }
   }
   return 1;
}

int check_for_output_file_objects(NS_ProblemDesc *pd)
{
   int i, type;
   char *s;

   log(2, "Checking for output file objects");
   for(i=0; i<pd->nb_output_objects; i++) {
      if (pd->output_objects[i]->object_type == NETSOLVE_FILE) {
	 if ((type = MLGetType(stdlink)) != MLTKSTR) {
	    log_and_printf_msg(1, "Error: argument %d has invalid type", i+1);
	    log_and_printf_msg(1, "       NetSolve type %d expected, ML type '%c' passed", 
			       pd->input_objects[i]->data_type, type);
	    return 0;
	 }
	 if (get_string_from_ml(&s)) {
	    pd->output_objects[i]->attributes.file_attributes.filename = strdup(s);

	    log(8, "  arg #%d of type file = %s", i+1, s);
	    free(s);
	 }
	 else {
	    s = strdup(MLErrorMessage(stdlink));
	    MLClearError(stdlink);
	    log_and_printf_msg(1, "Error retriving arg %d", i+1);
	    log_and_printf_msg(1, "MLError: %s", s);
	    free((void *) s);
	    return 0;
	 }
      }
   }
   return 1;
}

int get_num_out_file_objs(NS_ProblemDesc *pd)
{
   int i, n=0;

   for(i=0; i<pd->nb_output_objects; i++) 
     if (pd->output_objects[i]->object_type == NETSOLVE_FILE) n++;
   return n;
}

int init_output_objects(NS_ProblemDesc *pd)
{
  int i;
  int object_type,data_type;

  for (i=0;i<pd->nb_output_objects;i++)
  {
    object_type = pd->output_objects[i]->object_type;
    data_type = pd->output_objects[i]->data_type;

    switch(object_type)
    {
      case NETSOLVE_MATRIX:
      {
        free(pd->output_objects[i]);
        pd->output_objects[i] = createMatrixObject(
                data_type,
                NULL,-1,-1,-1,-1);
        break;
      }
      case NETSOLVE_VECTOR:
      {
        free(pd->output_objects[i]);
        pd->output_objects[i] = createVectorObject(
                data_type,
                NULL,-1);
        break;
      }
      case NETSOLVE_SCALAR:
      {
        free(pd->output_objects[i]);
        pd->output_objects[i] = createScalarObject(
                data_type,NULL);
        break;
      }
      case NETSOLVE_FILE:
      {
        char buffer[256];
        sprintf(buffer,"./%s-output%d-%d",pd->nickname,i,rand());
        free(pd->output_objects[i]);
        pd->output_objects[i] = createFileObject(buffer);
        break;
      }
      case NETSOLVE_PACKEDFILES:
      {
        char buffer[256];
        sprintf(buffer,"./%s-output%d-%d",pd->nickname,i,rand());
        pd->output_objects[i] = createPackedFilesObject(buffer,NULL,-1);
        break;
      }

      case NETSOLVE_STRING:
      {
        free(pd->output_objects[i]);
        pd->output_objects[i] = createStringObject(NULL);
        break;
      }
    }
  }
  return NetSolveOK;
}


/* clean up function */
static 
void ML_to_gateway_cu(int rc, char *code, NS_ProblemDesc *pd)
{
   char *s;
   
   log(8, "ML_to_gateway_cu");
   freeProblemDesc(pd);

   free(code);
   if (rc != MLEOK) {
      s = strdup(MLErrorMessage(stdlink));
      MLClearError(stdlink);
      nsmath_msg(s);
      log(2, s);
      free(s);
   }
}

NS_ProblemDesc *ML_to_gateway()
{
#define ML2GATEWAY_CU ML_to_gateway_cu(rc, code, pd)
   
   int		rc, ni;
   ML_SYMBOL  	s;
   long		arg_count;
   char         *code, *err;
   NS_ProblemDesc  *pd;

   log(1, "ML_to_gateway()");

/* init */
   rc = 0;
   code = NULL;
   
/* get the problem name and the number of argumets passed */
   if (MLGetNext(stdlink) == MLTKFUNC &&
       MLGetFunction(stdlink, &s, &arg_count)) {
      code = strdup(s);
      MLDisownSymbol(stdlink, s);
   }
   else {
      err = strdup(MLErrorMessage(stdlink));
      MLClearError(stdlink);
      ignore_ml_input();
      log_and_printf_msg(1, "Error retrieving function name: %s", err);
      free((void *) err);
      ML2GATEWAY_CU;
      return NULL;
   }
   log(2, "  problem name = %s", code);
   
    /* agent_name init here */
    if(flag==1)
     netslinit(NULL); 

      flag=0;

/* get the problem description */
   if (netsolveInfo(code, &pd) == -1) {
      ignore_ml_input();
      log(2, "\tError: %s", netsolve_error(ns_errno));
      ML2GATEWAY_CU;
      return NULL;
   }
   log(8, "  description of %s retrieved", code);

   ni = pd->nb_input_objects;
/*   fprintf(stderr,"pd->major = %d\n",pd->major); */
       
   if (arg_count != ni + get_num_out_file_objs(pd)) {
      ignore_ml_input();
      log_and_printf_msg(2, "Error: %d arguments expected, %ld passed.", 
			pd->nb_input_objects, arg_count);
      ns_errno = NetSolveBadValues;
      ML2GATEWAY_CU;

      return NULL;
   }
					      
   if (!check_and_copy_objects(pd)) {
      ML2GATEWAY_CU;
      return NULL;
   }
   
   if ((ns_errno = init_output_objects(pd)) != NetSolveOK) {
      ML2GATEWAY_CU;
      return NULL;
   }
   
/* for every output object we need to provide a file name for it */
   if (!check_for_output_file_objects(pd)) {
      ML2GATEWAY_CU;
      return NULL;
   }
   
   free(code);
   return pd;
}

int gateway_send(NS_ProblemDesc *pd)
{
   int elapsed;
 
   log(1, "gateway_send()");

   submit_problem(NS_BLOCK,NS_NOASSIGNMENT,NULL,pd, pd->input_objects, pd->output_objects,&elapsed,0);
   return ns_errno;
}

void gateway_to_ML(NS_ProblemDesc *pd)
{
   int 		i, j, k, object_type, data_type, value;
   int          my_major=ROW_MAJOR;
   long		depth, dims[3];
   char 	*heads[] = {"List", "List", "List"};
   char 	s[2], *sptr;
   int  	result_sent = 0;
   NS_Object 	*out_obj;
   void   	*ptr;
   
   log(1, "gateway_to_ML()");
   if (pd->nb_output_objects > 1) {
      MLPutNext(stdlink, MLTKFUNC);
      MLPutArgCount(stdlink, pd->nb_output_objects);
      MLPutSymbol(stdlink, "List");
   }
   
   for(i=0; i < pd->nb_output_objects; i++) {
      object_type = pd->output_objects[i]->object_type;
      data_type = pd->output_objects[i]->data_type;
      log(8, "  putting arg #%d to ML, object type %d, data type %d",
	  i+1, object_type, data_type);
      
      out_obj = pd->output_objects[i];
      switch (object_type) {
       case NETSOLVE_SCALAR:
	 result_sent |= 1;
	 
	 switch (data_type) {
	  case NETSOLVE_B:
	    value = *((char *) out_obj->attributes.scalar_attributes.ptr);
	    MLPutInteger(stdlink, value);
	    break;
	    
	  case NETSOLVE_I:
	    value = *((int *) out_obj->attributes.scalar_attributes.ptr);
	    MLPutInteger(stdlink, value);
	    log(2, "  arg#%d of type int = %d", i+1, value);
	    break;
	    
	  case NETSOLVE_S:
	    MLPutFloat(stdlink, *((float *) out_obj->attributes.scalar_attributes.ptr));
	    log(2, "  arg#%d of type float = %f", i+1, *((float *) out_obj->attributes.scalar_attributes.ptr));
	    break;
	    
	  case NETSOLVE_D:
	    MLPutDouble(stdlink, *((double *) out_obj->attributes.scalar_attributes.ptr));
	    log(2, "  arg#%d of type double = %lf", i+1, *((double *) out_obj->attributes.scalar_attributes.ptr));
	    break;
	    
	  case NETSOLVE_C:
	    put_complex_to_ml(out_obj->attributes.scalar_attributes.ptr, sizeof(scomplex));
	    break;
	    
	  case NETSOLVE_Z:
	    put_complex_to_ml(out_obj->attributes.scalar_attributes.ptr, sizeof(dcomplex));
	    break;
	    
	  case NETSOLVE_CHAR:
	    s[0] = *((char *) out_obj->attributes.scalar_attributes.ptr);
	    s[1] = 0;
	    put_string_to_ml(s);
	    break;
	    
	  default:
	    log_and_printf_msg(2, "NetSolve data type %d unknown", 
			       pd->output_objects[i]->data_type);
	    nsmath_return_null();
	 }
	 break;
	
       case NETSOLVE_STRING:
	 put_string_to_ml((char *) out_obj->attributes.string_attributes.ptr);
	 break;
	 
       case NETSOLVE_MATRIX:
       case NETSOLVE_VECTOR:
	 result_sent |= 1;
	 if (pd->output_objects[i]->object_type == NETSOLVE_MATRIX) {
	    dims[0] = pd->output_objects[i]->attributes.matrix_attributes.m;
	    dims[1] = pd->output_objects[i]->attributes.matrix_attributes.n;
	 }
	 else {
	    dims[0] = pd->output_objects[i]->attributes.vector_attributes.m;
	    dims[1] = 1;
	 }
	 depth = (pd->output_objects[i]->object_type == NETSOLVE_MATRIX) ? 2 : 1;
	 log(8, "  depth = %ld, dims[0] = %ld, dims[1] = %ld", 
	     depth, dims[0], dims[1]);
	 
	 if (object_type == NETSOLVE_MATRIX)
	   ptr = out_obj->attributes.matrix_attributes.ptr;
	 else
	   ptr = out_obj->attributes.vector_attributes.ptr;
         /******************************************
         if (object_type == NETSOLVE_MATRIX)
        {
          fprintf(STDERR "gateway_to_ML()\n");
          fprintf(STDERR "major=%d\n",pd->output_objects[i]->attributes.matrix_attributes.major);
       }
        if(object_type == NETSOLVE_MATRIX&&pd->output_objects[i]->attributes.matrix_attributes.major!=ROW_MAJOR)
         pd->output_objects[i]->attributes.matrix_attributes.major = ROW_MAJOR;

         if (transposeMatrix(out_obj) == -1)
                    {
#ifdef VIEW
          fprintf(STDERR "Error while transposing matrix\n");
#endif
          nsmath_return_null();
                   } 
*****************************************/


	 switch (pd->output_objects[i]->data_type) {
	  case NETSOLVE_I:
	    MLPutIntegerArray(stdlink, (int *) ptr, dims, heads, depth);
	    break;
	    
	  case NETSOLVE_S:
	    MLPutFloatArray(stdlink, (float *) ptr, dims, heads, depth);
	    break;
	    
	  case NETSOLVE_D:
	    MLPutDoubleArray(stdlink, (double *) ptr, dims, heads, depth);
	    break;
	    
	  case NETSOLVE_C:
	    MLPutNext(stdlink, MLTKFUNC);
	    MLPutArgCount(stdlink, dims[0]);
	    MLPutSymbol(stdlink, "List");
			  
	    for(j=0; j<dims[0]; j++) {
	       if (depth>1) {
		  MLPutNext(stdlink, MLTKFUNC);
		  MLPutArgCount(stdlink, dims[1]);
		  MLPutSymbol(stdlink, "List");
	       }
	       for(k=0; k<dims[1]; k++) 
		    put_complex_to_ml((scomplex *) ptr + j*dims[1]+k, sizeof(scomplex));
	    }
	    break;
	    
	  case NETSOLVE_Z:
	    MLPutNext(stdlink, MLTKFUNC);
	    MLPutArgCount(stdlink, dims[0]);
	    MLPutSymbol(stdlink, "List");
			  
	    for(j=0; j<dims[0]; j++) {
	       if (depth>1) {
		  MLPutNext(stdlink, MLTKFUNC);
		  MLPutArgCount(stdlink, dims[1]);
		  MLPutSymbol(stdlink, "List");
	       }
	       for(k=0; k<dims[1]; k++) 
		    put_complex_to_ml((dcomplex *) ptr + j*dims[1]+k, sizeof(dcomplex));
	    }
	    break;
	    
	  case NETSOLVE_B:
	    MLPutNext(stdlink, MLTKFUNC);
	    MLPutArgCount(stdlink, dims[0]);
	    MLPutSymbol(stdlink, "List");
			  
	    for(j=0; j<dims[0]; j++) {
	       if (depth>1) {
		  MLPutNext(stdlink, MLTKFUNC);
		  MLPutArgCount(stdlink, dims[1]);
		  MLPutSymbol(stdlink, "List");
	       }
	       for(k=0; k<dims[1]; k++) 
		    MLPutInteger(stdlink, ((char *) ptr)[j*dims[1] + k]);
	    }
	    break;
	    
	  case NETSOLVE_CHAR:
	    if (depth == 2) {
	       MLPutNext(stdlink, MLTKFUNC);
	       MLPutArgCount(stdlink, dims[0]);
	       MLPutSymbol(stdlink, "List");
	    }
	    k = dims[depth-1];
	    sptr = (char *) malloc(k + 1);
	    sptr[k] = '\0';
	    for(j=0; j < ((depth == 2)? dims[0] : 1); j++) {
	       strncpy(sptr, (char *) ptr + j*k, k);
	       put_string_to_ml(sptr);
	    }
	    free(sptr);
	    break;
	   
	  default:
	    log_and_printf_msg(2,  "NetSolve data type %d unsupported",
			       pd->output_objects[i]->data_type); 
	    nsmath_return_null();
	 }
 /********************************************
         if(pd->output_objects[i]->object_type == NETSOLVE_MATRIX) {
         fprintf(stderr,"i=%d\n",i);
             fprintf(stderr,"my_major=%d\n",my_major);
            fprintf(stderr,"pd->major=%d\n",pd->major);
             if (my_major != pd->major)
               if (transposeMatrix(pd->output_objects[i]) == -1)
                    {
#ifdef VIEW
          fprintf(STDERR "Error while transposing matrix\n");
#endif
       nsmath_return_null();
                    }
} 
*******************************************/


	 break;
	 
       case NETSOLVE_UPF:
       case NETSOLVE_FILE:
	 break;
	 
       default:
	 log_and_printf_msg(2, "NetSolve object type %d unknown", 
			    pd->output_objects[i]->object_type);
	 nsmath_return_null();
	 
      }
      if (MLError(stdlink) != MLEOK) 
	log(2, "MLError(%d): %s", MLErrorMessage(stdlink));
      if (!result_sent) nsmath_return_null();
   }
}
