/*****************************************************************/
/*      iserv_requests.c                                         */
/*      Jakob Oestergaard                                        */
/*---------------------------------------------------------------*/
/*  request handlers for the infoserver                          */
/*****************************************************************/

#include "infoserver.h"
#include "infoserver_int.h"

#include <unistd.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <time.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <assert.h>

/*
 * Input string management
 */

#ifndef MIN
# define MIN(a,b) ((a) < (b) ? (a) : (b))
#endif

char * input_parser_line = 0;
int input_parser_pos = 0;
int input_parser_len = 0;

int buf_yyinput(char * buf, int maxl)
{
  int n = MIN(maxl, input_parser_len - input_parser_pos);
  if(n > 0) {
    memcpy(buf, input_parser_line + input_parser_pos, n);
    input_parser_pos += n;
  }
  return n;
}


/*
 * Main reqest handling routine
 */
void iserv_receive_request(int fd)
{
  /*
   * Agent sent us new data
   */
  int rc;
  static char * ibuf = 0;

  parser_mode = NS_IS_PMode_Interactive;

  /*
   * Allocate buffer on heap
   * (prevents stack smashing)
   */
  if(!ibuf) {
    ibuf = (char*)malloc(NS_IS_MAX_REQLEN+1);
    if(!ibuf) {
      fprintf(stderr, "Not enough memory for input buffer allocation\n");
      exit(1);
    }
  }

  /*
   * Get input
   */
  rc = recv(fd, ibuf, NS_IS_MAX_REQLEN, 0);
  if(rc < 0) {
    fprintf(stderr, "Error (%s) while receiving input\n",
	    strerror(errno));
  }
  if(rc <= 0) {
    /*
     * Other end closed connection - unchain input
     */
    unchain_client(fd);
    fprintf(stderr, "Closed connection.\n");
    return;
  }
  assert(rc <= NS_IS_MAX_REQLEN);
  while(rc > 0 && (ibuf[rc-1] == '\r' || ibuf[rc-1] == '\n')) rc--;
  ibuf[rc] = 0;

  /*
   * Setup global (hackery) stuff
   */
  parse_client = fd;
  input_parser_line = ibuf;
  input_parser_pos = 0;
  input_parser_len = rc;

  /*
   * Parse input
   */
  yyrestart(0);
  if(yyparse()) {
    fprintf(stderr, "In: \"%s\"\n", ibuf);
  }
}

void treat_quit(int fd)
{
  unchain_client(fd);
}

void treat_stats(int fd)
{
  char buf[NS_IS_MAX_REQLEN];
  time_t now = time(0);

  push_to_client(fd, "DBMS Statistics:                              \n");
  push_to_client(fd, "----------------------------------------------\n");
  sprintf(buf, "Records in core       :  %9u\n", dbms_stat_records());
  push_to_client(fd, buf);
  sprintf(buf, "Average keys/record   :  %9.3f\n", dbms_stat_keyrec());
  push_to_client(fd, buf);
  sprintf(buf, "Total unique keys     :  %9u\n", dbms_stat_ukeys());
  push_to_client(fd, buf);
  sprintf(buf, "Total inserts         :  %9u\n", req_stat_puts());
  push_to_client(fd, buf);
  if(req_stat_lastput()) {
    int lit = (int)(now - req_stat_lastput());
    sprintf(buf, "Last insert happened  :  %9i sec. ago\n", lit);
    push_to_client(fd, buf);
  }
  sprintf(buf, "Total get requests    :  %9u\n", req_stat_gets());
  push_to_client(fd, buf);
  if(req_stat_lastget()) {
    int lgt = (int)(now - req_stat_lastget());
    sprintf(buf, "Last get happened     :  %9i sec. ago\n", lgt);
    push_to_client(fd, buf);
  }
  push_to_client(fd, "----------------------------------------------\n");
  sprintf(buf, "Active clients        :  %9u\n", req_stat_clients());
  push_to_client(fd, buf);
  push_to_client(fd, "----------------------------------------------\n"); 
}
