/*****************************************************************/
/*      iserv_persistence.c                                      */
/*      Jakob Oestergaard                                        */
/*---------------------------------------------------------------*/
/*  Persistence routines for the InfoServer DBMS                 */
/*****************************************************************/

#include "infoserver.h"
#include "infoserver_int.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>

static FILE * dbf = 0;
const char db_filename[] = "./infoserver.db";

/*
 * We provide several routines here:
 *
 *  iserv_readdb()
 *    -  loads the entire on-disk database into core
 *  iserv_putevent(ev)
 *    -  writes the given event to persistent store
 *
 *******************************************************
 *
 * File format:
 *
 *  All events are chronologically written.
 *
 * The format is very simple:
 *  KEY_i="VALUE_i" KEY_j="VALUE_j" ...  \n
 *
 *******************************************************
 */

void iserv_readdb(void)
{
  char * linebuffer;
  unsigned line = 0;

  parser_mode = NS_IS_PMode_LogFile;

  if(dbf) {
    fclose(dbf);
    fprintf(stderr, "Closed open database file\n");
  }

  if(!(dbf = fopen(db_filename, "r"))) {
    fprintf(stderr, "Couldn't open database file \"%s\"\n",
	    db_filename);
    return;
  }
  if(!(linebuffer = (char*)malloc(NS_IS_MAX_REQLEN))) {
    fprintf(stderr, "Couldn't allocate line buffer for db read\n");
    return;    
  }

  /*
   * Step thru database file, reading, parsing and inserting
   * line by line.
   * We make sure there's a "put" statement in the beginning
   * of each line - this is just to make the parser happy.
   */
  strcpy(linebuffer, "put ");
  while(!feof(dbf)) {
    char * citer;
    /*
     * Read line
     */
    if(!fgets(linebuffer + 4, NS_IS_MAX_REQLEN, dbf))
      break;
    line++;
    /*
     * Kill new-line 
     */
    citer = linebuffer;
    while(*citer && *citer != '\r' && *citer != '\n')
      citer++;
    *citer = 0;
    /*
     * Parse line
     */
    parse_client = -1;
    input_parser_line = linebuffer;
    input_parser_pos = 0;
    input_parser_len = strlen(linebuffer);
    yyrestart(0);
    yyparse();
  }
  fprintf(stderr, "Database %s loaded.\n", db_filename);
  /*
   * Close read-only database
   */
  fclose(dbf);
  dbf = 0;
}


void iserv_putrecord(const NS_IS_Record * rec)
{
  unsigned i;
  /*
   * Make sure database file is open
   */
  if(!dbf) {
    dbf = fopen(db_filename, "a");
    if(!dbf) {
      fprintf(stderr, "Couldn't open database file \"%s\" for writing\n", db_filename);
      return;
    }
  }
  /*
   * Write entry
   */  
  for(i = 0; i != rec->valuelen; i++)
    if(rec->value[i]) {
      const char * key = find_index_key(i);
      fprintf(dbf, "%s=\"%s\" ", key, rec->value[i]);
    }
  fprintf(dbf, ";\n");

  /*
   * Flush the data (don't sync(), just get the data out of libc)
   */
  fflush(dbf);
}
