/*****************************************************************/
/*      iserv_main.c                                             */
/*      Jakob Oestergaard                                        */
/*---------------------------------------------------------------*/
/*  main loop of the infoserver                                  */
/*****************************************************************/

#include "infoserver.h"
#include "infoserver_int.h"

#include <unistd.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <sys/time.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <fcntl.h>

#ifndef MAX
#define MAX(a,b) ((a) > (b) ? (a) : (b))
#endif

#ifndef SOL_TCP
/*
 * On GNU/Linux this one is defined, on
 * Solaris it must be set to the protocol
 * number of TCP
 */
# define SOL_TCP 6
#endif

NS_IS_ClientConnection * is_clients = 0;

/*
 * Statistics routines
 */
static unsigned n_clients = 0;


unsigned req_stat_clients(void)
{
  return n_clients;
}


void iserv_main_loop(void)
{
  listen(infoserver_input_fd, 5);
  listen(infoserver_client_fd, 5);

  while(1) {
    fd_set inputs, outputs;
    int rc;
    int maxfd;
    NS_IS_ClientConnection * iter;

    FD_ZERO(&inputs);
    FD_ZERO(&outputs);

    /*
     * Possible input filedescriptors are:
     *  input socket (for new connections)
     *  client socket (for new connections)
     *  all input connections
     *  all client conections
     */
    FD_SET(infoserver_input_fd, &inputs);
    FD_SET(infoserver_client_fd, &inputs);
    maxfd = MAX(infoserver_input_fd, infoserver_client_fd);
    
    for(iter = is_clients; iter; iter = iter->next) {
      /*
       * In order to avoid a trivial DoS attack, sending
       * requests but not reading back results, we only
       * consider a client for reading if it does not
       * have pending output
       */
      if(iter->first_pending_line) 
	continue;
      /*
       * Ok, set fd
       */
      FD_SET(iter->fd, &inputs);
      maxfd = MAX(maxfd, iter->fd);
    }

    /*
     * Possible output filedescriptors are:
     *  all client connections with pending messages
     *  all inputs with pending messages (only errors)
     */
    for(iter = is_clients; iter; iter = iter->next)
      if(iter->first_pending_line) {
	FD_SET(iter->fd, &outputs); 
	maxfd = MAX(maxfd, iter->fd);
      }

    /*
     * Now select
     */
    while(-1 == (rc = select(maxfd+1, &inputs, &outputs, 0, 0)))
      if(errno != EINTR) break;

    if(rc == -1) {
      fprintf(stderr, "InfoServer got select error: %s\n",
	      strerror(errno));
      exit(1);
    }

    /*
     * Treat new input connections
     */
    if(FD_ISSET(infoserver_input_fd, &inputs)) {
      iserv_new_input();
    }
    if(FD_ISSET(infoserver_client_fd, &inputs)) {
      iserv_new_client();
    }

    /*
     * Treat regular incoming data on existing connections
     */
    for(iter = is_clients; iter; ) {
      if(FD_ISSET(iter->fd, &inputs)) {
	FD_CLR(iter->fd, &inputs);
	iserv_receive_request(iter->fd);
	iter = is_clients;
      } else iter = iter->next;
    }
    
    /*
     * Manage pending output
     */
    for(iter = is_clients; iter; iter = iter->next) {
      if(FD_ISSET(iter->fd, &outputs)) {
	FD_CLR(iter->fd, &outputs);
	iserv_feed_client(iter);
      }
    }
  }
}

void iserv_new_input(void)
{
  int newi;
  char buf[NS_IS_MAX_REQLEN];
  char hbuf[512];
  const int itrue = 1;

  /*
   * Accept new input
   */
  newi = accept(infoserver_input_fd, 0, 0);
  if(newi < 0) {
    fprintf(stderr, "InfoServer got bad input (%s)\n",
	    strerror(errno));
    return;
  }

  if(setsockopt(newi, SOL_TCP, TCP_NODELAY, &itrue, sizeof(itrue))) {
    fprintf(stderr, "Setting TCP_NODELAY failed: %s\n", strerror(errno));
  }

  chain_client(newi);

  listen(infoserver_input_fd, 5);

  /*
   * Greet
   */
  gethostname(hbuf, 512);
  sprintf(buf, "Agent Interface to InfoServer at %s\n", hbuf);
  push_to_client(newi, buf);
}


void iserv_new_client(void)
{
  int newc;
  char buf[NS_IS_MAX_REQLEN];
  char hbuf[512];
  const int itrue = 1;
  /*
   * Accept new client
   */
  newc = accept(infoserver_client_fd, 0, 0);
  if(newc < 0) {
    fprintf(stderr, "InfoServer got bad client (%s)\n",
	    strerror(errno));
    return;
  }

  if(setsockopt(newc, SOL_TCP, TCP_NODELAY, &itrue, sizeof(itrue))) {
    fprintf(stderr, "Setting TCP_NODELAY failed: %s\n", strerror(errno));
  }

  chain_client(newc);

  listen(infoserver_client_fd, 5);

  /*
   * Greet
   */
  gethostname(hbuf, 512);
  sprintf(buf, "Welcome to the InfoServer at %s\n", hbuf);
  push_to_client(newc, buf);
}


void chain_client(int fd)
{
  NS_IS_ClientConnection * ncl;

  /*
   * Allocate new client and chain into client list 
   */
  ncl = (NS_IS_ClientConnection*)malloc(sizeof(NS_IS_ClientConnection));
  if(!ncl) {
    fprintf(stderr, "Out of memory allocating new client\n");
    return;
  }
  ncl->fd = fd;
  ncl->first_pending_line = 0;
  ncl->last_pending_line = 0;
  ncl->next = is_clients;
  is_clients = ncl;
  /*
   * Make sure client connection will not block
   */
  if(0 > fcntl(fd, F_SETFL, O_NONBLOCK)) {
    NS_IS_ClientConnection * citer, * piter = 0;
    fprintf(stderr, "InfoServer cannot make client connection not block (%s)\n",
	    strerror(errno));
    /*
     * In case we cannot make the connection non-blocking, we will simply
     * ditch the client. We cannot risk having the InfoServer blocking
     * due to a bad client.
     */
    for(citer = is_clients; citer; citer = citer->next)
      if(citer->fd) break;
      else piter = citer;
    if(piter)
      piter->next = citer->next;
    else
      is_clients = citer->next;
    close(citer->fd);
    free(citer);
  }
  /*
   * Account
   */
  n_clients++;
}

void unchain_client(int fd)
{
  NS_IS_BufLine * liter;
  NS_IS_ClientConnection * piter = 0, * iter;
  /*
   * locate
   */
  for(iter = is_clients; iter; iter = iter->next)
    if(iter->fd == fd) break;
    else piter = iter;
  if(!iter) {
    fprintf(stderr, "Tried to unchain nonexistant fd %i\n", fd);
    return;
  }
  /*
   * unchain 
   */
  if(piter)
    piter->next = iter->next;
  else
    is_clients = iter->next;
  /*
   * close 
   */
  close(iter->fd);
  /*
   * Kill buffer
   */
  while(iter->first_pending_line) {
    liter = iter->first_pending_line->next;
    free(iter->first_pending_line->line);
    free(iter->first_pending_line);
    iter->first_pending_line = liter;
  }  
  iter->first_pending_line = iter->last_pending_line = 0;
  free(iter);
  /*
   * Account
   */
  n_clients--;
}


void iserv_feed_client(NS_IS_ClientConnection* client)
{
  /*
   * Client can accept more data now
   *
   * We can only queue one request, so we we know that
   * as long as the max. request length (currently 1024bytes)
   * is reasonable, we can always send if select() says so.
   */
  int rc;
  NS_IS_BufLine * nlin;

  if(!client->first_pending_line) {
    fprintf(stderr, "Ouch! Caled iserv_feed_client with no pending lines!!\n");
    return;
  }

  rc = send(client->fd, client->first_pending_line->line, client->first_pending_line->len, 0);

  if(rc == -1 && errno == EAGAIN) {
    fprintf(stderr, "Odd - select said we could send, but sending %i bytes failed\n",
	    client->first_pending_line->len);
  } else if(rc == -1) {
    fprintf(stderr, "Error occured when feeding client (%s)\n", 
	    strerror(errno));
  } else if(rc != client->first_pending_line->len) {
    fprintf(stderr, "Odd! rc = %i  !=  pending = %i\n",
	    rc, client->first_pending_line->len);
  }

  /*
   * Kill what we sent (or what caused the failure)
   */
  nlin = client->first_pending_line->next;
  free(client->first_pending_line->line);
  client->first_pending_line->line = 0;
  free(client->first_pending_line);
  client->first_pending_line = nlin;
  if(!client->first_pending_line)
    client->last_pending_line = 0;

  return;  
}


void push_to_client(int fd, const char* line)
{
  NS_IS_ClientConnection * iter;
  NS_IS_BufLine * nlin;
  for(iter = is_clients; iter; iter = iter->next)
    if(iter->fd == fd) break;
  if(!iter) {
    fprintf(stderr, "No such client (%i)\n", fd);
    return;
  }
  /*
   * Duplicate line in client's line buffer
   */
  nlin = (NS_IS_BufLine*)malloc(sizeof(NS_IS_BufLine));
  if(!nlin) {
    fprintf(stderr, "Out of memory allocating line buffer for client output\n");
    return;
  }
  nlin->len = strlen(line);
#ifdef INDENT_OUTPUT
  nlin->line = (char*)malloc(nlin->len + 1 + 2);
#else
  nlin->line = strdup(line);
#endif
  if(!line) {
    fprintf(stderr, "Out of memory allocating line data buffer for client output\n");
    free(nlin);
    return;
  }
#ifdef INDENT_OUTPUT
  nlin->line[0] = '>';
  nlin->line[1] = ' ';
  memcpy(nlin->line + 2, line, nlin->len + 1);
  nlin->len += 2;
#endif

  nlin->next = 0;
  if(iter->last_pending_line)
    iter->last_pending_line->next = nlin;
  iter->last_pending_line = nlin;
  if(!iter->first_pending_line)
    iter->first_pending_line = nlin;

}
