/*****************************************************************/
/*      iserv_input.c                                            */
/*      Jakob Oestergaard                                        */
/*---------------------------------------------------------------*/
/*  routines for telling the InfoServer what's going on          */
/*****************************************************************/

#include "infoserver.h"
#include "infoserver_int.h"

#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <unistd.h>
#include <netdb.h>
#include <fcntl.h>
#include <assert.h>
#include <stdlib.h>

const char * infoserver_input_path = 0;
const char * infoserver_input_host = 0;
int infoserver_input_port = -1;

static int infoserver_fd = -1;

static int assure_infoserver_connection(void);

static char * obuffer = 0;

typedef struct _NS_IS_InputBuffer {
  char * key;
  char * value;
  struct _NS_IS_InputBuffer * next;
} NS_IS_InputBuffer;

static NS_IS_InputBuffer * partial_record_head = 0;
static NS_IS_InputBuffer * partial_record_tail = 0;

/*
 * For use by the agent if infoserver is remote
 */
void remote_infoserver(const char* hostname, int port)
{
  if(!(infoserver_input_host = strdup(hostname))) {
    fprintf(stderr, "Out of memory allocating hostname buffer for InfoServer\n");
    return;
  }
  if(port > 0)
    infoserver_input_port = port;
  else
    infoserver_input_port = NS_IS_INPUT_PORT;
}

static void empty_ipipe(void)
{
  /*
   * Log responses from InfoServer
   */
  if(infoserver_fd == -1)
    return;
  while(0 < recv(infoserver_fd, obuffer, NS_IS_MAX_REQLEN, 0)) {
    char * i = obuffer;
    while(*i && *i != '\r' && *i != '\n') i++;
    *i = 0;
    fprintf(stderr, "InfoServer said: \"%s\"\n", obuffer);
  }
}

/*
 * Add a keyval to the record buffer
 */
void info_add_keyval(const char* key, const char* val)
{
  NS_IS_InputBuffer * newrec;

  /*
   * Allocate buffer
   */
  if(!(newrec = (NS_IS_InputBuffer*)malloc(sizeof(NS_IS_InputBuffer)))) {
    fprintf(stderr, "Out of memory allocating buffer for \"%s\"=\"%s\"\n",
	    key, val);
    return;
  }
  /*
   * Fill in fields
   */
  if(!(newrec->key = strdup(key))) {
    fprintf(stderr, "Out of memory allocating key buffer for \"%s\"=\"%s\"\n",
	    key, val);
    return;
  }
  if(!(newrec->value = strdup(val))) {
    fprintf(stderr, "Out of memory allocating value buffer for \"%s\"=\"%s\"\n",
	    key, val);
    return;
  }
  newrec->next = 0;
  /*
   * Chain into end of partial record buffer
   */
  if(partial_record_tail) {
    assert(partial_record_head);
    partial_record_tail->next = newrec;
  } else {
    assert(!partial_record_head);
    partial_record_head = newrec;
  }
  partial_record_tail = newrec;
}

void info_add_keyval_int(const char * key, int val)
{
  /*
   * Convert and add
   */
  char ibuf[256];
  sprintf(ibuf, "%i", val);
  info_add_keyval(key, ibuf);
}

void info_add_keyval_ulong(const char * key, unsigned long val)
{
  /*
   * Convert and add
   */
  char ibuf[256];
  sprintf(ibuf, "%lu", val);
  info_add_keyval(key, ibuf);
}

void info_submit(void)
{
  int count, rc;
  NS_IS_InputBuffer * iter;
  /*
   * See if we're hooked up
   */
  if(assure_infoserver_connection())
    return;
  empty_ipipe();
  /*
   * Build string
   */
  count = sprintf(obuffer, "put ");
  for(iter = partial_record_head; iter; iter = iter->next) {
    /* 
     * Print if safe
     */
    if(strlen(iter->key) + strlen(iter->value) + 5 + count >= NS_IS_MAX_REQLEN) {
      fprintf(stderr, "Request buffer about to overflow. Skipping remaining keyvals\n");
      break;
    }
    count += sprintf(obuffer + count, "%s=\"%s\" ", iter->key, iter->value);
  }
  if(count > 0) obuffer[count-1] = ';';
  /*
   * Now remove partial record buffer
   */
  iter = partial_record_head;
  while(iter) {
    /*
     * Kill and move on
     */
    NS_IS_InputBuffer * del = iter;
    iter = iter->next;
    if(del->key) {
      free(del->key);
      del->key = 0;
    }
    if(del->value) {
      free(del->value);
      del->value = 0;
    }
    free(del);
  }
  partial_record_head = partial_record_tail = 0;

  /*
   * Send string to server
   */
  rc = send(infoserver_fd, obuffer, count, 0);
  if(count != rc) {
    if(rc == -1) {
      fprintf(stderr, "InfoServer serverevent send error (%s)\n", strerror(errno));
      close(infoserver_fd);
      infoserver_fd = -1;
    } else {
      fprintf(stderr, "Send to InfoServer mysteriously failed\n");
    }
  }
  empty_ipipe();
}


static int assure_infoserver_connection(void)
{
  if(!obuffer) {
    obuffer = (char*)malloc(NS_IS_MAX_REQLEN);
    if(!obuffer) {
      fprintf(stderr, "Not enough memory for InfoServer request buffer allocation\n");
      return -1;
    }
  }

  if(infoserver_fd != -1)
    return 0;

  /*
   * Ok, let's reconnect
   */
  if(infoserver_input_path) {
    struct sockaddr_un addr;
    /*
     * Connect to local UN*X socket
     */
    if(0 > (infoserver_fd = socket(PF_UNIX, SOCK_STREAM, 0))) {
      fprintf(stderr, "Couldn't create infoserver connection socket (%s)\n",
	      strerror(errno));
      infoserver_fd = -1;
      return -1;
    }
    addr.sun_family = AF_UNIX;
    strcpy(addr.sun_path, infoserver_input_path);
    if(0 != connect(infoserver_fd, (struct sockaddr*)&addr, sizeof(addr))) {
      fprintf(stderr, "Couldn't connect to infoserver connection (%s) socket (%s)\n",
	      infoserver_input_path, strerror(errno));
      close(infoserver_fd);
      infoserver_fd = -1;
      return -1;
    }
  } else if(infoserver_input_host) {
    struct sockaddr_in addr;
    struct hostent * ent;
    /*
     * Conect to remote TCP socket
     */
    if(0 > (infoserver_fd = socket(PF_INET, SOCK_STREAM, 0))) {
      fprintf(stderr, "Couldn't create infoserver remote connection socket (%s)\n",
	      strerror(errno));
      infoserver_fd = -1;
      return -1;
    }
    addr.sin_family = AF_INET;
    addr.sin_port = htons(infoserver_input_port);
    ent = gethostbyname(infoserver_input_host);
    if(!ent) {
      fprintf(stderr, "Cannot look up name %s\n", infoserver_input_host);
      close(infoserver_fd);
      infoserver_fd = -1;
      return -1;
    }
    memcpy(&addr.sin_addr, *ent->h_addr_list, sizeof(addr.sin_addr));
    if(0 > connect(infoserver_fd, (struct sockaddr*)&addr, sizeof(addr))) {
      fprintf(stderr, "Couldn't connect to remote server (%s)\n",
	      strerror(errno));
      close(infoserver_fd);
      infoserver_fd = -1;
      return -1;
    }
  } else {
    fprintf(stderr, "InfoServer Client doesn't call the InfoServer routines appropriately\n"
	    "There must be either a UN*X socket path *or* a remote host name\n");
    return -1;
  }

  /*
   * Never block on reads or writes
   */
  fcntl(infoserver_fd, F_SETFL, O_NONBLOCK);

  return 0;
}

