/*******************************************************************/
/* Example of the C call to NetSolve                               */
/* This program sends :                                            */
/*                                                                 */
/*    - One blocking request for the problem 'dgesv'               */
/*    - One non-blocking request for the problem 'dgesv'           */
/*                                                                 */
/*  and                                                            */
/*                                                                 */
/*    - One blocking request for the problem 'linsol'              */
/*    - One non-blocking request for the problem 'linsol'          */
/*                                                                 */
/* The problem 'linsol' is a simplified version of 'dgesv'         */
/*                                                                 */
/* The matrices are stored column-wise in a Fortran fashion        */
/*                                                                 */
/* WARNING : The matrix may be singular, in which case NetSolve    */
/*           will print out an error message.                      */
/*                                                                 */
/*******************************************************************/

#include <stdio.h>
#include "netsolve.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

main(int argc,char **argv)
{
  int m;              /* Size of the matrix and right-hand side      */
  double *a1,*b1;     /* Matrix and right-hand side for the 1st call */
  double *a2,*b2;     /* Matrix and right-hand side for the 2nd call */
  double *a3,*b3;     /* Matrix and right-hand side for the 3rd call */
  double *a4,*b4;     /* Matrix and right-hand side for the 4th call */
  int *pivot;         /* Vector of pivots returned by 'dgesv'        */
  int ierr;           /* 'dgesv' error code                          */

  int i;              /* Loop index                                  */
  int init=1325;      /* Seed of the random number generator         */
  int info;           /* NetSolve error code                         */
  int request;        /* NetSolve request handler                    */

  if (argc != 2)
  {
    fprintf(stderr,"Usage : %s <size>\n",argv[0]);
    exit(0);
  }
  if ((m = atoi(argv[1])) <= 0)
  {
    fprintf(stderr,"'%s' : Should be a positive integer\n",argv[1]);
    exit(0);
  }

  /*
   * Generating the random mxm matrices, as well as the
   * random right hand sides.
   */

  fprintf(stderr,"Generating the problem ...\n");

  a1 = (double *)malloc(m*m*sizeof(double));
  a2 = (double *)malloc(m*m*sizeof(double));
  a3 = (double *)malloc(m*m*sizeof(double));
  a4 = (double *)malloc(m*m*sizeof(double));
  for (i=0;i<m*m;i++) {
    init = 2315*init % 65536;
    a1[i] = (double)((double)init - 32768.0) / 16384.0;
    a2[i] = a1[i];  /*                                                    */
    a3[i] = a1[i];  /* In this example, we solve 4 times the same problem */
    a4[i] = a1[i];  /*                                                    */
  }

  b1 = (double *)malloc(m*sizeof(double));
  b2 = (double *)malloc(m*sizeof(double));
  b3 = (double *)malloc(m*sizeof(double));
  b4 = (double *)malloc(m*sizeof(double));
  for (i=0;i<m;i++) {
    init = 2315*init % 65536;
    b1[i] = (double)((double)init - 32768.0) / 16384.0; 
    b2[i] = b1[i];
    b3[i] = b1[i];
    b4[i] = b1[i];
  }
  pivot = (int *)malloc(m*sizeof(double));

  /* Calling Netsolve for 'dgesv' in a blocking fashion */
  /* For 'dgesv', the right-hand side is overwritten    */
  /* with the solution                                  */

  netslmajor("Col");

  fprintf(stderr,"Calling NetSolve for 'dgesv', blocking :\n");
  info = netsl("dgesv()",m,1,a1,m,pivot,b1,m,&ierr);
  if (info <0)
  {
    netslerr(info);
    exit(0);
  }
  if (ierr != 0)
    fprintf(stderr,"Cannot solve for this Matrix and right-hand side\n");
  else
  {
    fprintf(stderr,"Solution :\n");
    for (i=0;i<m;i++)
      fprintf(stderr,"--> %f\n",b1[i]);
  }

  free(a1);
  free(b1);

  /* Calling Netsolve for 'dgesv' in a non-blocking fashion */
  /* For 'dgesv', the right-hand side is overwritten        */
  /* with the solution                                      */

  fprintf(stderr,"Calling NetSolve for 'dgesv', non-blocking :\n");
  request = netslnb("dgesv()",m,1,a2,m,pivot,b2,m,&ierr);
  if (request <0)
  {
    netslerr(request);
    exit(0);
  }
  fprintf(stderr,"Request #%d being processed\n",request);
  fprintf(stderr,"Probing......\n");
  info = netslpr(request);
  while(info == NetSolveNotReady)
  {
    sleep(4);
    fprintf(stderr,".");
    fflush(stderr);
    info = netslpr(request);
  }
    fprintf(stderr,"\n");
    if (info == NetSolveOK)
    {
      info = netslwt(request);
    } 
  if (info < 0)
    netslerr(info);
  else
  {
     if (ierr != 0)
       fprintf(stderr,"Cannot solve for this Matrix and right-hand side\n");
     else
     {
       fprintf(stderr,"Solution :\n");
       for (i=0;i<m;i++)
         fprintf(stderr,"\t--> %f\n",b2[i]);
     }
  }

  free(a2);
  free(b2);
  free(pivot);

  /* Calling Netsolve for 'linsol' in a blocking fashion */
  /* For 'linsol', the right-hand side is overwritten    */
  /* with the solution                                  */

  fprintf(stderr,"Calling NetSolve for 'linsol', blocking :\n");
  info = netsl("linsol()",m,1,a3,m,b3,m);
  if (info <0)
  {
    netslerr(info);
  }
  else
  {
    fprintf(stderr,"*************\n");
    fprintf(stderr,"** Success **\n");
    fprintf(stderr,"*************\n");
    fprintf(stderr,"Solution :\n");
    for (i=0;i<m;i++)
      fprintf(stderr,"\t --> %f\n",b3[i]);
  }

  free(a3);
  free(b3);

  /* Calling Netsolve for 'linsol' in a non-blocking fashion */
  /* For 'linsol', the right-hand side is overwritten        */
  /* with the solution                                      */

  fprintf(stderr,"Calling NetSolve for 'linsol', non-blocking :\n");
  request = netslnb("linsol()",m,1,a4,m,b4,m);
  if (info <0)
  {
    netslerr(info);
    exit(0);
  }
  fprintf(stderr,"Request #%d being processed\n",request);
  fprintf(stderr,"Probing......\n");
  info = netslpr(request);
  while(info == NetSolveNotReady)
  {
    sleep(4);
    fprintf(stderr,".");
    fflush(stderr);
    info = netslpr(request);
  }
    fprintf(stderr,"\n");
    if (info == NetSolveOK)
    {
      info = netslwt(request);
    } 
  if (info < 0)
    netslerr(info);
  else
  {
    fprintf(stderr,"*************\n");
    fprintf(stderr,"** Success **\n");
    fprintf(stderr,"*************\n");
    fprintf(stderr,"Solution :\n");
    for (i=0;i<m;i++)
      fprintf(stderr,"\t--> %f\n",b4[i]);
  }

  free(a4);
  free(b4);

  return 1;
}
