/*****************************************************************/
/*      farming.c                                                */
/*      Henri Casanova                                           */
/*---------------------------------------------------------------*/
/* The example probgram using the NetSolve farming feature to    */
/* perform a 2-D block Squaring of a Square Matrix.              */
/* The command line args are "the size of the matrix" and the    */
/* block size. The smaller the block size, the bigger the number */
/* of requests to farm for a given matrix size. The program then */
/* checks the results by doing the multiplication locally.       */
/*                                                               */
/* The matrix is stored column-wise in Fortran fashion           */
/*                                                               */
/* Since this is only an example, we assume that the block size  */
/* divides the size of the matrix.                               */
/*                                                               */
/* The problem 'dmatmul' should eb availalbe on a server.        */
/*                                                               */
/*****************************************************************/

#include "netsolve.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

double *createMatrix(int);
void printMatrix(double*,int,int,int);

#define EPSILON 0.000001

/*
 * main()
 */
int main(int argc, char **argv)
{
  time_t date1,date2;
  int NSstatus;
  int n,*status,i,j;
  int blocksize;
  double *a;
  double *asquare;
  double *direct;
  int nb_requests;
  char buf[32];
  double **blockptr1;
  double **blockptr2;
  double **resultptr;
  int blocknb;
  int *blockm1;
  int *blockn1;
  int *blocklda1;
  int *blockm2;
  int *blockn2;
  int *blocklda2;
  int *resultlda;

  if (argc != 3)
  {
    fprintf(stderr,"Usage: %s <matrix size> <block size>\n",argv[0]);
    exit(-1); 
  }
  n = atoi(argv[1]);
  blocksize = atoi(argv[2]);
 
  if ((n<0) || (blocksize<0) || (blocksize>n)) 
  {
    fprintf(stderr,"Invalid arguments\n");
    return -1;
  }

  if (n % blocksize != 0)
  {
    fprintf(stderr,"Warning: using a blocksize of ");
    for (i=blocksize;i>0;i--)
    {
      if (n % i == 0)
      {
        blocksize = i;
        break;
      }
    }
    fprintf(stderr,"%d instead.\n",blocksize);
  }

  a = createMatrix(n);
  asquare = (double*)calloc(n*n,sizeof(double));
  direct = (double*)calloc(n*n,sizeof(double));

/*
  for (i=0;i<n;i++)
  {
   for (j=0;j<n;j++)
     fprintf(stderr,"%f ",a[j*n+i]);
   fprintf(stderr,"\n");
  }
*/
   
  /* Compute the number of blocks */
  blocknb = (n/blocksize);

  nb_requests = blocknb*blocknb;

  /* Allocate the farming memory */
  blockptr1 = (double **)calloc(nb_requests,sizeof(double*)); 
  blockptr2 = (double **)calloc(nb_requests,sizeof(double*)); 
  resultptr = (double **)calloc(nb_requests,sizeof(double*)); 
  blockm1 = (int *)calloc(nb_requests,sizeof(int)); 
  blockn1 = (int *)calloc(nb_requests,sizeof(int)); 
  blocklda1 = (int *)calloc(nb_requests,sizeof(int)); 
  blockm2 = (int *)calloc(nb_requests,sizeof(int)); 
  blockn2 = (int *)calloc(nb_requests,sizeof(int)); 
  blocklda2 = (int *)calloc(nb_requests,sizeof(int)); 

  resultlda = (int *)calloc(nb_requests,sizeof(int)); 

  /* Set up the farming pointers */
  for (j=0;j<blocknb;j++)  
  {
    for (i=0;i<blocknb;i++)  
    {
      resultptr[j*blocknb+i] = &(asquare[i*blocksize+j*n*blocksize]);

      blockptr1[j*blocknb+i] = &(a[i*blocksize]);
      blockm1[j*blocknb+i] = blocksize;
      blockn1[j*blocknb+i] = n;
      blocklda1[j*blocknb+i] = n;

      blockptr2[j*blocknb+i] = &(a[j*n*blocksize]);
      blockm2[j*blocknb+i] = n;
      blockn2[j*blocknb+i] = blocksize;
      blocklda2[j*blocknb+i] = n;

      resultlda[j*blocknb+i] = n;
    }
  }
      
  /* Call netsl_foral() */
  netslmajor("Col");

  sprintf(buf,"i=0,%d",blocknb*blocknb-1);
  fprintf(stderr,"Farming requests with NetSolve...\n");
  time(&date1);
  status = netsl_farm(buf,"dmatmul()",ns_int_array(blockm1,"$i"),
                                        ns_int_array(blockn2,"$i"),
                                        ns_int_array(blockn1,"$i"),
                                        ns_ptr_array((void**)blockptr1,"$i"),
                                        ns_int_array(blocklda1,"$i"),
                                        ns_ptr_array((void**)blockptr2,"$i"),
                                        ns_int_array(blocklda2,"$i"),
                                        ns_int_array(resultlda,"$i"),
                                        ns_ptr_array((void**)resultptr,"$i"));
  time(&date2);
  fprintf(stderr,"Elapsed: %d\n",(int)date2-(int)date1);
                                         
  if (status[0] != NetSolveOK)
  {
    for (i=1;i<=nb_requests;i++)
    {
      fprintf(stderr,"Requests #%d:",i);
      netslerr(status[i]);
    }
    
    netslerr(status[0]);
    exit(-1);
  }
  fprintf(stderr,"Done.\n");
 
  /* One netsolve call */
  fprintf(stderr,"Sending a SINGLE request to NetSolve...\n");
  time(&date1);
  NSstatus = netsl("dmatmul()",n,n,n,a,n,a,n,n,direct);
  if (NSstatus != NetSolveOK)
  {
    netslerr(NSstatus);
    exit(-1);
  }
  time(&date2);
  fprintf(stderr,"Done.\n");
  fprintf(stderr,"Elapsed: %d\n",(int)date2-(int)date1);

  /* Checking the result */

  for (i=0;i<n*n;i++)
  {
    if ((direct[i] - asquare[i] > EPSILON) ||
        (direct[i] - asquare[i] < -EPSILON))
    {
      fprintf(stderr,"Farming failed to compute a correct resut !!\n");
      exit(-1);
    }
  }

  fprintf(stderr," ** Success **\n");
  return 1;
}

/*
 * createMatrix()
 */
double *createMatrix(int n)
{
  int i;
  double *new;
  int init = 1325;

  new = calloc(n*n,sizeof(double));
  for (i=0;i<n*n;i++)
  {
    init = 2315*init % 65536;
    new[i] = (double)((double)init - 32768.0) / 16384.0; 
  }
  return new;
}

/*
 * printMatrix()
 */
void printMatrix(double *a,int m,int n,int lda)
{
  int i,j;

  for (i=0;i<m;i++)
  {
    for (j=0;j<n;j++)
      fprintf(stderr,"%f ",a[j*lda+i]);
    fprintf(stderr,"\n");
  }
}
