/*****************************************************************/
/*      serverdesc.c                                             */
/*      Henri Casanova                                           */
/*---------------------------------------------------------------*/
/*****************************************************************/

#include "core.h"

/*
 * freeServerDesc()
 */
void freeServerDesc(void *d)
{
  NS_ServerDesc *sd = (NS_ServerDesc *)d;
  int i;

  if (sd == NULL)
    return;

  if (sd->workload_history != NULL);
    freeWorkloadHistory(sd->workload_history);
  if (sd->network_history != NULL);
    freeNetworkHistory(sd->network_history);
  if (sd->host_desc != NULL);
    freeHostDesc(sd->host_desc);
  for (i=0;i<sd->nb_restrictions;i++)
    freeRestrictionDesc((sd->restrictions)[i]);
  if (sd->restrictions != NULL)
    free(sd->restrictions);
  if (sd->type != NULL)
    free(sd->type);
  if (sd->nb_disks != 0)
  {
    freeCharStarList(sd->disks,sd->nb_disks);
    free(sd->diskspaces);
  }
  free(sd);
}

/*
 * compServerDescKey()
 *
 * Used for the linked list
 */
int compServerDescKey(void *key,void *d)
{
  NS_IPaddr_type IPaddr;
  
  if (key == NULL)
    return 1;

  IPaddr = ((NS_ServerDesc *)d)->host_desc->IPaddr;

  if (*((NS_IPaddr_type *)key) == IPaddr)
    return 1;

  return 0;
}

/*
 * newServerDesc()
 */
NS_ServerDesc *newServerDesc()
{
  return (NS_ServerDesc *)calloc(1,sizeof(NS_ServerDesc));
}

/*
 * sendServerDesc()
 *
 * Sends a Server descriptor on a Communicator
 */
int sendServerDesc(NS_Communicator *comm, NS_ServerDesc *sd)
{
  int i;

  if (sd == NULL)
  {
    ns_errno = NetSolveInternalError; 
    return -1;
  }
  if (sendHostDesc(comm,sd->host_desc) == -1)
    return -1;
  if (sendInt(comm,sd->port) == -1)
    return -1;
  if (sendWorkloadHistory(comm,sd->workload_history) == -1)
    return -1;
  if (sendNetworkHistory(comm,sd->network_history) == -1)
    return -1;
  if (sendInt(comm,sd->nb_restrictions) == -1)
    return -1;
  for (i=0;i<sd->nb_restrictions;i++)
    if (sendRestrictionDesc(comm,(sd->restrictions)[i]) == -1)
      return -1;
  if (sendInt(comm,sd->workload_threshhold) == -1)
    return -1;
  if (sendInt(comm,sd->broadcast) == -1)
    return -1;
  if (sendInt(comm,sd->server_status) == -1)
    return -1;
  if (sendInt(comm,sd->nb_failures) == -1)
    return -1;
  if (sendString(comm,sd->type) == -1)
    return -1;
  if (sendInt(comm,sd->nb_disks) == -1)
    return -1;
  for (i=0;i<sd->nb_disks;i++)
  {
    if (sendString(comm,(sd->disks)[i]) == -1)
      return -1;
  }
  return 1;
}

/*
 * recvServerDesc()
 *
 * Receives a Server descriptor from a Communicator
 */
NS_ServerDesc *recvServerDesc(NS_Communicator *comm)
{
  NS_ServerDesc *new;
  int i;

  new = newServerDesc();
  
  new->host_desc = recvHostDesc(comm);
  if (new->host_desc == NULL)
    return NULL;
  if (recvInt(comm,&(new->port)) == -1)
    return NULL;
  new->workload_history = recvWorkloadHistory(comm);
  if (new->workload_history == NULL)
    return NULL;
  new->network_history = recvNetworkHistory(comm);
  if (new->network_history == NULL)
    return NULL;

  if (recvInt(comm,&(new->nb_restrictions)) == -1)
    return NULL;
  new->restrictions = (NS_RestrictionDesc **)calloc(new->nb_restrictions,
                               sizeof(NS_RestrictionDesc *));
  for (i=0;i<new->nb_restrictions;i++)
    if ((new->restrictions[i] = recvRestrictionDesc(comm)) == NULL)
      return NULL;

  if (recvInt(comm,&(new->workload_threshhold)) == -1)
    return NULL;
  if (recvInt(comm,&(new->broadcast)) == -1)
    return NULL;
  if (recvInt(comm,&(new->server_status)) == -1)
    return NULL;
  if (recvInt(comm,&(new->nb_failures)) == -1)
    return NULL;
  if (recvString(comm,&(new->type)) == -1)
    return NULL;
  if (recvInt(comm,&(new->nb_disks)) == -1)
    return NULL;
  new->disks = (char **)calloc(new->nb_disks,sizeof(char*));
  for (i=0;i<new->nb_disks;i++)
    if (recvString(comm,&((new->disks)[i])) == -1)
      return NULL;
  return new;
}

/*
 * serverOrder()
 */
#ifdef sunos
int serverOrder(void *s1, void *s2)
#else
int serverOrder(const void *s1, const void *s2)
#endif
{
  NS_ServerDesc **pd1 = (NS_ServerDesc **)s1;
  NS_ServerDesc **pd2 = (NS_ServerDesc **)s2;

  if (strcmp((*pd1)->host_desc->hostname,(*pd2)->host_desc->hostname) > 0)
    return 1;
  if (strcmp((*pd1)->host_desc->hostname,(*pd2)->host_desc->hostname) < 0)
    return -1;
  return 0;
}

