/*****************************************************************/
/*      expressions.c                                            */
/*      Version 1.0                      Henri Casanova          */
/*****************************************************************/

#include "core.h"
#include "mnemonics.h"
#include "expressions.h"

/*
 * getOpArg1Arg2()
 *
 * extract 'op','arg1','arg2' from an op(x,y,z) expression.
 *
 * errorcheck = 1: error checking
 * errorcheck = 0: no error checking
 *
 */
int getOpArg1Arg2(char *string,char **op,char **arg1,char **arg2,int errorcheck)
{
  int nb_parent_seen = 0;
  char *buf;
  char *tmp;
  char *to_free;
  char *tmp_string=NULL;

#ifdef DEBUG
  fprintf(stderr,"get_op_arg1_arg2:'%s'\n",string);
#endif
  if (errorcheck)
  {
    tmp_string = (char *)strdup(string);
    if (tmp_string[strlen(tmp_string)-1] != ')')
    {
      fprintf(stderr,"Unbalanced parenthesis\n");
      *op = NULL;
      *arg1 = NULL;
      *arg2 = NULL;
      free(tmp_string);
      return 1;
    }
  }

  string[strlen(string)-1] = '\0';

  to_free = buf = (char *)strdup(string+3);
  /* Get the op */
  tmp = (char *)strchr(buf,',');
  if (tmp == NULL)
  {
    *op = NULL;
    *arg1 = NULL;
    *arg2 = NULL;
    free(tmp_string);
    free(to_free);
    return 1;
  }
  *tmp = '\0';
  *op = (char *)strdup(buf);
  buf = tmp+1;
  tmp = buf;
  while(1)
  {
    if (errorcheck)
    {
      if (*tmp == '\0')
      {
        fprintf(stderr,"Unbalanced parenthesis\n");
        *arg1 = NULL;
        *arg2 = NULL;
        free(tmp_string);
        free(to_free);
        return 1;
      }
    }
    if (*tmp == '(')
      nb_parent_seen++;
    if (*tmp == ')')
      nb_parent_seen--;
    if (errorcheck)
    {
      if (nb_parent_seen <0)
      {
        fprintf(stderr,"Unbalanced parenthesis\n");
        *arg1 = NULL;
        *arg2 = NULL;
        free(tmp_string);
        free(to_free);
        return 1;
      }
    }
    if ((*tmp == ',')&&(nb_parent_seen == 0))
      break;
    tmp++;
  }
  *tmp = '\0';
  *arg1 = (char *)strdup(buf);
  buf = tmp+1;
  tmp = buf;
  nb_parent_seen = 0;
  while(1)
  {
/*
    if (*tmp == '\0')
    {
      if (errorcheck)
      {
        fprintf(stderr,"TOTA\n");
        fprintf(stderr,"Unbalanced parenthesis\n");
        *arg2 = NULL;
        free(tmp_string);
        free(to_free);
        return 1;
      }
      break;
    }
*/
    if (*tmp == '(')
      nb_parent_seen++;
    if (*tmp == ')')
      nb_parent_seen--;
    if (errorcheck)
    {
      if (nb_parent_seen <0)
      {
        fprintf(stderr,"Unbalanced parenthesis\n");
        *arg2 = NULL;
        free(tmp_string);
        free(to_free);
        return 1;
      }
    }
    if ((*tmp == '\0')&&(nb_parent_seen == 0))
      break;
    tmp++;
  }
  *tmp = '\0';
  *arg2 = (char *)strdup(buf);
  buf = tmp+1;
  if (errorcheck)
    free(tmp_string);
  free(to_free);
#ifdef DEBUG
  fprintf(stderr,"arg1 = '%s'\n",*arg1);
  fprintf(stderr,"arg2 = '%s'\n",*arg2);
#endif
  return 1;
}


/*
 * getArrayIndex()
 *
 * extract 'array','index' from an array(x,y) expression.
 *
 */
int getArrayIndex(char *string,char **array,char **index,int errorcheck)
{
  int nb_parent_seen = 0;
  char *buf;
  char *tmp;
  char *to_free;
  char *tmp_string=NULL;

#ifdef DEBUG
  fprintf(stderr,"get_array_index:'%s'\n",string);
#endif
  if (errorcheck)
  {
    tmp_string = (char *)strdup(string);
  
    if (tmp_string[strlen(tmp_string)-1] != ')')
    {
      fprintf(stderr,"Unbalanced parenthesis\n");
      *array = NULL;
      *index = NULL;
      return 1;
    }
  }
  string[strlen(string)-1] = '\0';

  to_free = buf = (char *)strdup(string+6);
  tmp = buf;
  while(1)
  {
    if (errorcheck)
    {
      if (*tmp == '\0')
      {
        fprintf(stderr,"Unbalanced parenthesis\n");
        *array = NULL;
        *index = NULL;
        free(to_free);
        free(tmp_string);
        return 1;
      }
    }
    if (*tmp == '(')
      nb_parent_seen++;
    if (*tmp == ')')
      nb_parent_seen--;
    if (errorcheck)
    {
      if (nb_parent_seen <0)
      {
        fprintf(stderr,"Unbalanced parenthesis\n ");
        *array = NULL;
        *index = NULL;
        free(to_free);
        free(tmp_string);
        return 1;
      }
    }
    if ((*tmp == ',')&&(nb_parent_seen == 0))
      break;
    tmp++;
  }
  *tmp = '\0';
  *array = (char *)strdup(buf);
  buf = tmp+1;
  tmp = buf;
  nb_parent_seen = 0;
  while(1)
  {
    if (*tmp == '\0')
    {
      if (errorcheck)
      {
        if (nb_parent_seen != 0)
        {
          fprintf(stderr,"Unbalanced parenthesis\n");
          *index = NULL;
          free(to_free);
          free(tmp_string);
          return 1;
        }
      }
      break;
    }
    if (*tmp == '(')
      nb_parent_seen++;
    if (*tmp == ')')
      nb_parent_seen--;
    if (errorcheck)
    {
      if (nb_parent_seen <0)
      {
        fprintf(stderr,"Unbalanced parenthesis\n");
        *index = NULL;
        free(to_free);
        free(tmp_string);
        return 1;
      }
    }
    if ((*tmp == ',')&&(nb_parent_seen == 0))
      break;
    tmp++;
  }
  *tmp = '\0';
  *index = (char *)strdup(buf);
  buf = tmp+1;
  if (errorcheck)
    free(tmp_string);
  free(to_free);
#ifdef DEBUG
  fprintf(stderr,"array = '%s'\n",*array);
  fprintf(stderr,"index = '%s'\n",*index);
#endif
  return 1;
}

/*
 * getIfThenElse()
 */
int getIfThenElse(char *string,char **lhs,char **rhs,
                char **ifstmt,char **elsestmt,int errorcheck)
{
  int nbpar = 0;
  int length;
  int comma[2];
  int nb=0;
  int i;
  int egal = 0;
  char *start;

#ifdef DEBUG
  fprintf(stderr,"get_if_then_else:'%s'\n",string);
#endif
  for (i=0;i<2;i++)
    comma[i] = 0;

  start = (char *)strdup(string);
  length = strlen(start);
  start[--length] = '\0';

  /* Looking for ','s */

  for (i=0;i<length;i++)
  {
    if (start[i] == ',')
      if (nbpar == 0)
      {
        if (nb == 2)
        {
          if (errorcheck)
          {
            fprintf(stderr,"An if expression must be of the form :\n");
            fprintf(stderr," if(<expr1>,<expr2>,<expr3>)\n");
          }
          free(start);
          return -1;
        }
        else
          comma[nb++] = i;
      }
    if (start[i] == '(')
      nbpar++;
    if (start[i] == ')')
      nbpar--;
  }
  if ((comma[0] == 0)||
      (comma[1] == 0)||
      (comma[0] == comma[1])||
      (nbpar != 0))
  {
    if (errorcheck)
    {
      fprintf(stderr,"An if expression must be of the form :\n");
      fprintf(stderr," if(<expr1>,<expr2>,<expr3>)\n");
    }
    free(start);
    return -1;
  }

  /* now, we have the 'if' and 'else' */

  *(start+comma[0]) = '\0';
  *(start+comma[1]) = '\0';
  *ifstmt   = (char *)strdup(start+comma[0]+1);
  *elsestmt = (char *)strdup(start+comma[1]+1);

  /* Deal with the expression */

  for (i=0;i<comma[0];i++)
  {
    if (start[i] == '=')
      if (nbpar == 0)
        egal = i;
    if (start[i] == '(')
      nbpar++;
    if (start[i] == ')')
      nbpar--;
  }
  if ((nbpar != 0)||(egal == 0))
  {
    if (errorcheck)
    {
      fprintf(stderr,"The expression must be of the form :\n");
      fprintf(stderr,"   <expr1> = <expr2>\n");
    }
    free(start);
    return -1;
  }
  *(start+egal) = '\0';
  *lhs = (char *)strdup(start);
  *rhs = (char *)strdup(start+egal+1);
  free(start);
#ifdef DEBUG
  fprintf(stderr,"IF: '%s=%s'\n",*lhs,*rhs);
  fprintf(stderr,"THEN: '%s'\n",*ifstmt);
  fprintf(stderr,"ELSE: '%s'\n",*elsestmt);
#endif
  return 1;
}

/*
 * evalFormula()
 */
int evalFormula(NS_ProblemDesc *pd,char *formula)
{
  if (!strncmp(formula,"op(",3))
  {
    char *op,*arg1,*arg2;
    int val1=0;
    int val2=0;
    int result=0;
    getOpArg1Arg2(formula,&op,&arg1,&arg2,0);
    val1 = evalFormula(pd,arg1);
    val1 = evalFormula(pd,arg2);
    if (!strcmp(op,"+"))
      result = val1+val2;
    else if (!strcmp(op,"-"))
      result = val1-val2;
    else if (!strcmp(op,"*"))
      result = val1*val2;
    free(op);free(arg1);free(arg2);
    return result;
  }
  else if (!strncmp(formula,"array(",6))
  {
    char *array,*index;
    int indexval;
    int result=0;
    NS_Object *obj;
    int objnumber,IorO;
    getArrayIndex(formula,&array,&index,0);
    indexval = evalFormula(pd,index);
    objnumber = getIndex(array);
    IorO = getIO(array);
    if (IorO == INPUT)
      obj = pd->input_objects[objnumber];
    else
      obj = pd->output_objects[objnumber];
    switch(obj->object_type)
    {
      case NETSOLVE_MATRIX:
        switch(obj->data_type)
        {
          case NETSOLVE_I:
            result = ((int*)(obj->attributes.matrix_attributes.ptr))[indexval];
            break;
          case NETSOLVE_CHAR:
            result = (int)(((char*)(obj->attributes.matrix_attributes.ptr))[indexval]);
            break;
        }
      case NETSOLVE_SPARSEMATRIX:
        switch(obj->data_type)
        {
          case NETSOLVE_I:
            result = ((int*)(obj->attributes.sparsematrix_attributes.ptr))[indexval];
            break;
          case NETSOLVE_CHAR:
            result = (int)(((char*)(obj->attributes.sparsematrix_attributes.ptr))[indexval]);
            break;
        }
      case NETSOLVE_VECTOR:
        switch(obj->data_type)
        {
          case NETSOLVE_I:
            result = ((int*)(obj->attributes.vector_attributes.ptr))[indexval];
            break;
          case NETSOLVE_CHAR:
            result = (int)(((char*)(obj->attributes.vector_attributes.ptr))[indexval]);
            break;
        }
      case NETSOLVE_SCALAR:
        switch(obj->data_type)
        {
          case NETSOLVE_I:
            result = ((int*)(obj->attributes.scalar_attributes.ptr))[indexval];
            break;
          case NETSOLVE_CHAR:
            result = (int)(((char*)(obj->attributes.scalar_attributes.ptr))[indexval]);
            break;
        }
    }
    free(array);free(index);
    return result;
  }
  else if (!strncmp(formula,"'",1))
  {
    return (int)(formula[1]);
  }
  else if (!strncmp(formula,"if(",3))
  {
    char *lhs,*rhs,*ifstmt,*elsestmt;
    int lhv,rhv,result;

    getIfThenElse(formula+3,&lhs,&rhs,&ifstmt,&elsestmt,0);
    lhv = evalFormula(pd,lhs);
    rhv = evalFormula(pd,rhs);
    if (lhv == rhv)
      result = evalFormula(pd,ifstmt);
    else
      result = evalFormula(pd,elsestmt);
    free(lhs);free(rhs);free(ifstmt);free(elsestmt);
    return result; 
  }
  else if (isInteger(formula))  
  {
    return atoi(formula);
  }
  else /* MNEMONIC */
  {
    int index = getIndex(formula);
    int IorO = getIO(formula);
    NS_Object *obj;
  
    if (IorO == INPUT)
      obj = pd->input_objects[index];
    else 
      obj = pd->output_objects[index];

    switch(formula[0])
    {
      case 'm':
        switch(obj->object_type)
        {
          case NETSOLVE_MATRIX:
            return obj->attributes.matrix_attributes.m;
          case NETSOLVE_SPARSEMATRIX:
            return obj->attributes.sparsematrix_attributes.m;
          case NETSOLVE_VECTOR:
            return obj->attributes.vector_attributes.m;
          case NETSOLVE_PACKEDFILES:
            return obj->attributes.packedfiles_attributes.m;
        }
        break;
      case 'n':
        switch(obj->object_type)
        {
          case NETSOLVE_MATRIX:
            return obj->attributes.matrix_attributes.n;
          case NETSOLVE_SPARSEMATRIX:
            return obj->attributes.sparsematrix_attributes.n;
        }
      case 'l':
        switch(obj->object_type)
        {
          case NETSOLVE_MATRIX:
            return obj->attributes.matrix_attributes.l;
        }
      case 'f':
        switch(obj->object_type)
        {
          case NETSOLVE_SPARSEMATRIX:
            return obj->attributes.sparsematrix_attributes.f;
        }
    }
     
  }
  return -1;
}
