/*****************************************************************/
/*      communicator.c                                           */
/*      Henri Casanova                                           */
/*---------------------------------------------------------------*/
/*****************************************************************/

#ifndef WIN32
#include <rpc/types.h>
#include <rpc/xdr.h>
#include "core.h"
#include <fcntl.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <unistd.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <signal.h>
#include <errno.h>
#endif

#ifdef WIN32
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <io.h>
#include <signal.h>
#include "types.h"
#include "xdr.h"
#include "core.h"
#endif

bool_t xdr_scomplex(XDR *,scomplex *);
bool_t xdr_dcomplex(XDR *,dcomplex *);

/**
 ** Global variables for the xdr sizes
 **/
int XDR_SIZEOF_INT;
int XDR_SIZEOF_FLOAT;
int XDR_SIZEOF_DOUBLE;
int XDR_SIZEOF_CHAR;
int XDR_SIZEOF_SCOMPLEX;
int XDR_SIZEOF_DCOMPLEX;
int XDR_SIZEOF_U_CHAR;

#ifndef WIN32
/*
 * private prototypes
 */
#ifdef sunos
int twrite(int,char*,unsigned);
#else
ssize_t twrite(int,const void*,size_t);
#endif
int tread(int,char*,int);
#endif

/*
 * setXDRSizes()
 *
 * Computes the sizes of diverse data types once they are
 * XDR encoded. Should be done with the function xdr_sizeof(),
 * but it is not available on all platforms (e.g. ALPHA)
 */
int setXDRSizes()
{
  XDR xdr_stream;
  char buffer[128];
  int cc=1;
  int t_int=2;
  float t_float=2.0;
  double t_double=2.0;
  scomplex t_scomplex;
  dcomplex t_dcomplex;
  char t_char='a';
  unsigned char t_u_char='a';

  t_scomplex.r = 2.0;
  t_scomplex.i = 2.0;
  t_dcomplex.r = 2.0;
  t_dcomplex.i = 2.0;
  
  xdrmem_create(&xdr_stream,buffer,128,XDR_FREE);
  xdrmem_create(&xdr_stream,buffer,128,XDR_ENCODE);
  cc = cc && xdr_int(&xdr_stream,&t_int);
  XDR_SIZEOF_INT = xdr_getpos(&xdr_stream);
  xdr_destroy(&xdr_stream);

  xdrmem_create(&xdr_stream,buffer,128,XDR_ENCODE);
  cc = cc && xdr_float(&xdr_stream,&t_float);
  XDR_SIZEOF_FLOAT = xdr_getpos(&xdr_stream);
  xdr_destroy(&xdr_stream);

  xdrmem_create(&xdr_stream,buffer,128,XDR_FREE);
  xdrmem_create(&xdr_stream,buffer,128,XDR_ENCODE);
  cc = cc && xdr_double(&xdr_stream,&t_double);
  XDR_SIZEOF_DOUBLE = xdr_getpos(&xdr_stream);
  xdr_destroy(&xdr_stream);

  xdrmem_create(&xdr_stream,buffer,128,XDR_FREE);
  xdrmem_create(&xdr_stream,buffer,128,XDR_ENCODE);
  cc = cc && xdr_char(&xdr_stream,&t_char);
  XDR_SIZEOF_CHAR = xdr_getpos(&xdr_stream);
  xdr_destroy(&xdr_stream);

  xdrmem_create(&xdr_stream,buffer,128,XDR_FREE);
  xdrmem_create(&xdr_stream,buffer,128,XDR_ENCODE);
  cc = cc && xdr_scomplex(&xdr_stream,&t_scomplex);
  XDR_SIZEOF_SCOMPLEX = xdr_getpos(&xdr_stream);
  xdr_destroy(&xdr_stream);

  xdrmem_create(&xdr_stream,buffer,128,XDR_FREE);
  xdrmem_create(&xdr_stream,buffer,128,XDR_ENCODE);
  cc = cc && xdr_dcomplex(&xdr_stream,&t_dcomplex);
  XDR_SIZEOF_DCOMPLEX = xdr_getpos(&xdr_stream);
  xdr_destroy(&xdr_stream);

  xdrmem_create(&xdr_stream,buffer,128,XDR_FREE);
  xdrmem_create(&xdr_stream,buffer,128,XDR_ENCODE);
#ifndef PLAT_T3E
  cc = cc && xdr_u_char(&xdr_stream,&t_u_char);
#else
  cc = cc && xdr_u_char(&xdr_stream,(char *)(&t_u_char));
#endif
  XDR_SIZEOF_U_CHAR = xdr_getpos(&xdr_stream);
  xdr_destroy(&xdr_stream);

  if (cc == 0)
  {
    ns_errno = NetSolveSystemError;
    return -1;
  }
  else
    return 1;
}

#ifndef WIN32
/*
 * tread()
 * 
 * reads n bytes from the socket sock. blocks until the n bytes are
 * received.
 * returns the number of bytes read, or -1 if failure.
 */
int tread(int d,char *b,int n)
{
  int e;
  int x = n;
  static int sig=1;

  if (sig)
  {
    (void)signal(SIGPIPE, SIG_IGN);
    sig = 0;
  }

  while (x > 0) {
    if ((e = read(d, b, x)) == -1)
      return e;
    if (!e)
      return n - x;
    b += e;
    x -= e;
  }
  return n - x;
}
#endif

#ifndef WIN32
/*
 * twrite()
 */
#ifdef sunos
int twrite(int fildes, char *buf,unsigned nbyte)
#else
ssize_t twrite(int fildes, const void *buf,size_t nbyte)
#endif
{
  static int sig=1;
  if (sig)
  {
    (void)signal(SIGPIPE, SIG_IGN);
    sig=0;
  }
  return write(fildes,buf,nbyte);
}
#endif


/*
 * xdr_scomplex()
 *
 * XDR encode a single precision complex
 */
bool_t xdr_scomplex(XDR *xdrptr,scomplex *ptr)
{
  return (xdr_float(xdrptr,&ptr->r) &&
          xdr_float(xdrptr,&ptr->i));
}

/*
 * xdr_dcomplex()
 *
 * XDR encode a double precision complex
 */
bool_t xdr_dcomplex(XDR *xdrptr,dcomplex *ptr)
{
  return (xdr_double(xdrptr,&ptr->r) &&
          xdr_double(xdrptr,&ptr->i));
}


/*
 * newCommunicator()
 *
 * allocate the space for the communicator and
 * assigns the fields
 */
NS_Communicator *newCommunicator(NS_Socket_type sock,int encoding)
{
  NS_Communicator *comm;
  static int firsttime=1;

  if (encoding == DATA_XDR)
  {
    if (firsttime) 
    {
      setXDRSizes();
      firsttime = 0;
    }
  }
  comm = (NS_Communicator *)calloc(1,sizeof(NS_Communicator));
  comm->sock = sock;
  comm->encoding = encoding;
  return comm;
}

/*
 * initTransaction()
 *
 * initiates a communication by sending a u_short 
 * or a u_long (depending on NTOHS_SCHEME_[CRAY|SUN],
 * under the network format to signal if XDR 
 * is being used or not. Returns a new communicator.
 */
NS_Communicator *initTransaction(NS_Socket_type sock,int encoding)
{
  if (send8BitFlag(sock,encoding) == -1)
    return NULL;
  else
    return newCommunicator(sock,encoding);
}

/*
 * acceptTransaction()
 *
 * Receives a u_short or a u_long depending on
 * NTOHS_SCHEME_[SUN|CRAY] and determines the encoding.
 * returns a new communicator.
 */
NS_Communicator *acceptTransaction(NS_Socket_type sock)
{
  int encoding;

  if (recv8BitFlag(sock,&encoding) == -1)
    return NULL;
  else
    return newCommunicator(sock,encoding);
}

/*
 * recv8BitFlag()
 */
int recv8BitFlag(NS_Socket_type sock, int *encoding)
{
  char flag; /* sizeof(char) >= 1 */
  int cc;

#ifndef WIN32
  cc = tread(sock,&flag,1);
#endif
#ifdef WIN32
  cc = ReceiveData(sock,&flag,1);
#endif
  CHECK_TIMEOUT(cc);
  if (cc != 1)
  {
    ns_errno = NetSolveNetworkError;
    return -1;
  }
  /* Process the flag */
  if (flag == 0)
    *encoding = DATA_RAW;
  else
    *encoding = DATA_XDR;
  return 0;
}

/*
 * send8BitFlag()
 */
int send8BitFlag(NS_Socket_type sock, int encoding)
{
  char flag; /* sizeof(char) >= 1 */
  int cc;
  int mask0 = 0;
  int mask1 = ~0;

  /* Create the flag */
  if (encoding == DATA_RAW)
    BCOPY(&mask0,&flag,1);
  if( encoding == DATA_XDR)
    BCOPY(&mask1,&flag,1);

#ifndef WIN32
  cc = twrite(sock,&flag,1);
  CHECK_TIMEOUT(cc);
#endif
#ifdef WIN32
  cc = SendData(sock,&flag,1);
#endif
  if (cc != 1)
  {
    ns_errno = NetSolveNetworkError;
    return -1;
  }
  return 0;
}

/*
 * endTransaction()
 *
 * Terminates a transaction, kill the connection
 * and free the Communicator.
 */
void endTransaction(NS_Communicator *comm)
{
  closeSocket(comm->sock);
  free(comm);
}

/*
 * netsolve_sizeof()
 *
 * Returns the size in bytes of Netsolve data types
 */
int netsolve_sizeof(int data_type)
{
  switch(data_type)
  {
    case NETSOLVE_I:
      return sizeof(int);
    case NETSOLVE_D:
      return sizeof(double);
    case NETSOLVE_S:
      return sizeof(float);
    case NETSOLVE_C:
      return sizeof(scomplex);
    case NETSOLVE_Z:
      return sizeof(dcomplex);
    case NETSOLVE_CHAR:
      return sizeof(char);
    case NETSOLVE_B:
      return sizeof(char);
    default:
      ns_errno = NetSolveInternalError;
      return -1;
  }
}

/*
 * netsolve_xdrsizeof()
 *
 * Returns the size in bytes of Netsolve data types
 * once they are XDR encoded
 */
int netsolve_xdrsizeof(int data_type)
{
  switch(data_type)
  {
    case NETSOLVE_I:
      return XDR_SIZEOF_INT;
    case NETSOLVE_D:
      return XDR_SIZEOF_DOUBLE;
    case NETSOLVE_S:
      return XDR_SIZEOF_FLOAT;
    case NETSOLVE_C:
      return XDR_SIZEOF_SCOMPLEX;
    case NETSOLVE_Z:
      return XDR_SIZEOF_DCOMPLEX;
    case NETSOLVE_CHAR:
      return XDR_SIZEOF_CHAR;
    case NETSOLVE_B:
      return sizeof(char);
    default:
      ns_errno = NetSolveInternalError; 
      return -1;
  }
}

/*
 * sendArray()
 *
 * Send an array of typed data on the current communicator
 */
int sendArray(NS_Communicator *comm,int data_type,void *data,int nb)
{
  XDR xdr_stream;
  char *buffer = NULL;
  void *tosend = NULL;
  int size;
  int cc=1;

  if (comm->encoding == DATA_XDR)
    size = nb*netsolve_xdrsizeof(data_type);
  else
    size = nb*netsolve_sizeof(data_type);

  if (comm->encoding == DATA_XDR)
  {
    buffer = (char *)calloc(size,sizeof(char));
    xdrmem_create(&xdr_stream,buffer,size,XDR_FREE);
    xdrmem_create(&xdr_stream,buffer,size,XDR_ENCODE);
    
    switch(data_type)
    {
    case NETSOLVE_I:
      if (xdr_vector(&xdr_stream,data,nb,sizeof(int),(xdrproc_t)xdr_int) != 1)
        cc =  -1;
      tosend = buffer;
      break;
    case NETSOLVE_S:
      if (xdr_vector(&xdr_stream,data,nb,sizeof(float),(xdrproc_t)xdr_float) != 1)
        cc =  -1;
       tosend = buffer;
       break;
    case NETSOLVE_D:
      if (xdr_vector(&xdr_stream,data,nb,sizeof(double),(xdrproc_t)xdr_double) != 1)
        cc = -1;
      tosend = buffer;
      break;
    case NETSOLVE_CHAR:
      if (xdr_vector(&xdr_stream,data,nb,sizeof(char),(xdrproc_t)xdr_char) != 1)
        cc = -1;
      tosend = buffer;
      break;
    case NETSOLVE_C:
      if (xdr_vector(&xdr_stream,data,nb,sizeof(scomplex),(xdrproc_t)xdr_scomplex) != 1)
        cc = -1;
      tosend = buffer;
      break;
    case NETSOLVE_Z:
      if (xdr_vector(&xdr_stream,data,nb,sizeof(dcomplex),(xdrproc_t)xdr_dcomplex) != 1)
        cc = -1;
      tosend = buffer;
      break;
    case NETSOLVE_B: /* No XDR encoding there */
      tosend = data;
      break;
    }
  }
  else
  {
    tosend = data;
  }

  if (cc == -1)
  {
    if (comm->encoding == DATA_XDR)
    {
      free(buffer);
      xdr_destroy(&xdr_stream);
    }
    ns_errno = NetSolveNetworkError;
    return -1;
  }

#ifndef WIN32
  cc = twrite(comm->sock,tosend,size);
  CHECK_TIMEOUT(cc);
#endif
#ifdef WIN32
  cc = SendData(comm->sock,tosend,size);
#endif

  if (comm->encoding == DATA_XDR)
  {
    free(buffer);
    xdr_destroy(&xdr_stream);
  }
  if (cc != size)
  {
    ns_errno = NetSolveNetworkError;
    return -1;
  }
  else
    return nb;
}

/*
 * recvArray()
 *
 * Receive an array of typed data on the current communicator
 */
int recvArray(NS_Communicator *comm,int data_type,void *data,int nb)
{
  XDR xdr_stream;
  char *buffer = NULL;
  int size;
  int cc;
  char *torecv;

  if (comm->encoding == DATA_XDR)
    size = nb*netsolve_xdrsizeof(data_type);
  else
    size = nb*netsolve_sizeof(data_type);

  if (comm->encoding == DATA_XDR)
  {
    switch(data_type)
    {
      case NETSOLVE_B:
        torecv = data;
        break;
      default:
        buffer = (char *)calloc(size,sizeof(char));
        torecv = buffer;
        break; 
    }
  }
  else
  {
    torecv = data;
  }

#ifndef WIN32
  cc = tread(comm->sock,torecv,size);
  CHECK_TIMEOUT(cc);
  if (cc == -1)
  {
    if (buffer)
      free(buffer);
    ns_errno = NetSolveNetworkError;
    return -1;
  }
#endif /* UNIX */
#ifdef WIN32
  cc = ReceiveData(comm->sock,torecv,size);
  if (cc == SOCKET_ERROR) 
  {
    if (buffer)
      free(buffer);
    ns_errno = NetSolveNetworkError;
    return -1;
  }
#endif
 
  if (comm->encoding == DATA_XDR)
  {
    /* (jakob) XXX: If buffer is 0, xdr_vector will segfault */
    xdrmem_create(&xdr_stream,buffer,size,XDR_FREE);
    xdrmem_create(&xdr_stream,buffer,size,XDR_DECODE);
    
    switch(data_type)
    {
    case NETSOLVE_I:
      if (xdr_vector(&xdr_stream,data,nb,sizeof(int),(xdrproc_t)xdr_int) != 1)
        cc = -1;
      break;
    case NETSOLVE_S:
      if (xdr_vector(&xdr_stream,data,nb,sizeof(float),(xdrproc_t)xdr_float) != 1)
        cc = -1;
      break;
    case NETSOLVE_D:
      if (xdr_vector(&xdr_stream,data,nb,sizeof(double),(xdrproc_t)xdr_double) != 1)
        cc = -1;
      break;
    case NETSOLVE_CHAR:
      if (xdr_vector(&xdr_stream,data,nb,sizeof(char),(xdrproc_t)xdr_char) != 1)
        cc = -1;
      break;
    case NETSOLVE_C:
      if (xdr_vector(&xdr_stream,data,nb,sizeof(scomplex),(xdrproc_t)xdr_scomplex) != 1)
        cc = -1;
      break;
    case NETSOLVE_Z:
      if (xdr_vector(&xdr_stream,data,nb,sizeof(dcomplex),(xdrproc_t)xdr_dcomplex) != 1)
        cc = -1;
      break;
    case NETSOLVE_B: /* No XDR encoding there */
      /* Nothing */
      break;
    }
  }

  if (comm->encoding == DATA_XDR)
  {
    free(buffer);
    xdr_destroy(&xdr_stream);
  }
  if (cc == size)
    return nb;
  else 
  {
    ns_errno = NetSolveNetworkError;
    return -1;
  }
}

/* 
 * sendInt()
 *
 * Sends an integer (to make the code look nicer)
 */
int sendInt(NS_Communicator *comm,int data)
{
  int tmp = data;
  return sendArray(comm,NETSOLVE_I,&tmp,1);
}

/*
 * recvInt()
 *
 * Receives an integer (to make the code look nicer)
 */
int recvInt(NS_Communicator *comm,int *ptr)
{
  return recvArray(comm,NETSOLVE_I,ptr,1);
}

/*
 * sendString()
 *
 * Send a string, prepended with its length. Returns the number
 * of bytes sent.
 */
int sendString(NS_Communicator *comm,char *s)
{
  int size;
  char *tosend;
 
  if (s == NULL)
    tosend = strdup(""); 
  else
    tosend = s;

  size = strlen(tosend)+1;
 
  if (sendInt(comm,size) != 1)
  {
    ns_errno = NetSolveNetworkError;
    return -1;
  }
  if (sendArray(comm,NETSOLVE_CHAR,tosend,size) != size)
  {
    ns_errno = NetSolveNetworkError;
    return -1;
  }
  if (s == NULL)
    free(tosend);
  return size;
}

/*
 * recvString()
 *
 * Receives a string, and allocates the space for it !!
 * Returnes the number of bytes received.
 */
int recvString(NS_Communicator *comm, char **s)
{
  int size;

  if (recvInt(comm,&size) != 1)
  {
    ns_errno = NetSolveNetworkError;
    return -1;
  }
  *s = (char *)calloc(size,sizeof(char));
  /* (jakob) XXX:  We don't check for null pointer return ! */
  if (recvArray(comm,NETSOLVE_CHAR,*s,size) != size)
  {
    free(*s);
    *s = NULL;
    return -1;
  }
  return size;
}


/*
 * sendIPaddr
 *
 * Sends an IPaddr. Takes care of the XDR screw up.
 */
int sendIPaddr(NS_Communicator *comm,NS_IPaddr_type *IPaddr)
{
  XDR xdr_stream;
  char *buffer = NULL;
  int size;
  void *tosend;
  int cc=0;

  if (comm->encoding == DATA_XDR)
  {
    size = 4*XDR_SIZEOF_U_CHAR; 
    buffer = (char *)calloc(size,sizeof(char));
    xdrmem_create(&xdr_stream,buffer,size,XDR_FREE);
    xdrmem_create(&xdr_stream,buffer,size,XDR_ENCODE);
#ifndef PLAT_T3E
    if (xdr_u_char(&xdr_stream,&(((unsigned char *)IPaddr)[0])) != 1)
      cc = -1;
    if (xdr_u_char(&xdr_stream,&(((unsigned char *)IPaddr)[1])) != 1)
      cc = -1;
    if (xdr_u_char(&xdr_stream,&(((unsigned char *)IPaddr)[2])) != 1)
      cc = -1;
    if (xdr_u_char(&xdr_stream,&(((unsigned char *)IPaddr)[3])) != 1)
      cc = -1;
#else
    if (xdr_u_char(&xdr_stream,&(((char *)IPaddr)[0])) != 1)
      cc = -1;
    if (xdr_u_char(&xdr_stream,&(((char *)IPaddr)[1])) != 1)
      cc = -1;
    if (xdr_u_char(&xdr_stream,&(((char *)IPaddr)[2])) != 1)
      cc = -1;
    if (xdr_u_char(&xdr_stream,&(((char *)IPaddr)[3])) != 1)
      cc = -1;
#endif
    tosend = buffer;
  }
  else
  {
    size = sizeof(NS_IPaddr_type);
    tosend = IPaddr;
  }

  if (cc == -1)
  {
    if (comm->encoding == DATA_XDR)
    {
      free(buffer);
      xdr_destroy(&xdr_stream);
    }
    ns_errno = NetSolveNetworkError;
    return -1;
  }

#ifndef WIN32
  cc = twrite(comm->sock,tosend,size);
  CHECK_TIMEOUT(cc);
#endif
#ifdef WIN32
  cc = SendData(comm->sock,tosend,size);
#endif
  if (comm->encoding == DATA_XDR)
  {
    free(buffer);
    xdr_destroy(&xdr_stream);
  }
  if (cc == size)
    return 1;
  else
  {
    ns_errno = NetSolveNetworkError;
    return -1; 
  }
}


/*
 * recvIPaddr
 *
 * Receives an IPaddr. Takes care of the XDR screw up.
 */
int recvIPaddr(NS_Communicator *comm,NS_IPaddr_type *IPaddr)
{
  XDR xdr_stream;
  char *buffer = NULL;
  int size;
  void *torecv;
  int cc;

  
  if (comm->encoding == DATA_XDR)
  {
    size = 4*XDR_SIZEOF_U_CHAR; 
    buffer = (char *)calloc(size,sizeof(char));
    torecv = buffer;
  }
  else
  {
    size = sizeof(NS_IPaddr_type);
    torecv = IPaddr; 
  }

#ifndef WIN32
  cc = tread(comm->sock,torecv,size);
  CHECK_TIMEOUT(cc);
  if (cc == -1)
  {
    if (buffer)
      free(buffer);
    ns_errno = NetSolveNetworkError;
    return -1;
  } 
#endif
#ifdef WIN32
  cc = ReceiveData(comm->sock,torecv,size);
  if (cc == SOCKET_ERROR)
  {
    if (buffer)
      free(buffer);
    ns_errno = NetSolveNetworkError;
    return -1;
  } 
#endif

  if (comm->encoding == DATA_XDR)
  {
    xdrmem_create(&xdr_stream,buffer,size,XDR_FREE);
    xdrmem_create(&xdr_stream,buffer,size,XDR_DECODE);
#ifndef PLAT_T3E
    if (xdr_u_char(&xdr_stream,&(((unsigned char *)IPaddr)[0])) != 1)
      cc = -1;
    if (xdr_u_char(&xdr_stream,&(((unsigned char *)IPaddr)[1])) != 1)
      cc = -1;
    if (xdr_u_char(&xdr_stream,&(((unsigned char *)IPaddr)[2])) != 1)
      cc = -1;
    if (xdr_u_char(&xdr_stream,&(((unsigned char *)IPaddr)[3])) != 1)
      cc = -1;
#else
    if (xdr_u_char(&xdr_stream,&(((char *)IPaddr)[0])) != 1)
      cc = -1;
    if (xdr_u_char(&xdr_stream,&(((char *)IPaddr)[1])) != 1)
      cc = -1;
    if (xdr_u_char(&xdr_stream,&(((char *)IPaddr)[2])) != 1)
      cc = -1;
    if (xdr_u_char(&xdr_stream,&(((char *)IPaddr)[3])) != 1)
      cc = -1;
#endif
  }

  if (comm->encoding == DATA_XDR)
  {
    free(buffer);
    xdr_destroy(&xdr_stream);
  }
  if (cc == -1)
  {
    ns_errno = NetSolveNetworkError;
    return -1;
  }
  else
    return 1;
}

/*
 * sendArrayFromFile
 *
 * Sends an array from a file to the communicator.
 */
int sendArrayFromFile(NS_Communicator *comm,int data_type,int fd,int nb)
{
  char buffer[BUFFSIZE];
  int count=nb; 
  int maximum=BUFFSIZE/netsolve_sizeof(data_type);
  int loaded;

  while(count != 0)
  {
    /* Load data in the buffer from the file */
    loaded = MIN(maximum,count);
    if (read(fd,buffer,loaded*netsolve_sizeof(data_type)) !=
             netsolve_sizeof(data_type)*loaded)
    {
      ns_errno = NetSolveFileError;
      return -1;
    }
    /* Send that on the communicator */
    if (sendArray(comm,data_type,buffer,loaded) != loaded)
      return -1;
    count-=loaded;
  }
  return nb;
}

/*
 * revcArrayToFile
 *
 * Receives an arry from the communicator into a file
 */
int recvArrayToFile(NS_Communicator *comm,int data_type,int fd,int nb)
{
  char buffer[BUFFSIZE];
  int count=nb;
  int maximum=BUFFSIZE/netsolve_sizeof(data_type);
  int loaded;

  while(count != 0)
  {
    /* Receive data on the communicator */
    loaded = MIN(maximum,count);
    if (recvArray(comm,data_type,buffer,loaded) != loaded)
      return -1;
    /* Store that from the buffer into the file */
    if (write(fd,buffer,loaded*netsolve_sizeof(data_type)) !=
             netsolve_sizeof(data_type)*loaded)
    {
      ns_errno = NetSolveFileError;
      return -1;
    }
    count-=loaded;
  }
  return nb;
}

/*
 * sendFileAsString()
 */
int sendFileAsString(NS_Communicator *comm,char *filename)
{
  char *buffer;
  struct stat st;
  int fd;

  if (stat(filename,&st))
  {
    ns_errno = NetSolveFileError;
    return -1;
  }

  fd = open(filename,O_RDONLY,0666);
  if (fd < 0)
  {
    ns_errno = NetSolveFileError;
    return -1; 
  }

  buffer = (char *)calloc(st.st_size+1,sizeof(char));
  if (read(fd,buffer,st.st_size*sizeof(char)) != st.st_size*sizeof(char))
  {
    perror("read()");
    free(buffer);
    close(fd);
    ns_errno = NetSolveFileError;
    return -1;
  }
  if (sendString(comm,buffer) == -1)
  {
    free(buffer);
    close(fd);
    return -1;
  }
  free(buffer);
  close(fd);
  return 1;
}
 
int recvDsiObj(NS_Communicator *comm,DSI_OBJECT* dsi_obj){
  /* receive parameters for the distributed object */

#ifdef DSI_IBP
char* readCap;
char* writeCap;
char* manageCap;
char* host;
int port;
#endif

  if (recvString(comm,&(dsi_obj->name)) == -1)
    return -1;

  if (recvInt(comm,&(dsi_obj->dsi_file->flag)) == -1)
    return -1;
  if (recvInt(comm,&(dsi_obj->dsi_file->perm)) == -1)
    return -1;
  if (recvInt(comm,(int*)&(dsi_obj->dsi_file->uid)) == -1)
    return -1;
  if (recvInt(comm,&(dsi_obj->dsi_file->storage_system)) == -1)
    return -1;

  if(dsi_obj->dsi_file->storage_system == IBP){
  /* send the cap set */
#ifdef DSI_IBP
    dsi_obj->dsi_file->dsi_specifics.ibp.depot =
      (struct ibp_depot*)malloc(sizeof(struct ibp_depot));
    if (recvString(comm,&host) == -1)
      return -1;
    strcpy(dsi_obj->dsi_file->dsi_specifics.ibp.depot->host, host);
    if (recvInt(comm,&port) == -1)
      return -1; 
    dsi_obj->dsi_file->dsi_specifics.ibp.depot->port = port;

    /* receive the cap structure */
    dsi_obj->dsi_file->dsi_specifics.ibp.cap =
      (struct ibp_set_of_caps*)malloc(sizeof(struct ibp_set_of_caps));
    if (recvString(comm,&readCap) == -1)
      return -1;
    dsi_obj->dsi_file->dsi_specifics.ibp.cap->readCap = readCap;
    if (recvString(comm,&writeCap) == -1)
      return -1;
    dsi_obj->dsi_file->dsi_specifics.ibp.cap->writeCap = writeCap;
    if (recvString(comm,&manageCap) == -1)
      return -1;
    dsi_obj->dsi_file->dsi_specifics.ibp.cap->manageCap = manageCap;
#endif
  }

  if (recvInt(comm,&(dsi_obj->dsi_file->written_count)) == -1)
     return -1;
  if (recvInt(comm,&(dsi_obj->offset)) == -1)
    return -1;

  return 0;
}

int sendDsiObj(NS_Communicator *comm,DSI_OBJECT* dsi_obj){
/* have to send attributes of DSI_FILE* and  offset */

  if(sendString(comm,dsi_obj->name) == -1)
    return -1;

/* sending DSI_FILE* */
  if (sendInt(comm,dsi_obj->dsi_file->flag) == -1)
    return -1;
  if (sendInt(comm,dsi_obj->dsi_file->perm) == -1)
    return -1;
  if (sendInt(comm,dsi_obj->dsi_file->uid) == -1)
    return -1;
  if (sendInt(comm,dsi_obj->dsi_file->storage_system) == -1)
    return -1;
  if(dsi_obj->dsi_file->storage_system == IBP){
  /* send the cap set */
#ifdef DSI_IBP
    /* send the depot structure */
    if(sendString(comm,dsi_obj->dsi_file->dsi_specifics.ibp.depot->host) == -1)
      return -1;
    if(sendInt(comm,dsi_obj->dsi_file->dsi_specifics.ibp.depot->port) == -1)
      return -1;

    /* send the cap set */
    if(sendString(comm,dsi_obj->dsi_file->dsi_specifics.ibp.cap->readCap) == -1)
      return -1;
    if(sendString(comm,dsi_obj->dsi_file->dsi_specifics.ibp.cap->writeCap) == -1)
      return -1;
    if(sendString(comm,dsi_obj->dsi_file->dsi_specifics.ibp.cap->manageCap) == -1)
      return -1;
#endif
  }
  if (sendInt(comm,dsi_obj->dsi_file->written_count) == -1)
     return -1;
  if (sendInt(comm,dsi_obj->offset) == -1)
    return -1;

  return 0;
}
