/*****************************************************************/
/*      process_pseudocode.c                                     */
/*      Henri Casanova, Dorian Arnold                            */
/*****************************************************************/

#include "core.h"
#include "codegenerator.h"
#include "mnemonics.h"
#include "expressions.h"
#include "netsolveutil.h"

/*
 * processCode()
 *
 * Actually processes the code itself, with replacement of the macros
 *
 */

int processCode(FILE *f,NS_ProblemDesc *pd)
{
  int status;
  char prev_last_previous = ' ';
  char last_previous = ';';

  while(1)
  {
    if (read_line(f) == 0)
    {
      fprintf(stderr,"Interrupted @CODE section at line %d\n",linecount);
      return -1;
    }
    if (strncmp(line,"@END_CODE",9) == 0) /* End of the code */
      return 1;
    status = processLine(&prev_last_previous,&last_previous,pd);
    if (status < 0)
      return status;
  }
}

/*
 * processLine()
 *
 *  Actually processes a line
 *
 */

int processLine(char *prev_last_previous, char *last_previous,NS_ProblemDesc *pd)
{
  char outputline[MAX_LINE_LENGTH];
  int output_offset = 0;
  int input_offset = 0;

  int waiting_quote = 0;
  char *replacement;
 
  int i;
  int first = 1;


  for (i=0;i<MAX_LINE_LENGTH;i++)
    outputline[i] = '\0';
  
  while(line[input_offset] != '\n')
  {
    char mnemonic[MNEMONIC_LENGTH+1];

    /** Carriage return after each comma **/
    if ((line[input_offset] == '"')&&(waiting_quote == 0))
    {
      waiting_quote = 1;
      outputline[output_offset++] = line[input_offset++];
      continue;
    }
    if ((line[input_offset] == '"')&&(waiting_quote == 1))
    {
      waiting_quote = 0;
      outputline[output_offset++] = line[input_offset++];
      continue;
    }
    if (line[input_offset] == ',')
    {
      first = 0;
      outputline[output_offset++] = line[input_offset++];
      if (waiting_quote == 0)
      {
        outputline[output_offset] = '\n';
        write_line(outputline);
        for (i=0;i<MAX_LINE_LENGTH;i++)
            outputline[i] = '\0';
        output_offset = 0;
      }
      if (line[input_offset] == '\n')
      {
        if (input_offset-2 >= 0)
          *prev_last_previous = line[input_offset-2];
        *last_previous = line[input_offset-1];
        write_line(outputline);
        return 1;
      }
      continue;
    }
    /** Carriage return after each = **/
    if ((line[input_offset] == '=')&&
        (line[input_offset-1] != '=')&&
        (line[input_offset+1] != '='))
    {
      first = 0;
      outputline[output_offset++] = line[input_offset++];
      if (waiting_quote == 0)
      {
        outputline[output_offset] = '\n';
        write_line(outputline);
        for (i=0;i<MAX_LINE_LENGTH;i++)
          outputline[i] = '\0';
        output_offset = 0;
      }
      if (line[input_offset] == '\n')
      {
        if (input_offset-2 >= 0)
          *prev_last_previous = line[input_offset-2];
        *last_previous = line[input_offset-1];
        write_line(outputline);
        return 1;
      }
      continue;
    }

    /** Not a special character **/
    if ((line[input_offset] != '@')&&
        (line[input_offset] != '\\'))
    {
      if (line[input_offset] != ' ' && line[input_offset] != '\t')
        first = 0;
      outputline[output_offset++] = line[input_offset++];
      if (line[input_offset] == '\n')
      {
        if (input_offset-2 >= 0)
          *prev_last_previous = line[input_offset-2];
        *last_previous = line[input_offset-1];
        outputline[output_offset] = '\n';
        write_line(outputline);
        return 1;
      }
      continue;
    }
    
    /** De-specified character **/
    if (line[input_offset] == '\\')
    {
      first = 0;
      input_offset++;
      outputline[output_offset++] = line[input_offset++];
      if (line[input_offset] == '\n')
      {
        if (input_offset-2 >= 0)
          *prev_last_previous = line[input_offset-2];
        *last_previous = line[input_offset-1];
        outputline[output_offset] = '\n';
        write_line(outputline);
        return 1;
      }
      continue;
    }

    /** FROM NOW : SPECIAL CHARACTER **/

    input_offset++;
    for (i=0;i<MNEMONIC_LENGTH+1;i++)
      mnemonic[i] = '\0';
    for (i=0;i<MNEMONIC_LENGTH;i++)
    {
      if (line[input_offset] == '@')
        break;
      mnemonic[i] = line[input_offset++]; 
    }
    if (i==MNEMONIC_LENGTH)
    {
      fprintf(stderr,"Unterminated or too long mnemonic at line %d\n",linecount);
      return -1;
    }

    /* Check the validity of the mnemonic */
    i = 0;
    while(mnemonic[i] != '\0')
    {
      if ((mnemonic[i] != 'I') && (mnemonic[i] != 'O') &&
          (mnemonic[i] != 'l') && (mnemonic[i] != 'd') &&
          (mnemonic[i] != 'm') && (mnemonic[i] != 'n') &&
          (mnemonic[i] != 'i') && (mnemonic[i] != 'p') &&
          (mnemonic[i] != 'f') &&
          (mnemonic[i] != '0') && (mnemonic[i] != '1') &&
          (mnemonic[i] != '2') && (mnemonic[i] != '3') &&
          (mnemonic[i] != '4') && (mnemonic[i] != '5') &&
          (mnemonic[i] != '6') && (mnemonic[i] != '7') &&
          (mnemonic[i] != '8') && (mnemonic[i] != '9'))
      {
        fprintf(stderr,"Illegal mnemonic character '%c ' in '%s ' at line %d\n",
                mnemonic[i],mnemonic,linecount);
        return -1;
      }
      i++;
    }

    input_offset++;
    if (first == 1)
    {
      if (*last_previous != ';' && *last_previous != '}' 
          && *last_previous != '{')
        first = 0;
      if (*last_previous == '/' && *prev_last_previous == '*')
        first = 1;
    }
    i = input_offset;
    while((line[i] == ' ')||(line[i] == ')')) i++;
    if (line[i] == '[')
      first = 0;
    replacement = processMnemonic(first,mnemonic,&output_offset,pd);
    if (replacement == NULL)
    {
      return -1;
    }

    i = 0;
    while(line[i] != '\n')
    {
      if (line[i] != ' ' && line[i] != '\t')
      {
        if (i-1 >= 0)
          *last_previous = line[i-1];
        *last_previous = line[i];
      }
      i++;
    }
    for (i=0;replacement[i]!='\0';i++)
      outputline[output_offset++] = replacement[i];
    
    free(replacement);
  }
  fprintf(stderr," finished\n");
  return 1;
}

/*
 * processMnemonic()
 *
 *  Do the substitution in the target file
 */

char *processMnemonic(int first, char *mnemonic,int *output_offset,NS_ProblemDesc *pd)
{
  char buf[MAX_LINE_LENGTH];
  char tmp[MAX_LINE_LENGTH];
  int i;
  int data_type;
  int object_type;

  for (i=0;i<MAX_LINE_LENGTH;i++)
    buf[i] = '\0';

  switch (mnemonic[0])
  {
    case 'I' :         /* INPUT PARAMETER */
    {
      int index;
      if (!isInteger((char *)mnemonic+1))
      {
        fprintf(stderr,"Invalid mnemonic '%s' at line %d\n",
                       mnemonic,linecount);
        return NULL;
      }
      index = atoi((char *)mnemonic+1);

      if (index >= pd->nb_input_objects)
      {
        fprintf(stderr,
           "Line %d :Only %d objects in input to %s in mnemonic @%s@\n",
           linecount,pd->nb_input_objects,pd->nickname,mnemonic);
        return NULL;
      }
      data_type = pd->input_objects[index]->data_type;
      object_type = pd->input_objects[index]->object_type;

      switch(data_type)
      {
        case NETSOLVE_NOTYPE:
          sprintf(buf,"(");
          break;
        case NETSOLVE_B:
          if (!first)
            sprintf(buf,"((char *)");
          else
            sprintf(buf,"("); 
          break;
        case NETSOLVE_I:
          if (!first)
            sprintf(buf,"((int *)");
          else
            sprintf(buf,"("); 
          break;
        case NETSOLVE_CHAR:
          if (!first)
          {
            if ((object_type != NETSOLVE_PACKEDFILES) &&
                (object_type != NETSOLVE_STRINGLIST))
              sprintf(buf,"((char *)");
            else
              sprintf(buf,"((char **)");
          }
          else
          {
            sprintf(buf,"("); 
          }
          break;
        case NETSOLVE_S:
          if (!first)
            sprintf(buf,"((float *)");
          else
            sprintf(buf,"("); 
          break;
        case NETSOLVE_D:
          if (!first)
            sprintf(buf,"((double *)");
          else
            sprintf(buf,"("); 
          break;
        case NETSOLVE_C:
          if (!first)
            sprintf(buf,"((scomplex *)");
          else
            sprintf(buf,"("); 
          break;
        case NETSOLVE_Z:
          if (!first)
            sprintf(buf,"((dcomplex *)");
          else
            sprintf(buf,"("); 
          break;
        default:
          fprintf(stderr,"Unknown data type\n");
          return NULL;
      }
      switch(object_type)
      {
        case NETSOLVE_MATRIX:
          sprintf(tmp,"input_objects[%d]->attributes.matrix_attributes.ptr)",index);
          strcat(buf,tmp);
          break;
        case NETSOLVE_SPARSEMATRIX:
          sprintf(tmp,"input_objects[%d]->attributes.sparsematrix_attributes.ptr)",index);
          strcat(buf,tmp);
          break;
        case NETSOLVE_VECTOR:
          sprintf(tmp,"input_objects[%d]->attributes.vector_attributes.ptr)",index);
          strcat(buf,tmp);
          break;
        case NETSOLVE_SCALAR:
          sprintf(tmp,"input_objects[%d]->attributes.scalar_attributes.ptr)",index);
          strcat(buf,tmp);
          break;
        case NETSOLVE_FILE:
          sprintf(tmp,"input_objects[%d]->attributes.file_attributes.filename)",index);
          strcat(buf,tmp);
          break;
        case NETSOLVE_PACKEDFILES:
          sprintf(tmp,"input_objects[%d]->attributes.packedfiles_attributes.filenames)",index);
          strcat(buf,tmp);
          break;
        case NETSOLVE_STRINGLIST:
          sprintf(tmp,"input_objects[%d]->attributes.stringlist_attributes.strings)",index);
          strcat(buf,tmp);
          break;
        case NETSOLVE_UPF:
          sprintf(tmp,"upf%d)",index);
          strcat(buf,tmp);
          break;
        case NETSOLVE_STRING:
          sprintf(tmp,"input_objects[%d]->attributes.string_attributes.ptr)",index);
          strcat(buf,tmp);
          break;
        default:
          break;
      }
      return (strdup(buf));
    }
    case 'O' :             /* OUTPUT PARAMETER */
    {
      int index;

      if (!isInteger((char *)mnemonic+1))
      {
        fprintf(stderr,"Invalid mnemonic '%s' at line %d\n",
                       mnemonic,linecount);
        return NULL;
      }
      index = atoi((char *)mnemonic+1);

      if (index >= pd->nb_output_objects)
      {
        fprintf(stderr,
           "Line %d :Only %d objects in output to %s in mnemonic @%s@\n",
           linecount,pd->nb_output_objects,pd->nickname,mnemonic);
        return NULL;
      }
      data_type = pd->output_objects[index]->data_type;
      object_type = pd->output_objects[index]->object_type;

      switch(data_type)
      {
        case NETSOLVE_NOTYPE:
          sprintf(buf,"(");
          break;
        case NETSOLVE_B:
          if (!first)
            sprintf(buf,"((char *)");
          else
            sprintf(buf,"(");
          break;
        case NETSOLVE_I:
          if (!first)
            sprintf(buf,"((int *)");
          else
            sprintf(buf,"(");
          break;
        case NETSOLVE_CHAR:
          if (!first)
          {
            if (object_type != NETSOLVE_PACKEDFILES)
              sprintf(buf,"((char *)");
            else
              sprintf(buf,"((char **)");
          }
          else
            sprintf(buf,"(");
          break;
        case NETSOLVE_S:
          if (!first)
            sprintf(buf,"((float *)");
          else
            sprintf(buf,"(");
          break;
        case NETSOLVE_D:
          if (!first)
            sprintf(buf,"((double *)");
          else
            sprintf(buf,"(");
          break;
        case NETSOLVE_C:
          if (!first)
            sprintf(buf,"((scomplex *)");
          else
            sprintf(buf,"(");
          break;
        case NETSOLVE_Z:
          if (!first)
            sprintf(buf,"((dcomplex *)");
          else
            sprintf(buf,"(");
          break;
        default:
          fprintf(stderr,"Unknown data type\n");
          return NULL;
      }
      switch(object_type)
      {
        case NETSOLVE_MATRIX:
          sprintf(tmp,"output_objects[%d]->attributes.matrix_attributes.ptr)",index);
          strcat(buf,tmp);
          break;
        case NETSOLVE_SPARSEMATRIX:
          sprintf(tmp,"output_objects[%d]->attributes.sparsematrix_attributes.ptr)",index);
          strcat(buf,tmp);
          break;
        case NETSOLVE_VECTOR:
          sprintf(tmp,"output_objects[%d]->attributes.vector_attributes.ptr)",index);
          strcat(buf,tmp);
          break;
        case NETSOLVE_SCALAR:
          sprintf(tmp,"output_objects[%d]->attributes.scalar_attributes.ptr)",index);
          strcat(buf,tmp);
          break;
        case NETSOLVE_FILE:
          sprintf(tmp,"output_objects[%d]->attributes.file_attributes.filename)",index);
          strcat(buf,tmp);
          break;
        case NETSOLVE_PACKEDFILES:
          sprintf(tmp,"output_objects[%d]->attributes.packedfiles_attributes.filenames)",index);
          strcat(buf,tmp);
          break;
        case NETSOLVE_STRING:
          sprintf(tmp,"output_objects[%d]->attributes.string_attributes.ptr)",index);
          strcat(buf,tmp);
          break;
        default:
          break;
      }
      return (strdup(buf));

    }
    case 'm' :  /* M DIMENSION */
    {
      switch (mnemonic[1])
      {
        case 'I' :            /* In input */
        {
          int index;
          int object_type;

          if (!isInteger((char *)mnemonic+2))
          {
            fprintf(stderr,"Invalid mnemonic '%s' at line %d\n",
                           mnemonic,linecount); 
            return NULL;
          }
          index = atoi((char *)mnemonic+2);
          if (index >= pd->nb_input_objects)
          {
            fprintf(stderr,
              "Line %d :Only %d objects in input to %s in mnemonic @%s@\n",
              linecount,pd->nb_input_objects,pd->nickname,mnemonic);
            return NULL;
          }
          object_type = pd->input_objects[index]->object_type;
          switch(object_type)
          {
            case NETSOLVE_MATRIX:
              sprintf(buf,"((int *)(&(input_objects[%d]->attributes.matrix_attributes.m)))",index);
              break;
            case NETSOLVE_SPARSEMATRIX:
              sprintf(buf,"((int *)(&(input_objects[%d]->attributes.sparsematrix_attributes.m)))",index);
              break;
            case NETSOLVE_VECTOR:
              sprintf(buf,"((int *)(&(input_objects[%d]->attributes.vector_attributes.m)))",index);
              break;
            case NETSOLVE_STRINGLIST:
              sprintf(buf,"((int *)(&(input_objects[%d]->attributes.stringlist_attributes.m)))",index);
              break;
            case NETSOLVE_PACKEDFILES:
              sprintf(buf,"((int *)(&(input_objects[%d]->attributes.packedfiles_attributes.m)))",index);
              break;
            default:
                fprintf(stderr,
                  "Line %d :Object #%d in input to %s does not have a 'm-dimension'\n",
                  linecount,index,pd->nickname);
                return NULL;
          }
          return (strdup(buf));
        }
        case 'O' :             /* IN OUTPUT */
        {
          int index;
          int object_type;

          if (!isInteger((char *)mnemonic+2))
          {
            fprintf(stderr,"Invalid mnemonic '%s' at line %d\n",mnemonic,linecount);
            return NULL;
          }
          index = atoi((char *)mnemonic+2);
          if (index >= pd->nb_output_objects)
          {
            fprintf(stderr,
              "Line %d :Only %d objects in output to %s in mnemonic @%s@\n",
              linecount,pd->nb_output_objects,pd->nickname,mnemonic);
            return NULL;
          }
          object_type = pd->output_objects[index]->object_type;
          switch(object_type)
          {
            case NETSOLVE_MATRIX:
              sprintf(buf,"((int *)(&(output_objects[%d]->attributes.matrix_attributes.m)))",index);
              break;
            case NETSOLVE_SPARSEMATRIX:
              sprintf(buf,"((int *)(&(output_objects[%d]->attributes.sparsematrix_attributes.m)))",index);
              break;
            case NETSOLVE_VECTOR:
              sprintf(buf,"((int *)(&(output_objects[%d]->attributes.vector_attributes.m)))",index);
              break;
            case NETSOLVE_PACKEDFILES:
              sprintf(buf,"((int *)(&(output_objects[%d]->attributes.packedfiles_attributes.m)))",index);
              break;

            default:
                fprintf(stderr,
                  "Line %d :Object #%d in output to %s does not have a 'm-dimension'\n",linecount,index,pd->nickname);
                 return NULL;
          }
          return (strdup(buf));
        }
        default :     /* ERROR */
        {
          fprintf(stderr,"Line %d : Invalid mnemonic @%s @\n",
                  linecount,mnemonic);
          return NULL;
        }
      }
    }
    case 'n' : /* N Dimension */
    {
      switch (mnemonic[1])
      {
        case 'I' :            /* In input */
        {
          int index;
          int object_type;

          if (isInteger((char *)mnemonic+2) == -1)
          {
            fprintf(stderr,"Invalid mnemonic '%s' at line %d\n",
                    mnemonic,linecount);
            return NULL;
          }
          index = atoi((char *)mnemonic+2);
          if (index >= pd->nb_input_objects)
          {
            fprintf(stderr,
              "Line %d :Only %d objects in input to %s in mnemonic @%s@\n",
              linecount,pd->nb_input_objects,pd->nickname,mnemonic);
            return NULL;
          }
          object_type = pd->input_objects[index]->object_type;
          switch(object_type)
          {
            case NETSOLVE_MATRIX:
              sprintf(buf,"((int *)(&(input_objects[%d]->attributes.matrix_attributes.n)))",index);
              break;
            case NETSOLVE_SPARSEMATRIX:
              sprintf(buf,"((int *)(&(input_objects[%d]->attributes.sparsematrix_attributes.n)))",index);
              break;
            default:
              fprintf(stderr,
                  "Line %d :Object #%d in input to %s does not have a 'n-dimension'\n",
                  linecount,index,pd->nickname);
              return NULL;
          }
          return (strdup(buf));
        }
        case 'O' :             /* IN OUTPUT */
        {
          int index;
          int object_type;

          if (isInteger((char *)mnemonic+2) == -1)
          {
            fprintf(stderr,"Invalid mnemonic '%s' at line %d\n",mnemonic,linecount);
            return NULL;
          }
          index = atoi((char *)mnemonic+2);
          if (index >= pd->nb_output_objects)
          {
            fprintf(stderr,
              "Line %d :Only %d objects in output to %s in mnemonic @%s@\n",
              linecount,pd->nb_output_objects,pd->nickname,mnemonic);
            return NULL;
          }
          object_type = pd->output_objects[index]->object_type;
          switch(object_type)
          {
            case NETSOLVE_MATRIX:
              sprintf(buf,"((int *)(&(output_objects[%d]->attributes.matrix_attributes.n)))",index);
              break;
            case NETSOLVE_SPARSEMATRIX:
              sprintf(buf,"((int *)(&(output_objects[%d]->attributes.sparsematrix_attributes.n)))",index);
              break;
            default:
              fprintf(stderr,
                  "Line %d :Object #%d in output to %s does not have a 'n-dimension'\n",
                  linecount,index,pd->nickname);
              return NULL;
          }
          return (strdup(buf));
        }
        default :     /* ERROR */
        {
          fprintf(stderr,"Line %d : Invalid mnemonic @%s @\n",
                  linecount,mnemonic);
          return NULL;
        }
      }
    }
    case 'l' : /* Leading Dimension */
    {
      switch (mnemonic[1])
      {
        case 'I' :            /* In input */
        {
          int index;
          int object_type;

          if (isInteger((char *)mnemonic+2) == -1)
          {
            fprintf(stderr,"Invalid mnemonic '%s' at line %d\n",mnemonic,linecount);
            return NULL;
          }
          index = atoi((char *)mnemonic+2);
          if (index >= pd->nb_input_objects)
          {
            fprintf(stderr,
              "Line %d :Only %d objects in input to %s in mnemonic @%s@\n",
              linecount,pd->nb_input_objects,pd->nickname,mnemonic);
            return NULL;
          }
          object_type = pd->input_objects[index]->object_type;
          if (object_type != NETSOLVE_MATRIX)
          {
            fprintf(stderr,
              "Line %d :Object #%d in input to %s does not have a 'leading dimension'\n",
              linecount,index,pd->nickname);
            return NULL;
          }
          sprintf(buf,"((int *)(&(input_objects[%d]->attributes.matrix_attributes.l)))",index);
          return (strdup(buf));
        }
        case 'O' :             /* IN OUTPUT */
        {
          int index;
          int object_type;

          if (isInteger((char *)mnemonic+2) == -1)
          {
            fprintf(stderr,"Invalid mnemonic '%s' at line %d\n",mnemonic,linecount);
            return NULL;
          }
          index = atoi((char *)mnemonic+2);
          if (index >= pd->nb_output_objects)
          {
            fprintf(stderr,
              "Line %d :Only %d objects in output to %s in mnemonic @%s@\n",
              linecount,pd->nb_output_objects,pd->nickname,mnemonic);
            return NULL;
          }
          object_type = pd->output_objects[index]->object_type;
          if (object_type != NETSOLVE_MATRIX)
          {
            fprintf(stderr,
              "Line %d :Object #%d in output to %s does not have a 'leading dimension'\n",
              linecount,index,pd->nickname);
            return NULL;
          }
          sprintf(buf,"((int *)(&(output_objects[%d]->attributes.matrix_attributes.l)))",index);
          return (strdup(buf));
        }
        default :     /* ERROR */
        {
          fprintf(stderr,"Line %d : Invalid mnemonic @%s @\n",
                  linecount,mnemonic);
          return NULL;
        }
      }
    }
    case 'f' : /* Non zeroes */
    {
      switch (mnemonic[1])
      {
        case 'I' :            /* In input */
        {
          int index;
          int object_type;

          if (isInteger((char *)mnemonic+2) == -1)
          {
            fprintf(stderr,"Invalid mnemonic '%s' at line %d\n",mnemonic,linecount);
            return NULL;
          }
          index = atoi((char *)mnemonic+2);
          if (index >= pd->nb_input_objects)
          {
            fprintf(stderr,
              "Line %d :Only %d objects in input to %s in mnemonic @%s@\n",
              linecount,pd->nb_input_objects,pd->nickname,mnemonic);
            return NULL;
          }
          object_type = pd->input_objects[index]->object_type;
          if (object_type != NETSOLVE_SPARSEMATRIX)
          {
            fprintf(stderr,
              "Line %d :Object #%d in input to %s does not have a 'number of filled postions'\n",
              linecount,index,pd->nickname);
            return NULL;
          }
          sprintf(buf,"((int *)(&(input_objects[%d]->attributes.sparsematrix_attributes.f)))",index);
          return (strdup(buf));
        }
        case 'O' :             /* IN OUTPUT */
        {
          int index;
          int object_type;

          if (isInteger((char *)mnemonic+2) == -1)
          {
            fprintf(stderr,"Invalid mnemonic '%s' at line %d\n",mnemonic,linecount);
            return NULL;
          }
          index = atoi((char *)mnemonic+2);
          if (index >= pd->nb_output_objects)
          {
            fprintf(stderr,
              "Line %d :Only %d objects in output to %s in mnemonic @%s@\n",
              linecount,pd->nb_output_objects,pd->nickname,mnemonic);
            return NULL;
          }
          object_type = pd->output_objects[index]->object_type;
          if (object_type != NETSOLVE_SPARSEMATRIX)
          {
            fprintf(stderr,
              "Line %d :Object #%d in output to %s does not have a 'number of filled positions'\n",
              linecount,index,pd->nickname);
            return NULL;
          }
          sprintf(buf,"((int *)(&(output_objects[%d]->attributes.sparsematrix_attributes.f)))",index);
          return (strdup(buf));
        }
        default :     /* ERROR */
        {
          fprintf(stderr,"Line %d : Invalid mnemonic @%s @\n",
                  linecount,mnemonic);
          return NULL;
        }
      }
    }
    case 'p' : /* Leading Dimension */
    {
      switch (mnemonic[1])
      {
        case 'I' :            /* In input */
        {
          int index;
          int object_type;

          if (isInteger((char *)mnemonic+2) == -1)
          {
            fprintf(stderr,"Invalid mnemonic '%s' at line %d\n",mnemonic,linecount);
            return NULL;
          }
          index = atoi((char *)mnemonic+2);
          if (index >= pd->nb_input_objects)
          {
            fprintf(stderr,
              "Line %d :Only %d objects in input to %s in mnemonic @%s@\n",
              linecount,pd->nb_input_objects,pd->nickname,mnemonic);
            return NULL;
          }
          object_type = pd->input_objects[index]->object_type;
          if (object_type != NETSOLVE_SPARSEMATRIX)
          {
            fprintf(stderr,
              "Line %d :Object #%d in input to %s does not have 'row/col indices'\n",
              linecount,index,pd->nickname);
            return NULL;
          }
          sprintf(buf,"((int *)((input_objects[%d]->attributes.sparsematrix_attributes.rc_ptr)))",
                  index);
          return (strdup(buf));
        }
        case 'O' :             /* IN OUTPUT */
        {
          int index;
          int object_type;

          if (isInteger((char *)mnemonic+2) == -1)
          {
            fprintf(stderr,"Invalid mnemonic '%s' at line %d\n",mnemonic,linecount);
            return NULL;
          }
          index = atoi((char *)mnemonic+2);
          if (index >= pd->nb_output_objects)
          {
            fprintf(stderr,
              "Line %d :Only %d objects in output to %s in mnemonic @%s@\n",
              linecount,pd->nb_output_objects,pd->nickname,mnemonic);
            return NULL;
          }
          object_type = pd->output_objects[index]->object_type;
          if (object_type != NETSOLVE_SPARSEMATRIX)
          {
            fprintf(stderr,
              "Line %d :Object #%d in output to %s does not have 'row/col indices'\n",
              linecount,index,pd->nickname);
            return NULL;
          }
          sprintf(buf,"((int *)((output_objects[%d]->attributes.sparsematrix_attributes.rc_ptr)))",
                  index);
          return (strdup(buf));
        }
        default :     /* ERROR */
        {
          fprintf(stderr,"Line %d : Invalid mnemonic @%s @\n",
                  linecount,mnemonic);
          return NULL;
        }
      }
    }
    case 'i' : /* Leading Dimension */
    {
      switch (mnemonic[1])
      {
        case 'I' :            /* In input */
        {
          int index;
          int object_type;

          if (isInteger((char *)mnemonic+2) == -1)
          {
            fprintf(stderr,"Invalid mnemonic '%s' at line %d\n",mnemonic,linecount);
            return NULL;
          }
          index = atoi((char *)mnemonic+2);
          if (index >= pd->nb_input_objects)
          {
            fprintf(stderr,
              "Line %d :Only %d objects in input to %s in mnemonic @%s@\n",
              linecount,pd->nb_input_objects,pd->nickname,mnemonic);
            return NULL;
          }
          object_type = pd->input_objects[index]->object_type;
          if (object_type != NETSOLVE_SPARSEMATRIX)
          {
            fprintf(stderr,
              "Line %d :Object #%d in input to %s does not have 'row/col indices'\n",
              linecount,index,pd->nickname);
            return NULL;
          }
          sprintf(buf,"((int *)((input_objects[%d]->attributes.sparsematrix_attributes.rc_index)))",
                  index);
          return (strdup(buf));
        }
        case 'O' :             /* IN OUTPUT */
        {
          int index;
          int object_type;

          if (isInteger((char *)mnemonic+2) == -1)
          {
            fprintf(stderr,"Invalid mnemonic '%s' at line %d\n",mnemonic,linecount);
            return NULL;
          }
          index = atoi((char *)mnemonic+2);
          if (index >= pd->nb_output_objects)
          {
            fprintf(stderr,
              "Line %d :Only %d objects in output to %s in mnemonic @%s@\n",
              linecount,pd->nb_output_objects,pd->nickname,mnemonic);
            return NULL;
          }
          object_type = pd->output_objects[index]->object_type;
          if (object_type != NETSOLVE_SPARSEMATRIX)
          {
            fprintf(stderr,
              "Line %d :Object #%d in output to %s does not have 'row/col indices'\n",
              linecount,index,pd->nickname);
            return NULL;
          }
          sprintf(buf,"((int *)((output_objects[%d]->attributes.sparsematrix_attributes.rc_index)))",
                  index);
          return (strdup(buf));
        }
        default :     /* ERROR */
        {
          fprintf(stderr,"Line %d : Invalid mnemonic @%s @\n",
                  linecount,mnemonic);
          return NULL;
        }
      }
    }
    case 'd' : /* ScaLAPACK Description */
    {
      switch (mnemonic[1])
      {
        case 'I' :            /* In input */
        {
          int index;
          int object_type;

          if (isInteger((char *)mnemonic+2) == -1)
          {
            fprintf(stderr,"Invalid mnemonic '%s' at line %d\n",mnemonic,linecount);
            return NULL;
          }
          index = atoi((char *)mnemonic+2);
          if (index >= pd->nb_input_objects)
          {
            fprintf(stderr,
              "Line %d :Only %d objects in input to %s in mnemonic @%s@\n",
              linecount,pd->nb_input_objects,pd->nickname,mnemonic);
            return NULL;
          }
          object_type = pd->input_objects[index]->object_type;

          if ((object_type != NETSOLVE_MATRIX)&&
              (object_type != NETSOLVE_VECTOR))
          {
            fprintf(stderr,
              "Line %d :Object #%d in input to %s does not have a 'descriptor'\n",
              linecount,index,pd->nickname);
            return NULL;
          }
          
          if (!first)
          {
            if (object_type == NETSOLVE_MATRIX)
              sprintf(buf,"((int *)(input_objects[%d]->attributes.matrix_attributes.d))",index);
            if (object_type == NETSOLVE_VECTOR)
              sprintf(buf,"((int *)(input_objects[%d]->attributes.vector_attributes.d))",index);
          }
          else
          {
            if (object_type == NETSOLVE_MATRIX)
              sprintf(buf,"(input_objects[%d]->attributes.matrix_attributes.d)",index);
            if (object_type == NETSOLVE_VECTOR)
              sprintf(buf,"(input_objects[%d]->attributes.vector_attributes.d)",index);
          }

          return (strdup(buf));
        }
        case 'O' :             /* IN OUTPUT */
        {
          int index;
          int object_type;

          if (isInteger((char *)mnemonic+2) == -1)
          {
            fprintf(stderr,"Invalid mnemonic '%s' at line %d\n",mnemonic,linecount);
            return NULL;
          }
          index = atoi((char *)mnemonic+2);
          if (index >= pd->nb_output_objects)
          {
            fprintf(stderr,
              "Line %d :Only %d objects in output to %s in mnemonic @%s@\n",
              linecount,pd->nb_output_objects,pd->nickname,mnemonic);
            return NULL;
          }
          object_type = pd->output_objects[index]->object_type;
          if ((object_type != NETSOLVE_MATRIX)&&
             (object_type != NETSOLVE_VECTOR))
          {
            fprintf(stderr,
              "Line %d :Object #%d in output to %s does not have a 'descriptor'\n",
              linecount,index,pd->nickname);
            return NULL;
          }
          if (!first)
          {
            if (object_type == NETSOLVE_MATRIX)
              sprintf(buf,"((int *)(output_objects[%d]->attributes.matrix_attributes.d))",index);
            if (object_type == NETSOLVE_VECTOR)
              sprintf(buf,"((int *)(output_objects[%d]->attributes.vector_attributes.d))",index);
          }
          else
          {
            if (object_type == NETSOLVE_MATRIX)
              sprintf(buf,"(output_objects[%d]->attributes.matrix_attributes.d)",index);
            if (object_type == NETSOLVE_VECTOR)
              sprintf(buf,"(output_objects[%d]->attributes.vector_attributes.d)",index);
          }
          return (strdup(buf));
        }
        default :     /* ERROR */
        {
          fprintf(stderr,"Line %d : Invalid mnemonic @%s @\n",
                  linecount,mnemonic);
          return NULL;
        }
      }
    }
    default : /* Should never happen */
    {
      fprintf(stderr,"Invalid mnemonic '%s' at line %d\n",mnemonic,linecount);
      return NULL;
    }
  }

}
