/*****************************************************************/
/*      process_problems_util.c                                  */
/*      Version 1.0                      Henri Casanova          */
/*****************************************************************/

#include "core.h"
#include "codegenerator.h"
#include "mnemonics.h"
#include "expressions.h"
#include "netsolveutil.h"

int parsePROBLEM(NS_ProblemDesc *pd,NS_ProblemDescAddendum *pdadd,char *filename)
{
  FILE *f;
  char *token1,*token2;

  f = fopen(filename,"r");
  if (f == NULL)
  {
    fprintf(stderr,"Impossible to open file '%s'\n",filename);
    return -1;
  }
  
  while(read_line(f) != NULL)
  {
    if (strncmp(line,"@PROBLEM ",9))
      continue;

    token1 = strtok(line," \t\n");
    token2 = strtok(NULL," \t\n");
    if ((token1 == NULL)||(token2 == NULL))
    {
      fprintf(stderr,"'@PROBLEM <problem name>' expected.\n");
      fclose(f);
      return -1;
    }
    pd->nickname = strdup(token2);
    fclose(f);
    /* UPDATE THE INITCODEFILE */
    fprintf(initcodefile,"/* Problem */\n");
    fprintf(initcodefile,"if (inList(list,nlist,\"%s\"))\n",pd->file);
    fprintf(initcodefile,"{\n");
    fprintf(initcodefile,"  pb_desc=newProblemDesc();\n\n");
    fprintf(initcodefile,"  pb_desc->nickname = strdup(\"%s\");\n",
                    pd->nickname);
    fprintf(initcodefile,"  pb_desc->file = strdup(\"%s\");\n",
                  pd->file);
    return 1;
  }
  fprintf(stderr,"Mission @PROBLEM clause\n");
  fclose(f);
  return -1;
}

int parseFUNCTION(NS_ProblemDesc *pd,NS_ProblemDescAddendum *pdadd,char *filename)
{
  FILE *f;
  char *token1,*token2;

  f = fopen(filename,"r");
  if (f == NULL)
  {
    fprintf(stderr,"Impossible to open file '%s'\n",filename);
    return -1;
  }
  
  pdadd->nb_funcs = 0;
  pdadd->funcs = NULL;
  while(read_line(f) != NULL)
  {
    if (strncmp(line,"@FUNCTION ",10))
      continue;

    token1 = strtok(line," \t\n");
    token2 = strtok(NULL," \t\n");
    if ((token1 == NULL)||(token2 == NULL))
    {
      fprintf(stderr,"%s:'@FUNCTION <problem name>' expected\n",
                   pd->nickname);
      fclose(f);
      return -1;
    }
    (pdadd->nb_funcs)++;
    pdadd->funcs = (char **)myRealloc(pdadd->funcs,
            (pdadd->nb_funcs)*sizeof(char*));
    pdadd->funcs[pdadd->nb_funcs-1] = strdup(token2);
  
  }
  fclose(f);
  return 1;
}

int parseINCLUDE(NS_ProblemDesc *pd,NS_ProblemDescAddendum *pdadd,char *filename)
{
  FILE *f;
  char *token1,*token2;

  f = fopen(filename,"r");
  if (f == NULL)
  {
    fprintf(stderr,"Impossible to open file '%s'\n",filename);
    return -1;
  }

  pdadd->nb_includes = 0;
  pdadd->includes = NULL;
  while(read_line(f) != NULL)
  {
    if (strncmp(line,"@INCLUDE ",9))
      continue;

    token1 = strtok(line," \t\n");
    token2 = strtok(NULL," \t\n");
    if ((token1 == NULL)||(token2 == NULL))
    {
      fprintf(stderr,"%s:'@INCLUDE <include file>' expected\n",pd->nickname);
      fclose(f);
      return -1;
    }
    (pdadd->nb_includes)++;
    pdadd->includes = (char **)myRealloc(pdadd->includes,
            (pdadd->nb_includes)*sizeof(char*));
    pdadd->includes[pdadd->nb_includes-1] = strdup(token2);
  }
  fclose(f);
  return 1;
}

int parseDASHI(NS_ProblemDesc *pd,NS_ProblemDescAddendum *pdadd,char *filename)
{
  FILE *f;
  char *token1,*token2;

  f = fopen(filename,"r");
  if (f == NULL)
  {
    fprintf(stderr,"Impossible to open file '%s'\n",filename);
    return -1;
  }

  pdadd->nb_dashis = 0;
  pdadd->dashis = NULL;
  while(read_line(f) != NULL)
  {
    if (strncmp(line,"@DASHI ",7))
      continue;

    token1 = strtok(line," \t\n");
    token2 = strtok(NULL," \t\n");
    if ((token1 == NULL)||(token2 == NULL))
    {
      fprintf(stderr,"%s:'@DASHI <-Ipath>' expected\n",pd->nickname);
      fclose(f);
      return -1;
    }
    (pdadd->nb_dashis)++;
    pdadd->dashis = (char **)myRealloc(pdadd->dashis,
            (pdadd->nb_dashis)*sizeof(char*));
    pdadd->dashis[pdadd->nb_dashis-1] = strdup(token2);
  }
  fclose(f);
  return 1;
}


int parseLIB(NS_ProblemDesc *pd,NS_ProblemDescAddendum *pdadd,char *filename)
{
  FILE *f;
  char *token1,*token2;

  f = fopen(filename,"r");
  if (f == NULL)
  {
    fprintf(stderr,"Impossible to open file '%s'\n",filename);
    return -1;
  }

  pdadd->nb_libs = 0;
  pdadd->libs = NULL;
  while(read_line(f) != NULL)
  {
    if (strncmp(line,"@LIB ",5))
      continue;

    token1 = strtok(line," \t\n");
    token2 = strtok(NULL,"\n");
    if ((token1 == NULL)||(token2 == NULL))
    {
      fprintf(stderr,"%s:'@LIB <library>' expected\n",pd->nickname);
      fclose(f);
      return -1;
    }
    (pdadd->nb_libs)++;
    pdadd->libs = (char **)myRealloc(pdadd->libs,
            (pdadd->nb_libs)*sizeof(char*));
    pdadd->libs[pdadd->nb_libs-1] = strdup(token2);
    /* UPDATE THE INITCODEFILE */
    if (!strncmp(token2,"FTP://",6))
      fprintf(initcodefile,"  pb_desc->ftp=1;\n");
  }
  fclose(f);
  return 1;
}


int parseLANGUAGE(NS_ProblemDesc *pd,NS_ProblemDescAddendum *pdadd,char *filename)
{
  FILE *f;
  char *token1,*token2;

  f = fopen(filename,"r");
  if (f == NULL)
  {
    fprintf(stderr,"Impossible to open file '%s'\n",filename);
    return -1;
  }

  while(read_line(f) != NULL)
  {
    if (strncmp(line,"@LANGUAGE ",10))
      continue;

    token1 = strtok(line," \t\n");
    token2 = strtok(NULL," \t\n");
    if ((token1 == NULL)||(token2 == NULL))
    {
      fprintf(stderr,"%s:'@LANGUAGE <language>' expected\n",pd->nickname);
      fclose(f);
      return -1;
    }
    if (strcmp(token2,"C") == 0)
      pdadd->language = C;
    else if (strcmp(token2,"FORTRAN") == 0)
      pdadd->language = F;
    else
    {
      fprintf(stderr,"%s:'%s' unknown language",pd->nickname,token2);
      fclose(f);
      return -1;
    }
    fclose(f);
    return 1;
  }
  fprintf(stderr,"%s:Missing language specification\n",pd->nickname);
  fclose(f);
  return -1;
}

int parseMAJOR(NS_ProblemDesc *pd,NS_ProblemDescAddendum *pdadd,char *filename)
{
  FILE *f;
  char *token1,*token2;

  f = fopen(filename,"r");
  if (f == NULL)
  {
    fprintf(stderr,"Impossible to open file '%s'\n",filename);
    return -1;
  }

  while(read_line(f) != NULL)
  {
    if (strncmp(line,"@MAJOR ",7))
      continue;

    token1 = strtok(line," \t\n");
    token2 = strtok(NULL," \t\n");
    if ((token1 == NULL)||(token2 == NULL))
    {
      fprintf(stderr,"%s:'@MAJOR <major>' expected\n",pd->nickname);
      fclose(f);
      return -1;
    }
    if (strcmp(token2,"ROW") == 0)
      pd->major = ROW_MAJOR;
    else if (strcmp(token2,"COL") == 0)
      pd->major = COL_MAJOR;
    else
    {
      fprintf(stderr,"%s:'%s' unknown major specification\n",pd->nickname,token2);
      fclose(f);
      return -1;
    }
    fclose(f);
    /* UPDATE THE INITCODEFILE */
    fprintf(initcodefile,"  pb_desc->major=%d;\n",pd->major);
    return 1;
  }
  fprintf(stderr,"%s:Missing major specification\n",pd->nickname);
  fclose(f);
  return -1;
}

int parsePATH(NS_ProblemDesc *pd,NS_ProblemDescAddendum *pdadd,char *filename)
{
  FILE *f;
  char *token1,*token2;
  char buffer[256];

  f = fopen(filename,"r");
  if (f == NULL)
  {
    fprintf(stderr,"Impossible to open file '%s'\n",filename);
    return -1;
  }

  while(read_line(f) != NULL)
  {
    if (strncmp(line,"@PATH ",6))
      continue;

    token1 = strtok(line," \t\n");
    token2 = strtok(NULL," \t\n");
    if ((token1 == NULL)||(token2 == NULL))
    {
      fprintf(stderr,"%s:'@PATH <path>' expected\n",pd->nickname);
      fclose(f);
      return -1;
    }
    if ((token2[0] != '/')||(token2[strlen(token2)-1] != '/'))
    {
      fprintf(stderr,"%s: A path should begin and end with '/'\n",
              pd->nickname);
      fclose(f);
      return -1;
    }
    sprintf(buffer,"%s%s",token2,pd->nickname);
    pd->name=strdup(buffer);
    fclose(f);
    /* UPDATE THE INITCODEFILE */
    fprintf(initcodefile,"  pb_desc->name = strdup(\"%s\");\n",pd->name);
    return 1;
  }
  fprintf(stderr,"%s:Missing path specification\n",pd->nickname);
  fclose(f);
  return -1;
}

int parseDESCRIPTION(NS_ProblemDesc *pd,NS_ProblemDescAddendum *pdadd,char *filename)
{
  FILE *f;
  char *cleaned_line;
  char buf[256];

  f = fopen(filename,"r");
  if (f == NULL)
  {
    fprintf(stderr,"Impossible to open file '%s'\n",filename);
    return -1;
  }

  while(read_line(f) != NULL)
  {
    if (strncmp(line,"@DESCRIPTION",12))
      continue;
    pd->description = strdup(" ");
    fprintf(initcodefile,"   pb_desc->description = strdup(\" \");\n");
    if (read_line(f) == NULL)
    {
      fprintf(stderr,"%s: interrupted problem description\n",pd->nickname);
      fclose(f);
      return -1;
    }
    while (line[0] != '@')
    {
      cleaned_line = cleaned(line);
      strcpy(buf,cleaned_line);
      buf[strlen(buf)-1] = '\0';
      fprintf(initcodefile,"  strcpy(buf,\"%s\");\n",cleaned_line);
      free(cleaned_line);
      pd->description = (char *)myRealloc(
               pd->description,
               strlen(pd->description)+
               strlen(buf)+2);
      fprintf(initcodefile,"  pb_desc->description = (char *)myRealloc(");
      fprintf(initcodefile,"pb_desc->description,%d);\n",
               strlen(pd->description)+strlen(buf)+2);
      strcat(pd->description,buf);
      strcat(pd->description,"\n");
      fprintf(initcodefile,"  strcat(pb_desc->description,buf);\n");
      if (read_line(f) == NULL)
      {
        fprintf(stderr,"%s: interrupted problem description\n",pd->nickname);
        fclose(f);
        return -1;
      }
    }
    fclose(f);
    return 1;
  }
  fprintf(stderr,"%s:Missing problem description\n",pd->nickname);
  fclose(f);
  return -1;
}


int parseINPUT(NS_ProblemDesc *pd,NS_ProblemDescAddendum *pdadd,char *filename)
{
  FILE *f;
  int i;
  char * token1, * token2;

  f = fopen(filename,"r");
  if (f == NULL)
  {
    fprintf(stderr,"Impossible to open file '%s'\n",filename);
    return -1;
  }

  while(read_line(f) != NULL)
  {
    if (strncmp(line,"@INPUT ",7))
      continue;

    token1 = strtok(line," \t\n");
    token2 = strtok(NULL," \t\n");
    if ((token1 == NULL)||(token2 == NULL))
    {
      fprintf(stderr,"%s:'@INPUT <number>' expected\n",pd->nickname);
      fclose(f);
      return -1;
    }
    if (!isInteger(token2))
    {
      fprintf(stderr,"%s:'%s': invalid number of input\n",pd->nickname,token2);
      fclose(f);
      return -1;
    }
    pd->nb_input_objects = atoi(token2);

    if(pd->nb_input_objects <= 0){
      fprintf(stderr, "%s: Number of input objects must be greater than 0 (%d)\n",
              pd->nickname, pd->nb_input_objects);
      return -1;
    }
    pd->input_objects = (NS_Object **)calloc(
            pd->nb_input_objects,sizeof(NS_Object *));

    fprintf(initcodefile,"  pb_desc->nb_input_objects = %d;\n",pd->nb_input_objects);
    fprintf(initcodefile,"  pb_desc->input_objects = ");
    fprintf(initcodefile,"(NS_Object **)calloc(%d,%s);\n",
                 pd->nb_input_objects,"sizeof(NS_Object *)");

    fclose(f);

    for (i=0;i<pd->nb_input_objects;i++)
    {
      pd->input_objects[i] = parseObject(pd,filename,'i',i);
      if (pd->input_objects[i] == NULL)
      {
        fprintf(stderr,"%s:Error while parsing input object #%d\n",pd->nickname,i);
        return -1;
      }
      fprintf(initcodefile,"pb_desc->input_objects[%d] = obj;\n",i);
    }
    return 1;
  }
  fprintf(stderr,"%s:Missing input specification\n",pd->nickname);
  fclose(f);
  return -1;
}

/*
 * parseObject()
 *
 */
NS_Object *parseObject(NS_ProblemDesc *pd,char *filename, char io, int index)
{
  NS_Object *obj;
  char *token1,*token2,*token3,*token4;
  char *cleaned_line;
  char buf[256];
  FILE *f;
  int seenio=0;
  int count=0;

  f = fopen(filename,"r");
  if (f == NULL)
  {
    fprintf(stderr,"Impossible to open file '%s'\n",filename);
    return NULL;
  }

  while(1)
  {
    if (read_line(f) == NULL)
    {
      fprintf(stderr,"%s: interrupted file\n",pd->nickname);
      fclose(f);
      return NULL;
    }
    if (!strncmp(line,"@OUTPUT",7))
      if (io == 'o')
        seenio = 1;
    if (!strncmp(line,"@INPUT",6))
      if (io == 'i')
        seenio = 1;

    if (!seenio)
      continue;
    if (!strncmp(line,"@OBJECT",7))
      count++;
 
    if (count == index+1)
      break;
  }

  obj = newObject();
  token1 = strtok(line," \t\n");
  token2 = strtok(NULL," \t\n");
  token3 = strtok(NULL," \t\n");
  if (token3 != NULL)
    token4 = strtok(NULL," \t\n");
  else
    token4 = NULL;
  if ((token1 == NULL)||(token2 == NULL))
  {
    fprintf(stderr,"%s:@OBJECT <object type> [data type] <name>' expected\n",
            pd->nickname);
    fclose(f);
    return NULL;
  }
  if (!strcmp(token2,"MATRIX"))
  {
    if ((token3 == NULL)||(token4 == NULL))
    {
      fprintf(stderr,"%s:Data type and name expected\n",pd->nickname);
      fclose(f);
      return NULL;
    }
    obj->name = strdup(token4);
    obj->object_type = NETSOLVE_MATRIX;
  }
  else if (!strcmp(token2,"SPARSEMATRIX"))
  {
    if ((token3 == NULL)||(token4 == NULL))
    {
      fprintf(stderr,"%s:Data type and name expected\n",pd->nickname);
      fclose(f);
      return NULL;
    }
    obj->name = strdup(token4);
    obj->object_type = NETSOLVE_SPARSEMATRIX;
  }
  else if (!strcmp(token2,"VECTOR"))
  {
    if ((token3 == NULL)||(token4 == NULL))
    {
      fprintf(stderr,"%s:Data type and name expected\n",pd->nickname);
      fclose(f);
      return NULL;
    }
    obj->name = strdup(token4);
    obj->object_type = NETSOLVE_VECTOR;
  }
  else if (!strcmp(token2,"SCALAR"))
  {
    if ((token3 == NULL)||(token4 == NULL))
    {
      fprintf(stderr,"%s:Data type and name expected\n",pd->nickname);
      fclose(f);
      return NULL;
    }
    obj->name = strdup(token4);
    obj->object_type = NETSOLVE_SCALAR;
  }
  else if (!strcmp(token2,"STRINGLIST"))
  {
    if (token3 == NULL)
    {
      fprintf(stderr,"%s:Name expected\n",pd->nickname);
      fclose(f);
      return NULL;
    }
    obj->name = strdup(token3);
    obj->object_type = NETSOLVE_STRINGLIST;
    obj->data_type = NETSOLVE_CHAR;
  }
  else if (!strcmp(token2,"PACKEDFILES"))
  {
    if (token3 == NULL)
    {
      fprintf(stderr,"%s:Name expected\n",pd->nickname);
      fclose(f);
      return NULL;
    }
    obj->name = strdup(token3);
    obj->object_type = NETSOLVE_PACKEDFILES;
    obj->data_type = NETSOLVE_CHAR;
  }
  else if (!strcmp(token2,"FILE"))
  {
    if (token3 == NULL)
    {
      fprintf(stderr,"%s:Name expected\n",pd->nickname);
      fclose(f);
      return NULL;
    }
    obj->name = strdup(token3);
    obj->object_type = NETSOLVE_FILE;
    obj->data_type = NETSOLVE_CHAR;
  }
  else if (!strcmp(token2,"UPF"))
  {
    if (token3 == NULL)
    {
      fprintf(stderr,"%s:Name expected\n",pd->nickname);
      fclose(f);
      return NULL;
    }
    obj->name = strdup(token3);
    obj->object_type = NETSOLVE_UPF;
    obj->data_type = NETSOLVE_CHAR;
  }
  else if (!strcmp(token2,"STRING"))
  {
    if (token3 == NULL)
    {
      fprintf(stderr,"%s:Name expected\n",pd->nickname);
      fclose(f);
      return NULL;
    }
    obj->name = strdup(token3);
    obj->object_type = NETSOLVE_STRING;
    obj->data_type = NETSOLVE_CHAR;
  }
  else
  {
    fprintf(stderr,
      "%s:Object type should be one of the following:\n",
      pd->nickname);
    fprintf(stderr,"        'MATRIX'\n");
    fprintf(stderr,"        'SPARSEMATRIX'\n");
    fprintf(stderr,"        'VECTOR'\n");
    fprintf(stderr,"        'SCALAR'\n");
    fprintf(stderr,"        'FILE'\n");
    fprintf(stderr,"        'UPF'\n");
    fprintf(stderr,"        'STRING'\n");
    fprintf(stderr,"        'STRINGLIST'\n");
    fprintf(stderr,"        'PACKEDFILES'\n");
    fclose(f);
    return NULL;
  }
  if (token3 != NULL)
  {
    if ((obj->object_type != NETSOLVE_PACKEDFILES)&&
        (obj->object_type != NETSOLVE_STRINGLIST)&&
        (obj->object_type != NETSOLVE_UPF)&&
        (obj->object_type != NETSOLVE_FILE)&&
        (obj->object_type != NETSOLVE_STRING))
    {
      if (strcmp(token3,"I") == 0)
        obj->data_type = NETSOLVE_I;
      else if (strcmp(token3,"B") == 0)
        obj->data_type = NETSOLVE_B;
      else if (strcmp(token3,"CHAR") == 0)
        obj->data_type = NETSOLVE_CHAR;
      else if (strcmp(token3,"S") == 0)
        obj->data_type = NETSOLVE_S;
      else if (strcmp(token3,"D") == 0)
        obj->data_type = NETSOLVE_D;
      else if (strcmp(token3,"C") == 0)
        obj->data_type = NETSOLVE_C;
      else if (strcmp(token3,"Z") == 0)
        obj->data_type = NETSOLVE_Z;
      else
      {
        fprintf(stderr,
          "%s:Data type should be one of the following:\n",
          pd->nickname);
        fprintf(stderr,"        'CHAR'     : char\n");
        fprintf(stderr,"        'B'        : byte\n");
        fprintf(stderr,"        'I'        : integer\n");
        fprintf(stderr,"        'S'        : single precision real\n");
        fprintf(stderr,"        'D'        : double precision real\n");
        fprintf(stderr,"        'C'        : single precision complex\n");
        fprintf(stderr,"        'Z'        : double precision complex\n");
        fclose(f);
        return NULL;
      }
    }
  }

  fprintf(initcodefile,"  obj = createEmptyObject(%d,%d);\n",
                               obj->object_type,obj->data_type);
  fprintf(initcodefile,"  obj->name = strdup(\"%s\");\n",obj->name);

  obj->description = strdup(" ");
  fprintf(initcodefile,"   obj->description = strdup(\" \");\n");
  if (read_line(f) == NULL)
  {
    fprintf(stderr,"%s: interrupted object description\n",pd->nickname);
    fclose(f);
    return NULL;
  }
  if (line[0] == '@')
  {
    fprintf(stderr,"%s: object description required\n",pd->nickname);
    fclose(f);
    return NULL;
  }
  while (line[0] != '@')
  {
    cleaned_line = cleaned(line);
    strcpy(buf,cleaned_line);
    buf[strlen(buf)-1] = '\0';
    fprintf(initcodefile,"  strcpy(buf,\"%s\");\n",cleaned_line);
    free(cleaned_line);
    obj->description = (char *)myRealloc(
             obj->description,
             strlen(obj->description)+
             strlen(buf)+2);
    fprintf(initcodefile,"  obj->description = (char *)myRealloc(");
    fprintf(initcodefile,"obj->description,%d);\n",
             strlen(obj->description)+strlen(buf)+2);
    strcat(obj->description,buf);
    strcat(obj->description,"\n");
    fprintf(initcodefile,"  strcat(obj->description,buf);\n");
    if (read_line(f) == NULL)
    {
      fprintf(stderr,"%s: interrupted object description\n",pd->nickname);
      fclose(f);
      return NULL;
    }
  }
  fclose(f);
  return obj;
}

int parseOUTPUT(NS_ProblemDesc *pd,NS_ProblemDescAddendum *pdadd,char *filename)
{
  FILE *f;
  char *token1,*token2;
  int i;

  f = fopen(filename,"r");
  if (f == NULL)
  {
    fprintf(stderr,"Impossible to open file '%s'\n",filename);
    return -1;
  }

  while(read_line(f) != NULL)
  {
    if (strncmp(line,"@OUTPUT ",8))
      continue;

    token1 = strtok(line," \t\n");
    token2 = strtok(NULL," \t\n");
    if ((token1 == NULL)||(token2 == NULL))
    {
      fprintf(stderr,"%s:'@OUTPUT <number>' expected\n",pd->nickname);
      fclose(f);
      return -1;
    }
    if (!isInteger(token2))
    {
      fprintf(stderr,"%s:'%s': invalid number of output\n",pd->nickname,token2);
      fclose(f);
      return -1;
    }
    pd->nb_output_objects = atoi(token2);
    if(pd->nb_output_objects <= 0){
      fprintf(stderr, "%s: Number of output objects must be greater than 0 (%d)\n",
              pd->nickname, pd->nb_output_objects);
      return -1;
    }
    pd->output_objects = (NS_Object **)calloc(
            pd->nb_output_objects,sizeof(NS_Object *));

    fprintf(initcodefile,"  pb_desc->nb_output_objects = %d;\n",pd->nb_output_objects);
    fprintf(initcodefile,"  pb_desc->output_objects = ");
    fprintf(initcodefile,"(NS_Object **)calloc(%d,%s);\n",
                 pd->nb_output_objects,"sizeof(NS_Object *)");

    fclose(f);
    for (i=0;i<pd->nb_output_objects;i++)
    {
      pd->output_objects[i] = parseObject(pd,filename,'o',i);
      if (pd->output_objects[i] == NULL)
      {
        fprintf(stderr,"%s:Error while parsing output object #%d\n",pd->nickname,i);
        return -1;
      }
      fprintf(initcodefile,"pb_desc->output_objects[%d] = obj;\n",i);
    }
    return 1;
  }
  fprintf(stderr,"%s:Missing output specification\n",pd->nickname);
  fclose(f);
  return -1;
}


int parseMATLAB_MERGE(NS_ProblemDesc *pd,NS_ProblemDescAddendum *pdadd,char *filename)
{
  FILE *f;
  char *token1,*token2,*token3;

  f = fopen(filename,"r");
  if (f == NULL)
  {
    fprintf(stderr,"Impossible to open file '%s'\n",filename);
    return -1;
  }

  while(read_line(f) != NULL)
  {
    if (strncmp(line,"@MATLAB_MERGE ",14))
      continue;

    token1 = strtok(line," \t\n");
    token2 = strtok(NULL," \t\n");
    token3 = strtok(NULL," \t\n");
    if ((token1 == NULL)||(token2 == NULL)||(token3 == NULL))
    {
      fprintf(stderr,"%s:'@MATLAB_MERGE <#> <#>' expected\n",pd->nickname);
      fclose(f);
      return -1;
    }
    if (!isInteger(token2))
    {
      fprintf(stderr,"%s:'%s' should be a number\n",
              pd->nickname,token2);
      fclose(f);
      return -1;
    }
    if (!isInteger(token3))
    {
      fprintf(stderr,"%s:'%s' should be a number\n",
              pd->nickname,token3);
      fclose(f);
      return -1;
    }
    if ((atoi(token2) >= pd->nb_output_objects)||
       (atoi(token3) >= pd->nb_output_objects))
    {
      fprintf(stderr,"%s: Matlab merge: only %d objects in output\n",
              pd->nickname,pd->nb_output_objects);
      fclose(f);
      return -1;
    }
    if ((atoi(token2) != (atoi(token3)-1)))
    {
      fprintf(stderr,"%s:Merges can occur only with consecutive objects\n",
              pd->nickname);
      fclose(f);
      return -1;
    }
    
    /* UPDATE THE INITCODEFILE */
    fprintf(initcodefile,"matlabtmp.index1 = %d;\n",atoi(token2));
    fprintf(initcodefile,"matlabtmp.index2 = %d;\n",atoi(token3));
    (pd->nb_matlab_merge)++;
    fprintf(initcodefile,"pb_desc->nb_matlab_merge=%d;\n",pd->nb_matlab_merge);
    fprintf(initcodefile,"pb_desc->matlab_merge= (NS_MatlabMerge *)");
    fprintf(initcodefile,"myRealloc(pb_desc->matlab_merge,");
    fprintf(initcodefile,"%d*sizeof(NS_MatlabMerge));\n",
                 pd->nb_matlab_merge);
    fprintf(initcodefile,"(pb_desc->matlab_merge)[%d].index1=matlabtmp.index1;\n\n",
            pd->nb_matlab_merge-1);
    fprintf(initcodefile,"(pb_desc->matlab_merge)[%d].index2=matlabtmp.index2;\n\n",
            pd->nb_matlab_merge-1);
  }
  fclose(f);
  return 1;
}

int parseCUSTOMIZED(NS_ProblemDesc *pd,NS_ProblemDescAddendum *pdadd,char *filename)
{
  FILE *f;
  char *token1,*token2;

  f = fopen(filename,"r");
  if (f == NULL)
  {
    fprintf(stderr,"Impossible to open file '%s'\n",filename);
    return -1;
  }

  while(read_line(f) != NULL)
  {
    if (strncmp(line,"@CUSTOMIZED ",12))
      continue;

    token1 = strtok(line," \t\n");
    token2 = strtok(NULL," \t\n");
    if ((token1 == NULL)||(token2 == NULL))
    {
      fprintf(stderr,"%s:'@CUSTOMIZED <name>' expected\n",pd->nickname);
      fclose(f);
      return -1;
    }
    fclose(f);
    pd->customized=strdup(token2);
    /* UPDATE THE INITCODEFILE */
    fprintf(initcodefile,"  pb_desc->customized = strdup(\"%s\");\n",pd->customized);
    return 1;
  }
  fclose(f);
  if (is_condor_server)
  {
    pd->customized = strdup("CONDOR");
    fprintf(initcodefile,"  pb_desc->customized = strdup(\"%s\");\n",pd->customized);
    return 1;
  }
  pd->customized = strdup("NONE");
  fprintf(initcodefile,"  pb_desc->customized = strdup(\"%s\");\n",pd->customized);
  return 1;
}

int parseCOMPLEXITY(NS_ProblemDesc *pd,NS_ProblemDescAddendum *pdadd,char *filename)
{
  FILE *f;
  char *token1,*token2;

  f = fopen(filename,"r");
  if (f == NULL)
  {
    fprintf(stderr,"Impossible to open file '%s'\n",filename);
    return -1;
  }

  while(read_line(f) != NULL)
  {
    if (strncmp(line,"@COMPLEXITY ",6))
      continue;

    token1 = strtok(line," \t\n");
    token2 = strtok(NULL," \t\n");
    if ((token1 == NULL)||(token2 == NULL))
    {
      fprintf(stderr,"%s:'@COMPLEXITY <string>' expected\n",pd->nickname);
      fclose(f);
      return -1;
    }
    pd->complexity=strdup(token2);
    fclose(f);
    /* UPDATE THE INITCODEFILE */
    fprintf(initcodefile,"  pb_desc->complexity = strdup(\"%s\");\n",
             pd->complexity);
    return 1;
  }
  fprintf(stderr,"%s:Missing complexity specification\n",pd->nickname);
  fclose(f);
  return -1;
}

int parseCALLINGSEQUENCE(NS_ProblemDesc *pd,NS_ProblemDescAddendum *pdadd,char *filename)
{
  FILE *f;

  f = fopen(filename,"r");
  if (f == NULL)
  {
    fprintf(stderr,"Impossible to open file '%s'\n",filename);
    return -1;
  }

  while(read_line(f) != NULL)
  {
    if (strncmp(line,"@CALLINGSEQUENCE",16))
      continue;

    if (parseCallingSequence(f,pd) == -1)
    {
      fclose(f);
      return -1;
    }

    fclose(f);
    return 1;
  }
  fprintf(stderr,"%s:Missing calling sequence specification\n",pd->nickname);
  fclose(f);
  return -1;
}

/* 
 * Sudesh: adding this function to free list memory 
 */

void free_list(char **list,int num_list){

  int i = 0;
  
  for(i = 0;i< num_list;i++){
   if(list[i] != NULL)
    free(list[i]); 
  }

   free(list);
}

/*
 * parseCallingSequence()
 */
int parseCallingSequence(FILE *f, NS_ProblemDesc *pd)
{
  char **mnemonic_list;
  char **formula_list;
  char **const_list;
  int mnemonic_index = 0;
  int nb_arg = 0;
  int nb_const = 0;
  int nb_formula = 0;
  char *token1,*token2,*token3;
  int i = 0;

  /* Maximum case */
  mnemonic_list =
        (char **)calloc(5*(pd->nb_input_objects+pd->nb_output_objects),
        sizeof(char *));
  formula_list =
        (char **)calloc(5*(pd->nb_input_objects+pd->nb_output_objects),
        sizeof(char *));
  const_list =
        (char **)calloc(5*(pd->nb_input_objects+pd->nb_output_objects),
        sizeof(char *));

  /* Sudesh : adding this loop */
  for(i = 0;i<5;i++){
    mnemonic_list[i] = NULL;
    formula_list[i] = NULL;
    const_list[i] = NULL;
  }
 
  while(read_line(f) != NULL)
  {
    while(strncmp(line,"@ARG",4) == 0)
    {
      int type_mnemonic;
      /* char *token_const; */
  
      token1 = strtok(line," \t\n");
      token2 = strtok(NULL,"=\n");
      /* token_const = strtok(NULL," \t\n"); */
      if (token2 == NULL)
      {
        fprintf(stderr,"Argument list expected at line %d\n",
              linecount);
        return -1;
      }
      /* Check the mnemonics */
      mnemonic_list[mnemonic_index] = strdup(token2);
      if (strncmp(token1,"@ARG",4) != 0)
      {
        fprintf(stderr,"Invalid line at line %d\n",linecount);
        return -1;
      }
      mnemonic_index++;
      nb_arg++;
      token3 = strtok(token2," \t,");
      if (token3 == NULL)
      {
        fprintf(stderr,"Invalid mnemonic at line %d\n",linecount);
        return -1;
      }
      else
      {
        type_mnemonic = checkMnemonic(token3,pd);
      }
      while(token3 != NULL)
      {
        int type;
  
        type= checkMnemonic(token3,pd);
        if (type<0)
          return -1;
        if (type != type_mnemonic)
        {
          fprintf(stderr,"%s:Incoherent integer/pointer mnemonic specification\n",pd->nickname);
          return -1;
        }
  
        token3 = strtok(NULL," \t,");
      }
      if (read_line(f) == NULL)
      {
        fprintf(stderr,"Format section interrupted at line %d\n",linecount);
        return -1;
      }
    }
  
    /* Constants */
    while(strncmp(line,"@CONST",4) == 0)
    {
      int type_mnemonic;
      char *token_const;
  
      token1 = strtok(line," \t\n");
      token2 = strtok(NULL,"=\n");
      token_const = strtok(NULL," \t\n");
      if (token2 == NULL)
      {
        fprintf(stderr,"Argument expected at line %d\n",
              linecount);
        return -1;
      }
      /* Check the mnemonics */
      mnemonic_list[mnemonic_index] = strdup(token2);
  
      if (token_const == NULL)
      {
        fprintf(stderr,"A constant value must be specified at line %d\n",linecount);
        return -1;
      }
      if (!isInteger(token_const))
      {
        fprintf(stderr,"Invalid constant value must be specified at line %d\n",linecount);
        return -1;
      }
      const_list[nb_const] = strdup(token_const);
  
      mnemonic_index++;
      nb_const++;
  
      type_mnemonic = checkMnemonic(token2,pd);
      if (type_mnemonic<0)
          return -1;
      if (type_mnemonic != 1)
      {
        fprintf(stderr,"Only an integer can be a constant\n");
        return -1;
      }
      if (read_line(f) == NULL)
      {
        fprintf(stderr,"Format section interrupted at line %d\n",linecount);
        return -1;
      }
    }
  
    /* Formulae */
  
    while(strncmp(line,"@COMP",5) == 0)
    {
      int  type_mnemonic;
      char *token_form_line[10];
      int i,j;
      int length = 0;
      char *token_form = NULL;
      char *there;
      char *tmp;
  
      token1 = strtok(line," \t\n");
      token1 = strdup(token1);
      token2 = strtok(NULL,"=\n");
      token2 = strdup(token2);
      /* Get the different lines */
      tmp = strtok(NULL,"\n");
      for (i=0;i<10;i++)
      {
  
        if (tmp == NULL)
        {
          i = 0;
          length = 0;
          break;
        }
        if (tmp[strlen(tmp)-1] == '\\')
        {
          tmp[strlen(tmp)-1] = '\0';
          token_form_line[i] = strdup(tmp);  
          length+=strlen(tmp);
        } 
        else
        {
          token_form_line[i] = strdup(tmp);
          length+=strlen(tmp);
          break;
        }
        if (read_line(f) == NULL)
        {
          fprintf(stderr,"Format section interrupted at line %d\n",linecount);
          return -1;
        }
        tmp = strtok(line,"\n");; 
      }
      i++;
  
      /* concatenate into a big stuff */
      if (length != 0)
        token_form = (char *)calloc(1+length,sizeof(char));
      else
        i = 0;
      there = token_form;
      for(j=0;j<i;j++)
      {
        strncpy(there,token_form_line[j],strlen(token_form_line[j]));
        there+=strlen(token_form_line[j]);
        free(token_form_line[j]);
      } 
      token_form[length] = '\0';
      if (token2 == NULL)
      {
        fprintf(stderr,"'@COMP <arg list> <formula>' expected at line %d\n",
                linecount);
        return -1;
      }
      if (token_form == NULL)
      {
        fprintf(stderr,"A expression must be specified at line %d\n",
                linecount);
        return -1;
      }
      if (checkExpression(token_form,mnemonic_list,mnemonic_index,pd) < 0)
      {
        return -1;
      }
      mnemonic_list[mnemonic_index] = strdup(token2);
    	  formula_list[nb_formula] = strdup(token_form);
      nb_formula++;
      mnemonic_index++;
      type_mnemonic = checkMnemonic(token2,pd);
      if (type_mnemonic <0)
        return -1;
      if (type_mnemonic != 1)
      {
        fprintf(stderr,"Only an integer can be computed by an ");
        fprintf(stderr,"expression at line %d\n",linecount);
        return -1;
      }
      if (read_line(f) == NULL)
      {
        fprintf(stderr,"Format section interrupted at line %d\n",linecount);
        return -1;
      }

      /* Sudesh: freeing token_form, removes 150bytes of MLK*/
        free(token_form); 
    }

    /* Now the list of mnemonics is constructed */
    if (checkMnemonicList(mnemonic_list,pd) < 0)
      return -1;
    else
    {    /* Update the problem descriptor */
      int j;
   
  
      pd->calling_sequence = (NS_CallingSequence *)
                   calloc(1,sizeof(NS_CallingSequence));
      fprintf(initcodefile,"  pb_desc->calling_sequence = (NS_CallingSequence *)");
      fprintf(initcodefile,"calloc(1,sizeof(NS_CallingSequence));\n");
      pd->calling_sequence->nb_args = nb_arg;
      pd->calling_sequence->nb_constants = nb_const;
      pd->calling_sequence->nb_formulae = nb_formula;
      fprintf(initcodefile,
          "  pb_desc->calling_sequence->nb_args = %d;\n",nb_arg);
      fprintf(initcodefile,
          "  pb_desc->calling_sequence->nb_constants = %d;\n",nb_const);
      fprintf(initcodefile,
          "  pb_desc->calling_sequence->nb_formulae = %d;\n",nb_formula);
  
      pd->calling_sequence->arg_to_mnemonic = 
               (char **)calloc(nb_arg,sizeof(char *));
      fprintf(initcodefile,"  pb_desc->calling_sequence->arg_to_mnemonic =\n");
      fprintf(initcodefile,
            "       (char **)calloc(%d,sizeof(char *));\n",nb_arg);
      pd->calling_sequence->constants =
               (NS_Constant *)calloc(nb_const,sizeof(NS_Constant));
      fprintf(initcodefile,"  pb_desc->calling_sequence->constants =");
      fprintf(initcodefile,
           "    (NS_Constant *)calloc(%d,sizeof(NS_Constant));\n",nb_const);
      pd->calling_sequence->formulae =
               (NS_Formula *)calloc(nb_formula,sizeof(NS_Formula));
      fprintf(initcodefile,"  pb_desc->calling_sequence->formulae =");
      fprintf(initcodefile,
           "    (NS_Formula *)calloc(%d,sizeof(NS_Formula));\n",nb_formula);
  
      for (j=0;j<nb_arg;j++)
      {
        pd->calling_sequence->arg_to_mnemonic[j] = 
                   strdup(mnemonic_list[j]);
        fprintf(initcodefile,"  pb_desc->calling_sequence->arg_to_mnemonic[%d] =",j);
        fprintf(initcodefile,"strdup(\"%s\");\n",mnemonic_list[j]);
        free(mnemonic_list[j]);

        /* Sudesh : setting NULL to make sure not to free it again*/
        mnemonic_list[j] = NULL;
      }
      for (j=0;j<nb_const;j++)
      {
        pd->calling_sequence->constants[j].value =
          strdup(const_list[j]);
        pd->calling_sequence->constants[j].mnemonic = 
          strdup(mnemonic_list[nb_arg+j]);
  
  
        if (((pd->calling_sequence->constants[j].mnemonic)[0] == 'O')||
           (((pd->calling_sequence->constants[j].mnemonic)[1]) == 'O')||
           (((pd->calling_sequence->constants[j].mnemonic)[0]) == 'I'))
        {
          fprintf(stderr,
            "'%s' cannot be a constant (at line %d)\n",
              pd->calling_sequence->constants[j].mnemonic,
              linecount);
          return -1;
        }
        fprintf(initcodefile,"  pb_desc->calling_sequence->constants[%d].value=",j);
        fprintf(initcodefile,"strdup(\"%s\");\n",const_list[j]); 
        fprintf(initcodefile,"  pb_desc->calling_sequence->constants[%d].mnemonic=",j);
        fprintf(initcodefile,"strdup(\"%s\");\n",mnemonic_list[nb_arg+j]); 
  
      }
      for (j=0;j<nb_formula;j++)
      {
        pd->calling_sequence->formulae[j].expression =
          strdup(formula_list[j]);
        pd->calling_sequence->formulae[j].mnemonic = 
          strdup(mnemonic_list[nb_arg+j]);
  
  
        if (((pd->calling_sequence->formulae[j].mnemonic)[0] == 'O')||
           (((pd->calling_sequence->formulae[j].mnemonic)[1]) == 'O')||
           (((pd->calling_sequence->formulae[j].mnemonic)[0]) == 'I'))
        {
          fprintf(stderr,
            "'%s' cannot be a constant (at line %d)\n",
              pd->calling_sequence->formulae[j].mnemonic,
              linecount);
          return -1;
        }
        fprintf(initcodefile,"  pb_desc->calling_sequence->formulae[%d].expression=",j);
        fprintf(initcodefile,"strdup(\"%s\");\n",formula_list[j]);
        fprintf(initcodefile,"  pb_desc->calling_sequence->formulae[%d].mnemonic=",j);
        fprintf(initcodefile,"strdup(\"%s\");\n",mnemonic_list[nb_arg+j]);
      }
    }

    /* Sudesh: adding this 3 free_list() calls  */
    free_list(mnemonic_list,5);
    free_list(formula_list,5);
    free_list(const_list,5);
    return 1;
  }
    /* Sudesh: adding this 3 free_list() calls */
    free_list(mnemonic_list,5);
    free_list(formula_list,5);
    free_list(const_list,5);
  return 0;
}

int parseCODE(NS_ProblemDesc *pd,NS_ProblemDescAddendum *pdadd,char *filename)
{
  FILE *f;
  char buf[256];
  char *upper_code = 0;

  f = fopen(filename,"r");
  if (f == NULL)
  {
    fprintf(stderr,"Impossible to open file '%s'\n",filename);
    return -1;
  }

  while(read_line(f) != NULL)
  {
    if (strncmp(line,"@CODE",5))
      continue;

    /* Insert the pseudo-header */
    sprintf(buf,"#if (defined(flag_%s)||(!defined(UPF)))\n",pd->nickname);
    write_line(buf);
    write_line("/****************************/\n");
    sprintf(buf,"/* Problem : %s */\n",pd->nickname);
    write_line(buf);
    write_line("/****************************/\n");
    sprintf(buf,"if (inList(list,\"%s\")) {\n",pd->file);
    upper_code = strdup(pd->nickname);
    netsolveCaps(upper_code);
    sprintf(buf,"if (strcasecmp(buf1,\"%s\") == 0)\n{\n\n",upper_code);
    write_line(buf);

    /* Process the code */

    fprintf(stderr,"*** %s ***\n",pd->nickname);
    if (processCode(f,pd) < 0)
    {
      fprintf(stderr,"Impossible to generate the code for : %s\n",
           pd->nickname);
      exit(-1);
    }

    /* Insert the end of problem section */
    write_line("\n}\n\n");
    write_line("#endif\n");
    fclose(f);
    free(upper_code);
    return 1;
  }
  fprintf(stderr,"%s:Missing code section\n",pd->nickname);
  fclose(f);
  if (upper_code != NULL)
    free(upper_code);
  return -1;
}

