/* test_bmmc_mpi.c

   Code to test the BMMC-MPI permutation functions.
   */

/* $Id: test_bmmc_mpi.c,v 1.3 1997/05/06 22:02:30 thc Exp $
   $Log: test_bmmc_mpi.c,v $
   Revision 1.3  1997/05/06 22:02:30  thc
   Added copyright notice.

   Revision 1.2  1997/05/05 14:18:41  thc
   Cosmetic changes.

   Revision 1.1  1997/04/18 18:23:59  thc
   Initial revision

   */

/*
 * Copyright (C) 1997, Thomas H. Cormen, thc@cs.dartmouth.edu
 *
 * This software may be freely copied, modified, and redistributed,
 * provided that this copyright notice is preserved on all copies.
 *
 * There is no warranty or other guarantee of fitness for this
 * software, and it is provided solely "as is".  Bug reports or fixes
 * may be sent to the author, who may or may not act on them as he
 * desires.
 *
 * Rights are granted to use this software in any non-commercial
 * enterprise.  For commercial rights to this software, please contact
 * the author.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "mpi.h"
#include "bit_matrix_types.h"
#include "bit_matrix_fns.h"
#include "bmmc_mpi.h"

/* Fill a buffer with identity data. */
static void fill_buffer(long int *buffer,
			int n,
                        int p,
			int f,
			int proc_id);

/* Check whether the result of a BMMC permutation is correct. */
static int check_BMMC(bit_matrix A,
		      matrix_column c,
		      long int *data,
		      int n,
		      int p,
		      int f,
		      int proc_id);

/* Get a random BMMC permutation. */
static void get_random_BMMC(bit_matrix A,
			    matrix_column *c,
			    int proc_id,
			    int n);

/* Generate a random matrix. */
static void generate_random_matrix(bit_matrix A,
				   int m,
				   int n);

/* Read a characteristic matrix and complement vector from stdin. */
static void read_BMMC(bit_matrix A,
		      matrix_column *c,
		      int proc_id,
		      int n);


void main(int argc, char **argv)
{
#define CHECK_ERROR(code,func)						\
  if (code != BMMC_MPI_OK)						\
    {									\
      fprintf(stderr, "Error code %d from %s, exiting\n", code, func);	\
      MPI_Abort(MPI_COMM_WORLD, 1);					\
      exit(1);								\
    }

  /* An arbitrary magic number. */
#define RING_IN 95

  int n;			/* log of problem size */
  long int N;			/* problem size */
  int random_A = 0;		/* should we generate a random matrix? */
  int random_c = 0;		/* should we generate a random comp vector? */
  int arg = 1;			/* current arg number being processed */
  int trials = 1;		/* how many trials to run */
  int p;			/* log of number of processors */
  int P;			/* number of processors */
  int f;			/* least signif processor number bit */
  int proc_id;			/* my processor id */
  long int *data;		/* data buffer */
  long int *temp;		/* temp buffer */
  int trial_number;		/* which trial we're on */
  bit_matrix A;			/* characteristic matrix */
  matrix_column c;		/* complement vector */
  BMMC_MPI_factor_info info;	/* factor information */
  int error_code = BMMC_MPI_OK;	/* an error code */
  MPI_Status status;		/* MPI message status */

  MPI_Init(&argc, &argv);

  /* Process command line arguments. */
  if (argc < 3 || argc > 5)
    {
      fprintf(stderr, "Usage: %s [-r|m] n f [trials]\n", argv[0]);
      MPI_Abort(MPI_COMM_WORLD, 1);
      exit(1);
    }

  /* See if the -r (random matrix and comp vector) or -m (random
     matrix, comp vector is 0) option is used. */
  if (strcmp(argv[arg], "-r") == 0)
    {
      random_A = random_c = 1;	/* random matrix and comp vector */
      arg++;			/* so skip over this arg */
    }
  else if (strcmp(argv[arg], "-m") == 0)
    {
      random_A = 1;		/* random matrix */
      arg++;			/* so skip over this arg */
    }

  n = atoi(argv[arg++]);
  N = ((long int) 1) << n;

  if (arg < argc)
    f = atoi(argv[arg++]);
  else
    {
      fprintf(stderr, "Usage: %s [-r|m] n f [trials]\n", argv[0]);
      MPI_Abort(MPI_COMM_WORLD, 1);
      exit(1);
    }

  if (arg < argc)
    trials = atoi(argv[arg]);

  /* Find out how many processors there are and which one I am. */
  MPI_Comm_rank(MPI_COMM_WORLD, &proc_id);
  MPI_Comm_size(MPI_COMM_WORLD, &P);

  /* Ring in. */
  if (proc_id == 0)
    {
      int other_id;		/* other processor id's */

      /* Tell each other processor to ring in, and wait for it to have
	 done so. */
      fprintf(stderr, "Processor %d is %s\n", proc_id, getenv("HOST"));
      for (other_id = 1; other_id < P; other_id++)
	{
	  MPI_Send(NULL, 0, MPI_BYTE, other_id, RING_IN, MPI_COMM_WORLD);
	  MPI_Recv(NULL, 0, MPI_BYTE, other_id, RING_IN, MPI_COMM_WORLD,
		   &status);
	}
    }
  else
    {
      /* Wait to be told to ring in, do so, and tell processor 0 that
	 we did. */
      MPI_Recv(NULL, 0, MPI_BYTE, 0, RING_IN, MPI_COMM_WORLD, &status);
      fprintf(stderr, "Processor %d is %s\n", proc_id, getenv("HOST"));
      MPI_Send(NULL, 0, MPI_BYTE, 0, RING_IN, MPI_COMM_WORLD);
    }

  /* Determine the log of the problem size. */
  for (p = 0; (1 << p) < P; p++)
    ;
  if (1 << p != P)
    {
      fprintf(stderr, "Error: number of processors is %d, "
	      "but must be a power of 2\n", P);
      MPI_Abort(MPI_COMM_WORLD, 1);
      exit(1);
    }	      

  /* Allocate two buffers of the correct size. */
  data = calloc(N/P, sizeof(long int));
  temp = calloc(N/P, sizeof(long int));

  /* Allocate the matrices A and A_inv. */
  A = allocate_bit_matrix(n);

  /* Run the desired number of trials. */
  for (trial_number = 0; trial_number < trials; trial_number++)
    {
      if (proc_id == 0)
	printf("Starting trial %d.\n", trial_number+1);

      /* Get the test data, either by reading it in, or by randomly
	 generating it. */
      if (random_A)
	{
	  get_random_BMMC(A, &c, proc_id, n);
	  if (!random_c)
	    c = 0;
	}
      else
	read_BMMC(A, &c, proc_id, n);

      /* Print out the characteristic matrix and complement vector. */
      if (proc_id == 0)
	{
	  int i;
	  print_bit_matrix(A, n, n, "A");
	  printf("c:\n   ");
	  for (i = 0; i < n; i++)
	    printf("%d ", (int) ((c >> i) & 1));
	  printf("\n\n");
	}

      fill_buffer(data, n, p, f, proc_id);

      /* Do the preprocessing. */
      if (proc_id == 0)
	printf("Preprocessing...\n");
      error_code = factor_BMMC_MPI(A, c, n, p, f, &info);
      CHECK_ERROR(error_code, "factor_BMMC_MPI");

      /* Perform the permutation. */
      if (proc_id == 0)
	printf("Permuting...\n");
      error_code = perform_BMMC_MPI(&info, n, p, proc_id, MPI_COMM_WORLD,
				    sizeof(long int), data, temp);
      CHECK_ERROR(error_code, "perform_BMMC_MPI");

      /* Free up all the factoring information. */
      free_BMMC_MPI_factor_info(&info);

#if 0
      error_code = BMMC_MPI(A, c, n, p, f, proc_id, MPI_COMM_WORLD,
			    sizeof(long int), data, temp);
      CHECK_ERROR(error_code, "BMMC_MPI");
#endif

#if 0
      /* If testing BMMC_MPI_proc_minor, make sure that f is set to 0
	 before entering the main loop!! */
      error_code = BMMC_MPI_proc_minor(A, c, n, p, proc_id, MPI_COMM_WORLD,
				       sizeof(long int), data, temp);
      CHECK_ERROR(error_code, "BMMC_MPI_proc_minor");
#endif

#if 0
      /* If testing BMMC_MPI_proc_major, make sure that f is set to n-p
	 before entering the main loop!! */
      error_code = BMMC_MPI_proc_major(A, c, n, p, proc_id, MPI_COMM_WORLD,
				       sizeof(long int), data, temp);
      CHECK_ERROR(error_code, "BMMC_MPI_proc_major");
#endif

      /* Check the result. */
      if (proc_id == 0)
	printf("Checking...");
      if (check_BMMC(A, c, data, n, p, f, proc_id) && proc_id == 0)
	printf("correct\n");
    }

  free_bit_matrix(A);
  free(temp);
  free(data);

  MPI_Finalize();
  exit(0);
}


/* Fill a buffer with identity data. */
static void fill_buffer(long int *buffer,
			int n,
                        int p,
			int f,
			int proc_id)
{
  long int offset;		/* offset into buffer */
  long int low_mask;		/* mask for low bits */
  long int proc_part;		/* processor part of the data */
  long int vpr;			/* N/P */

  vpr = ((long int) 1) << (n-p);
  proc_part = ((long int) proc_id) << f;
  low_mask = ~((~((long int) 0)) << f);

  for (offset = 0; offset < vpr; offset++)
      buffer[offset] = proc_part | (offset & low_mask) |
	                 ((offset >> f) << (f+p));
}


/* Check whether the result of a BMMC permutation is correct.
   Return 1 if correct, 0 if error. */
static int check_BMMC(bit_matrix A,
		      matrix_column c,
		      long int *data,
		      int n,
		      int p,
		      int f,
		      int proc_id)
{
  bit_matrix A_inv = allocate_bit_matrix(n); /* inverse of A */
  long int offset;		/* offset into buffer */
  long int low_mask;		/* mask for low bits */
  long int proc_part;		/* processor part of a target index */
  long int vpr;			/* N/P */
  long int source_index;	/* corresponding source index */
  int correct = 1;		/* return value */

  vpr = ((long int) 1) << (n-p);
  proc_part = ((long int) proc_id) << f;
  low_mask = ~((~((long int) 0)) << f);
  invert_bit_matrix(A_inv, A, n);

#ifdef DEBUG
  printf("Proc %d result:\n", proc_id);
#endif

  for (offset = 0; offset < vpr; offset++)
    {
#ifdef DEBUG
      printf("%ld: %lx\n", offset, data[offset]);
#endif

      source_index = bit_matrix_vector_multiply(A_inv,
		       (proc_part | (offset & low_mask) |
			((offset >> f) << (f+p))) ^ c, n);
      if (data[offset] != source_index)
	{
	  if (correct)
	    printf("\n");	/* if first error, need a new line */
	  correct = 0;
	  printf("Mismatch on processor %d, offset = %ld: "
		 "expect %lx, got %lx\n", proc_id, offset,
		 source_index, data[offset]);
	}
    }

  free_bit_matrix(A_inv);

  return correct;
}

/* Get a random BMMC permutation. */
static void get_random_BMMC(bit_matrix A,
			    matrix_column *c,
			    int proc_id,
			    int n)
{
  int rank;			/* rank of the matrix */
  matrix_column basis;		/* basis...not really used */

  /* Only the leader should try to come up with a nonsingular matrix
     and complement vector.  It then broadcasts them. */
  if (proc_id == 0)
    {
      do
	{
	  generate_random_matrix(A, n, n);
	  rank = find_bit_matrix_basis(&basis, A, n, n);
	}
      while (rank < n);
      generate_random_matrix((bit_matrix) c, n, 1);
    }

  MPI_Bcast(A, n * sizeof(matrix_column) / sizeof(unsigned long),
	    MPI_UNSIGNED_LONG, 0, MPI_COMM_WORLD);
  MPI_Bcast(c, sizeof(matrix_column) / sizeof(unsigned long),
	    MPI_UNSIGNED_LONG, 0, MPI_COMM_WORLD);
}


/* Generate a random matrix. */
void generate_random_matrix(bit_matrix A,
			    int m,
			    int n)
{
  int i, j;

  for (j = 0; j < n; j++)
    {
      A[j] = 0;

      for (i = 0; i < m; i++)
	if (random() & 1)
	  A[j] |= ((matrix_column) 1) << i;
    }
}


/* Read a characteristic matrix and complement vector from stdin. */
static void read_BMMC(bit_matrix A,
		      matrix_column *c,
		      int proc_id,
		      int n)
{
  int i, j;			/* row and column numbers */
  int bit;			/* bit read in */

  /* Only the leader should read in the characteristic matrix and
     complement vector.  It then broadcasts them. */
  if (proc_id == 0)
    {
      /* Zero out A and c. */
      for (j = 0; j < n; j++)
	A[j] = 0;
      *c = 0;

      /* Read in A and c from stdin, setting 1's appropriately. */
      for (i = 0; i < n; i++)
	for (j = 0; j < n; j++)
	  {
	    scanf("%d", &bit);
	    if (bit)
	      A[j] |= ((matrix_column) 1) << i;
	  }

      for (i = 0; i < n; i++)
	{
	  scanf("%d", &bit);
	  if (bit)
	    *c |= ((matrix_column) 1) << i;
	}
    }

  MPI_Bcast(A, n * sizeof(matrix_column) / sizeof(unsigned long),
	    MPI_UNSIGNED_LONG, 0, MPI_COMM_WORLD);
  MPI_Bcast(c, sizeof(matrix_column) / sizeof(unsigned long),
	    MPI_UNSIGNED_LONG, 0, MPI_COMM_WORLD);
}
