/* bmmc_mpi.h

   Header file for BMMC-MPI permutation functions.
   */

/* $Id: bmmc_mpi.h,v 1.2 1997/05/06 22:02:30 thc Exp $
   $Log: bmmc_mpi.h,v $
   Revision 1.2  1997/05/06 22:02:30  thc
   Added copyright notice.

   Revision 1.1  1997/04/18 18:23:59  thc
   Initial revision

   */

/*
 * Copyright (C) 1997, Thomas H. Cormen, thc@cs.dartmouth.edu
 *
 * This software may be freely copied, modified, and redistributed,
 * provided that this copyright notice is preserved on all copies.
 *
 * There is no warranty or other guarantee of fitness for this
 * software, and it is provided solely "as is".  Bug reports or fixes
 * may be sent to the author, who may or may not act on them as he
 * desires.
 *
 * Rights are granted to use this software in any non-commercial
 * enterprise.  For commercial rights to this software, please contact
 * the author.
 */

#ifndef BMMC_MPI_H_
#define BMMC_MPI_H_

#include "bit_matrix_types.h"
#include "mpi.h"

/* Error codes. */
#define BMMC_MPI_OK          0x0
#define BMMC_MPI_SINGULAR    0x1
#define BMMC_MPI_BAD_FACTOR  0x2
#define BMMC_MPI_MPI_ERROR   0x4


/* It's convenient to have all the factored information in one place. */
struct BMMC_MPI_factor_info {
  bit_matrix alpha_S;
  bit_matrix alpha_Sbar;
  bit_matrix beta;
  bit_matrix gamma_S;
  bit_matrix delta;
  bit_matrix delta_inv;
  bit_matrix in_proc_perm;
  matrix_column c_offset;
  matrix_column c_proc;
  int card_S;
};

typedef struct BMMC_MPI_factor_info BMMC_MPI_factor_info;


/* Once allocated, an easy way to free all the factored information. */
void free_BMMC_MPI_factor_info(BMMC_MPI_factor_info *info);


/* Create the factored information.
   Inputs:
     A: matrix to factor
     c: complement vector
     n: log of problem size (= rows and columns of A)
     p: log of number of processors
     f: bits f through f+p-1 of an index indicate which processor
        the element is on.  Bit numbers run 0 to n-1.

   Output:
     Fields of info are allocated and filled in.
     Return value is 0 if successful, nonzero if error.

   Note:
     The caller should call free_BMMC_MPI_factor_info later on, though
     it is not necessary if factor_BMMC_MPI returns an error code
     other than 0.
   */
int factor_BMMC_MPI(bit_matrix A,
		    matrix_column c,
		    int n,
		    int p,
		    int f,
		    BMMC_MPI_factor_info *info);


/* Perform the BMMC permutation, given the factored information.
   Inputs:
     info: factored information
     n:    log of problem size (= rows and columns of A)
     p:    log of number of processors
     rank: rank of this processor, in the range 0 to 2^p-1
     comm: MPI communicator
     size: size of each record, in bytes
     data: pointer to array of 2^(n-p) records, of above size, to permute
     temp: pointer to array of same size as data, for temp use

   Outputs:
     data is overwritten with the result of the permutation.
     Return value is 0 if successful, nonzero if error.

   It is the caller's responsibility to call
   free_BMMC_MPI_factor_info(info) and to deallocate temp and any
   other parameters given to this function afterward.
   */
int perform_BMMC_MPI(BMMC_MPI_factor_info *info,
		     int n,
		     int p,
		     int rank,
		     MPI_Comm comm,
		     int size,
		     void *data,
		     void *temp);

/**********************************************************************/

/* The preceding functions are a minimal set.  The following are
   wrappers to make calls more convenient. */

/* Perform a BMMC permutation.
   Inputs:
     A:    matrix to factor
     c:    complement vector
     n:    log of problem size (= rows and columns of A)
     p:    log of number of processors
     f:    bits f through f+p-1 of an index indicate which processor
           the element is on.  Bit numbers run 0 to n-1.
     rank: rank of this processor, in the range 0 to 2^p-1
     comm: MPI communicator
     size: size of each record, in bytes
     data: pointer to array of 2^(n-p) records, of above size, to permute
     temp: pointer to array of same size as data, for temp use

   Outputs:
     data is overwritten with the result of the permutation.
     Return value is 0 if successful, nonzero if error.

   It is the caller's responsibility deallocate temp and any other
   parameters given to this function afterward.
   */
int BMMC_MPI(bit_matrix A,
	     matrix_column c,
	     int n,
	     int p,
	     int f,
	     int rank,
	     MPI_Comm comm,
	     int size,
	     void *data,
	     void *temp);

/* The following functions are the same as BMMC_MPI_factor_info and
   BMMC_MPI except that they do not take the f parameter.

   For BMMC_MPI_factor_info_proc_major and BMMC_MPI_proc_major, data
   is assumed to be in processor-major order, so that the most
   significant p bits contain the processor number and the least
   significant n-p bits contain the offset within the processor.  The
   processor number of element i is floor(i/(N/P)), where N = 2^n and
   P = 2^p.

   For BMMC_MPI_factor_info_proc_minor and BMMC_MPI_proc_minor, data
   is assumed to be in processor-minor order, so that the least
   significant p bits contain the processor number and the most
   significant n-p bits contain the offset within the processor.  The
   processor number of element i is i mod P.
   */

int factor_BMMC_MPI_proc_major(bit_matrix A,
			       matrix_column c,
			       int n,
			       int p,
			       BMMC_MPI_factor_info *info);

int factor_BMMC_MPI_proc_minor(bit_matrix A,
			       matrix_column c,
			       int n,
			       int p,
			       BMMC_MPI_factor_info *info);

int BMMC_MPI_proc_major(bit_matrix A,
			matrix_column c,
			int n,
			int p,
			int rank,
			MPI_Comm comm,
			int size,
			void *data,
			void *temp);

int BMMC_MPI_proc_minor(bit_matrix A,
			matrix_column c,
			int n,
			int p,
			int rank,
			MPI_Comm comm,
			int size,
			void *data,
			void *temp);

#endif
