/*
 * Trace analysis pipe (create a sound based on a trace file)
 *		J-Y Peterschmitt - 03/93
 *
 * tr_sendnum.c
 * 
 * trace_file : stdin > tr_sendnum nb length scale > stdout : sound
 *
 * nb		: maximum number of waiting messages
 * length	: number of sound samples generated for each event,
 *		  or each period of scale microseconds (execution time)
 * scale	: when there is no new event happening at the current 
 *		  time, we skip scale microseconds (execution time).
 *		  In fact, we play a sound of length samples.
 *
 * ==> scale microseconds of execution time (time described by the
 * timestamps in the trace file) will be associated with length sound
 * samples (a sound lasting length / SMP_RATE seconds). When nothing
 * at all is happening, we play a "blank" sound (low frequency, low
 * intensity) of length samples.
 *
 * The program maps the number of waiting messages (from 1 to nb) to
 * frequencies (from low to high frequencies). 
 * Then, sequentially analysing the input trace file, the program
 * plays the frequency associated with the current number of waiting
 * messages.
 *
 */

#include <stdio.h>
#include <math.h>
#include <memory.h>
#include <ulaw2linear.h>
#include <audio_errno.h>
#include <audio_hdr.h>

#define PERROR(str_error) fprintf(stderr, "%s\n", str_error)

#define PG_TRACESTART	1      	/* subset of PICL			*/
#define PG_SEND		4      	/* event types used			*/
#define PG_RECV		6      	/* by ParaGraph				*/
#define PG_RECV_BLOCKING 7
#define PG_RECV_WAKING	8
#define PG_COMPSTATS	11
#define PG_TRACEEXIT	19
#define PG_BLOCKBEGIN	20
#define PG_BLOCKEND	21
#define PG_TRACEMARK	22

#define MAX_LINES 10		/* number of lines in the trace		*/
				/* buffer				*/
#define LINE_SIZE 100		/* max number of chars / line		*/
#define MAX_INFO 80		/* length of the info string		*/
#define OUT_BUFSIZE 8000	/* size of the sound buffer		*/
#define SMP_RATE 8000		/* number of samples / sec.		*/
#define PI 3.141593
#define F0 300			/* Min freq of the generated		*/
				/* sounds	      			*/
#define FN 3520			/* Max freq				*/
#define MAX_PROC 128		/* Max number of processors		*/
#define MAX_WAIT (2 * MAX_PROC)	/* Max number of messages, waiting to	*/
				/* be received, at a given time		*/
#define DFLT_MAXWAIT 8
#define DFLT_LENGTH 200		/* default length of a sound		*/
				/* event       				*/
#define MIN_LENGTH 50
#define DFLT_UNIT 5		/* length of a sound event in		*/
				/* microseconds		      		*/
#define MIN_UNIT 1
#define BLANK_FREQ 150		/* characteristics of the		*/
#define BLANK_MAG 0.04		/* blank sound				*/
#define INTENSITY 150.0		/* "intensity" of the generated		*/
				/* sounds	       			*/

typedef struct
{
    char str[LINE_SIZE];
} Trace;

Trace trace[MAX_LINES];		/* trace buffer				*/

short tsin[SMP_RATE];		/* samples of a 1 HZ sine		*/

int overflow_proc = MAX_PROC;	/* current value of the overflow limit	*/
int overflow_wait = DFLT_MAXWAIT;
int *freq_lut;			/* frequency lookup table 		*/
float *mag_lut;			/* magnitude lookup table		*/
int *nb_waiting;		/* number of messages sent by a given	*/
				/* processor, but not received yet	*/
int tot_waiting;		/* total number of waiting messages	*/
unsigned char out_buf[OUT_BUFSIZE], *p_out;	/* sound buffer		*/
int fd_out, out_bufsize;	/* number of samples in the sound buffer*/
int max_wait, length, unit;	/* command line parameters' value	*/

int current_time = -1;		/* curent time (timestamp) in		*/
				/* microseconds. This time is		*/
				/*increased as we analyze the trace file*/
int nb_sending = 0;		/* number of sending processors		*/

/* Initialization function						*/

void tr_init()
{
    int		i;
    double	inc, angle;
    
    /* compute the sine table used to generate the sound 		*/
    
    for(angle = -PI, inc = 2*PI / (SMP_RATE-1), i = 0;
	i < SMP_RATE;
	i++, angle += inc)
    {
	tsin[i] = audio_d2s(sin(angle));
    }
    
    /* make sure the pointer to the current sample points to		*/
    /* the beginning of the out buffer					*/
    p_out = out_buf;
    out_bufsize = 0;
    tot_waiting = 0;
}

/* Open the stdout stream to start the pipe				*/

void pipe_open()
{
    fd_out = fileno(stdout);
}

/* Create a valid header for the generated sound	       		*/

void pipe_header()
{
    Audio_hdr au_desc;
    char au_info[MAX_INFO];
    
    au_desc.sample_rate = SMP_RATE;
    au_desc.samples_per_unit = 1;
    au_desc.bytes_per_unit = 1;
    au_desc.channels = 1;
    au_desc.encoding = AUDIO_ENCODING_ULAW;
    au_desc.data_size = AUDIO_UNKNOWN_SIZE;    
    sprintf(au_info, "Trace sound file generated by tr_sendnum");
    
    /* send the header to stdout	      				*/
    audio_write_filehdr(fd_out, &au_desc, au_info, 
			(unsigned) strlen(au_info));
}

/* Close the pipe      							*/

void pipe_close()
{
    (void) close(fd_out);
}

/* Read as many lines as possible, to fill the trace buffer		*/

int getlines()
{
    int i;
    
    for(i = 0; i < MAX_LINES; i++)
    {
	if(fgets(trace[i].str, LINE_SIZE, stdin) == NULL)
	{
	    break;     			/* end of file reached	       	*/
	}
    }	
    return(i);
}

/* Empty the sound buffer, if there is not enough room to store		*/
/* the samples corresponding to the next event			       	*/

void flush_outbuf()
{
    if((out_bufsize + length) >= OUT_BUFSIZE)
    {
	write(fd_out, (char *) out_buf, out_bufsize);
	p_out = out_buf;
	out_bufsize = 0;
    }
}

/* Create the frequency and magnitude lookup tables		       	*/

void create_lut(nb)
int nb;
{
    int i, freq;
    float lambda;

    /* NOTE : we use the same mapping between the number of waiting	*/
    /* messages and the frequency, as between the proc number and the	*/
    /* frequency							*/
    
    /* There are two ways to compute the mapping between		*/
    /* the proc number and the frequency of the sound			*/
    /* - a linear distribution between the min freq allowed		*/
    /*		and the max frequency		 			*/
    /* - a logarithmic distribution between the min and the		*/
    /*		max freq      						*/
    /* In both cases, the magnitude of the sound is chosen		*/
    /* so that the product "freq * mag" remains constant,		*/
    /* in order to deal with sounds of the same "intensity"		*/
    
    /* linear distribution						*/
    /* f(k) = f(0) + k * (f(N) - f(0)) / N		   		*/
    /* where N = MAX_PROC_NO (max number of processors)			*/
    
    /* logarithmic distribution				       		*/
    /* f(k) = f(0) * exp( k/N * log(f(N) / f(0)))	       		*/
    /* where N = MAX_PROC_NO (max number of processors)			*/
    
    lambda = (float) log((double) (FN / F0)) / nb;
    for(i = 0; i < nb; i++)
    {
	freq = F0 * exp((double) (i * lambda));
	*(freq_lut + i) = freq;
	*(mag_lut + i) = INTENSITY / freq;
    }
}

/* Create the next LENGTH sound samples			      		*/

void create_au()
{
    static int ind = 0;
    int i, freq;
    float mag;
    
    if(tot_waiting == 0)
    {
	freq = BLANK_FREQ;
	mag = BLANK_MAG;
    }
    else if((tot_waiting > 0) && (tot_waiting <= max_wait))
    {
	freq = *(freq_lut + tot_waiting - 1);
	mag = *(mag_lut + tot_waiting - 1);
    }
    else if(tot_waiting > max_wait)
    {
	freq = *(freq_lut + max_wait - 1);
	mag = *(mag_lut + max_wait - 1);
    }
    else
    {
	PERROR("create_au : tot_waiting < 0 !");
	freq = BLANK_FREQ;
	mag = BLANK_MAG;
    }
    
    for(i = 0; 
	i < length; 
	i++, ind = (ind + freq) % SMP_RATE)
    {
	*p_out = audio_s2u((short) (mag * tsin[ind]));
	p_out++;
	out_bufsize++;
    }
}

/* Case of a send event	      						*/

void send2au(str)
char *str;
{
    int proc;
    
    if(sscanf(str, "%*d %*d %*d %d", &proc) != 1)
    {
	PERROR("send2au : unable to get proc number");
    }
    else if((proc + 1) > MAX_PROC)
    {
	/* PERROR("send2au : too many processors");			*/
	if((proc + 1) > overflow_proc)
	{
	    PERROR("send2au : too many processors");
	    overflow_proc = proc + 1;
	    fprintf(stderr, "\t\t%d processors\n", overflow_proc);
	}
    }
    else if(*(nb_waiting + proc) > 0)
    {
	/* This processor has already sent a message			*/
	/* (but the message hasn't been received yet)			*/
	*(nb_waiting + proc) += 1;
	tot_waiting++;
    }
    else
    {
	/* This processor was not sending messages			*/
	nb_sending++;
	*(nb_waiting + proc) = 1;
	tot_waiting++;
    }
    if(tot_waiting > max_wait)
    {
	/* PERROR("send2au : too many waiting messages");		*/
	if(tot_waiting > overflow_wait)
	{
	    PERROR("send2au : too many waiting messages");
	    overflow_wait = tot_waiting;
	    fprintf(stderr, "\t\t%d waiting messages\n", overflow_wait);
	}
    }
}

/* Case of a receive event						*/

void recv2au(str)
char *str;
{
    int proc;
    
    /* We get the proc number of the SENDING processor, not		*/
    /* of the receiving one						*/
    if(sscanf(str, "%*d %*d %*d %*d %d", &proc) != 1)
    {
	PERROR("recv2au : unable to get proc number");
    }
    else if((proc + 1) > MAX_PROC)
    {
	/* PERROR("recv2au : too many processors");			*/
	if((proc + 1) > overflow_proc)
	{
	    PERROR("recv2au : too many processors");
	    overflow_proc = proc + 1;
	    fprintf(stderr, "\t\t%d processors\n", overflow_proc);
	}
    }
    else if(*(nb_waiting + proc) <= 0)
    {
	PERROR("recv2au : receiving an unsent message!");
    }
    else
    {
	*(nb_waiting + proc) -= 1;
	if(*(nb_waiting + proc) == 0)
	{
	    /* All the messages sent by proc have been received		*/
	    nb_sending--;	
	}
	tot_waiting--;
    }
}

/* Analyse the content of the trace buffer				*/

void trace2au(nb)
int nb;
{
    int i, type, t_sec, t_musec;
    int trace_time;
    
    for(i = 0; i < nb; i++)
    {
	/* get the time of the new event				*/
	if(sscanf(trace[i].str, "%d %d %d", &type, 
		  &t_sec, &t_musec) !=  3)
	{
	    PERROR("Error analysing trace line. Line skipped");
	    continue;
	}
	trace_time = (int) t_sec * 1e6 + t_musec;
	
	if(current_time < 0)
	    current_time = trace_time;	/* time initialization		*/
	
	
	if(current_time < trace_time)
	{
	    /* We need to catch up with the trace time			*/
	    do
	    {
		flush_outbuf();
		create_au();
		current_time += unit;
	    } while(current_time < trace_time);
	}
	
	flush_outbuf();
	switch(type)
	{
	  case PG_TRACESTART :
	      break;
	    case PG_SEND :
		send2au(trace[i].str);
	      create_au();
	      break;
	    case PG_RECV :
		recv2au(trace[i].str);
	      create_au();
	      break;
	    case PG_RECV_BLOCKING :
		break;
	    case PG_RECV_WAKING :
		recv2au(trace[i].str);
	      create_au();
	      break;
	    case PG_COMPSTATS :
		break;
	    case PG_TRACEEXIT :
		break;
	    case PG_BLOCKBEGIN :
		break;
	    case PG_BLOCKEND :
		break;
	    case PG_TRACEMARK :
		break;
	      default :
		  ;
	  }
    }
}

/* on-line "manual" 							*/

void usage()
{
    PERROR("\nWrong parameters. Please specify 3 integers");
    PERROR("\tand check the range of the supplied parameters");
    PERROR("Usage : tr_sendnum max length scale");
    PERROR("\tnb : maximum number of waiting messages");
    PERROR("\tlength : number of sound samples generated for each event");
    PERROR("\tscale : time scale");
}

/* main program								*/

main(argc, argv)
int argc;
char *argv[];
{
    int i, nb_lines;
    
    if(argc == 4)
    {
	if((sscanf(argv[1], "%d", &max_wait) != 1) ||
	   (max_wait < 1) || (max_wait > MAX_WAIT) ||
	   (sscanf(argv[2], "%d", &length) != 1) || 
	   (length < MIN_LENGTH) ||
	   (length > OUT_BUFSIZE) ||
	   (sscanf(argv[3], "%d", &unit) != 1) ||
	   (unit < MIN_UNIT))
	{
	    usage();
	    exit();
	}
    }
    else
    {
	PERROR("Using default parameters...");
	max_wait = DFLT_MAXWAIT;
	length = DFLT_LENGTH;
	unit = DFLT_UNIT;
    }
    overflow_wait = max_wait;
    
    /* allocate memory for misc tables					*/
    
    freq_lut = (int *) malloc(max_wait * sizeof(int));
    if(freq_lut == NULL)
    {
	PERROR("Error allocating memory for freq_lut");
	exit();
    }
    
    mag_lut = (float *) malloc(max_wait * sizeof(float));
    if(mag_lut == NULL)
    {
	PERROR("Error allocating memory for mag_lut");
	exit();
    }
    
    nb_waiting = (int *) malloc(MAX_PROC * sizeof(int));
    if(nb_waiting == NULL)
    {
	PERROR("Error allocating memory for nb_waiting");
	exit();
    }
    for(i = 0; i < MAX_PROC; i++)
	*(nb_waiting + i) = 0;
    
    create_lut(max_wait);
    tr_init();
    pipe_open();
    pipe_header();
    
    while((nb_lines = getlines()) > 0)
    {
	trace2au(nb_lines);
    }

    /* force the flushing of the output buffer a last time		*/
    length = OUT_BUFSIZE;
    flush_outbuf();
    
    fflush(stdout);
    pipe_close();
}
