#define MAINLOG
#include "pargrid2.h"

/*+ main - the main routine for this grid program
           We are running the solver on two different sets of
	   processors simultaneously

     Input Parameters:
     argc, argv - the usual
     argv[1] - the number of processors in the x direction
     argv[2] - the number of processors in the y direction
     argv[3] - the number of points in the x direction on each processor
               do not use less than 3 points
     argv[4] - the number of points in the y direction on each processor
               do not use less than 3 points
     argv[5] - the number of points in the z direction on each processor
               do not use less than 3 points

     To Run, see tools.

     Notes: The grid program solves a linear system associated with
            a 3-D grid distributed across the processors.  The 3-D
            grid is partitioned in two dimensions amongst the
            processors.  The processors on the left and right ends
            of the grid use a 7 pt stencil, those in the middle use
            a 27-pt stencil.  No real problem is being solved, we
            are just demonstrating the use of the linear system solver.

 +*/

main(argc,argv)
int	argc;
char	**argv;
{
	int	main2();

	PICall(main2,argc,argv);
	exit(0);
}

int main2(argc,argv)
int	argc;
char	**argv;
{
	par_grid grid;
	BSprocinfo *procinfo;
	int	t, work;
	int	np;
	ProcSet	*procset;
	int	xpos, ypos;

	/* initialize the logging */
	MLOG_INIT();

	/* get the number of processors and check it */
	if (argc < 6) {
		SETERRC(ARG_ERR,"Argument list too small\n");
		return;
	}
	sscanf(argv[1],"%d",&grid.worker_x);
	sscanf(argv[2],"%d",&grid.worker_y);
	grid.worker_z = 1;
	if (NUMNODES != grid.worker_x*grid.worker_y*grid.worker_z) {
		SETERRC(ARG_ERR,"Number of processors is not correct\n");
		return;
	}
	if ((NUMNODES % 2) != 0) {
		SETERRC(ARG_ERR,"Number of processors is not divisible by 2\n");
		return;
	}
	if ((grid.worker_x % 2) != 0) {
		SETERRC(ARG_ERR,"Number of processors in X direction is not divisible by 2\n");
		return;
	}
	grid.worker_x /= 2;

	/* initialize the two processor sets */
	/* we are going to split only in the x direction */
	/* so the user must give us a number of processors divisible by 2 */
	/* AND the number in the X direction divisible by 2 */
	procset = PSPartition((MYPROCID / (NUMNODES/2)),(ProcSet *) NULL);
	PSSetMeshSize(procset,grid.worker_x,grid.worker_y);
	PSCompile(procset);

	/* set up the context for BlockSolve */
	procinfo = BScreate_ctx(); CHKERR(0);
	/* tell it that this matrix has no i-nodes or cliques */
	BSctx_set_si(procinfo,TRUE); CHKERR(0);
	/* tell it to print out some information on the reordering */
	BSctx_set_pr(procinfo,TRUE); CHKERR(0);
	/* tell it the number of processors and the procset */
	BSctx_set_ps(procinfo,procset); CHKERR(0);
	BSctx_set_np(procinfo,PSNUMNODES(procinfo->procset)); CHKERR(0);

	if (PSNUMNODES(procinfo->procset) != grid.worker_x*grid.worker_y*grid.worker_z) {
		SETERRC(ARG_ERR,"Number of processors in procset is not correct\n");
		return;
	}

	/* read in grid parameters */
	if (PSISROOT(procinfo->procset)) {
		printf("Number of workers (x,y,z): %d %d %d\n",
			grid.worker_x,grid.worker_y,grid.worker_z);
	}
	sscanf(argv[3],"%d",&grid.num_x);
	sscanf(argv[4],"%d",&grid.num_y);
	sscanf(argv[5],"%d",&grid.num_z);

	/* local grid size and type */
	/* the end processors are 7-pt stencils, the others are 27-pt */
	PSMESHLOC(procinfo->procset,&xpos,&ypos);
	if ((xpos == grid.worker_x-1) || (xpos == 0)) {
		grid.l_num_x = grid.num_x;
		grid.type = 7;
	} else {
		grid.l_num_x = grid.num_x / 2;
		grid.type = 27;
	}
	grid.l_num_y = grid.num_y;
	grid.l_num_z = grid.num_z;
	grid.num_x = grid.l_num_x;
	GISUM(&grid.num_x,1,&work,procinfo->procset);
	grid.num_y = grid.l_num_y;
	GISUM((&grid.num_y),1,&work,procinfo->procset);
	grid.num_z = grid.l_num_z;
	GISUM((&grid.num_z),1,&work,procinfo->procset);

	if (PSISROOT(procinfo->procset)) {
		printf("Discretizations (x,y,z): %d %d %d\n",
			grid.num_x,grid.num_y,grid.num_z);
	}

	/* call the worker */
	worker(&grid,procinfo); 

	/* print out the log info and return */
	MLOG_PRINT(procinfo->procset);
	BSfree_ctx(procinfo); CHKERR(0);

	return;
}
