/*@ BSsparse.h - Include file for the sparse solver package.  
    This is the include file for public use.
 @*/

/* ****************************************************************** */
/* NOTICE: Argonne National Laboratory is in the process of           */
/* copyrighting the BlockSolve program.  Essentially this copyright   */
/* will allow completely free distribution of the BlockSolve program  */
/* to anyone and everyone.  We also will grant the right to anyone    */
/* to modify the program but NOT to distribute the modified version.  */
/* Our only reason for this restriction is that we don't want to      */
/* chase other people's bugs or try to debug possibly modified code.  */
/*                                                                    */
/* Authors: Mark Jones and Paul Plassmann                             */
/*                                                                    */
/* Version: 2.0                                                       */
/*                                                                    */
/* Last Modified: 10/94                                               */
/*                                                                    */
/* ****************************************************************** */

#ifndef __BSsparseh
#define __BSsparseh

/* include the "system" dependent routines */
#include "BSdepend.h"

/* include the special message passing stuff that I wrote */
#include "BMmsg.h"

/* the usual system stuff */
#include <stdio.h>
#include <limits.h>
#include <math.h>

/* ****************************************************************** */
/* Definitions of double/single precision stuff */
/* ****************************************************************** */
#if defined(FORTRANCAPS)
#define dgemm_ DGEMM
#define dtrmm_ DTRMM
#define dcopy_ DCOPY
#define daxpy_ DAXPY
#define dgemv_ DGEMV
#define dtrmv_ DTRMV
#define dpotrf_ DPOTRF
#define dtrtri_ DTRTRI
#define dlamch_ DLAMCH
#define sdot_ SDOT
#define sgemm_ SGEMM
#define strmm_ STRMM
#define scopy_ SCOPY
#define saxpy_ SAXPY
#define sgemv_ SGEMV
#define strmv_ STRMV
#define spotrf_ SPOTRF
#define strtri_ STRTRI
#define slamch_ SLAMCH
#elif !defined(FORTRANUNDERSCORE)
#define dgemm_ dgemm
#define dtrmm_ dtrmm
#define dcopy_ dcopy
#define daxpy_ daxpy
#define dgemv_ dgemv
#define dtrmv_ dtrmv
#define dpotrf_ dpotrf
#define dtrtri_ dtrtri
#define dlamch_ dlamch
#define sdot_ sdot
#define sgemm_ sgemm
#define strmm_ strmm
#define scopy_ scopy
#define saxpy_ saxpy
#define sgemv_ sgemv
#define strmv_ strmv
#define spotrf_ spotrf
#define strtri_ strtri
#define slamch_ slamch
#endif

/* turn on compilation of the double precision version if */
/* BSDOUBLE is defined, otherwise single precision */
#define BSDOUBLE

#ifdef BSDOUBLE
#define FLOAT double
#define DGEMM dgemm_
#define DTRMM dtrmm_
#define DCOPY dcopy_
#define DAXPY daxpy_
#define DGEMV dgemv_
#define DTRMV dtrmv_
#define DPOTRF dpotrf_
#define DTRTRI dtrtri_
#define DLAMCH dlamch_
#define GFLSUM GDSUM
#else
#define FLOAT float
#define DGEMM sgemm_
#define DTRMM strmm_
#define DCOPY scopy_
#define DAXPY saxpy_
#define DGEMV sgemv_
#define DTRMV strmv_
#define DPOTRF spotrf_
#define DTRTRI strtri_
#define DLAMCH slamch_
#define GFLSUM GFSUM
#endif
/* ****************************************************************** */
/* End of Definitions */
/* ****************************************************************** */

/* include some logging stuff that I wrote */
#include "BSlog.h"

/* ****************************************************************** */
/* Definitions of misc. constants */
/* ****************************************************************** */

#define	MEM_ERROR -25
#define	FILE_ERROR -26
#define	BB_ERROR -27
#define	COLOR1_ERROR -28
#define	COLOR2_ERROR -29
#define	INODE_ERROR -30
#define	MSG_ERROR -30
#define CG_NEG_ERROR -31
#define	COLOR_ERROR -32
#define	FACTOR_ERROR -33
#define	MATRIX_ERROR0 -50
#define	MATRIX_ERROR1 -51
#define	MATRIX_ERROR2 -52
#define	MATRIX_ERROR3 -53
#define	MATRIX_ERROR4 -54
#define	MATRIX_ERROR5 -55
#define	MATRIX_ERROR6 -56
#define	MATRIX_ERROR7 -57
#define	MATRIX_ERROR8 -58
#define	MATRIX_ERROR9 -59
#define	MATRIX_ERROR10 -60
#define	MATRIX_ERROR11 -61
#define	MATRIX_ERROR12 -62
#define	MATRIX_ERROR13 -63
#define	MATRIX_ERROR14 -64
#define	MATRIX_ERROR15 -65
#define	MATRIX_ERROR16 -66
#define	MATRIX_ERROR17 -67
#define	REPERM_ERROR -68

/* type of coloring */
#define IDO 10
#define SDO 20

/* what kind of preconditioning */
#define PRE_DIAG 0
#define PRE_STICCG 1
#define PRE_SSOR 2
#define PRE_BJACOBI 2

/* ****************************************************************** */
/* End of section */
/* ****************************************************************** */


/* ****************************************************************** */
/* Definitions of data structures */
/* ****************************************************************** */

/* store a local dense matrix */
typedef struct __BSdense {
	int	size;          /* size of the dense matrix */
	int	local_ind;     /* local index of first element of dense matrix */
	FLOAT	*matrix;   /* the matrix itself */
} BSdense;

/* shows which inodes belong to each clique */
/* the arrays are one element longer than num_cols for ease of use */
typedef struct __BScl_2_inode {
	int	num_cols;      /* number of columns */
	int	*g_offset;     /* global number of beginning of each clique */
	int	*proc;         /* processor who owns clique */
	int	*inode_index;  /* index into inode structure */
	BSdense *d_mats;	/* array of pointers to dense matrices */
} BScl_2_inode;

/* i_node structure keeps several identical rows/cols in place */
/* we are storing on the half of the symmetric matrix, in addition */
/* the diagonal portions are stored with the clique structure */
typedef struct __BSinode {
	int	num_cols; /* number of cols in i_node */
	int	gcol_num; /* global column number of first row of i_node */
					/* global column numbers are consecutive */
	int	*o_gcol_num; /* original global column number of each row of i_node */
					/* original global column numbers are NOT consecutive */
	int	length;	  /* the length of each col */
	int	*row_num; /* the row numbers (we store local, not global #'s */
	FLOAT	*nz;      /* the nz values */
} BSinode;

/* list of inodes */
/* the array is one element longer than length for ease of use */
typedef struct __BSinode_list {
	int	 length; /* the number of i_nodes */
	BSinode	*list; /* array of i_nodes */
} BSinode_list;

/* permutation of rows/cols of a matrix */
typedef	struct __BSpermutation {
	int	length;
	int	*perm;
} BSpermutation;

/* distribution of a numbering */
/* i.e. how many 0's, how many 1's, etc. */
typedef struct __BSdistribution {
	int	max;
	int	*distribution;
} BSdistribution;

/* numbering of rows/cols of a matrix */
typedef	struct __BSnumbering {
	int	length;
	int	*numbers;
} BSnumbering;

/* global/local mapping */
typedef struct __BSmapping {
	/* not all of these members may be present, some may be NULL */
	/* we are trying to define a flexible structure */
	void	*vlocal2global; /* data for mapping local to global */
	void (*flocal2global)(); /* a function for mapping local to global */
	void (*free_l2g)(); /* a function for free'ing the l2g data */
	void	*vglobal2local; /* data for mapping global to local */
	void (*fglobal2local)(); /* a function mapping global to local */
	void (*free_g2l)(); /* a function for free'ing the g2l data */
	void	*vglobal2proc; /* data for mapping global to proc */
	void (*fglobal2proc)(); /* a function mapping global to proc */
	void (*free_g2p)(); /* a function for free'ing the g2p data */
} BSmapping;

/* row of a sparse matrix */
typedef struct __BSsprow {
	int	diag_ind;	/* index of diagonal in row */
	int	length;		/* num. of nz in row */
	int	*col;		/* col numbers */
	FLOAT	*nz;	/* nz values */
} BSsprow;

/* sparse matrix */
typedef struct __BSspmat {
	int	num_rows;		/* number of local rows */
	int	global_num_rows;/* number of global rows */
	BSmapping	*map;	/* mapping from local to global, etc */
	BSsprow	**rows;		/* the sparse rows */
} BSspmat;

/* processor information and execution time information */
typedef struct __BSprocinfo {
	int	my_id;			/* processor id */
	int	nprocs;			/* number of processors */
	ProcSet	*procset;	/* set of processors to operate on (see tools) */
	int	max_clique_size; /* the maximum clique size to allow */
	int	max_inode_size; /* the maximum i-node size to allow */
	int	coloring_type;  /* the type of coloring to do */
	int	error_check;    /* if TRUE, then do error checking */
	int	retain;         /* if TRUE, then keep around some extra */
	                    /* information to allow for fast permutation */
	                    /* for a matrix with the same structure */
	int	print;			/* if TRUE, then print reordering information */
	int	single;			/* if TRUE, the no cliques or i-nodes are found */
} BSprocinfo;

/* offset mapping structure (keep track of who has what row, etc. */
typedef struct __BSoff_map {
	int	length;		/* length of the map (number of processors, usually) */
	int	my_ind;		/* my index into the map */
	int	*proc_id;	/* processor id's (sorted by offset) */
	int	*offset;	/* sorted list of offsets */
} BSoff_map;

/* information for quickly permuting a matrix with the same structure */
typedef struct __BSreperm {
	BSpermutation *inode_perm;    /* the original inode permutation */
	BSdistribution *inode_distr;  /* the original inode distribution */
} BSreperm;

/* Here is a very important data structure, this is the structure     */
/* returned by BSmain_perm.  It contains everything needed for        */
/* parallel matrix multiply, back solve, etc.  It also may or may not */
/* contain the information for quickly permuting a matrix with the    */
/* exact same structure again                                         */
/* IMPORTANT: the map pointer is only a copy, do not destroy          */
/* if the original is destroyed, we are in trouble                    */
typedef struct __BSpar_mat {
	int	num_rows;                /* number of local rows */
	int	global_num_rows;         /* number of global rows */
	int	local_nnz;				 /* number of nonzeroes on this processor */
	BSpermutation *perm;         /* the local permutation of the columns */
	BSpermutation *inv_perm;     /* the inverse of perm */
	BSnumbering *global_row_num; /* the global numbers of the rows */
	BSnumbering *color2clique;   /* map from colors to cliques */
	BScl_2_inode *clique2inode;  /* map from cliques to inodes */
	BSinode_list *inodes;        /* the array of inodes */
	FLOAT	*diag;               /* a copy of the original permuted diagonal */
	FLOAT	*save_diag;          /* a copy of the current (scaled) diagonal */
	                             /* this is useful if we share cliques */
	                             /* it can be NULL (which means the diagonal */
	                             /* is all 1's */
	                             /* can legally be equal to diag */
	FLOAT	*scale_diag;         /* the diagonal the matrix was scaled by */
	                             /* if NULL, then no scaling was done */
	                             /* can legally be equal to diag */
	BSmapping *map;              /* a POINTER to the original mapping */
	BSreperm *reperm;            /* information for quick permutation (opt) */
} BSpar_mat;

/* data structure used to contain communication information */
typedef struct __BScomm {
	BMcomp_msg *to_msg;		/* compiled message structure */
	BMcomp_msg *from_msg;	/* compiled message structure */
} BScomm;

/* ****************************************************************** */
/* End of section */
/* ****************************************************************** */


/* ****************************************************************** */
/* Declarations of my functions */
/* ****************************************************************** */

BSpar_mat *BSmain_perm();
BSoff_map	*BSmake_off_map();
void BSglob2loc();
void BSglob2proc();
void BSloc2glob();
void BSfree_spmat();
void BSfree_easymat();
void BSfree_off_map();
void BSfreel2g();
void BSfreeg2l();
void BSfree_par_mat();
void BSfree_copy_par_mat();
void BSmain_reperm();
void BSget_diag();
void BSscale_diag();
BSpar_mat *BScopy_par_mat();
void BScopy_nz();
void BSset_diag();
BScomm *BSsetup_forward();
void BSforward();
void BSbackward();
void BSfree_comm();
void BSfree_comm_data();
int	BSpar_solve();
int	BSpar_isolve();
void BSinv_diag_block();
void BSfor_solve();
void BSb_for_solve();
void BSback_solve();
void BSb_back_solve();
int	BSfactor();
BScomm *BSsetup_factor();
void BSb_forward();
void BSb_backward();
void BStri_solve();
void BStri_mult();
void BSsave_diag();
void BSbjacobi();
void	BSmat_subtract();
void	BSctx_set_id();
void	BSctx_set_np();
void	BSctx_set_ps();
void	BSctx_set_cs();
void	BSctx_set_is();
void	BSctx_set_ct();
void	BSctx_set_err();
void	BSctx_set_rt();
void	BSctx_set_pr();
void	BSctx_set_si();
BSprocinfo	*BScreate_ctx();
void	BSfree_ctx();
int	BSoffset();
BSspmat	*BSeasy_A();

/* ****************************************************************** */
/* End of section */
/* ****************************************************************** */
#endif
