/*@ BSlog.h - This file defines the type of logging done in the
              BlockSolve package.  It is hoped that this will
              eventually be converted to use the logging routines
              written by Bill and Rusty.  Therefore the user should
              not attempt to use this package in their own code.  It
              would be reasonable to print out the information gathered
              by this logging facility, however.

	System Description:
.   This logging system allows for two kinds of phases: one time
.   occurences whose time of occurence and duration must be recorded
.   and things that are executed many times for which we want
.   aggregate information (this are accumulated events).

.   The facility is initialized by calling the macro MLOG_INIT.
.   The facility is terminated and statistics printed by calling the
.   macro MLOG_PRINT.  A one-time occurence is recorded by calling
.   the macro MLOG_ELM to signal the beginning of the event and
.   MLOG_IT to signal the end.  An accumulated event is signaled by
.   calling MLOG_ELM to signal the beginning and MLOG_ACC to signal
.   the end.

.   Somewhere in the users main program the preprocessor variable
.   MAINLOG must be defined (#define MAINLOG) if logging is turned
.   on.  This allows the declaration of certain global variables.

 @*/

#ifndef __BSlogh
#define __BSlogh

#ifdef MLOG

/* define the clock to use */
#define Mclock() SYGetCPUTime()

/* we allow a maximum of 75 events and 75 accumulated events */
#define	MLOG_MAX_EVNTS 75
#define	MLOG_MAX_ACCUM 75

/* define a structure that contains all information regarding an event */
typedef struct __MLOG_log_type {
	double	time_stamp;
	double	total_time;
	int	event_num;
	int	flops;
} MLOG_log_type;

#endif

/* a set of global variables that are used in logging */
#ifdef MAINLOG

#ifdef MLOG
int MLOG_flops;
double MLOG_time_stamp;
int MLOG_sequence_num;
MLOG_log_type MLOG_event_log[MLOG_MAX_EVNTS];
MLOG_log_type MLOG_accum_log[MLOG_MAX_ACCUM];
#endif

#else

#ifdef MLOG
extern int MLOG_flops;
extern double MLOG_time_stamp;
extern int MLOG_sequence_num;
extern MLOG_log_type MLOG_event_log[MLOG_MAX_EVNTS];
extern MLOG_log_type MLOG_accum_log[MLOG_MAX_ACCUM];
#endif

#endif



#ifdef MLOG

/* define the error code associated with a over-full event log */
#define	LOG_FULL -8

/* print out the logging information */
#define MLOG_PRINT(procset) \
{ \
	int	i_99, j_99, count_99; \
	double	d_max, d_sum, t_99; \
	MLOG_log_type *log_99; \
	for (i_99=0;i_99<MLOG_MAX_ACCUM;i_99++) { \
		if (MLOG_accum_log[i_99].event_num > 0) { \
			log_99 = &(MLOG_accum_log[i_99]); \
			j_99 = log_99->event_num; \
			log_99->total_time /= j_99; \
			if (PSISROOT(procset)) { \
				printf("LOG: EVENT %2d (ACC) occurred %d times\n",i_99, \
				log_99->event_num); \
			} \
			d_sum = log_99->flops; \
			d_max = log_99->flops; \
			GDSUM(&d_sum,1,&t_99,procset); \
			GDMAX(&d_max,1,&t_99,procset); \
			if (PSISROOT(procset)) { \
				printf("   LOG: FLOP avg = %e, max = %e, total = %e\n", \
					d_sum/PSNUMNODES(procset),d_max,d_sum); \
			} \
			d_sum = log_99->total_time; \
			d_max = log_99->total_time; \
			GDSUM(&d_sum,1,&t_99,procset); \
			GDMAX(&d_max,1,&t_99,procset); \
			if (PSISROOT(procset)) { \
				printf("   LOG: TIME avg = %e, max = %e\n", \
					d_sum/PSNUMNODES(procset),d_max); \
			} \
		} \
	} \
	for (i_99=0;i_99<MLOG_sequence_num;i_99++) { \
		if (PSISROOT(procset)) { \
			printf("LOG: EVENT %2d (SEQ) occurred at %e\n", \
				MLOG_event_log[i_99].event_num, \
				MLOG_event_log[i_99].time_stamp-MLOG_event_log[0].time_stamp); \
		} \
		d_sum = MLOG_event_log[i_99].flops; \
		d_max = MLOG_event_log[i_99].flops; \
		GDSUM(&d_sum,1,&t_99,procset); \
		GDMAX(&d_max,1,&t_99,procset); \
		if (PSISROOT(procset)) { \
			printf("   LOG: FLOP avg = %e, max = %e, total = %e\n", \
				d_sum/PSNUMNODES(procset),d_max,d_sum); \
		} \
		d_max = MLOG_event_log[i_99].total_time; \
		d_sum = MLOG_event_log[i_99].total_time; \
		GDSUM(&d_sum,1,&t_99,procset); \
		GDMAX(&d_max,1,&t_99,procset); \
		if (PSISROOT(procset)) { \
			printf("   LOG: TIME avg = %e, max = %e\n", \
				d_sum/PSNUMNODES(procset),d_max); \
		} \
	} \
}

/* initialize the logging facility */
#define	MLOG_INIT() \
{ \
	int	MLOG_99; \
	for (MLOG_99=0;MLOG_99<MLOG_MAX_ACCUM;MLOG_99++) { \
		MLOG_accum_log[MLOG_99].event_num = 0; \
		MLOG_accum_log[MLOG_99].total_time = 0.0; \
		MLOG_accum_log[MLOG_99].flops = 0; \
	} \
	MLOG_sequence_num = 0; \
	MLOG_flops = 0; \
	MLOG_time_stamp = Mclock(); \
}

#else
#define MLOG_INIT()
#define MLOG_PRINT(a)
#endif

#ifdef MLOG 
/* record a one-time event */
#define MLOG_IT(event) \
{ \
	if (MLOG_sequence_num >= MLOG_MAX_EVNTS) { \
		SETERRC(LOG_FULL,"Log full\n"); \
	} else {\
		MLOG_event_log[MLOG_sequence_num].time_stamp = Mclock(); \
		MLOG_event_log[MLOG_sequence_num].event_num = event; \
		MLOG_event_log[MLOG_sequence_num].total_time = \
			MLOG_event_log[MLOG_sequence_num].time_stamp - MLOG_time_stamp; \
		MLOG_event_log[MLOG_sequence_num].flops = MLOG_flops; \
		MLOG_sequence_num++; \
		MLOG_flops = 0; \
		MLOG_time_stamp = Mclock(); \
	} \
}
/* signal the beginning of an event */
#define MLOG_ELM(procset) \
{ \
	GSYNC(procset); \
	MLOG_flops = 0; \
	MLOG_time_stamp = Mclock(); \
}
/* record an occurence of an accumulated event */
#define MLOG_ACC(event) \
{ \
	double t_99; \
	if (event >= MLOG_MAX_ACCUM) { \
		MY_SETERRC(LOG_FULL,"Log full\n"); \
	} \
	MLOG_accum_log[event].time_stamp = Mclock(); \
	(MLOG_accum_log[event].event_num)++; \
	MLOG_accum_log[event].total_time += \
		(MLOG_accum_log[event].time_stamp - MLOG_time_stamp); \
	t_99 = MLOG_accum_log[event].flops; \
	t_99 *= (MLOG_accum_log[event].event_num-1); \
	t_99 += MLOG_flops; \
	MLOG_accum_log[event].flops = t_99 / MLOG_accum_log[event].event_num; \
	MLOG_flops = 0; \
	MLOG_time_stamp = Mclock(); \
}
#else
#define MLOG_IT(event)
#define MLOG_ACC(event)
#define MLOG_ELM(procset)
#endif

/* log floating point operations */
#ifdef MLOG
#ifdef MCOUNT
#define MLOG_flop(num) MLOG_flops += num
#else
#define MLOG_flop(num)
#endif
#else
#define MLOG_flop(num)
#endif

/* the set of logging events for BlockSolve */
#define INODE 1
#define CLIQUE 2
#define COLOR 3
#define PERMUTE 4
#define MM_FORWARD 5
#define MM_BACKWARD 6
#define MS_FORWARD 7
#define MS_BACKWARD 8
#define FACTOR 9
#define BMM_FORWARD 10
#define BMM_BACKWARD 11
#define BMS_FORWARD 12
#define BMS_BACKWARD 13
#define BJ_FORWARD 14
#define OMM_FORWARD 15
#define OMM_BACKWARD 16
#define OBMM_FORWARD 17
#define OBMM_BACKWARD 18
#define BALANCED_COLOR 19

#endif
