#ifndef _SP_base
#define _SP_base 1
#endif
/*
//                       Harwell-Boeing File I/O in C
//            Extracted from  SparseLib++ Sparse Matrix Library
//
//             National Institute of Standards and Technology, MD.
//                         University of Notre Dame, IN.
//              Authors:  R. Pozo, K. Remington, A. Lumsdaine
//                     (C) 1995 All Rights Reserved
//
//                                NOTICE
//
// Permission to use, copy, modify, and distribute this software and
// its documentation for any purpose and without fee is hereby granted
// provided that the above copyright notice appear in all copies and
// that both the copyright notice and this permission notice appear in
// supporting documentation.
//
// Neither the Authors nor the Institutions (National Institute of Standards
// and Technology, University of Notre Dame) make any representations about 
// the suitability of this software for any purpose. This software is 
// provided "as is" without expressed or implied warranty.
//
// SparseLib++ was funded in part by the U.S. Department of Energy, the
// National Science Foundation, the University of Notre Dame, and the State 
// of Tennessee.
 ---------------------------------------------------------------------
 05.31.96 -- Bug fix for correctly parsing real fortran formats which
             do not include a character separator between the fractional
             part and the exponent of the the real number
             (e.g. 1.4444444-004).  
             Also, minor change to writeHB functions so that real formats
             are not printed in header for pattern matrices.  KAR
 ---------------------------------------------------------------------
*/
/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
/*                 I/O for Harwell-Boeing files                             */
/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
/*                                                                          */
/*  Several user I/O functions for Harwell-Boeing files are provided,       */
/*  together with various utility functions.                                */
/*  For a description of the Harwell-Boeing standard, see:                  */
/*                                                                          */
/*            Duff, et al.,  ACM TOMS Vol.15, No.1, March 1989              */
/*                                                                          */
/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
/*  Function:                                                               */
/*  void readHB_info(const char *filename, int *M, int *N, int *nz,         */
/*                                   char **Type, int *nrhs)                */
/*  Description:                                                            */
/*                                                                          */
/*  The readHB_info function opens and reads the header information from    */
/*  the specified Harwell-Boeing file, and reports back the number of rows  */
/*  and columns in the stored matrix (M and N), the number of nonzeros in   */
/*  the matrix (nz), the 3-character matrix type(Type), and the number of   */
/*  right-hand-sides stored along with the matrix (nrhs).                   */
/*                                                                          */
/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
/*  Function:                                                               */
/*                                                                          */
/*  void readHB_newmat_double(const char *filename, int *M, int *N, *int nz,*/
/*        int **colptr, int **rowind,  double**val)                         */
/*  void readHB_newmat_float(const char *filename, int *M, int *N, *int nz, */
/*        int **colptr, int **rowind, float**val)                           */
/*                                                                          */
/*  Description:                                                            */
/*                                                                          */
/*  This function opens and reads the specified file, interpreting its      */
/*  contents as a sparse matrix stored in the Harwell/Boeing standard       */
/*  format.  The function allocates vectors to hold the index and nonzero   */
/*  value information, and returns pointers to these vectors along with     */
/*  matrix dimension and number of nonzeros.                                */
/*                                                                          */
/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
/*  Function:                                                               */
/*                                                                          */
/*  void readHB_mat_double(const char *filename, int *colptr, int *rowind,  */
/*             double*val)                                                  */
/*  void readHB_mat_float(const char *filename, int *colptr, int *rowind,   */
/*             float*val)                                                   */
/*                                                                          */
/*  Description:                                                            */
/*                                                                          */
/*  This function opens and reads the specified file, interpreting its      */
/*  contents as a sparse matrix stored in the Harwell/Boeing standard       */
/*  format and using _pre-allocated_ vectors to hold the index and nonzero  */
/*  value information.                                                      */
/*                                                                          */
/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
/*  Function:                                                               */
/*                                                                          */
/*  void readHB_newrhs_double(const char *filename, double **b, int j)      */
/*  void readHB_newrhs_float(const char *filename, float **b, int j)        */
/*                                                                          */
/*  Description:                                                            */
/*                                                                          */
/*  This function opens and reads the specified file, returning a right-    */
/*  hand-side vector b.  If the file specifies multiple right-hand-sides    */
/*  (that is, a right-hand-side matrix B), then the optional argument j     */
/*  may be used to select the (j+1)st column of B.                          */
/*                                                                          */
/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
/*  Function:                                                               */
/*                                                                          */
/*  void readHB_rhs_double(const char *filename, double *b, int j)          */
/*  void readHB_rhs_float(const char *filename, float *b, int j)            */
/*                                                                          */
/*  Description:                                                            */
/*                                                                          */
/*  This function opens and reads the specified file, returning a right-    */
/*  hand-side vector b in _pre-allocated storage.  If the file specifies    */
/*  multiple right-hand-sides (that is, a right-hand-side matrix B), then   */
/*  the optional argument j may be used to select the (j+1)st column of B.  */
/*                                                                          */
/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
/*  Function:                                                               */
/*                                                                          */
/*  void writeHB_mat_double(const char *filename, int M, int N,             */
/*         int nz, const int *colptr, const int *rowind,                    */
/*         const double *val, int nrhs, const double *rhs,                  */
/*         const char *Type, const char *Title, const char *Key)            */
/*  void writeHB_mat_float(const char *filename, int M, int N,              */
/*         int nz, const int *colptr, const int *rowind,                    */
/*         const float *val, int nrhs, const float *rhs,                    */
/*         const char *Type, const char *Title, const char *Key)            */
/*                                                                          */
/*  Description:                                                            */
/*                                                                          */
/*  The writeHB function opens the named file and writes the specified      */
/*  matrix and optional right-hand-side(s) to that file in Harwell-Boeing   */
/*  format.                                                                 */
/*                                                                          */
/****************************************************************************/
#include "../misc.h"
/*
#include "iohb.h"
#include<stdio.h>
#include<stdlib.h>
#include<malloc.h>
*/

/* Utility function declarations: */
static void readHB_header(FILE*, char*, char*, char*, int*, int*, int*, int*,
                          char*, char*, char*, char*, int*, int*, int*, int*, char*);
static void nullchk(char*);
static void ParseIfmt(char*, int*, int*);
static void ParseRfmt(char*, int*, int*, int*);
static void convertDtoE(char*);
static char * substr(char*, int, int);
static char * substr_after(char*, char);
static char * substr_before(char*, char);
static char * substr_through(char*, char);
static void upcase(char*);
static int charLocation(char*, char);


void readHB_info(const char* filename, int* M, int* N, int* nz, char** Type, 
                                                      int* nrhs)
{
/****************************************************************************/
/*  The readHB_info function opens and reads the header information from    */
/*  the specified Harwell-Boeing file, and reports back the number of rows  */
/*  and columns in the stored matrix (M and N), the number of nonzeros in   */
/*  the matrix (nz), and the number of right-hand-sides stored along with   */
/*  the matrix (nrhs).                                                      */
/*                                                                          */
/*  For a description of the Harwell Boeing standard, see:                  */
/*            Duff, et al.,  ACM TOMS Vol.15, No.1, March 1989              */
/*                                                                          */
/*    ----------                                                            */
/*    **CAVEAT**                                                            */
/*    ----------                                                            */
/*  **  If the input file does not adhere to the H/B format, the  **        */
/*  **             results will be unpredictable.                 **        */
/*                                                                          */
/****************************************************************************/
    FILE *in_file;
    char Title[73], Key[9], Rhstype[4];
    char Ptrfmt[17], Indfmt[17], Valfmt[21], Rhsfmt[21];
    int Ptrcrd, Indcrd, Valcrd, Rhscrd; 
    int Nrow, Ncol, Nnzero;
    int Nrhs;
    char *mat_type;
    mat_type = (char *) malloc(4);
/*
    ----------------------------------------------
    start mod by cca, 97mar13, initialize mat_type
    ----------------------------------------------
*/
    mat_type[0] = mat_type[1] = mat_type[2] = mat_type[3] = '\0' ;
/*
    --------------------------------------------
    end mod by cca, 97mar13, initialize mat_type
    --------------------------------------------
*/
    
    in_file = fopen( filename, "r");
    if (in_file == NULL)
    {
       fprintf(stderr,"Error: Cannot open file: %s\n",filename);
       exit(1);
    }

    readHB_header(in_file, Title, Key, mat_type, &Nrow, &Ncol, &Nnzero, &Nrhs,
                  Ptrfmt, Indfmt, Valfmt, Rhsfmt, 
                  &Ptrcrd, &Indcrd, &Valcrd, &Rhscrd, Rhstype);
    *Type = mat_type;
    *M    = Nrow;
    *N    = Ncol;
    *nz   = Nnzero;
    if (Rhscrd != 0) {*nrhs = Nrhs;}
                else {*nrhs = 0;}

/*  In verbose mode, print some of the header information:   */
/*
    if (verbose == 1)
    {
        printf("Reading from Harwell-Boeing file %s (verbose on)...\n",filename);
        printf("  Title: %s\n",Title);
        printf("  Key:   %s\n",Key);
        printf("  The stored matrix is %i by %i with %i nonzeros.\n", 
                *M, *N, *nz );
        printf("  %i right-hand--side(s) stored.\n",*nrhs);
    }
*/
 
    return;

}


/*************************************************************************/
/*  Read header information from the named H/B file...                   */
/*************************************************************************/

static void  readHB_header(FILE* in_file, char* Title, char* Key, char* Type, 
                    int* Nrow, int* Ncol, int* Nnzero, int* Nrhs,
                    char* Ptrfmt, char* Indfmt, char* Valfmt, char* Rhsfmt, 
                    int* Ptrcrd, int* Indcrd, int* Valcrd, int* Rhscrd, 
                    char *Rhstype)
{
    char line[82];
    char *line_ptr;
    int Totcrd;
    int Neltvl, Nrhsix;

/*  First line:   */
    line_ptr = fgets(line, 82, in_file);
    nullchk(line_ptr);
    (void) sscanf(line, "%72c%8c\n", Title, Key);
    *(Key+8) = (char) NULL;
    *(Title+72) = (char) NULL;

/*  Second line:  */
    line_ptr = fgets(line, 82, in_file);
    nullchk(line_ptr);
    (void) sscanf(line, "%i%i%i%i%i", &Totcrd, Ptrcrd, Indcrd, 
                                          Valcrd, Rhscrd);

/*  Third line:   */
    line_ptr = fgets(line, 82, in_file);
    nullchk(line_ptr);
    (void) sscanf(line, "%3c%i%i%i%i", Type, Nrow, Ncol, 
                                          Nnzero, &Neltvl);
    upcase(Type);

/*  Fourth line:  */
    line_ptr = fgets(line, 82, in_file);
    nullchk(line_ptr);
    (void) sscanf(line, "%16c%16c%20c%20c", Ptrfmt, Indfmt, Valfmt, Rhsfmt);
   
/*  (Optional) Fifth line: */
    if (*Rhscrd != 0 )
    { 
       line_ptr = fgets(line, 82, in_file);
       nullchk(line_ptr);
       (void) sscanf(line, "%3c%i%i", Rhstype, Nrhs, &Nrhsix);
    }

    return;
}


void readHB_mat_double(const char* filename, int colptr[], int rowind[], 
                                                                 double val[])
{
/****************************************************************************/
/*  This function opens and reads the specified file, interpreting its      */
/*  contents as a sparse matrix stored in the Harwell/Boeing standard       */
/*  format and creating compressed column storage scheme vectors to hold    */
/*  the index and nonzero value information.                                */
/*                                                                          */
/*    ----------                                                            */
/*    **CAVEAT**                                                            */
/*    ----------                                                            */
/*  Parsing real formats from Fortran is tricky, and this file reader       */
/*  does not claim to be foolproof.   It has been tested for cases when     */
/*  the real values are printed consistently and evenly spaced on each      */
/*  line, with Fixed (F), and Exponential (E or D) formats.                 */
/*                                                                          */
/*  **  If the input file does not adhere to the H/B format, the  **        */
/*  **             results will be unpredictable.                 **        */
/*                                                                          */
/****************************************************************************/
    FILE *in_file;
    char Title[73], Key[8], Type[4], Rhstype[4];
    char Ptrfmt[17], Indfmt[17], Valfmt[21], Rhsfmt[21];
    int Ptrcrd, Indcrd, Valcrd, Rhscrd;
    int Nrow, Ncol, Nnzero, Nentries;
    int Nrhs;
    char line[82];
    char* line_ptr;
    int Ptrperline, Ptrwidth, Indperline, Indwidth;
    int Valperline, Valwidth;
    int Valflag;           /* Indicates 'E','D', or 'F' float format */
    int i, j, ind, col, offset;
    int count, last;
    char* ThisElement;

    in_file = fopen( filename, "r");
    if (in_file == NULL)
    {
       fprintf(stderr,"Error: Cannot open file: %s\n",filename);
       exit(1);
    }

    readHB_header(in_file, Title, Key, Type, &Nrow, &Ncol, &Nnzero, &Nrhs,
                  Ptrfmt, Indfmt, Valfmt, Rhsfmt,
                  &Ptrcrd, &Indcrd, &Valcrd, &Rhscrd, Rhstype);

/*  Parse the array input formats from Line 3 of HB file  */
    ParseIfmt(Ptrfmt,&Ptrperline,&Ptrwidth);
    ParseIfmt(Indfmt,&Indperline,&Indwidth);
    if ( Type[0] != 'P' ) {          /* Skip if pattern only  */
    ParseRfmt(Valfmt,&Valperline,&Valwidth,&Valflag);
    }

/*  Read column pointer array:   */

    offset = 1-_SP_base;  /* if base 0 storage is declared (via macro definition), */
                          /* then storage entries are offset by 1                  */

    count=0;
    for (i=0;i<Ptrcrd;i++)
    {
       line_ptr = fgets(line, 82, in_file);
       nullchk(line_ptr);
       col =  0;
       for (ind = 0;ind<Ptrperline;ind++)
       {
          if (count > Ncol) break;
          ThisElement = substr(line,col,Ptrwidth);
          colptr[count] = atoi(ThisElement)-offset;
          count++; col += Ptrwidth;
       }
    }

/*  Read row index array:  */

    count = 0;
    for (i=0;i<Indcrd;i++)
    {
       line_ptr = fgets(line, 82, in_file);
       nullchk(line_ptr);
       col =  0;
       for (ind = 0;ind<Indperline;ind++)
       {
          if (count == Nnzero) break;
          ThisElement = substr(line,col,Indwidth);
          rowind[count] = atoi(ThisElement)-offset;
          count++; col += Indwidth;
       }
    }

/*  Read array of values:  */

    if ( Type[0] != 'P' ) {          /* Skip if pattern only  */

       if ( Type[0] == 'C' ) Nentries = 2*Nnzero;
           else Nentries = Nnzero;

    count = 0;
    for (i=0;i<Valcrd;i++)
    {
       line_ptr = fgets(line, 82, in_file);
       nullchk(line_ptr);
       if (Valflag == 'D') convertDtoE(line);
       col =  0;
       for (ind = 0;ind<Valperline;ind++)
       {
          if (count == Nentries) break;
          ThisElement = substr(line,col,Valwidth);
          if ( charLocation(ThisElement,Valflag) == -1 ) {
             /* insert a char prefix for exp */
             last = strlen(ThisElement);
             for (j=last+1;j>=0;j--) {
                ThisElement[j] = ThisElement[j-1];
                if ( ThisElement[j] == '+' || ThisElement[j] == '-' ) {
                   ThisElement[j-1] = Valflag;
                   break;
                }
             }
          }
          val[count] = atof(ThisElement);
          count++; col += Valwidth;
       }
    }
    }

    return;
}

void readHB_newmat_double(const char* filename, int* M, int* N, int* nonzeros, int** colptr, 
                          int** rowind, double** val)
{
	int nrhs;
        char *Type;

	readHB_info(filename, M, N, nonzeros, &Type, &nrhs);

        *colptr = (int *)malloc((*N+1)*sizeof(int));
        *rowind = (int *)malloc(*nonzeros*sizeof(int));
        if ( Type[0] == 'C' ) {
   fprintf(stderr, "Warning: Reading complex data from HB file %s.\n",filename);
   fprintf(stderr, "         Real and imaginary parts will be interlaced in val[].\n");
           /* Malloc enough space for real AND imaginary parts of val[] */
           *val = (double *)malloc(*nonzeros*sizeof(double)*2);
        } else {
           if ( Type[0] != 'P' ) {   
             /* Malloc enough space for real array val[] */
             *val = (double *)malloc(*nonzeros*sizeof(double));
           }
        }  /* No val[] space needed if pattern only */
	readHB_mat_double(filename, *colptr, *rowind, *val);

}


void readHB_mat_float(const char* filename, int* colptr, int* rowind, float* val)
{
/****************************************************************************/
/*  This function opens and reads the specified file, interpreting its      */
/*  contents as a sparse matrix stored in the Harwell/Boeing standard       */
/*  format and creating compressed column storage scheme vectors to hold    */
/*  the index and nonzero value information.                                */
/*                                                                          */
/*    ----------                                                            */
/*    **CAVEAT**                                                            */
/*    ----------                                                            */
/*  Parsing real formats from Fortran is tricky, and this file reader       */
/*  does not claim to be foolproof.   It has been tested for cases when     */
/*  the real values are printed consistently and evenly spaced on each      */
/*  line, with Fixed (F), and Exponential (E or D) formats.                 */
/*                                                                          */
/*  **  If the input file does not adhere to the H/B format, the  **        */
/*  **             results will be unpredictable.                 **        */
/*                                                                          */
/****************************************************************************/
    FILE *in_file;
    char Title[72], Key[9], Type[4], Rhstype[4];
    char Ptrfmt[17], Indfmt[17], Valfmt[21], Rhsfmt[21];
    int Ptrcrd, Indcrd, Valcrd, Rhscrd;
    int Nrow, Ncol, Nnzero, Nentries;
    int Nrhs;
    char line[82];
    char* line_ptr;
    int Ptrperline, Ptrwidth, Indperline, Indwidth;
    int Valperline, Valwidth;
    int Valflag;           /* Indicates 'E','D', or 'F' float format */
    int i, ind, col, offset;
    int count;
    char* ThisElement;

    in_file = fopen( filename, "r");
    if (in_file == NULL)
    {
       fprintf(stderr,"Error: Cannot open file: %s\n",filename);
       exit(1);
    }

    readHB_header(in_file, Title, Key, Type, &Nrow, &Ncol, &Nnzero, &Nrhs,
                  Ptrfmt, Indfmt, Valfmt, Rhsfmt,
                  &Ptrcrd, &Indcrd, &Valcrd, &Rhscrd, Rhstype);

/*  Parse the array input formats from Line 3 of HB file  */
    ParseIfmt(Ptrfmt,&Ptrperline,&Ptrwidth);
    ParseIfmt(Indfmt,&Indperline,&Indwidth);
    if ( Type[0] != 'P' ) {          /* Skip if pattern only  */
    ParseRfmt(Valfmt,&Valperline,&Valwidth,&Valflag);
    }

/*  Read column pointer array:   */

    offset = 1-_SP_base;  /* if base 0 storage is declared (via macro definition), */
                          /* then storage entries are offset by 1                  */
    count=0;
    for (i=0;i<Ptrcrd;i++)
    {
       line_ptr = fgets(line, 82, in_file);
       nullchk(line_ptr);
       col =  0;
       for (ind = 0;ind<Ptrperline;ind++)
       {
          if (count > Ncol) break;
          ThisElement = substr(line,col,Ptrwidth);
          colptr[count] = atoi(ThisElement)-offset;
          count++; col += Ptrwidth;
       }
    }

/*  Read row index array:  */

    count = 0;
    for (i=0;i<Indcrd;i++)
    {
       line_ptr = fgets(line, 82, in_file);
       nullchk(line_ptr);
       col =  0;
       for (ind = 0;ind<Indperline;ind++)
       {
          if (count == Nnzero) break;
          ThisElement = substr(line,col,Indwidth);
          rowind[count] = atoi(ThisElement)-offset;
          count++; col += Indwidth;
       }
    }

/*  Read array of values:  */

    if ( Type[0] != 'P' ) {          /* Skip if pattern only  */

       if ( Type[0] == 'C' ) Nentries = 2*Nnzero;
           else Nentries = Nnzero;

    count = 0;
    for (i=0;i<Valcrd;i++)
    {
       line_ptr = fgets(line, 82, in_file);
       nullchk(line_ptr);
       if (Valflag == 'D') convertDtoE(line);
       col =  0;
       for (ind = 0;ind<Valperline;ind++)
       {
          if (count == Nentries) break;
          ThisElement = substr(line,col,Valwidth);
          val[count] = atof(ThisElement);
          count++; col += Valwidth;
       }
    }

    }

    return;
}


void readHB_newmat_float(const char* filename, int* M, int* N, int* nonzeros, int** colptr, 
                         int** rowind, float** val)
{
	int nrhs;
        char *Type;

	readHB_info(filename, M, N, nonzeros, &Type, &nrhs);
	*colptr = (int *)malloc((*N+1)*sizeof(int));
	*rowind = (int *)malloc(*nonzeros*sizeof(int));
        if ( Type[0] == 'C' ) {
           fprintf(stderr, "Warning: Reading complex data from HB file %s.\n",filename);
           fprintf(stderr, "         Real and imaginary parts will be interlaced in val[].\n");
           /* Malloc enough space for real AND imaginary parts of val[] */
           *val = (float *)malloc(*nonzeros*sizeof(float)*2);
        } else {
           if ( Type[0] != 'P' ) {   
             /* Malloc enough space for real array val[] */
             *val = (float *)malloc(*nonzeros*sizeof(float));
           }
        }  /* No val[] space needed if pattern only */

	readHB_mat_float(filename, *colptr, *rowind, *val);

}





void readHB_rhs_double(const char* filename, double b[], int j)
{
/****************************************************************************/
/*  This function opens and reads the specified file, returning a right-    */
/*  hand-side vector b.  If the file specifies multiple right-hand-sides    */
/*  (that is, a right-hand-side matrix B), then the optional argument j     */
/*  may be used to select the (j+1)st column of B.                          */
/*                                                                          */
/*    ----------                                                            */
/*    **CAVEAT**                                                            */
/*    ----------                                                            */
/*  Parsing real formats from Fortran is tricky, and this file reader       */
/*  does not claim to be foolproof.   It has been tested for cases when     */
/*  the real values are printed consistently and evenly spaced on each      */
/*  line, with Fixed (F), and Exponential (E or D) formats.                 */
/*                                                                          */
/*  **  If the input file does not adhere to the H/B format, the  **        */
/*  **             results will be unpredictable.                 **        */
/*                                                                          */
/****************************************************************************/
    int i, n, null_entries, lines_left;
    FILE *in_file;
    char Title[73], Key[9], Type[4], Rhstype[4];
    char Ptrfmt[17], Indfmt[17], Valfmt[21], Rhsfmt[21];
    int Ptrcrd, Indcrd, Valcrd, Rhscrd;
    int Nrow, Ncol, Nnzero, Nentries;
    int Nrhs;
    int Rhsperline, Rhswidth;
    int Rhsflag;
    char buffer[BUFSIZ];
    int ind, col;
    int count;
    char *line_ptr;
    char line[BUFSIZ];
    char *ThisElement;

    if ((in_file = fopen( filename, "r")) == NULL) 
    {
      fprintf(stderr,"Error: Cannot open file: %s\n",filename);
      exit(1);
     }

    readHB_header(in_file, Title, Key, Type, &Nrow, &Ncol, &Nnzero, &Nrhs,
                  Ptrfmt, Indfmt, Valfmt, Rhsfmt,
                  &Ptrcrd, &Indcrd, &Valcrd, &Rhscrd, Rhstype);

    if (Nrhs <= 0)
    {
      fprintf(stderr, "Error: Attempt to read rhs when none is present.\n");
      exit(1);
      return;
    }
    if (Rhstype[0] != 'F')
    {
      fprintf(stderr,"Error: Attempt to read rhs which is not stored in Full form.\n");
      fprintf(stderr,"       Rhs must be specified as full. \n");
      exit(1);
      return;
    }

    /* If reading complex data, allow for interleaved real and imaginary values. */ 
    if ( Type[0] == 'C' ) {
       Nentries = 2*Nrow;
     } else {
       Nentries = Nrow;
    }

    ParseRfmt(Rhsfmt, &Rhsperline, &Rhswidth, &Rhsflag);

/*  Lines to skip before starting to read RHS values... */
    n = Ptrcrd + Indcrd + Valcrd + j*Nentries/Rhsperline;

/*  number of entries on the line of interest to skip before */
/*  reading RHS values...                                    */
    null_entries = j*Nentries%Rhsperline;
    lines_left = (int) ( .5 + (double) (Nentries-Rhsperline+null_entries)/ 
                                            (double) Rhsperline );

    for (i = 0; i < n; i++)
      fgets(buffer, BUFSIZ, in_file);

    count = 0;

/*  Handle first line separately, in case j != 0 and the right-hand-side */
/*  of interest starts mid-line (after null_entries entries)...          */

    line_ptr = fgets(line, 82, in_file);
    nullchk(line_ptr);
    col = 0;
    for (ind = 0; ind < Rhsperline; ind++)
    {
      if (ind < null_entries)
        col += Rhswidth;
      else
      {
        if (count > Nentries-1)
          return;
        ThisElement = substr(line, col, Rhswidth);
        b[count] = atof(ThisElement);
        count++; col += Rhswidth;
      }
    }

/*  Handle subsequent lines...              */
    for (i=count;i<Nentries;i++) {
       if ( (i-count)%Rhsperline == 0 ) {
           line_ptr = fgets(line, 82, in_file);
           nullchk(line_ptr);
           col = 0;
       }
       ThisElement = substr(line, col, Rhswidth);
       b[i] = atof(ThisElement);
       col += Rhswidth;
    }

    return;
}

void readHB_newrhs_double(const char* filename, double** b, int j)
{
        int nrhs, M, N, nonzeros;
        char *Type;
	readHB_info(filename, &M, &N, &nonzeros, &Type, &nrhs);
        if ( nrhs == 0 ) {
          fprintf(stderr,"Error: Requested read of rhs %d when none are present.\n", j+1);
          exit(1);
        } else if ( j >= 0 && j < nrhs ) {
          if ( Type[0] == 'C' ) {
            fprintf(stderr, "Warning: Reading complex rhs from HB file %s.",filename);
            fprintf(stderr, "         Real and imaginary parts will be interlaced in b[].");
            *b = (double *)malloc(M*sizeof(double)*2);
	    readHB_rhs_double(filename, *b, j);
          } else {
            *b = (double *)malloc(M*sizeof(double));
	    readHB_rhs_double(filename, *b, j);
          }
          return; 
        } else {
          fprintf(stderr,"Error: Requested read of rhs %d when only %d present.\n", j+1, nrhs);
          exit(1);
        }
}

void readHB_rhs_float(const char* filename, float b[], int j)
{
/****************************************************************************/
/*  This function opens and reads the specified file, returning a right-    */
/*  hand-side vector b.  If the file specifies multiple right-hand-sides    */
/*  (that is, a right-hand-side matrix B), then the optional argument j     */
/*  may be used to select the (j+1)st column of B.                          */
/*                                                                          */
/*    ----------                                                            */
/*    **CAVEAT**                                                            */
/*    ----------                                                            */
/*  Parsing real formats from Fortran is tricky, and this file reader       */
/*  does not claim to be foolproof.   It has been tested for cases when     */
/*  the real values are printed consistently and evenly spaced on each      */
/*  line, with Fixed (F), and Exponential (E or D) formats.                 */
/*                                                                          */
/*  **  If the input file does not adhere to the H/B format, the  **        */
/*  **             results will be unpredictable.                 **        */
/*                                                                          */
/****************************************************************************/
    int i, n, null_entries, lines_left;
    FILE *in_file;
    char Title[73], Key[9], Type[4], Rhstype[4];
    char Ptrfmt[17], Indfmt[17], Valfmt[21], Rhsfmt[21];
    int Ptrcrd, Indcrd, Valcrd, Rhscrd;
    int Nrow, Ncol, Nnzero, Nentries;
    int Nrhs;
    int Rhsperline, Rhswidth;
    int Rhsflag;
    char buffer[BUFSIZ];
    int ind, col;
    int count;
    char *line_ptr;
    char line[BUFSIZ];
    char *ThisElement;

    if ((in_file = fopen( filename, "r")) == NULL) 
    {
      fprintf(stderr,"Error: Cannot open file: %s\n",filename);
      exit(1);
     }

    readHB_header(in_file, Title, Key, Type, &Nrow, &Ncol, &Nnzero, &Nrhs,
                  Ptrfmt, Indfmt, Valfmt, Rhsfmt,
                  &Ptrcrd, &Indcrd, &Valcrd, &Rhscrd, Rhstype);

    if (Nrhs <= 0)
    {
      fprintf(stderr,"Error: Attempt to read rhs when none is present.\n");
      exit(1);
      return;
    }
    if (Rhstype[0] != 'F')
    {
      fprintf(stderr,"Error: Attempt to read rhs which is not stored in Full form.\n");
      fprintf(stderr,"       Rhs must be specified as full. \n");
      exit(1);
      return;
    }

    /* If reading complex data, allow for interleaved real and imaginary values. */ 
    if ( Type[0] == 'C' ) {
       Nentries = 2*Nrow;
       if ( sizeof(b) < Nentries*sizeof(float) ) {
          fprintf(stderr,"Not enough storage for complex rhs. \n");
          fprintf(stderr,"Allocate 2*Nrow*sizeof(float) and try again.\n");
          exit(1);
       }
     } else {
       Nentries = Nrow;
       if ( sizeof(b) < Nentries*sizeof(float) ) {
          fprintf(stderr,"Not enough storage for real rhs. \n");
          fprintf(stderr,"Allocate Nrow*sizeof(float) and try again.\n");
          exit(1);
       }
    }

    ParseRfmt(Rhsfmt, &Rhsperline, &Rhswidth, &Rhsflag);

/*  Lines to skip before starting to read RHS values... */
    n = Ptrcrd + Indcrd + Valcrd + j*Nrow/Rhsperline;

/*  number of entries on the line of interest to skip before */
/*  reading RHS values...                                    */
    null_entries = j*Nrow%Rhsperline;
    lines_left = (int)( .5 + (double) (Nrow-Rhsperline+null_entries)/
                                             (double) Rhsperline);

    for (i = 0; i < n; i++)
      fgets(buffer, BUFSIZ, in_file);

    count = 0;

/*  Handle first line separately, in case j != 0 and the right-hand-side */
/*  of interest starts mid-line (after null_entries entries)...          */

    line_ptr = fgets(line, 82, in_file);
    nullchk(line_ptr);
    col = 0;
    for (ind = 0; ind < Rhsperline; ind++)
    {
      if (ind < null_entries)
        col += Rhswidth;
      else
      {
        if (count > Nrow-1)
          break;
        ThisElement = substr(line, col, Rhswidth);
        b[count] = atof(ThisElement);
        count++; col += Rhswidth;
      }
    }

/*  Handle subsequent lines...              */
    for (i=count;i<Nrow;i++) {
       if ( (i-count)%Rhsperline == 0 ) {
           line_ptr = fgets(line, 82, in_file);
           nullchk(line_ptr);
           col = 0;
       }
       ThisElement = substr(line, col, Rhswidth);
       b[i] = atof(ThisElement);
       col += Rhswidth;
    }

    return;
}

void readHB_newrhs_float(const char* filename, float** b, int j)
{
        int nrhs, M, N, nonzeros;
        char *Type;
	readHB_info(filename, &M, &N, &nonzeros, &Type, &nrhs);
        if ( j >= 0 && j < nrhs ) {
          if ( Type[0] == 'C' ) {
            fprintf(stderr, "Warning: Reading complex rhs from HB file %s.",filename);
            fprintf(stderr, "         Real and imaginary parts will be interlaced in b[].");
            *b = (float *)malloc(M*sizeof(float)*2);
	    readHB_rhs_float(filename, *b, j);
          } else {
            *b = (float *)malloc(M*sizeof(float));
	    readHB_rhs_float(filename, *b, j);
          }
          return; 
        } else {
          fprintf(stderr,"Error: Requested read of rhs %d when none is present.\n", j+1);
          exit(1);
        }
}


void writeHB_mat_double(const char* filename, int M, int N, 
                        int nz, const int colptr[], const int rowind[], 
                        const double val[], int nrhs, const double rhs[], 
                        const char* Type, const char* Title, const char* Key)
{
/****************************************************************************/
/*  The writeHB function opens the named file and writes the specified      */
/*  matrix and optional right-hand-side(s) to that file in Harwell-Boeing   */
/*  format.                                                                 */
/*                                                                          */
/*  For a description of the Harwell Boeing standard, see:                  */
/*            Duff, et al.,  ACM TOMS Vol.15, No.1, March 1989              */
/*                                                                          */
/****************************************************************************/


    FILE *out_file;

    int totcrd, ptrcrd, indcrd, valcrd, rhscrd;
    int nvalentries, nrhsentries;
    char *ptrfmt, *indfmt, *valfmt, *rhsfmt;
    int i;
    int entry, finfo;
    int offset;

    if ( Type[0] == 'C' ) {
         nvalentries = 2*nz;
         nrhsentries = 2*M;
    } else {
         nvalentries = nz;
         nrhsentries = M;
    }

    if ( *filename != '\0' ) {
       out_file = fopen( filename, "w");
    } else {
       out_file = stdout;
    }

    ptrcrd = (N+1)/8;
    if ( (N+1)%8 != 0) ptrcrd++;

    indcrd = nz/8;
    if ( nz%8 != 0) indcrd++;

    valcrd = nvalentries/4;
    if ( nvalentries%4 != 0) valcrd++;

    rhscrd = nrhs*nrhsentries/4; 
    if ( nrhs*nrhsentries%4 != 0) rhscrd++;

    totcrd = 4+ptrcrd+indcrd+valcrd+rhscrd;

    ptrfmt = "(8I10)          ";
    indfmt = ptrfmt;
    valfmt = "(4E20.16)           ";
    rhsfmt = valfmt;

/*  Print header information:  */

    fprintf(out_file,"%-72s%-8s\n%14d%14d%14d%14d%14d\n",Title, Key, totcrd,
            ptrcrd, indcrd, valcrd, rhscrd);
    fprintf(out_file,"%3s%11s%14d%14d%14d\n",Type,"          ", M, N, nz);
    if ( Type[0] == 'P' ) {    /* Do not print real formats for pattern only */
      fprintf(out_file,"%16s%16s\n", ptrfmt, indfmt);
    } else {
      fprintf(out_file,"%16s%16s%20s%20s\n", ptrfmt, indfmt, valfmt, rhsfmt);
    }
    if ( nrhs != 0 ) {
/*     Print optional fifth header line for right-hand-side information : */
       fprintf(out_file,"F             %d\n", nrhs);
    }

    offset = 1-_SP_base;  /* if base 0 storage is declared (via macro definition), */
                          /* then storage entries are offset by 1                  */

/*  Print column pointers:   */
    for (i=0;i<N+1;i++)
    {
       entry = colptr[i]+offset;
       fprintf(out_file,"%10d",entry);
       if ( (i+1)%8 == 0 ) fprintf(out_file,"\n");
    }

   if ( (N+1) % 8 != 0 ) fprintf(out_file,"\n");

/*  Print row indices:       */
    for (i=0;i<nz;i++)
    {
       entry = rowind[i]+offset;
       fprintf(out_file,"%10d",entry);
       if ( (i+1)%8 == 0 ) fprintf(out_file,"\n");
    }

   if ( nz % 8 != 0 ) fprintf(out_file,"\n");

/*  Print values:            */

    if ( Type[0] != 'P' ) {          /* Skip if pattern only  */

    for (i=0;i<nvalentries;i++)
    {
       fprintf(out_file,"% 20.12E",val[i]);
       if ( (i+1)%4 == 0 ) fprintf(out_file,"\n");
    }

    if ( nvalentries % 4 != 0 ) fprintf(out_file,"\n");

/*  Print right hand sides:  */
    if ( nrhs > 0 ) {
       for (i=0;i<nrhs*nrhsentries;i++)
       {
          fprintf(out_file,"% 20.12E",rhs[i]);
          if ( (i+1)%4 == 0 ) fprintf(out_file,"\n");
       }
    }

    }

    finfo = fclose(out_file);
    if (finfo != 0) fprintf(stderr,"Error closing file in writeHB_mat_double().\n");

    return;
}


void writeHB_mat_float(const char* filename, int M, int N, 
                        int nz, const int colptr[], const int rowind[], 
                        const float val[], int nrhs, const float rhs[], 
                        const char* Type, const char* Title, const char* Key)
{
/****************************************************************************/
/*  The writeHB function opens the named file and writes the specified      */
/*  matrix and optional right-hand-side(s) to that file in Harwell-Boeing   */
/*  format.                                                                 */
/*                                                                          */
/*  For a description of the Harwell Boeing standard, see:                  */
/*            Duff, et al.,  ACM TOMS Vol.15, No.1, March 1989              */
/*                                                                          */
/****************************************************************************/


    FILE *out_file;

    int totcrd, ptrcrd, indcrd, valcrd, rhscrd;
    int nvalentries, nrhsentries;
    char *ptrfmt, *indfmt, *valfmt, *rhsfmt;
    int i;
    int entry, finfo;
    int offset;

    if ( Type[0] == 'C' ) {
         nvalentries = 2*nz;
         nrhsentries = 2*M;
    } else {
         nvalentries = nz;
         nrhsentries = M;
    }

    if ( *filename != '\0' ) {
       out_file = fopen( filename, "w");
    } else {
       out_file = stdout;
    }

    ptrcrd = (N+1)/8;
    if ( (N+1)%8 != 0) ptrcrd++;

    indcrd = nz/8;
    if ( nz%8 != 0) indcrd++;

    valcrd = nvalentries/4;
    if ( nvalentries%4 != 0) valcrd++;

    rhscrd = nrhs*nrhsentries/4; 
    if ( nrhs*nrhsentries%4 != 0) rhscrd++;

    totcrd = 4+ptrcrd+indcrd+valcrd+rhscrd;

    ptrfmt = "(8I10)          ";
    indfmt = ptrfmt;
    valfmt = "(4E20.16)           ";
    rhsfmt = valfmt;

/*  Print header information:  */

    fprintf(out_file,"%-72s%-8s\n%14d%14d%14d%14d%14d\n",Title, Key, totcrd,
            ptrcrd, indcrd, valcrd, rhscrd);
    fprintf(out_file,"%3s%11s%14d%14d%14d\n",Type,"          ", M, N, nz);
    if ( Type[0] == 'P' ) {    /* Do not print real formats for pattern only */
      fprintf(out_file,"%16s%16s\n", ptrfmt, indfmt);
    } else {
      fprintf(out_file,"%16s%16s%20s%20s\n", ptrfmt, indfmt, valfmt, rhsfmt);
    }
    if ( nrhs != 0 ) {
/*     Print optional fifth header line for right-hand-side information : */
       fprintf(out_file,"F             %d\n", nrhs);
    }

    offset = 1-_SP_base;  /* if base 0 storage is declared (via macro definition), */
                          /* then storage entries are offset by 1                  */

/*  Print column pointers:   */
    for (i=0;i<N+1;i++)
    {
       entry = colptr[i]+offset;
       fprintf(out_file,"%10d",entry);
       if ( (i+1)%8 == 0 ) fprintf(out_file,"\n");
    }

   if ( (N+1) % 8 != 0 ) fprintf(out_file,"\n");

/*  Print row indices:       */
    for (i=0;i<nz;i++)
    {
       entry = rowind[i]+offset;
       fprintf(out_file,"%10d",entry);
       if ( (i+1)%8 == 0 ) fprintf(out_file,"\n");
    }

   if ( nz % 8 != 0 ) fprintf(out_file,"\n");

/*  Print values:            */
    if ( Type[0] != 'P' ) {          /* Skip if pattern only  */
    for (i=0;i<nvalentries;i++)
    {
       fprintf(out_file,"% 20.12E",val[i]);
       if ( (i+1)%4 == 0 ) fprintf(out_file,"\n");
    }

    if ( nvalentries % 4 != 0 ) fprintf(out_file,"\n");

/*  Print right hand sides:  */
    if ( nrhs > 0 ) {
       for (i=0;i<nrhs*nrhsentries;i++)
       {
          fprintf(out_file,"% 20.12E",rhs[i]);
          if ( (i+1)%4 == 0 ) fprintf(out_file,"\n");
       }
    }
    }

    finfo = fclose(out_file);
    if (finfo != 0) fprintf(stderr,"Error closing file in writeHB_mat_float().\n");

    return;
}

void writetxt_mat_double(const char* filename, int M, int N, 
                        int nz, const int colptr[], const int rowind[], 
                        const double val[], 
                        const char* Type, const char* Title, const char* Key)
{
/****************************************************************************/
/*  The writetxt function opens the named file and writes the specified     */
/*  compressed column matrix and optional right-hand-side(s) to that file   */
/*  in plain text format.                                                   */
/****************************************************************************/
    FILE *out_file; 

    int rowp1, colp1, indxval;
    int maxi, maxj;
    int i, j;

    if ( *filename != '\0' ) {
       out_file = fopen( filename, "w");
    } else {
       out_file = stdout;
    }

    maxi = M-1+_SP_base;
    maxj = N-1+_SP_base;

    fprintf(out_file,"%d %d %d\n", M, N, nz);
    if ( Type[0] == 'C' ) { 
/*  Loop through columns */
    for (j = 0; j < N ; j++)
       for (i=colptr[j];i<colptr[j+1];i++)
       {
          indxval = 2*(i-_SP_base);
          rowp1 = rowind[i-_SP_base]+1-_SP_base;
          colp1 = j + 1;
          fprintf(out_file,"%d%1s%d%1s% le%1s% le\n", rowp1, " ",
                     colp1," ", val[indxval], " ",val[indxval+1]);
       }
    } else if ( Type[0] != 'P' ) { 
/*  Loop through columns */
    for (j = 0; j < N ; j++)
       for (i=colptr[j];i<colptr[j+1];i++)
       {
          rowp1 = rowind[i-_SP_base]+1-_SP_base;
          colp1 = j + 1;
          fprintf(out_file,"%d%1s%d%1s% le\n", rowp1, " ",
                                           colp1," ", val[i-_SP_base]);
       }
    } else {
/*  Loop through columns */
    for (j = 0; j < N ; j++)
       for (i=colptr[j];i<colptr[j+1];i++)
       {
          rowp1 = rowind[i-_SP_base]+1-_SP_base;
          colp1 = j + 1;
          fprintf(out_file,"%d%1s%d%2s\n", rowp1, " ",
                                           colp1, " 0");
       }

    }

    fclose(out_file);

    return;
}


void writetxt_mat_float(const char* filename, int M, int N, 
                        int nz, const int colptr[], const int rowind[], 
                        const float val[], 
                        const char* Type, const char* Title, const char* Key)
{
/****************************************************************************/
/*  The writetxt function opens the named file and writes the specified     */
/*  compressed column matrix and optional right-hand-side(s) to that file   */
/*  in plain text format.                                                   */
/****************************************************************************/
    FILE *out_file; 

    int rowp1, colp1, indxval;
    int maxi, maxj;
    int i, j;

    if ( *filename != '\0' ) {
       out_file = fopen( filename, "w");
    } else {
       out_file = stdout;
    }

    maxi = M-1+_SP_base;
    maxj = N-1+_SP_base;

    fprintf(out_file,"%d %d %d\n", M, N, nz);
    if ( Type[0] == 'C' ) { 
/*  Loop through columns */
    for (j = 0; j < N ; j++)
       for (i=colptr[j];i<colptr[j+1];i++)
       {
          indxval = 2*(i-_SP_base);
          rowp1 = rowind[i-_SP_base]+1-_SP_base;
          colp1 = j + 1;
          fprintf(out_file,"%d%1s%d%1s% le%1s% le\n", rowp1, " ",
                  colp1," ", val[indxval], " ", val[indxval+1]);
       }
    } else if ( Type[0] != 'P' ) { 
/*  Loop through columns */
    for (j = 0; j < N ; j++)
       for (i=colptr[j];i<colptr[j+1];i++)
       {
          rowp1 = rowind[i-_SP_base]+1-_SP_base;
          colp1 = j + 1;
          fprintf(out_file,"%d%1s%d%1s% le\n", rowp1, " ",
                                           colp1," ", val[i-_SP_base]);
       }
    } else { 
/*  Loop through columns */
    for (j = 0; j < N ; j++)
       for (i=colptr[j];i<colptr[j+1];i++)
       {
          rowp1 = rowind[i-_SP_base]+1-_SP_base;
          colp1 = j + 1;
          fprintf(out_file,"%d%1s%d%2s\n", rowp1, " ",
                                           colp1, " 0");
       }

    }

    fclose(out_file);

    return;
}

/****************************************************************************/

              /********************************************/
              /*  Utility functions supporting readHB:    */
              /********************************************/

/****************************************************************************/

/**************************************************/
/*  Check to ensure header information is present */
/**************************************************/
static void nullchk(char* line_ptr)
{
    if (line_ptr == NULL)
    {
          printf("Cannot complete reading file information.\n ");
          exit(1);
    }
}

/*************************************************/
/*  Parse an *integer* format field to determine */
/*  width and number of elements per line.       */
/*************************************************/
static void ParseIfmt(char* fmt, int* perline, int* width)
{
    char *PPL, *WID;
    upcase(fmt);
    PPL = substr_before(fmt,'I');
    PPL = substr_after(PPL,'(');
    *perline = atoi(PPL);
    WID = substr_after(fmt,'I');
    WID = substr(WID,0,strlen(WID)-1);
    *width = atoi(WID);
}

/*************************************************/
/*  Parse a *real* format field to determine     */
/*  width and number of elements per line.       */
/*  Also sets flag indicating 'E' 'F' 'P' or 'D' */
/*  format.                                      */
/*************************************************/
static void ParseRfmt(char* fmt, int* perline, int* width, int* flag)
{
    int foundE, foundD, foundF, foundP;
    char *PPL, *WID;

    if (fmt == NULL )
    {
      *perline = 0; *width = 0; *flag = '\0'; return;
    }

    upcase(fmt);
    foundP = charLocation(fmt,'P');
    foundE = charLocation(fmt,'E');
    foundD = charLocation(fmt,'D');
    foundF = charLocation(fmt,'F');      /* Fixed format */
    if (foundP != -1 )            /* Remove any scaling factor, which */
    {                             /* affects output only, not input */
      fmt = fmt + foundP;
      if ( fmt[1] == ',' ) fmt++;
      fmt[0] = '(';
    }
    if (foundE != -1 )
    {
      *flag = 'E';
      PPL = substr_before(fmt,'E');
      PPL = substr_after(PPL,'(');
      *perline = atoi(PPL);
      WID = substr_after(fmt,'E');
      WID = substr_through(WID,'.');
      WID = substr(WID,0,strlen(WID)-1);
      *width = atoi(WID);
    }
    else if (foundD != -1)
    {
      *flag = 'D';
      PPL = substr_before(fmt,'D');
      PPL = substr_after(PPL,'(');
      *perline = atoi(PPL);
      WID = substr_after(fmt,'D');
      WID = substr_through(WID,'.');
      WID = substr(WID,0,strlen(WID)-1);
      *width = atoi(WID);
    }
    else if (foundF != -1)
    {
      *flag = 'F';
      PPL = substr_before(fmt,'F');
      PPL = substr_after(PPL,'(');
      *perline = atoi(PPL);
      WID = substr_after(fmt,'F');
      WID = substr_through(WID,'.');
      WID = substr(WID,0,strlen(WID)-1);
      *width = atoi(WID);
    }
    else
    {
      printf("Real format in H/B file not supported.\n");
      exit(1);
    }

}


/*****************************************************/
/*  Converts a line with real data in 'D' format     */
/*  to 'E' format by simply changing the character   */
/*  'D' to 'E' by adding 1 to it.                    */
/*****************************************************/
static void convertDtoE(char* line)
{
    int len, i;
    len = strlen(line);
    for (i=0;i<len;i++)
       if ( line[i] == 'D' || line[i] == 'd' ) line[i] = line[i]+1;
    return;
}


/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
/*       small string processing library for iohb.c              */
/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/

/*****************************************************************/
/*  This file contains some minimal string functions to support  */
/*  support parsing the data formats in a Harwell-Boeing file.   */
/*****************************************************************/

static char* substr(char* S, int pos, int len)
{
    int i;
    char *SubS;
    SubS = (char *)malloc(len+1);
    if ( (pos+len) > strlen(S))
    {
       fprintf(stderr,"Error: Substring (%s, %d, %d) will read beyond string boundary.\n",
               S, pos, len);
       exit(1);
    }
    for (i=0;i<len;i++)
       SubS[i] = S[pos+i];
    SubS[len] = '\0';
    return SubS;
}

static char* substr_after(char* S, char M)
{
/*  Return the substring of S from the char after the first   */
/*  occurrence of M to the end of the string                  */
    int i, pos, SubSlen;
    char * SubS;
    pos = 0;
    for (i=0;i< strlen(S);i++)
    {
       if ( S[i] == M )
       {
          pos = i+1;
          break;
       }
    }
    if (pos != 0)
    {
       SubSlen = strlen(S) - pos;
       SubS = (char *)malloc(SubSlen);
       for (i=0;i< SubSlen;i++)
          SubS[i] = S[pos+i];
       SubS[SubSlen] = '\0';
       return SubS;
     }
     else
     {
       fprintf(stderr,"substr_after: Character %c not found in input string.\n", M);
       exit(1);
     }

       return NULL;       
 }



static char* substr_before(char* S, char M)
{
/*  Return the substring of S from the first char of S to the   */
/*  char before the first occurrence of M                       */
    int i, pos, SubSlen;
    char* SubS;
    pos = 0;
    for (i=0;i< strlen(S);i++)
    {
       if ( S[i] == M )
       {
          pos = i-1;
          break;
       }
    }
    if (pos != 0)
    {
       SubSlen = pos+1;
       SubS = (char *)malloc(SubSlen);
       for (i=0;i< SubSlen;i++)
          SubS[i] = S[i];
       SubS[SubSlen] = '\0';
       return SubS;
     }
     else
     {
       fprintf(stderr,"substr_before: Character %c not found in input string.\n", M);
       exit(1);
     }
     return NULL;   
 }


static char* substr_through(char* S, char M)
{
/*  Similer to substr_before, but include M         */
    int i, pos, len, SubSlen;
    char *SubS;
    len = strlen(S);
    pos = 0;
    for (i=0;i< len;i++)
    {
       if ( S[i] == M )
       {
          pos = i;
          break;
       }
    }
    if (pos != 0)
    {
       SubSlen = pos+1;
       SubS = (char *)malloc(SubSlen);
       for (i=0;i< SubSlen;i++)
          SubS[i] = S[i];
       SubS[SubSlen] = '\0';
       return SubS;
     }
     else
     {
       fprintf(stderr,"substr_through: Character %c not found in input string.\n", M);
       exit(1);
     }
       return NULL;
 }


static void upcase(char* S)
{
/*  Convert S to uppercase     */
    int i,len;
    len = strlen(S);
    for (i=0;i< len;i++)
       if ( S[i] >= 'a' && S[i] <= 'z' ) S[i] = S[i] - 32;
}


static int charLocation(char* S, char M)
{
/*  Return the location of the first occurrence of M in S  */
/*  Return -1 if M does not occur in S                  */
    int i, pos, len;
    len = strlen(S);
    pos = -1;
    for (i=0;i< len;i++)
    {
       if ( S[i] == M )
       {
          pos = i;
          break;
       }
    }
    return pos;
}

