/*  misc.h  */

#include "../cfiles.h"
#include "../ETree.h"
#include "../Graph.h"
#include "../MSMD.h"
#include "../GPart.h"

/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods in ND.c --------------------------------------------------
------------------------------------------------------------------------
*/
/*
   ------------------------------------------------------------
   purpose -- main procedure. if the input region is 1 x 1 x 1,
              the node is put into the permutation vector.
              otherwise the region is split into three pieces,
              two subregions and a separator, and recursive 
              calls are made to order the subregions

   input --

      n1         -- number of points in the first direction
      n2         -- number of points in the second direction
      n3         -- number of points in the third direction
      new_to_old -- pointer to the permutation vector
      west       -- west coordinate
      east       -- east coordinate
      south      -- south coordinate
      north      -- north coordinate
      bottom     -- bottom coordinate
      top        -- top coordinate

   created -- 95nov15, cca
   ------------------------------------------------------------
*/
void
mkNDperm ( 
   int   n1, 
   int   n2, 
   int   n3, 
   int   new_to_old[], 
   int   west, 
   int   east, 
   int   south, 
   int   north, 
   int   bottom, 
   int   top 
) ;
/*
   -----------------------------------------------------
   purpose -- to print a vector on a 2-d grid

   input --

      n1   -- number of points in the first direction
      n2   -- number of points in the second direction
      ivec -- integer vector to be printed in %4d format
      fp   -- file pointer

   created -- 95nov16, cca
   -----------------------------------------------------
*/
void
fp2DGrid ( 
   int    n1, 
   int    n2, 
   int    ivec[], 
   FILE   *fp 
) ;
/*
   -----------------------------------------------------
   purpose -- to print a vector on a 3-d grid

   input --

      n1   -- number of points in the first direction
      n2   -- number of points in the second direction
      n3   -- number of points in the third direction
      ivec -- integer vector to be printed in %4d format
      fp   -- file pointer

   created -- 95nov16, cca
   -----------------------------------------------------
*/
void
fp3DGrid ( 
   int    n1, 
   int    n2, 
   int    n3, 
   int    ivec[], 
   FILE   *fp 
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods in localND.c ---------------------------------------------
------------------------------------------------------------------------
*/
/*
   ------------------------------------------------------------
   compute an old-to-new ordering for 
   local nested dissection in two dimensions

   n1       -- number of grid points in first direction
   n2       -- number of grid points in second direction
   p1       -- number of domains in first direction
   p2       -- number of domains in second direction
   dsizes1  -- domain sizes in first direction, size p1
               if NULL, then we construct our own
   dsizes2  -- domain sizes in second direction, size p2
               if NULL, then we construct our own
   oldToNew -- old-to-new permutation vector

   note : the following must hold
      n1 > 0, n2 >0, n1 >= 2*p1 - 1, n2 >= 2*p2 - 1, p2 > 1
      sum(dsizes1) = n1 - p1 + 1 and sum(dsizes2) = n2 - p2 + 1

   created -- 95nov16, cca
   ------------------------------------------------------------
*/
void
localND2D ( 
   int   n1, 
   int   n2, 
   int   p1, 
   int   p2, 
   int   dsizes1[], 
   int   dsizes2[], 
   int   oldToNew[] 
) ;
/*
   ------------------------------------------------------------
   compute an old-to-new ordering for 
   local nested dissection in three dimensions

   n1       -- number of grid points in first direction
   n2       -- number of grid points in second direction
   n3       -- number of grid points in third direction
   p1       -- number of domains in first direction
   p2       -- number of domains in second direction
   p3       -- number of domains in third direction
   dsizes1  -- domain sizes in first direction, size p1
               if NULL, then we construct our own
   dsizes2  -- domain sizes in second direction, size p2
               if NULL, then we construct our own
   dsizes3  -- domain sizes in third direction, size p3
               if NULL, then we construct our own
   oldToNew -- old-to-new permutation vector

   note : the following must hold
      n1 > 0, n2 >0, n3 > 0,
      n1 >= 2*p1 - 1, n2 >= 2*p2 - 1, n3 >= 2*p3 - 1, p3 > 1
      sum(dsizes1) = n1 - p1 + 1, sum(dsizes2) = n2 - p2 + 1
      sum(dsizes3) = n3 - p3 + 1

   created -- 95nov16, cca
   ------------------------------------------------------------
*/
void
localND3D ( 
   int   n1, 
   int   n2, 
   int   n3, 
   int   p1, 
   int   p2, 
   int   p3,
   int   dsizes1[], 
   int   dsizes2[], 
   int   dsizes3[], 
   int   oldToNew[] 
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods in iohb.c ------------------------------------------------
------------------------------------------------------------------------
*/
/*
//                       Harwell-Boeing File I/O in C
//            Extracted from  SparseLib++ Sparse Matrix Library
//
//             National Institute of Standards and Technology, MD.
//                         University of Notre Dame, IN.
//              Authors:  R. Pozo, K. Remington, A. Lumsdaine
//                     (C) 1995 All Rights Reserved
//
//                                NOTICE
//
// Permission to use, copy, modify, and distribute this software and
// its documentation for any purpose and without fee is hereby granted
// provided that the above copyright notice appear in all copies and
// that both the copyright notice and this permission notice appear in
// supporting documentation.
//
// Neither the Authors nor the Institutions (National Institute of Standards
// and Technology, University of Notre Dame) make any representations about 
// the suitability of this software for any purpose. This software is 
// provided "as is" without expressed or implied warranty.
//
// SparseLib++ was funded in part by the U.S. Department of Energy, the
// National Science Foundation, the University of Notre Dame, and the State 
// of Tennessee.
 ---------------------------------------------------------------------
 05.31.96 -- Bug fix for correctly parsing real fortran formats which
             do not include a character separator between the fractional
             part and the exponent of the the real number
             (e.g. 1.4444444-004).  
             Also, minor change to writeHB functions so that real formats
             are not printed in header for pattern matrices.  KAR
 ---------------------------------------------------------------------
*/
/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
/*                 I/O for Harwell-Boeing files                             */
/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
/*                                                                          */
/*  Several user I/O functions for Harwell-Boeing files are provided,       */
/*  together with various utility functions.                                */
/*  For a description of the Harwell-Boeing standard, see:                  */
/*                                                                          */
/*            Duff, et al.,  ACM TOMS Vol.15, No.1, March 1989              */
/*                                                                          */
/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
/*  Function:                                                               */
/*  void readHB_info(const char *filename, int *M, int *N, int *nz,         */
/*                                   char **Type, int *nrhs)                */
/*  Description:                                                            */
/*                                                                          */
/*  The readHB_info function opens and reads the header information from    */
/*  the specified Harwell-Boeing file, and reports back the number of rows  */
/*  and columns in the stored matrix (M and N), the number of nonzeros in   */
/*  the matrix (nz), the 3-character matrix type(Type), and the number of   */
/*  right-hand-sides stored along with the matrix (nrhs).                   */
/*                                                                          */
/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
/*  Function:                                                               */
/*                                                                          */
/*  void readHB_newmat_double(const char *filename, int *M, int *N, *int nz,*/
/*        int **colptr, int **rowind,  double**val)                         */
/*  void readHB_newmat_float(const char *filename, int *M, int *N, *int nz, */
/*        int **colptr, int **rowind, float**val)                           */
/*                                                                          */
/*  Description:                                                            */
/*                                                                          */
/*  This function opens and reads the specified file, interpreting its      */
/*  contents as a sparse matrix stored in the Harwell/Boeing standard       */
/*  format.  The function allocates vectors to hold the index and nonzero   */
/*  value information, and returns pointers to these vectors along with     */
/*  matrix dimension and number of nonzeros.                                */
/*                                                                          */
/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
/*  Function:                                                               */
/*                                                                          */
/*  void readHB_mat_double(const char *filename, int *colptr, int *rowind,  */
/*             double*val)                                                  */
/*  void readHB_mat_float(const char *filename, int *colptr, int *rowind,   */
/*             float*val)                                                   */
/*                                                                          */
/*  Description:                                                            */
/*                                                                          */
/*  This function opens and reads the specified file, interpreting its      */
/*  contents as a sparse matrix stored in the Harwell/Boeing standard       */
/*  format and using _pre-allocated_ vectors to hold the index and nonzero  */
/*  value information.                                                      */
/*                                                                          */
/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
/*  Function:                                                               */
/*                                                                          */
/*  void readHB_newrhs_double(const char *filename, double **b, int j)      */
/*  void readHB_newrhs_float(const char *filename, float **b, int j)        */
/*                                                                          */
/*  Description:                                                            */
/*                                                                          */
/*  This function opens and reads the specified file, returning a right-    */
/*  hand-side vector b.  If the file specifies multiple right-hand-sides    */
/*  (that is, a right-hand-side matrix B), then the optional argument j     */
/*  may be used to select the (j+1)st column of B.                          */
/*                                                                          */
/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
/*  Function:                                                               */
/*                                                                          */
/*  void readHB_rhs_double(const char *filename, double *b, int j)          */
/*  void readHB_rhs_float(const char *filename, float *b, int j)            */
/*                                                                          */
/*  Description:                                                            */
/*                                                                          */
/*  This function opens and reads the specified file, returning a right-    */
/*  hand-side vector b in _pre-allocated storage.  If the file specifies    */
/*  multiple right-hand-sides (that is, a right-hand-side matrix B), then   */
/*  the optional argument j may be used to select the (j+1)st column of B.  */
/*                                                                          */
/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
/*  Function:                                                               */
/*                                                                          */
/*  void writeHB_mat_double(const char *filename, int M, int N,             */
/*         int nz, const int *colptr, const int *rowind,                    */
/*         const double *val, int nrhs, const double *rhs,                  */
/*         const char *Type, const char *Title, const char *Key)            */
/*  void writeHB_mat_float(const char *filename, int M, int N,              */
/*         int nz, const int *colptr, const int *rowind,                    */
/*         const float *val, int nrhs, const float *rhs,                    */
/*         const char *Type, const char *Title, const char *Key)            */
/*                                                                          */
/*  Description:                                                            */
/*                                                                          */
/*  The writeHB function opens the named file and writes the specified      */
/*  matrix and optional right-hand-side(s) to that file in Harwell-Boeing   */
/*  format.                                                                 */
/*                                                                          */
/****************************************************************************/
/*--------------------------------------------------------------------*/
void 
readHB_info ( 
   const char   *filename, 
   int          *M, 
   int          *N, 
   int          *nz, 
   char         **Type, 
   int          *nrhs
) ;

void 
readHB_mat_double ( 
   const char   *filename, 
   int          colptr[], 
   int          rowind[], 
   double       val[]
) ;
void 
readHB_newmat_double (
   const char  *filename, 
   int         *M, 
   int         *N, 
   int         *nonzeros, 
   int         **colptr, 
   int         **rowind, 
   double      **val
) ;
void 
readHB_mat_float (
   const char    *filename, 
   int           *colptr, 
   int           *rowind, 
   float         *val
) ;
void 
readHB_newmat_float (
   const char   *filename, 
   int          *M, 
   int          *N, 
   int          *nonzeros, 
   int          **colptr, 
   int          **rowind, 
   float        **val
) ;
void 
readHB_rhs_double (
   const char   *filename, 
   double       b[], 
   int          j 
) ;
void 
readHB_newrhs_double ( 
   const char   *filename, 
   double       **b, 
   int          j
) ;

void 
readHB_rhs_float (
   const char   *filename, 
   float        b[], 
   int          j
) ;
void 
readHB_newrhs_float (
   const char   *filename, 
   float        **b, 
   int          j
) ;
void 
writeHB_mat_double (
   const char     *filename, 
   int            M, 
   int            N, 
   int            nz, 
   const int      colptr[], 
   const int      rowind[], 
   const double   val[], 
   int            nrhs, 
   const double   rhs[], 
   const char     *Type, 
   const char     *Title, 
   const char     *Key
) ;
void 
writeHB_mat_float (
   const char    *filename, 
   int           M, 
   int           N, 
   int           nz, 
   const int     colptr[], 
   const int     rowind[], 
   const float   val[], 
   int           nrhs, 
   const float   rhs[], 
   const char    *Type, 
   const char    *Title, 
   const char    *Key
) ;
void 
writetxt_mat_double (
   const char     *filename, 
   int            M, 
   int            N, 
   int            nz, 
   const int      colptr[], 
   const int      rowind[], 
   const double   val[], 
   const char     *Type, 
   const char     *Title, 
   const char     *Key
) ;
void 
writetxt_mat_float (
   const char     *filename, 
   int            M, 
   int            N, 
   int            nz, 
   const int      colptr[], 
   const int      rowind[], 
   const float    val[], 
   const char     *Type, 
   const char     *Title, 
   const char     *Key
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods in bestNDMMD.c -------------------------------------------
------------------------------------------------------------------------
*/
/*
   ---------------------------------------------------------------------
   purpose -- to find a multi-stage minimum degree ordering of a graph.
 
   level  -- starting level. an initial call will have level = 0.
             recursive calls have level > 0.
   graph  -- graph to be ordered
   stages -- stages vector that specifies what stage a vertex will
             be eliminated.
   seed   -- random number seed
   compressFlag -- type of compression to use
      compressFlag % 4 = 0 --> no compression at all
      compressFlag % 4 = 1 --> after each elimination step,
         consider only those nodes that are 2-adjacent
      compressFlag % 4 = 1 --> after each elimination step,
         consider all nodes 
      compressFlag / 4 >= 1 --> compress before any stage
   prioType -- priority type
      0 -- zero priority, independent elimination
      1 -- exact external degree for each vertex
      2 -- approximate external degree for each vertex
      3 -- exact external degree for 2-adjacent vertices,
           approximate external degree for others
   stepType -- elimination step type
      stepType = 0 -- eliminate only one vertex before 
         updating reach set
      stepType = 1 -- eliminate an independent set of vertices 
         with minimum priority before updating reach set
      stepType > 1 -- eliminate an independent set of vertices 
         whose priority lies within stepType*minimum priority
         before updating reach set
   pnfind  -- pointer to address to hold # of front indices
   pnzf    -- pointer to address to hold # of factor entries
   pops    -- pointer to address to hold # of factor operations
   istats  -- int statistics array
      istats[6*istage+0] -- # of vertices in domains
      istats[6*istage+1] -- weight of vertices in domains
      istats[6*istage+2] -- # factor entries in domains
      istats[6*istage+3] -- # of vertices in schur complement
      istats[6*istage+4] -- weight of vertices in schur complement
      istats[6*istage+5] -- # factor entries in schur complement
   dstats  -- double statistics array
      dstats[6*istage+0] -- # of factor operations in domains
      dstats[6*istage+1] -- # of factor operations in schur complement
   msglvl  -- message level
   msgFile -- message file
 
   return value --
      a pointer to an ETree object that holds the ordering
 
   created -- 97apr12, cca
   ---------------------------------------------------------------------
*/
ETree *
bestNDMMD (
   int      level,
   Graph    *graph,
   int      stages[],
   int      seed,
   int      compressFlag,
   int      prioType,
   double   stepType,
   int      *pnfind,
   int      *pnzf,
   double   *pops,
   int      istats[],
   double   dstats[],
   int      msglvl,
   FILE     *msgFile
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods in orderViaND.c ------------------------------------------
------------------------------------------------------------------------
*/
/*
   ------------------------------------------------------------------
   purpose -- return an ETree object for a nested dissection ordering
 
   graph -- graph to order
   maxdomainsize -- used to control the incomplete nested dissection 
     process. any subgraph whose weight is less than maxdomainsize
    is not split further.
   seed    -- random number seed
   msglvl  -- message level, 0 --> no output, 1 --> timings
   msgFile -- message file
 
   created -- 97nov06, cca
   ------------------------------------------------------------------
*/
ETree *
orderViaND (
   Graph   *graph,
   int     maxdomainsize,
   int     seed,
   int     msglvl,
   FILE    *msgFile
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods in orderViaMS.c ------------------------------------------
------------------------------------------------------------------------
*/
/*
   ------------------------------------------------------------------
   purpose -- return an ETree object for a multisection ordering
 
   graph -- graph to order
   maxdomainsize -- used to control the incomplete nested dissection 
     process. any subgraph whose weight is less than maxdomainsize
    is not split further.
   seed    -- random number seed
   msglvl  -- message level, 0 --> no output, 1 --> timings
   msgFile -- message file
 
   created -- 97nov06, cca
   ------------------------------------------------------------------
*/
ETree *
orderViaMS (
   Graph   *graph,
   int     maxdomainsize,
   int     seed,
   int     msglvl,
   FILE    *msgFile
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods in orderViaMMD.c -----------------------------------------
------------------------------------------------------------------------
*/
/*
   --------------------------------------------------------
   purpose -- return an ETree object for 
              a multiple minimum degree ordering
 
   graph -- graph to order
   seed    -- random number seed
   msglvl  -- message level, 0 --> no output, 1 --> timings
   msgFile -- message file
 
   created -- 97nov08, cca
   --------------------------------------------------------
*/
ETree *
orderViaMMD (
   Graph   *graph,
   int     seed,
   int     msglvl,
   FILE    *msgFile
) ;
/*--------------------------------------------------------------------*/
