/*  drawGraph.c  */

#include "../../Graph.h"
#include "../../Coords.h"
#include "../../timings.h"

/*--------------------------------------------------------------------*/

void
main ( int argc, char *argv[] )
/*
   -------------------------------------------------
   draw a graph.

   (1) read a Graph object
   (2) read a Coords object
   (3) read an IV object that contains a tag vector.
       if (v,w) is an edge in the graph
       and tags[v] == tags[w] then
          draw edge (v,w)
   (4) bbox[4] is the bounding box for the plot.
       bbox = { xsw, ysw, xne, yne }
       try bbox = { 0, 0, 500, 500 }
       because coordinates are measured in points,
       72 points per inch.
   (5) rect[4] contains the frame for the plot.
       to put a 20 point margin around the plot,
       rect[0] = bbox[0] + 20
       rect[1] = bbox[1] + 20
       rect[2] = bbox[2] - 20
       rect[3] = bbox[3] - 20

   created -- 96jun06, cca
   -------------------------------------------------
*/
{
char     *fnCoords, *fnEPS, *fnGraph, *fnTagsIV ;
Coords   *coords ;
double   a, b, d, height, linewidth1, linewidth2, 
         offset, radius, t1, t2, width, 
         xmax, xmin, xsize, xv, xw, x0, x1, 
         ymax, ymin, ysize, yv, yw, y0, y1 ;
float    bbox[4], rect[4] ;
int      ii, msglvl, nedge, nvtx, rc, v, vsize, w ;
int      *tags, *vadj ;
IV       *tagsIV ;
Graph    *g ;
FILE     *epsFile, *msgFile ;

if ( argc != 18 ) {
   fprintf(stdout, 
"\n\n usage : drawGraph msglvl msgFile GraphFile CoordsFile"
"\n         tagsIVfile epsFile bbox[4] rect[4] radius"
"\n    msglvl     -- message level"
"\n    msgFile    -- message file"
"\n    GraphFile  -- input graph file, must be *.graphf or *.graphb"
"\n    CoordsFile -- input Coords file, must be *.coordsf or *.coordsb"
"\n    tagsIVfile -- input IV file, must be *.ivf or *.ivb"
"\n                  contains the component ids"
"\n    epsFile    -- output postscript file, must be *.eps"
"\n    linewidth1 -- line width for edges connecting vertices"
"\n                  in the same component"
"\n    linewidth2 -- line width for edges connecting vertices"
"\n                  in the same component"
"\n    bbox[4]    -- bounding box for drawing"
"\n    rect[4]    -- rectangle to contain graph"
"\n    radius     -- radius for vertex circle"
      "\n") ;
   return ;
}
msglvl = atoi(argv[1]) ;
if ( strcmp(argv[2], "stdout") == 0 ) {
   msgFile = stdout ;
} else if ( (msgFile = fopen(argv[2], "a")) == NULL ) {
   fprintf(stderr, "\n fatal error in %s"
           "\n unable to open file %s\n",
           argv[0], argv[2]) ;
   return ;
}
fnGraph    = argv[3] ;
fnCoords   = argv[4] ;
fnTagsIV   = argv[5] ;
fnEPS      = argv[6] ;
linewidth1 = atof(argv[7]) ;
linewidth2 = atof(argv[8]) ;
bbox[0]    = atof(argv[9]) ;
bbox[1]    = atof(argv[10]) ;
bbox[2]    = atof(argv[11]) ;
bbox[3]    = atof(argv[12]) ;
rect[0]    = atof(argv[13]) ;
rect[1]    = atof(argv[14]) ;
rect[2]    = atof(argv[15]) ;
rect[3]    = atof(argv[16]) ;
radius     = atof(argv[17]) ;
fprintf(msgFile, 
        "\n drawGraph "
        "\n msglvl     -- %d" 
        "\n msgFile    -- %s" 
        "\n GraphFile  -- %s" 
        "\n CoordsFile -- %s" 
        "\n tagsIVfile -- %s" 
        "\n epsFile    -- %s" 
        "\n linewidth1 -- %f"
        "\n linewidth2 -- %f"
        "\n bbox[4] = { %f, %f, %f, %f }"
        "\n rect[4] = { %f, %f, %f, %f }"
        "\n radius     -- %f" 
        "\n",
        msglvl, argv[2], fnGraph, fnCoords, fnTagsIV, fnEPS,
        linewidth1, linewidth2, bbox[0], bbox[1], bbox[2], bbox[3],
        rect[0], rect[1], rect[2], rect[3], radius) ;
fflush(msgFile) ;
/*
   ------------------------
   read in the Graph object
   ------------------------
*/
g = Graph_new() ;
if ( strcmp(fnGraph, "none") == 0 ) {
   fprintf(msgFile, "\n no file to read from") ;
   exit(0) ;
}
MARKTIME(t1) ;
rc = Graph_readFromFile(g, fnGraph) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %9.5f : read in graph from file %s",
        t2 - t1, fnGraph) ;
nvtx = g->nvtx ;
if ( rc != 1 ) {
   fprintf(msgFile, "\n return value %d from Graph_readFromFile(%p,%s)",
           rc, g, fnGraph) ;
   exit(-1) ;
}
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n\n after reading Graph object from file %s",
           argv[3]) ;
   Graph_writeForHumanEye(g, msgFile) ;
   fflush(msgFile) ;
}
/*
   -------------------------
   read in the Coords object
   -------------------------
*/
if ( strcmp(fnCoords, "none") == 0 ) {
   fprintf(msgFile, "\n no file to read from") ;
   exit(0) ;
}
MARKTIME(t1) ;
coords = Coords_new() ;
rc = Coords_readFromFile(coords, fnCoords) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %9.5f : read in graph from file %s",
        t2 - t1, fnCoords) ;
if ( rc != 1 ) {
  fprintf(msgFile, "\n return value %d from Coords_readFromFile(%p,%s)",
           rc, coords, fnCoords) ;
   exit(-1) ;
}
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n\n after reading Coords object from file %s",
           argv[3]) ;
   Coords_writeForHumanEye(coords, msgFile) ;
   fflush(msgFile) ;
}
/*
   ---------------------
   read in the IV object
   ---------------------
*/
if ( strcmp(fnTagsIV, "none") == 0 ) {
   tags = NULL ;
} else {
   MARKTIME(t1) ;
   tagsIV = IV_new() ;
   rc = IV_readFromFile(tagsIV, fnTagsIV) ;
   MARKTIME(t2) ;
   fprintf(msgFile, "\n CPU %9.5f : read in graph from file %s",
           t2 - t1, fnTagsIV) ;
   if ( rc != 1 ) {
     fprintf(msgFile, "\n return value %d from IV_readFromFile(%p,%s)",
              rc, tagsIV, fnTagsIV) ;
      exit(-1) ;
   }
   if ( msglvl > 2 ) {
      fprintf(msgFile, "\n\n after reading IV object from file %s",
              argv[3]) ;
      IV_writeForHumanEye(tagsIV, msgFile) ;
      fflush(msgFile) ;
   }
   tags = IV_entries(tagsIV) ;
}
/*
   -----------------------------------
   write the preamble for the EPS file
   -----------------------------------
*/
if ( strcmp(fnEPS, "stdout") == 0 ) {
   epsFile = stdout ;
} else if ( (epsFile = fopen(fnEPS, "w")) == NULL ) {
   fprintf(stderr, "\n fatal error in %s"
           "\n unable to open file %s\n", argv[0], fnEPS) ;
   return ;
}
fprintf(epsFile,
   "%%!PS-Adobe-2.0 EPSF-1.2"
   "\n%%%%BoundingBox: %.1f %.1f %.1f %.1f",
   bbox[0], bbox[1], bbox[2], bbox[3]) ;

fprintf(epsFile,
   "\n /radius %.3f def"
   "\n /Helvetica findfont %.3f scalefont setfont"
   "\n /M {moveto} def"
   "\n /L {lineto} def"
   "\n /ACF { %% stack : x y radius"
   "\n    newpath 0 360 arc closepath fill "
   "\n } def"
   "\n /str 6 string def"
   "\n /drawLabel { %% x y label radius"
   "\n    /radius exch def"
   "\n    /label  exch def"
   "\n    /y      exch def"
   "\n    /x      exch def"
   "\n    gsave"
   "\n       1.0 setgray"
   "\n       x radius add y moveto"
   "\n       x y radius 0 360 arc"
   "\n       fill"
   "\n       0.0 setgray"
   "\n       x radius add y moveto"
   "\n       x y radius 0 360 arc"
   "\n       stroke"
   "\n       x y moveto"
   "\n       label stringwidth pop 2 div neg radius 2 div neg rmoveto"
   "\n       label show"
   "\n    grestore"
   "\n } def ", radius, 1.25*radius) ;
/*
   ---------------------------------------
   determine the transformation parameters
   ---------------------------------------
*/
xmin   = Coords_min(coords, 1) ;
xmax   = Coords_max(coords, 1) ;
ymin   = Coords_min(coords, 2) ;
ymax   = Coords_max(coords, 2) ;
fprintf(msgFile, 
        "\n xmin = %.3g, xmax = %.3g, ymin = %.3g, ymax = %.3g", 
        xmin, xmax, ymin, ymax) ;
xsize  = xmax - xmin ;
ysize  = ymax - ymin ;
width  = rect[2] - rect[0] ;
height = rect[3] - rect[1] ;
fprintf(msgFile, 
        "\n xsize = %.3g, ysize = %.3g, width = %.3g, height = %.3g", 
        xsize, ysize, width, height) ;
if ( ysize * width <= xsize * height ) {
   a = width / xsize ;
   b = rect[0] ;
   offset = (rect[3] - rect[1] - a * ysize)/2 ;
   d = rect[1] + offset - a * ymin ;
} else {
   a = height / ysize ;
   d = rect[1] ;
   offset = (rect[2] - rect[0] - a * xsize)/2 ;
   b = rect[0] + offset - a * xmin ;
}
if ( ysize * width <= xsize * height ) {
   a = width / xsize ;
} else {
   a = height / ysize ;
}
b = 0.5*(rect[2] + rect[0] - a*(xmin + xmax)) ;
d = 0.5*(rect[3] + rect[1] - a*(ymin + ymax)) ;
fprintf(msgFile, "\n width = %.3g, height = %.3g", width, height) ;
fprintf(msgFile, "\n xsize = %.3g, ysize = %.3g", xsize, ysize) ;
fprintf(msgFile, 
        "\n xmin = %.3g, xmax = %.3g, ymin = %.3g, ymax = %.3g",
        xmin, xmax, ymin, ymax) ;
fprintf(msgFile, "\n a = %.3g, b = %.3g, d = %.3g", a, b, d) ;
if ( tags == NULL ) {
/*
   --------------------------------
   no component ids, draw the edges
   --------------------------------
*/
   fprintf(epsFile,
           "\n gsave"
           "\n   %.3f setlinewidth"
           "\n   0.0 setgray", linewidth1) ;
   nedge = 0 ;
   for ( v = 0 ; v < nvtx ; v++ ) {
      Graph_adjAndSize(g, v, &vsize, &vadj) ;
      xv = Coords_value(coords, 1, v) ;
      yv = Coords_value(coords, 2, v) ;
      x0 = a * xv + b ;
      y0 = a * yv + d ;
      for ( ii = 0 ; ii < vsize ; ii++ ) {
         w = vadj[ii] ;
         if ( w < v ) {
            xw = Coords_value(coords, 1, w) ;
            yw = Coords_value(coords, 2, w) ;
            x1 = a * xw + b ;
            y1 = a * yw + d ;
            if ( nedge % 100 == 0 ) {
               fprintf(epsFile, "\n    newpath") ;
            }
            fprintf(epsFile, "\n       %.3g %.3g M %.3g %.3g L", 
                    x0, y0, x1, y1) ;
            if ( ++nedge % 100 == 0 ) {
               fprintf(epsFile, "\n    stroke") ;
            }
         }
      }
   }
   if ( nedge % 100 != 0 ) {
      fprintf(epsFile, "\n    stroke") ;
   }
   fprintf(epsFile,
           "\n grestore") ;
   fprintf(epsFile,
           "\n gsave"
           "\n   0.1 setlinewidth"
           "\n   0.0 setgray") ;
   if ( radius > 0.0 ) {
/*
      -----------------
      draw the vertices
      -----------------
*/
      for ( v = 0 ; v < nvtx ; v++ ) {
         xv = Coords_value(coords, 1, v) ;
         yv = Coords_value(coords, 2, v) ;
         x0 = a * xv + b ;
         y0 = a * yv + d ;
         fprintf(epsFile, "\n %.3f %.3f () radius drawLabel", 
                 x0, y0) ;
      }
   }
   fprintf(epsFile, "\n grestore") ;
} else {
/*
   -----------------------------------------
   component ids are present, draw the edges 
   between vertices in the same component
   -----------------------------------------
*/
   fprintf(epsFile,
           "\n gsave"
           "\n   %.3f setlinewidth"
           "\n   0.0 setgray", linewidth1) ;
   nedge = 0 ;
   for ( v = 0 ; v < nvtx ; v++ ) {
      Graph_adjAndSize(g, v, &vsize, &vadj) ;
      xv = Coords_value(coords, 1, v) ;
      yv = Coords_value(coords, 2, v) ;
      x0 = a * xv + b ;
      y0 = a * yv + d ;
      for ( ii = 0 ; ii < vsize ; ii++ ) {
         w = vadj[ii] ;
         if ( w < v && tags[w] == tags[v] ) {
            xw = Coords_value(coords, 1, w) ;
            yw = Coords_value(coords, 2, w) ;
            x1 = a * xw + b ;
            y1 = a * yw + d ;
            if ( nedge % 100 == 0 ) {
               fprintf(epsFile, "\n    newpath") ;
            }
            fprintf(epsFile, "\n       %.3g %.3g M %.3g %.3g L", 
                    x0, y0, x1, y1) ;
            if ( ++nedge % 100 == 0 ) {
               fprintf(epsFile, "\n    stroke") ;
            }
         }
      }
   }
   if ( nedge % 100 != 0 ) {
      fprintf(epsFile, "\n    stroke") ;
   }
   fprintf(epsFile,
        "\n grestore") ;
   fprintf(epsFile,
           "\n gsave"
           "\n   %.3f setlinewidth"
           "\n   0.0 setgray", linewidth2) ;
   nedge = 0 ;
   for ( v = 0 ; v < nvtx ; v++ ) {
      Graph_adjAndSize(g, v, &vsize, &vadj) ;
      xv = Coords_value(coords, 1, v) ;
      yv = Coords_value(coords, 2, v) ;
      x0 = a * xv + b ;
      y0 = a * yv + d ;
      for ( ii = 0 ; ii < vsize ; ii++ ) {
         w = vadj[ii] ;
         if ( w < v && tags[w] != tags[v] ) {
            xw = Coords_value(coords, 1, w) ;
            yw = Coords_value(coords, 2, w) ;
            x1 = a * xw + b ;
            y1 = a * yw + d ;
            if ( nedge % 100 == 0 ) {
               fprintf(epsFile, "\n    newpath") ;
            }
            fprintf(epsFile, "\n       %.3g %.3g M %.3g %.3g L", 
                    x0, y0, x1, y1) ;
            if ( ++nedge % 100 == 0 ) {
               fprintf(epsFile, "\n    stroke") ;
            }
         }
      }
   }
   if ( nedge % 100 != 0 ) {
      fprintf(epsFile, "\n    stroke") ;
   }
   fprintf(epsFile,
        "\n grestore") ;
   fprintf(epsFile,
           "\n gsave"
           "\n   0.1 setlinewidth"
           "\n   0.0 setgray") ;
   if ( radius > 0.0 ) {
/*
      -----------------
      draw the vertices
      -----------------
*/
      for ( v = 0 ; v < nvtx ; v++ ) {
         xv = Coords_value(coords, 1, v) ;
         yv = Coords_value(coords, 2, v) ;
         x0 = a * xv + b ;
         y0 = a * yv + d ;
         fprintf(epsFile, "\n %.3f %.3f (%d) radius drawLabel", 
                 x0, y0, tags[v]) ;
      }
   }
   fprintf(epsFile, "\n grestore") ;
}
fprintf(epsFile, "\n showpage") ;
fclose(epsFile) ;

fprintf(msgFile, "\n") ;
fclose(msgFile) ;

return ; }

/*--------------------------------------------------------------------*/
