/*  allInOne.c  */

#include "../../misc.h"
#include "../../DFrontMtx.h"
#include "../../SymbFac.h"

/*--------------------------------------------------------------------*/
void
main ( int argc, char *argv[] ) {
/*
   --------------------------------------------------
   all-in-one program
   (1) read in matrix entries and form DInpMtx object
   (2) form Graph object
   (3) order matrix and form front tree
   (4) get the permutation, permute the matrix and 
       front tree and get the symbolic factorization
   (5) compute the numeric factorization
   (6) read in right hand side entries
   (7) compute the solution

   created -- 97oct22, cca
   --------------------------------------------------
*/
/*--------------------------------------------------------------------*/
char               *matrixFileName, *rhsFileName ;
DA2                *rhsDA2, *solDA2 ;
DChv               *rootchv ;
DDenseMtxManager   *manager  ;
DFrontMtx          *frontmtx ;
DInpMtx            *inpmtx ;
double             value ;
double             *rhs ;
double             cpus[10] ;
ETree              *frontETree ;
FILE               *inputFile, *msgFile ;
Graph              *graph ;
int                ient, irow, jcol, msglvl, ncol, 
                   nedges, nent, nrow, seed ;
int                *newToOld, *oldToNew ;
int                stats[20] ;
IV                 *newToOldIV, *oldToNewIV ;
IVL                *adjIVL, *symbfacIVL ;
/*--------------------------------------------------------------------*/
/*
   --------------------
   get input parameters
   --------------------
*/
if ( argc != 6 ) {
   fprintf(stdout, 
      "\n usage: %s msglvl msgFile matrixFileName rhsFileName seed"
      "\n    msglvl -- message level"
      "\n    msgFile -- message file"
      "\n    matrixFileName -- matrix file name, format"
      "\n       nrow ncol nent"
      "\n       irow jcol entry"
      "\n        ..."
      "\n        note: indices are zero based"
      "\n    rhsFileName -- right hand side file name, format"
      "\n       nrow "
      "\n       entry[0]"
      "\n       ..."
      "\n       entry[nrow-1]"
      "\n    seed -- random number seed, used for ordering"
      "\n", argv[0]) ;
   return ;
}
msglvl = atoi(argv[1]) ;
if ( strcmp(argv[2], "stdout") == 0 ) {
   msgFile = stdout ;
} else if ( (msgFile = fopen(argv[2], "a")) == NULL ) {
   fprintf(stderr, "\n fatal error in %s"
           "\n unable to open file %s\n",
           argv[0], argv[2]) ;
   return ;
}
matrixFileName = argv[3] ;
rhsFileName    = argv[4] ;
seed           = atoi(argv[5]) ;
/*--------------------------------------------------------------------*/
/*
   ------------------------------------------------------------------
   read the entries from the input file and create the DInpMtx object
   ------------------------------------------------------------------
*/
inputFile = fopen(matrixFileName, "r") ;
fscanf(inputFile, "%d %d %d", &nrow, &ncol, &nent) ;
inpmtx = DInpMtx_new() ;
DInpMtx_init(inpmtx, 1, 2, nent, 0) ;
for ( ient = 0 ; ient < nent ; ient++ ) {
   fscanf(inputFile, "%d %d %le", &irow, &jcol, &value) ;
   DInpMtx_inputEntry(inpmtx, irow, jcol, value) ;
}
fclose(inputFile) ;
DInpMtx_changeStorageMode(inpmtx, 3) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n input matrix") ;
   DInpMtx_writeForHumanEye(inpmtx, msgFile) ;
   fflush(msgFile) ;
}
/*--------------------------------------------------------------------*/
/*
   -----------------------
   create the Graph object
   -----------------------
*/
graph = Graph_new() ;
adjIVL = DInpMtx_fullAdjacency(inpmtx) ;
nedges = IVL_tsize(adjIVL) ;
Graph_init2(graph, 0, nrow, 0, nedges, nrow, nedges, adjIVL,
            NULL, NULL) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n graph of the input matrix") ;
   Graph_writeForHumanEye(graph, msgFile) ;
   fflush(msgFile) ;
}
/*--------------------------------------------------------------------*/
/*
   ---------------
   order the graph
   ---------------
*/
frontETree = orderViaMMD(graph, seed, msglvl, msgFile) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n front tree from ordering") ;
   ETree_writeForHumanEye(frontETree, msgFile) ;
   fflush(msgFile) ;
}
/*--------------------------------------------------------------------*/
/*
   ---------------------------------------------
   get the permutation, permute the matrix and 
   front tree and get the symbolic factorization
   ---------------------------------------------
*/
oldToNewIV = ETree_oldToNewVtxPerm(frontETree) ;
oldToNew = IV_entries(oldToNewIV) ;
newToOldIV = ETree_newToOldVtxPerm(frontETree) ;
newToOld   = IV_entries(newToOldIV) ;
ETree_permuteVertices(frontETree, oldToNewIV) ;
DInpMtx_permute(inpmtx, oldToNew, oldToNew) ;
DInpMtx_mapToUpperTriangle(inpmtx) ;
symbfacIVL = SymbFac_initFromDInpMtx(frontETree, inpmtx) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n old-to-new permutation vector") ;
   IV_writeForHumanEye(oldToNewIV, msgFile) ;
   fprintf(msgFile, "\n\n new-to-old permutation vector") ;
   IV_writeForHumanEye(newToOldIV, msgFile) ;
   fprintf(msgFile, "\n\n front tree after permutation") ;
   ETree_writeForHumanEye(frontETree, msgFile) ;
   fprintf(msgFile, "\n\n input matrix after permutation") ;
   DInpMtx_writeForHumanEye(inpmtx, msgFile) ;
   fprintf(msgFile, "\n\n symbolic factorization") ;
   IVL_writeForHumanEye(symbfacIVL, msgFile) ;
   fflush(msgFile) ;
}
/*--------------------------------------------------------------------*/
/*
   ---------------------------------
   compute the numeric factorization
   ---------------------------------
*/
frontmtx = DFrontMtx_new() ;
DFrontMtx_init(frontmtx, frontETree, symbfacIVL, 0, 0, 0, 0, 0, NULL) ;
rootchv = DFrontMtx_factorDInpMtx(frontmtx, inpmtx, 100., 0.0, 
                                  cpus, stats, msglvl, msgFile) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n factor matrix") ;
   DFrontMtx_writeForHumanEye(frontmtx, msgFile) ;
   fflush(msgFile) ;
}
if ( rootchv != NULL ) {
   fprintf(msgFile, "\n\n matrix found to be singular\n") ;
   exit(-1) ;
}
/*--------------------------------------------------------------------*/
/*
   ----------------------------------
   read the right hand side entries
   ----------------------------------
*/
inputFile = fopen(rhsFileName, "r") ;
fscanf(inputFile, "%d", &nrow) ;
rhsDA2 = DA2_new() ;
DA2_init(rhsDA2, nrow, 1, 1, nrow, NULL) ;
rhs = DA2_entries(rhsDA2) ;
for ( irow = 0 ; irow < nrow ; irow++ ) {
   fscanf(inputFile, "%le", &value) ;
   rhs[irow] = value ;
}
fclose(inputFile) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n rhs matrix in original ordering") ;
   DA2_writeForHumanEye(rhsDA2, msgFile) ;
   fflush(msgFile) ;
}
/*
   ------------------------------------------------------
   permute the right hand side into the original ordering
   ------------------------------------------------------
*/
DA2_permuteRows(rhsDA2, nrow, oldToNew) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n right hand side matrix in new ordering") ;
   DA2_writeForHumanEye(rhsDA2, msgFile) ;
   fflush(msgFile) ;
}
/*--------------------------------------------------------------------*/
/*
   -----------------------
   solve the linear system
   -----------------------
*/
solDA2 = DA2_new() ;
DA2_init(solDA2, nrow, 1, 1, nrow, NULL) ;
DFrontMtx_postProcess(frontmtx, 0, 1, msglvl, msgFile) ;
manager = DDenseMtxManager_new() ;
DFrontMtx_solveDA2(frontmtx, solDA2, rhsDA2, manager, cpus) ;
DFrontMtx_postProcess(frontmtx, 0, 2, msglvl, msgFile) ;
DDenseMtxManager_free(manager) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n solution matrix in new ordering") ;
   DA2_writeForHumanEye(solDA2, msgFile) ;
   fflush(msgFile) ;
}
/*
   -----------------------------------------------
   permute the solution into the original ordering
   -----------------------------------------------
*/
DA2_permuteRows(solDA2, nrow, newToOld) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n solution matrix in original ordering") ;
   DA2_writeForHumanEye(solDA2, msgFile) ;
   fflush(msgFile) ;
}
/*--------------------------------------------------------------------*/
return ; }
/*--------------------------------------------------------------------*/
