/*  setBoxes.c  */

#include "../Tree.h"

#define MYDEBUG 1

/*--------------------------------------------------------------------*/
/*
   -------------------------------------------------------------
   purpose -- fill boxes arrays for display of a tree

   vmetric[] -- vector of metric on the nodes
   tmetric[] -- vector of metric on the subtrees
   xmin, xmax, ymin, ymax -- bounds on box for root
   west[], east[], south[], north[] -- vector to hold bounds for
                                       the nodes in the tree

   return value --
     1 --> success
     2 --> no success, maxnchild > 3

   created -- 96dec20, cca
   -------------------------------------------------------------
*/
int
Tree_setBoxesII (
   Tree     *tree,
   int      vmetric[],
   int      tmetric[],
   double   xmin,
   double   xmax,
   double   ymin,
   double   ymax,
   double   west[],
   double   east[],
   double   south[],
   double   north[]
) {
double   areaJ, heightJ, widthJ ;
int      J ;
int      *fch, *sib ;
/*
   ---------------
   check the input
   ---------------
*/
if (  tree == NULL || vmetric == NULL || tmetric == NULL 
   || west == NULL || east == NULL || south == NULL || north == NULL ) {
   fprintf(stderr, 
   "\n fatal error in Tree_setBoxesII(%p,%p,%p,%f,%f,%f,%f,%p,%p,%p,%p)"
   "\n bad input\n", 
   tree, vmetric, tmetric, xmin, xmax, ymin, ymax, 
   west, east, south, north) ;
   exit(-1) ;
}
fch = tree->fch ;
sib = tree->sib ;
/*
   -----------------------------
   perform a pre-order traversal
   -----------------------------
*/
for ( J = tree->root ; J != -1 ; J = sib[J] ) {
   west[J]  = xmin ;
   east[J]  = xmax ;
   south[J] = ymin ;
   north[J] = ymax ;
}
for ( J = Tree_preOTfirst(tree) ;
      J != -1 ;
      J = Tree_preOTnext(tree, J) ) {
   widthJ  = east[J]  - west[J]  ;
   heightJ = north[J] - south[J] ;
   areaJ   = widthJ * heightJ ;
#if MYDEBUG > 0
   fprintf(stdout, 
           "\n J = %5d, area = %9.3f, [%6.3f, %6.3f] x [%6.3f, %6.3f]",
           J, areaJ, west[J], east[J], south[J], north[J]) ;
#endif
   switch ( Tree_nchild(tree, J) ) {
   case 0 :
      break ;
   case 1 :
   case 2 : {
      int I ;

      I = fch[J] ;
      if ( widthJ >= heightJ ) {
         west[I]  = west[J] ;
         east[I]  = west[I] + (widthJ*tmetric[I])/tmetric[J] ;
         south[I] = south[J] ;
         north[I] = north[J] ;
         if ( (I = sib[I]) != -1 ) {
            east[I]  = east[J] ;
            west[I]  = east[I] - (widthJ*tmetric[I])/tmetric[J] ;
            south[I] = south[J] ;
            north[I] = north[J] ;
         }
      } else {
         south[I] = south[J] ;
         north[I] = south[I] + (heightJ*tmetric[I])/tmetric[J] ;
         west[I]  = west[J] ;
         east[I]  = east[J] ;
         if ( (I = sib[I]) != -1 ) {
            north[I] = north[J] ;
            south[I] = north[I] - (heightJ*tmetric[I])/tmetric[J] ;
            west[I]  = west[J] ;
            east[I]  = east[J] ;
         }
      }
      break ; }
   case 3 : {
      double   b, c, delta, r1 ;
      int      I1, I2, I3 ;

      I1 = fch[J]  ;
      I2 = sib[I1] ;
      I3 = sib[I2] ;
#if MYDEBUG > 0
   fprintf(stdout, 
        "\n case 3, I1 = %d, I2 = %d, I3 = %d" 
        "\n T[%d] = %d, T[%d] = %d, T[%d] = %d, T[%d] = %d",
        I1, I2, I3, J, tmetric[J], I1, tmetric[I1], I2, tmetric[I2],
        I3, tmetric[I3]) ;
#endif
      if ( widthJ >= heightJ ) {
         west[I1]  = west[J] ;
         east[I1]  = west[I1] + (widthJ * tmetric[I1]) / tmetric[J] ;
         south[I1] = south[J] ;
         north[I1] = north[J] ;
         b = heightJ + widthJ - (east[I1] - west[I1]) ;
         c = (vmetric[J] * widthJ * heightJ) / tmetric[J] ;
         r1 = 0.5*(b + sqrt(b*b - 4*c)) ;
         delta = c / r1 ;
         west[I2] = east[I1] + delta ;
         east[I2] = east[J] ;
         south[I2] = south[J] ;
         north[I2] = south[I2] 
              + ((areaJ*tmetric[I2])/tmetric[J])/(east[I2] - west[I2]) ;
         west[I3] = east[I1] + delta ;
         east[I3] = east[J] ;
         north[I3] = north[J] ;
         south[I3] = north[I3] 
              - ((areaJ*tmetric[I3])/tmetric[J])/(east[I3] - west[I3]) ;
      } else {
         south[I1] = south[J] ;
         north[I1] = south[I1] + (heightJ * tmetric[I1]) / tmetric[J] ;
         west[I1]  = west[J] ;
         east[I1]  = east[J] ;
         b = heightJ + widthJ - (north[I1] - south[I1]) ;
         c = (vmetric[J] * widthJ * heightJ) / tmetric[J] ;
         r1 = 0.5*(b + sqrt(b*b - 4*c)) ;
         delta = c / r1 ;
         south[I2] = north[I1] + delta ;
         north[I2] = north[J] ;
         west[I2]  = west[J] ;
         east[I2]  = west[I2] 
            + ((areaJ*tmetric[I2])/tmetric[J])/(north[I2] - south[I2]) ;
         south[I3] = north[I1] + delta ;
         north[I3] = north[J] ;
         east[I3]  = east[J] ;
         west[I3]  = east[I3] 
            - ((areaJ*tmetric[I3])/tmetric[J])/(north[I3] - south[I3]) ;
      }
      break ; }
   case 4 : {
      double   area1, area2, area3, area4, b, c, delta, 
               height1, height2, height3, height4, r1,
               width1, width2, width3, width4 ;
      int      I1, I2, I3, I4 ;

      I1 = fch[J]  ;
      I2 = sib[I1] ;
      I3 = sib[I2] ;
      I4 = sib[I3] ;
      area1 = (areaJ*tmetric[I1])/tmetric[J] ;
      area2 = (areaJ*tmetric[I2])/tmetric[J] ;
      area3 = (areaJ*tmetric[I3])/tmetric[J] ;
      area4 = (areaJ*tmetric[I4])/tmetric[J] ;
#if MYDEBUG > 0
   fprintf(stdout, 
        "\n case 4, I1 = %d, I2 = %d, I3 = %d, I4 = %d" 
        "\n T[%d] = %d, T[%d] = %d, T[%d] = %d, T[%d] = %d, T[%d] = %d",
        I1, I2, I3, I4, J, tmetric[J], I1, tmetric[I1], I2, tmetric[I2],
        I3, tmetric[I3], I4, tmetric[I4]) ;
#endif
      if ( widthJ >= heightJ ) {
         b = heightJ + widthJ ;
         c = (areaJ*vmetric[J])/tmetric[J] ;
         r1 = 0.5*(b + sqrt(b*b - 4*c)) ;
         delta = c / r1 ;
         height1 = height4 = (area1 + area4)/(widthJ - delta) ;
         height2 = height3 = (area2 + area3)/(widthJ - delta) ;
         width1  = area1 / height1 ;
         width2  = area2 / height2 ;
         width3  = area3 / height3 ;
         width4  = area4 / height4 ;
         west[I1]  = west[J] ;
         east[I1]  = west[I1] + width1 ;
         south[I1] = south[J] ;
         north[I1] = south[I1] + height1 ;
         east[I4]  = east[J] ;
         west[I4]  = east[I4] - width4 ;
         south[I4] = south[J] ;
         north[I4] = south[I4] + height4 ;
         west[I3]  = west[J] ;
         east[I3]  = west[I3] + width3 ;
         north[I3] = north[J] ;
         south[I3] = north[I3] - height3 ;
         east[I2]  = east[J] ;
         west[I2]  = east[I2] - width2 ;
         north[I2] = north[J] ;
         south[I2] = north[I2] - height2 ;
      } else {
         b = heightJ + widthJ ;
         c = (areaJ*vmetric[J])/tmetric[J] ;
         r1 = 0.5*(b + sqrt(b*b - 4*c)) ;
         delta = c / r1 ;
         width1 = width4 = (area1 + area4)/(heightJ - delta) ;
         width2 = width3 = (area2 + area3)/(heightJ - delta) ;
         height1 = area1 / width1 ;
         height2 = area2 / width2 ;
         height3 = area3 / width3 ;
         height4 = area4 / width4 ;
         west[I1] = west[J] ;
         east[I1] = west[I1] + width1 ;
         south[I1] = south[J] ;
         north[I1] = south[I1] + height1 ;
         west[I4] = west[J] ;
         east[I4] = west[I4] + width4 ;
         north[I4] = north[J] ;
         south[I4] = north[I4] - height4 ;
         east[I3] = east[J] ;
         west[I3] = east[I3] - width3 ;
         south[I3] = south[J] ;
         north[I3] = south[I3] + height3 ;
         east[I2] = east[J] ;
         west[I2] = east[I2] - width2 ;
         north[I2] = north[J] ;
         south[I2] = north[I2] - height2 ;
      }
      break ; }
   default :
      return(-1) ;
      break ;
   }
}
return(1) ; }

/*--------------------------------------------------------------------*/
/*
   -------------------------------------------------------------
   purpose -- fill boxes arrays for display of a tree

   vmetric[] -- vector of metric on the nodes
   tmetric[] -- vector of metric on the subtrees
   xmin, xmax, ymin, ymax -- bounds on box for root
   west[], east[], south[], north[] -- vector to hold bounds for
                                       the nodes in the tree

   return value --
     1 --> success
     2 --> no success, maxnchild > 3

   created -- 96dec20, cca
   -------------------------------------------------------------
*/
int
Tree_setBoxesDD (
   Tree     *tree,
   double   vmetric[],
   double   tmetric[],
   double   xmin,
   double   xmax,
   double   ymin,
   double   ymax,
   double   west[],
   double   east[],
   double   south[],
   double   north[]
) {
double   areaJ, heightJ, widthJ ;
int      J ;
int      *fch, *sib ;
/*
   ---------------
   check the input
   ---------------
*/
if (  tree == NULL || vmetric == NULL || tmetric == NULL 
   || west == NULL || east == NULL || south == NULL || north == NULL ) {
   fprintf(stderr, 
   "\n fatal error in Tree_setBoxesDD(%p,%p,%p,%f,%f,%f,%f,%p,%p,%p,%p)"
   "\n bad input\n", 
   tree, vmetric, tmetric, xmin, xmax, ymin, ymax, 
   west, east, south, north) ;
   exit(-1) ;
}
fch = tree->fch ;
sib = tree->sib ;
/*
   -----------------------------
   perform a pre-order traversal
   -----------------------------
*/
for ( J = tree->root ; J != -1 ; J = sib[J] ) {
   west[J]  = xmin ;
   east[J]  = xmax ;
   south[J] = ymin ;
   north[J] = ymax ;
}
for ( J = Tree_preOTfirst(tree) ;
      J != -1 ;
      J = Tree_preOTnext(tree, J) ) {
   widthJ  = east[J]  - west[J]  ;
   heightJ = north[J] - south[J] ;
   areaJ   = widthJ * heightJ ;
#if MYDEBUG > 0
   fprintf(stdout, 
           "\n J = %d, area = %6.3f, [%6.3f,%6.3f] x [%6.3f,%6.3f]",
           J, areaJ, west[J], east[J], south[J], north[J]) ;
#endif
   switch ( Tree_nchild(tree, J) ) {
   case 0 :
      break ;
   case 1 :
   case 2 : {
      int I ;

      I = fch[J] ;
      if ( widthJ >= heightJ ) {
         west[I]  = west[J] ;
         east[I]  = west[I] + widthJ*tmetric[I]/tmetric[J] ;
         south[I] = south[J] ;
         north[I] = north[J] ;
         if ( (I = sib[I]) != -1 ) {
            east[I]  = east[J] ;
            west[I]  = east[I] - widthJ*tmetric[I]/tmetric[J] ;
            south[I] = south[J] ;
            north[I] = north[J] ;
         }
      } else {
         south[I] = south[J] ;
         north[I] = south[I] + heightJ*tmetric[I]/tmetric[J] ;
         west[I]  = west[J] ;
         east[I]  = east[J] ;
         if ( (I = sib[I]) != -1 ) {
            north[I] = north[J] ;
            south[I] = north[I] - heightJ*tmetric[I]/tmetric[J] ;
            west[I]  = west[J] ;
            east[I]  = east[J] ;
         }
      }
      break ; }
   case 3 : {
      double   b, c, delta, r1 ;
      int      I1, I2, I3 ;

      I1 = fch[J]  ;
      I2 = sib[I1] ;
      I3 = sib[I2] ;
#if MYDEBUG > 0
   fprintf(stdout, 
        "\n case 3, I1 = %d, I2 = %d, I3 = %d" 
        "\n T[%d] = %6.3f, T[%d] = %6.3f, T[%d] = %6.3f, T[%d] = %6.3f",
        I1, I2, I3, J, tmetric[J], I1, tmetric[I1], I2, tmetric[I2],
        I3, tmetric[I3]) ;
#endif
      if ( widthJ >= heightJ ) {
         west[I1]  = west[J] ;
         east[I1]  = west[I1] + widthJ * tmetric[I1] / tmetric[J] ;
         south[I1] = south[J] ;
         north[I1] = north[J] ;
         b = heightJ + widthJ - (east[I1] - west[I1]) ;
         c = vmetric[J] * areaJ / tmetric[J] ;
         r1 = 0.5*(b + sqrt(b*b - 4*c)) ;
         delta = c / r1 ;
#if MYDEBUG > 0
         fprintf(stdout, 
             "\n vmetric[%d] = %.3f, areaJ = %.3f, tmetric[%d] = %.3f",
             J, vmetric[J], areaJ, J, tmetric[J]) ;
         fprintf(stdout, "\n b = %.3f, c = %.3f, delta = %6.3f", 
                 b, c, delta) ;
#endif
         west[I2] = east[I1] + delta ;
         east[I2] = east[J] ;
         south[I2] = south[J] ;
         north[I2] = south[I2] 
              + (tmetric[I2]/tmetric[J])*areaJ/(east[I2] - west[I2]) ;
         west[I3] = east[I1] + delta ;
         east[I3] = east[J] ;
         north[I3] = north[J] ;
         south[I3] = north[I3] 
              - (tmetric[I3]/tmetric[J])*areaJ/(east[I3] - west[I3]) ;
      } else {
         south[I1] = south[J] ;
         north[I1] = south[I1] + heightJ * tmetric[I1] / tmetric[J] ;
         west[I1]  = west[J] ;
         east[I1]  = east[J] ;
         b = heightJ + widthJ - (north[I1] - south[I1]) ;
         c = vmetric[J] * widthJ * heightJ / tmetric[J] ;
         r1 = 0.5*(b + sqrt(b*b - 4*c)) ;
         delta = c / r1 ;
         south[I2] = north[I1] + delta ;
         north[I2] = north[J] ;
         west[I2]  = west[J] ;
         east[I2]  = west[I2] 
              + (tmetric[I2]/tmetric[J])*areaJ/(north[I2] - south[I2]) ;
         south[I3] = north[I1] + delta ;
         north[I3] = north[J] ;
         east[I3]  = east[J] ;
         west[I3]  = east[I3] 
              - (tmetric[I3]/tmetric[J])*areaJ/(north[I3] - south[I3]) ;
      }
      break ; }
   case 4 : {
      double   area1, area2, area3, area4, b, c, delta, 
               height1, height2, height3, height4, r1,
               width1, width2, width3, width4 ;
      int      I1, I2, I3, I4 ;

      I1 = fch[J]  ;
      I2 = sib[I1] ;
      I3 = sib[I2] ;
      I4 = sib[I3] ;
      area1 = (areaJ*tmetric[I1])/tmetric[J] ;
      area2 = (areaJ*tmetric[I2])/tmetric[J] ;
      area3 = (areaJ*tmetric[I3])/tmetric[J] ;
      area4 = (areaJ*tmetric[I4])/tmetric[J] ;
#if MYDEBUG > 0
   fprintf(stdout, 
           "\n case 4, I1 = %d, I2 = %d, I3 = %d, I4 = %d" 
           "\n T[%d] = %6.3f, T[%d] = %6.3f, T[%d] = %6.3f"
           "\n T[%d] = %6.3f, T[%d] = %6.3f",
        I1, I2, I3, I4, J, tmetric[J], I1, tmetric[I1], I2, tmetric[I2],
        I3, tmetric[I3],
        I4, tmetric[I4]) ;
#endif
      if ( widthJ >= heightJ ) {
         b = heightJ + widthJ ;
         c = (areaJ*vmetric[J])/tmetric[J] ;
         r1 = 0.5*(b + sqrt(b*b - 4*c)) ;
         delta = c / r1 ;
#if MYDEBUG > 0
         fprintf(stdout, 
                 "\n b = %.3f, c = %.3f, r1 = %.3f, delta = %.3f",
                 b, c, r1, delta) ;
#endif
         height1 = height4 = (area1 + area4)/(widthJ - delta) ;
         height2 = height3 = (area2 + area3)/(widthJ - delta) ;
         width1  = area1 / height1 ;
         width2  = area2 / height2 ;
         width3  = area3 / height3 ;
         width4  = area4 / height4 ;
#if MYDEBUG > 0
         fprintf(stdout, 
                 "\n height1 = %.3f, width1 = %.3f"
                 "\n height2 = %.3f, width2 = %.3f"
                 "\n height3 = %.3f, width3 = %.3f"
                 "\n height4 = %.3f, width4 = %.3f",
                 height1, width1,
                 height2, width2,
                 height3, width3,
                 height4, width4) ;
#endif
         west[I1]  = west[J] ;
         east[I1]  = west[I1] + width1 ;
         south[I1] = south[J] ;
         north[I1] = south[I1] + height1 ;
         east[I4]  = east[J] ;
         west[I4]  = east[I4] - width4 ;
         south[I4] = south[J] ;
         north[I4] = south[I4] + height4 ;
         west[I3]  = west[J] ;
         east[I3]  = west[I3] + width3 ;
         north[I3] = north[J] ;
         south[I3] = north[I3] - height3 ;
         east[I2]  = east[J] ;
         west[I2]  = east[I2] - width2 ;
         north[I2] = north[J] ;
         south[I2] = north[I2] - height2 ;
      } else {
         b = heightJ + widthJ ;
         c = (areaJ*vmetric[J])/tmetric[J] ;
         r1 = 0.5*(b + sqrt(b*b - 4*c)) ;
         delta = c / r1 ;
         width1 = width4 = (area1 + area4)/(heightJ - delta) ;
         width2 = width3 = (area2 + area3)/(heightJ - delta) ;
         height1 = area1 / width1 ;
         height2 = area2 / width2 ;
         height3 = area3 / width3 ;
         height4 = area4 / width4 ;
         west[I1] = west[J] ;
         east[I1] = west[I1] + width1 ;
         south[I1] = south[J] ;
         north[I1] = south[I1] + height1 ;
         west[I4] = west[J] ;
         east[I4] = west[I4] + width4 ;
         north[I4] = north[J] ;
         south[I4] = north[I4] - height4 ;
         east[I3] = east[J] ;
         west[I3] = east[I3] - width3 ;
         south[I3] = south[J] ;
         north[I3] = south[I3] + height3 ;
         east[I2] = east[J] ;
         west[I2] = east[I2] - width2 ;
         north[I2] = north[J] ;
         south[I2] = north[I2] - height2 ;
      }
      break ; }
   default :
      return(-1) ;
      break ;
   }
}
return(1) ; }

/*--------------------------------------------------------------------*/
