/*  drawTree.c  */

#include "../Tree.h"

#define MYDEBUG 1

/*--------------------------------------------------------------------*/
/*
   ------------------------------------------------------------
   purpose -- to write an EPS file with a picture of a tree
              in Cartesian or polar form

   tagsIV   -- IV object that contains the labels of the nodes.
       if NULL then the node ids are used
   filename -- name of the file to be written
   heightflag -- flag to determine the y-coordinate of a node
      H --> height flag
         y(J) = 0 if J is a leaf
                1 + max_{p(I) = J} y(I) otherwise
      D --> depth flag, let h be the height of the tree
         y(J) = h if J is a root
                y(par(J)) - 1 otherwise
   coordflag -- flag to determine a Cartesion or polar plot
      0 --> Cartesian plot
      1 --> polar plot
   radius -- radius of the circle that represents a node
   bbox[] -- bounding box
      bbox[0] -- x_min
      bbox[1] -- y_min
      bbox[2] -- x_max
      bbox[3] -- y_max
   frame[] -- frame to hold tree, note tree is tangent to frame
      frame[0] -- x_min
      frame[1] -- y_min
      frame[2] -- x_max
      frame[3] -- y_max

   recommendations, 
      r = 2
      bbox[] = { 0, 0, 500, 200 } for nested dissection trees
               { 0, 0, 500, 500 } for minimum degree trees
      frame[0] = bbox[0] + 10
      frame[1] = bbox[1] + 10
      frame[2] = bbox[2] - 10
      frame[3] = bbox[3] - 10

   created -- 97apr12, cca
   ------------------------------------------------------------
*/
void
Tree_drawEPS (
   Tree     *tree,
   IV       *tagsIV,
   char     *filename,
   char     heightflag,
   char     coordflag,
   double   radius,
   double   bbox[],
   double   frame[]
) {
double   etamax, etamin, height, maxdepth, maxheight, 
         r, theta, ximax, ximin,
         xmax, xmin, xscale, ymax, ymin, yscale ;
double   *x, *y ;
DV       *xDV, *yDV ;
FILE     *fp ;
int      count, I, J, K, n, nleaves ;
int      *fch, *par, *sib ;
/*
   ---------------
   check the input
   ---------------
*/
if ( tree == NULL || filename == NULL 
   || (heightflag != 'H' && heightflag != 'D')
   || bbox == NULL || frame == NULL ) {
   fprintf(stderr, "\n fatal error in Tree_drawEPS(%p,%p,%c,%f,%p,%p)"
           "\n bad input\n", tree, filename, heightflag, radius,
           bbox, frame) ;
   exit(-1) ;
}
n   = tree->n ;
par = tree->par ;
fch = tree->fch ;
sib = tree->sib ;
/*
   -------------------
   get the coordinates
   -------------------
*/
yDV = DV_new() ;
DV_init(yDV, n, NULL) ;
y = DV_entries(yDV) ;
switch ( heightflag ) {
case 'D' :
   for ( J = Tree_preOTfirst(tree), maxdepth = 0 ;
         J != -1 ;
         J = Tree_preOTnext(tree, J) ) {
      if ( (K = par[J]) == -1 ) {
         y[J] = 0.0 ;
      } else {
         y[J] = y[K] + 1.0 ;
      }
      if ( maxdepth < y[J] ) {
         maxdepth = y[J] ;
      }
   }
   if ( coordflag == 0 ) {
      for ( J = 0 ; J < n ; J++ ) {
         y[J] = maxdepth - y[J] ;
      }
   }
   break ;
case 'H' :
   for ( J = Tree_postOTfirst(tree), maxheight = 0 ;
         J != -1 ;
         J = Tree_postOTnext(tree, J) ) {
      if ( (I = fch[J]) == -1 ) {
         y[J] = 0.0 ;
      } else {
         height = y[I] ;
         for ( I = sib[I] ; I != -1 ; I = sib[I] ) {
            if ( height < y[I] ) {
               height = y[I] ;
            }
         }
         y[J] = height + 1.0 ;
      }
      if ( maxheight < y[J] ) {
         maxheight = y[J] ;
      }
   }
   if ( coordflag == 1 ) {
      for ( J = 0 ; J < n ; J++ ) {
         y[J] = maxheight - y[J] ;
      }
   }
   break ;
default :
   break ;
}
#if MYDEBUG > 0
   fprintf(stdout, "\n\n y") ;
   DV_writeForHumanEye(yDV, stdout) ;
#endif
xDV = DV_new() ;
DV_init(xDV, n, NULL) ;
DV_fill(xDV, 0.0) ;
x = DV_entries(xDV) ;
nleaves = 0 ;
for ( J = Tree_postOTfirst(tree) ;
      J != -1 ;
      J = Tree_postOTnext(tree, J) ) {
   if ( fch[J] == -1 ) {
      x[J] = nleaves++ ;
   } else {
      for ( I = fch[J], count = 0 ; I != -1 ; I = sib[I] ) {
         x[J] += x[I] ;
         count++ ;
      }
      x[J] /= count ;
   }
}
if ( coordflag == 0 ) {
   for ( J = 0 ; J < n ; J++ ) {
      x[J] = x[J] / nleaves ;
   }
} else {
   for ( J = 0 ; J < n ; J++ ) {
      theta = 6.283185 * x[J] / nleaves ;
      r     = y[J] ;
      x[J]  = r * cos(theta) ;
      y[J]  = r * sin(theta) ;
   }
}
#if MYDEBUG > 0
   fprintf(stdout, "\n\n x") ;
   DV_writeForHumanEye(xDV, stdout) ;
#endif
/*
   ---------------------------------------------------------
   scale x[] and y[] to fit within the frame in bounding box 
   ---------------------------------------------------------
*/
xmin = frame[0] ;
ymin = frame[1] ;
xmax = frame[2] ;
ymax = frame[3] ;
ximax = DV_max(xDV) ;
ximin = DV_min(xDV) ;
etamax = DV_max(yDV) ;
etamin = DV_min(yDV) ;
xscale = (xmax - xmin - radius)/(ximax - ximin) ;
yscale = (ymax - ymin - radius)/(etamax - etamin) ;
#if MYDEBUG > 0
fprintf(stdout, "\n\n xscale = %.3g, yscale = %.3g", xscale, yscale) ;
#endif
for ( J = 0 ; J < n ; J++ ) {
   x[J] = xmin + radius + xscale*(x[J] - ximin) ;
   y[J] = ymin + radius + yscale*(y[J] - etamin) ;
}
/*
   -------------
   open the file
   -------------
*/
if ( (fp = fopen(filename, "w")) == NULL ) {
   fprintf(stderr, "\n unable to open file %s", filename) ;
   exit(-1) ;
}
/*
   ----------------------------
   print the header information
   ----------------------------
*/
fprintf(fp, 
        "%%!PS-Adobe-2.0 EPSF-1.2"
        "\n%%%%BoundingBox: %.3g %.3g %.3g %.3g",
        bbox[0], bbox[1], bbox[2], bbox[3]) ;
fprintf(fp, 
        "\n /CSH {"
        "\n %%"
        "\n %% center show a string"
        "\n %%"
        "\n %%  stack"
        "\n %%     string str"
        "\n %%"
        "\n dup stringwidth pop 2 div neg 0 rmoveto"
        "\n show"
        "\n } def") ;
fprintf(fp, 
        "\n /ML {"
        "\n %%"
        "\n %% moveto lineto"
        "\n %%"
        "\n %%  stack"
        "\n %%     x0 y0 x1 y1"
        "\n %%"
        "\n moveto lineto"
        "\n } def") ;
fprintf(fp, 
        "\n /FC {"
        "\n %%"
        "\n %% draw filled circle"
        "\n %%"
        "\n %%  stack"
        "\n %%     x y r"
        "\n %%"
        "\n newpath 2 index 1 index add 2 index moveto 0 360 arc fill"
        "\n } def") ;
fprintf(fp, 
        "\n /OC {"
        "\n %%"
        "\n %% draw open circle"
        "\n %%"
        "\n %%  stack"
        "\n %%     x y r"
        "\n %%"
        "\n newpath 2 index 1 index add 2 index moveto 0 360 arc stroke"
        "\n } def") ;
fprintf(fp, "\n\n %.3g %.3g %.3g %.3g rectstroke",
        frame[0], frame[1], frame[2], frame[3]) ;
/*
   --------------
   draw the edges
   --------------
*/
par = tree->par ;
count = 0 ;
for ( J = 0 ; J < n ; J++ ) {
   if ( (K = par[J]) != -1 ) {
      if ( count == 0 ) {
         fprintf(fp, "\n newpath") ;
      }
      fprintf(fp, "\n   %.3g %.3g %.3g %.3g ML",
              x[J], y[J], x[K], y[K]) ;
      count++ ;
      if ( count == 100 ) {
         fprintf(fp, "\n stroke") ;
         count = 0 ;
      }
   }
}
if ( count > 0 ) {
   fprintf(fp, "\n stroke") ;
}
/*
   -------------------------
   draw the nodes and labels
   -------------------------
*/
fprintf(fp, "\n\n gsave") ;
/*
fprintf(fp, "\n    /Helvetica-Bold findfont %.3g scalefont setfont",
        1.25*radius) ;
*/
fprintf(fp, "\n    /Helvetica-Bold findfont %.3g scalefont setfont",
        1.5*radius) ;
for ( J = 0 ; J < n ; J++ ) {
   fprintf(fp, "\n    1.0 setgray") ;
   fprintf(fp, " %.3g %.3g %.3g FC", x[J], y[J], radius) ;
   fprintf(fp, "\n    0.0 setgray") ;
   fprintf(fp, " %.3g %.3g %.3g OC", x[J], y[J], radius) ;
   fprintf(fp, "\n   %.3g %.3g moveto ", x[J], y[J] - radius/2) ;
   if ( tagsIV != NULL ) {
      fprintf(fp, " (%d) CSH", IV_entry(tagsIV, J)) ;
   } else {
      fprintf(fp, " (%d) CSH", J) ;
   }
}
fprintf(fp, "\n\n grestore") ;
fprintf(fp, "\n\n showpage") ;

return ; }

/*--------------------------------------------------------------------*/
