/*  Tree.h  */

#include "../cfiles.h"
#include "../IV.h"
#include "../DV.h"

/*--------------------------------------------------------------------*/
/*
   -----------------------
   simple tree object

   created -- 95nov15, cca
   -----------------------
*/
typedef struct _Tree   Tree ;
struct _Tree {
   int   n    ;
   int   root ;
   int   *par ;
   int   *fch ;
   int   *sib ;
} ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in basics.c ----------------------------------------
------------------------------------------------------------------------
*/
/*
   -----------------------------------------------
   purpose -- create and return a new Tree object

   created -- 95nov15, cca
   -----------------------------------------------
*/
Tree *
Tree_new ( 
   void
) ;
/*
   ------------------------------------------------------
   purpose -- set the default fields for the Tree object

   created -- 95nov15, cca
   ------------------------------------------------------
*/
void
Tree_setDefaultFields (
   Tree   *tree
) ;
/*
   --------------------------------
   purpose -- clear the data fields

   created -- 95nov15, cca
   --------------------------------
*/
void
Tree_clearData (
   Tree   *tree
) ;
/*
   --------------------------------
   purpose -- free the Tree object

   created -- 95nov15, cca
   --------------------------------
*/
void
Tree_free (
   Tree   *tree
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in init.c ------------------------------------------
------------------------------------------------------------------------
*/
/*
   -----------------------
   simplest constructor

   created -- 95nov15, cca
   -----------------------
*/
void
Tree_init1 (
   Tree   *tree,
   int    size
) ;
/*
   --------------------------------
   initialize given a parent vector

   created -- 95nov15, cca
   --------------------------------
*/
void
Tree_init2 (
   Tree   *tree,
   int    size,
   int    par[]
) ;
/*
   ---------------------------------
   initialize given the tree vectors

   created -- 95nov15, cca
   ---------------------------------
*/
void
Tree_init3 (
   Tree   *tree,
   int    size,
   int    par[],
   int    fch[],
   int    sib[]
) ;
/*
   ------------------------------------
   set the fch[], sib[] and root fields

   created -- 95nov15, cca
   ------------------------------------
*/
void
Tree_setFchSibRoot (
   Tree   *tree
) ;
/*
   -----------------------
   set the root field

   created -- 95nov15, cca
   -----------------------
*/
void
Tree_setRoot (
   Tree   *tree
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in util.c ------------------------------------------
------------------------------------------------------------------------
*/
/*
   -------------------------------------------------
   return the first vertex in a post-order traversal
   
   created -- 95nov15, cca
   -------------------------------------------------
*/
int
Tree_postOTfirst (
   Tree   *tree
) ;
/*
   ----------------------------------------------------------
   return the vertex that follows v in a post-order traversal
   ----------------------------------------------------------
*/
int
Tree_postOTnext (
   Tree   *tree,
   int    v
) ;
/*
   ------------------------------------------------
   return the first vertex in a pre-order traversal
   
   created -- 95nov15, cca
   ------------------------------------------------
*/
int
Tree_preOTfirst (
   Tree   *tree
) ;
/*
   ---------------------------------------------------------
   return the vertex that follows v in a pre-order traversal
   
   created -- 95nov15, cca
   ---------------------------------------------------------
*/
int
Tree_preOTnext (
   Tree   *tree,
   int    v
) ;
/*
   ---------------------------------------
   return the number of leaves in the tree

   created -- 95nov15, cca
   ---------------------------------------
*/
int
Tree_nleaves (
   Tree   *tree
) ;
/*
   -----------------------------------------------
   return the number of roots of the tree (forest)

   created -- 95nov15, cca
   -----------------------------------------------
*/
int
Tree_nroots (
   Tree   *tree
) ;
/*
   -----------------------------------------
   return the number of children of a vertex

   created -- 95nov15, cca
   -----------------------------------------
*/
int
Tree_nchild (
   Tree   *tree,
   int    v
) ;
/*
   -------------------------------------------
   this method returns an IV object that holds 
   the number of children for the tree nodes.
 
   created -- 96dec18, cca
   -------------------------------------------
*/
IV *
Tree_nchildIV (
   Tree   *tree
) ;
/*
   -----------------------------
   return the height of the tree
 
   created -- 96aug23, cca
   -----------------------------
*/
int
Tree_height (
   Tree   *tree
) ;
/*
   -------------------------------------------------------------
   return the maximum number of children of any node in the tree
 
   created -- 96sep05, cca
   -------------------------------------------------------------
*/
int
Tree_maxNchild (
   Tree   *tree
) ;
/*
   ---------------------------------------------
   return the number of bytes used by the object
   ---------------------------------------------
*/
int
Tree_sizeOf (
   Tree   *tree
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in metrics.c ---------------------------------------
------------------------------------------------------------------------
*/
/*
   ------------------------------------------------------
   create and return a subtree metric IV object
   input  : vmetricIV -- a metric defined on the vertices
   return : tmetricIV -- a metric defined on the subtrees
  
   created -- 96jun23, cca
   ------------------------------------------------------
*/
IV *
Tree_setSubtreeImetric (
   Tree   *tree,
   IV     *vmetricIV
) ;
/*
   ------------------------------------------------------
   create and return a subtree metric DV object
   input  : vmetricDV -- a metric defined on the vertices
   return : tmetricDV -- a metric defined on the subtrees
 
   created -- 96jun23, cca
   ------------------------------------------------------
*/
DV *
Tree_setSubtreeDmetric (
   Tree   *tree,
   DV     *vmetricDV
) ;
/*
   ------------------------------------------------------------
   create and return a depth metric IV object
   input  : vmetricIV -- a metric defined on the vertices
   output : dmetric[] -- a depth metric defined on the vertices
 
   dmetric[u] = vmetric[u] + dmetric[par[u]] if par[u] != -1
              = vmetric[u]                   if par[u] == -1
 
   created -- 96jun23, cca
   ------------------------------------------------------------
*/
IV *
Tree_setDepthImetric (
   Tree   *tree,
   IV     *vmetricIV
) ;
/*
   ------------------------------------------------------------
   create and return a depth metric DV object
   input  : vmetricDV -- a metric defined on the vertices
   output : dmetric[] -- a depth metric defined on the vertices
 
   dmetric[u] = vmetric[u] + dmetric[par[u]] if par[u] != -1
              = vmetric[u]                   if par[u] == -1
 
   created -- 96jun23, cca
   ------------------------------------------------------------
*/
DV *
Tree_setDepthDmetric (
   Tree   *tree,
   DV     *vmetricDV
) ;
/*
   ------------------------------------------------------------------
   create and return a height metric IV object
   input  : vmetricIV -- a metric defined on the vertices
   output : dmetricIV -- a depth metric defined on the vertices
 
   hmetric[v] = vmetric[v] + max{p(u) = v} hmetric[u] if fch[v] != -1
              = vmetric[v]                            if fch[v] == -1
 
   created -- 96jun23, cca
   ------------------------------------------------------------------
*/
IV *
Tree_setHeightImetric (
   Tree   *tree,
   IV     *vmetricIV
) ;
/*
   ------------------------------------------------------------------
   create and return a height metric DV object
   input  : vmetricDV -- a metric defined on the vertices
   output : dmetricDV -- a depth metric defined on the vertices
 
   hmetric[v] = vmetric[v] + max{p(u) = v} hmetric[u] if fch[v] != -1
              = vmetric[v]                            if fch[v] == -1
 
   created -- 96jun23, cca
   ------------------------------------------------------------------
*/
DV *
Tree_setHeightDmetric (
   Tree   *tree,
   DV     *vmetricDV
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in justify.c ---------------------------------------
------------------------------------------------------------------------
*/
/*
   ------------------------------------------------------------
   left-justify a tree by subtree size
   children are linked in ascending order of their subtree size
 
   created -- 95nov15, cca
   ------------------------------------------------------------
*/
void
Tree_leftJustify (
   Tree   *tree
) ;
/*
   ------------------------------------------------------
   left-justify a tree by a metric
   children are linked in ascending order of their metric
 
   created -- 96jun23, cca
   ------------------------------------------------------
*/
void
Tree_leftJustifyI (
   Tree   *tree,
   IV     *metricIV
) ;
/*
   ------------------------------------------------------
   left-justify a tree by a metric
   children are linked in ascending order of their metric
 
   created -- 96jun23, cca
   ------------------------------------------------------
*/
void
Tree_leftJustifyD (
   Tree   *tree,
   DV     *metricDV
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in perms.c -----------------------------------------
------------------------------------------------------------------------
*/
/*
   --------------------------------------
   fill the new-to-old permutation vector
 
   created -- 95nov15, cca
   --------------------------------------
*/
void
Tree_fillNewToOldPerm (
   Tree   *tree,
   int    newToOld[]
) ;
/*
   --------------------------------------
   fill the old-to-new permutation vector
 
   created -- 95nov15, cca
   --------------------------------------
*/
void
Tree_fillOldToNewPerm (
   Tree   *tree,
   int    oldToNew[]
) ;
/*
   ------------------------------------------------------
   fill the new-to-old and old-to-new permutation vectors
 
   created -- 95nov15, cca
   ------------------------------------------------------
*/
void
Tree_fillBothPerms (
   Tree   *tree,
   int    newToOld[],
   int    oldToNew[]
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in IO.c --------------------------------------------
------------------------------------------------------------------------
*/
/*
   ------------------------------------------------
   purpose -- to read in an Tree object from a file
 
   input --
 
      fn -- filename, must be *.treeb or *.treef
 
   return value -- 1 if success, 0 if failure
 
   created -- 95nov15, cca
   ------------------------------------------------
*/
int
Tree_readFromFile ( 
   Tree   *tree, 
   char   *fn 
) ;
/*
   -------------------------------------------------------
   purpose -- to read an Tree object from a formatted file
 
   return value -- 1 if success, 0 if failure
 
   created -- 95nov15, cca
   -------------------------------------------------------
*/
int
Tree_readFromFormattedFile (
   Tree   *tree,
   FILE   *fp
) ;
/*
   ---------------------------------------------------
   purpose -- to read an Tree object from a binary file
 
   return value -- 1 if success, 0  if failure
 
   created -- 95nov15, cca
   ---------------------------------------------------
*/
int
Tree_readFromBinaryFile (
   Tree    *tree,
   FILE   *fp
) ;
/*
   --------------------------------------------
   purpose -- to write an Tree object to a file
 
   input --
 
      fn -- filename
        *.treeb -- binary
        *.treef -- formatted
        anything else -- for human eye
 
   return value -- 1 if success, 0 otherwise
 
   created -- 95nov15, cca
   --------------------------------------------
*/
int
Tree_writeToFile (
   Tree   *tree,
   char   *fn
) ;
/*--------------------------------------------------------------------*/
/*
   ------------------------------------------------------
   purpose -- to write an Tree object to a formatted file
 
   return value -- 1 if success, 0 otherwise
 
   created -- 95nov15, cca
   ------------------------------------------------------
*/
int
Tree_writeToFormattedFile (
   Tree   *tree,
   FILE   *fp
) ;
/*
   ---------------------------------------------------
   purpose -- to write an Tree object to a binary file
 
   return value -- 1 if success, 0 otherwise
 
   created -- 95nov15, cca
   ---------------------------------------------------
*/
int
Tree_writeToBinaryFile (
   Tree    *tree,
   FILE   *fp
) ;
/*
   --------------------------------------------------
   purpose -- to write an Tree object for a human eye
 
   return value -- 1 if success, 0 otherwise
 
   created -- 95nov15, cca
   --------------------------------------------------
*/
int
Tree_writeForHumanEye (
   Tree    *tree,
   FILE   *fp
) ;
/*
   ----------------------------------------------------------
   purpose -- to write out the statistics for the Tree object
 
   return value -- 1 if success, 0 otherwise
 
   created -- 95nov15, cca
   ----------------------------------------------------------
*/
int
Tree_writeStats (
   Tree    *tree,
   FILE   *fp
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in compress.c --------------------------------------
------------------------------------------------------------------------
*/
/*
   --------------------------------------------
   create and return an IV object that contains
   the map from vertices to fundamental chains.
 
   return value -- # of fundamental chains
 
   created -- 96jun23, cca
   -------------------------------------------
*/
IV *
Tree_fundChainMap (
   Tree   *tree
) ;
/*
   -----------------------------------------------------------------
   compress a tree based on a map from old vertices to new vertices.
   the restriction on the map is that the set {u | map[u] = U} must
   be connected for all U.
 
   created  -- 95nov15, cca
   modified -- 96jan04, cca
      bug fixed, N computed incorrectly
   modified -- 96jun23, cca
      in calling sequence, int map[] converted to IV *mapIV 
   -----------------------------------------------------------------
*/
Tree *
Tree_compress (
   Tree   *tree,
   IV     *mapIV
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in permute.c ---------------------------------------
------------------------------------------------------------------------
*/
/*
   -----------------------
   return a permuted tree
 
   created -- 96jan04, cca
   -----------------------
*/
Tree *
Tree_permute (
   Tree   *tree,
   int    newToOld[],
   int    oldToNew[]
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in setBoxes.c --------------------------------------
------------------------------------------------------------------------
*/
/*
   -------------------------------------------------------------
   purpose -- fill boxes arrays for display of a tree
 
   vmetric[] -- vector of metric on the nodes
   tmetric[] -- vector of metric on the subtrees
   xmin, xmax, ymin, ymax -- bounds on box for root
   west[], east[], south[], north[] -- vector to hold bounds for
                                       the nodes in the tree
 
   return value --
     1 --> success
     2 --> no success, maxnchild > 3
 
   created -- 96dec20, cca
   -------------------------------------------------------------
*/
int
Tree_setBoxesII (
   Tree     *tree,
   int      vmetric[],
   int      tmetric[],
   double   xmin,
   double   xmax,
   double   ymin,
   double   ymax,
   double   west[],
   double   east[],
   double   south[],
   double   north[]
) ;
/*
   -------------------------------------------------------------
   purpose -- fill boxes arrays for display of a tree
 
   vmetric[] -- vector of metric on the nodes
   tmetric[] -- vector of metric on the subtrees
   xmin, xmax, ymin, ymax -- bounds on box for root
   west[], east[], south[], north[] -- vector to hold bounds for
                                       the nodes in the tree
 
   return value --
     1 --> success
     2 --> no success, maxnchild > 3
 
   created -- 96dec20, cca
   -------------------------------------------------------------
*/
int
Tree_setBoxesDD (
   Tree     *tree,
   double   vmetric[],
   double   tmetric[],
   double   xmin,
   double   xmax,
   double   ymin,
   double   ymax,
   double   west[],
   double   east[],
   double   south[],
   double   north[]
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in drawTree.c --------------------------------------
------------------------------------------------------------------------
*/
/*
   ------------------------------------------------------------
   purpose -- to write an EPS file with a picture of a tree
              in Cartesian or polar form
 
   tagsIV   -- IV object that contains the labels of the nodes.
       if NULL then the node ids are used
   filename -- name of the file to be written
   heightflag -- flag to determine the y-coordinate of a node
      H --> height flag
         y(J) = 0 if J is a leaf
                1 + max_{p(I) = J} y(I) otherwise
      D --> depth flag, let h be the height of the tree
         y(J) = h if J is a root
                y(par(J)) - 1 otherwise
   coordflag -- flag to determine a Cartesion or polar plot
      0 --> Cartesian plot
      1 --> polar plot
   radius -- radius of the circle that represents a node
   bbox[] -- bounding box
      bbox[0] -- x_min
      bbox[1] -- y_min
      bbox[2] -- x_max
      bbox[3] -- y_max
   frame[] -- frame to hold tree, note tree is tangent to frame
      frame[0] -- x_min
      frame[1] -- y_min
      frame[2] -- x_max
      frame[3] -- y_max
 
   recommendations, 
      r = 2
      bbox[] = { 0, 0, 500, 200 } for nested dissection trees
               { 0, 0, 500, 500 } for minimum degree trees
      frame[0] = bbox[0] + 10
      frame[1] = bbox[1] + 10
      frame[2] = bbox[2] - 10
      frame[3] = bbox[3] - 10
 
   created -- 97apr12, cca
   ------------------------------------------------------------
*/
void
Tree_drawEPS (
   Tree     *tree,
   IV       *tagsIV,
   char     *filename,
   char     heightflag,
   char     coordflag,
   double   radius,
   double   bbox[],
   double   frame[]
) ;
/*--------------------------------------------------------------------*/
