/*  symbfac.c  */

#include "../spoolesMPI.h"

#define MYDEBUG 0

/*--------------------------------------------------------------------*/
typedef struct _Msg Msg ;
struct _Msg {
   int           id    ;
   int           size  ;
   void          *base ;
   MPI_Request   req   ;
   Msg           *next ;
} ;
static void loadInternalIndices ( int myid, int nfront, int nvtx,
   int frontOwners[], int vtxToFront[], IVL *symbfacIVL,
   DPencil *pencil, int nLeft[], int msglvl, FILE *msgFile ) ;
static int mergeIndices ( int sizeJ, int indJ[], 
                          int sizeK, int indK[], int nleft ) ;
static Msg * Msg_new ( void ) ;
static void Msg_setDefaultFields ( Msg *msg ) ;
static void Msg_clearData ( Msg *msg ) ;
static void Msg_free ( Msg *msg ) ;
static Msg * wakeupFront ( int J, int myid, int owners[], 
   char supportTable[], ETree *frontETree, IVL *symbfacIVL,
   int firsttag, int stats[], MPI_Comm comm, 
   int msglvl, FILE *msgFile ) ;
static Msg * checkRecvMessages ( int J, Msg *firstmsg, int nLeft[],
   int nodwghts[], IVL *symbfacIVL, int msglvl, FILE *msgFile ) ;
static Msg * sendMessages ( int J, int myid, int owners[], int nfront, 
   int nproc, char supportTable[], int par[], IVL *symbfacIVL,
   Msg *firstMsgSent, int firsttag, int stats[], MPI_Comm comm, 
   int msglvl, FILE *msgFile ) ;
static Msg * checkSendMessages ( Msg *firstMsgSent,
   int msglvl, FILE *msgFile ) ;
/*--------------------------------------------------------------------*/
/*
   --------------------------------------------------------------------
   perform the symbolic factorization in parallel

   input --
      frontETree    -- front tree for the factorization
      frontOwnersIV -- map from fronts to owning process
      inpmtx        -- matrix object
      firsttag      -- first tag to be used for messages,
                       will use tag, ..., tag + nfront - 1
      msglvl        -- message level
      msgFile       -- message file
      comm          -- MPI communicator

   return value --
      symbfacIVL -- contains front indices for supported fronts

   created -- 97dec07, cca
   --------------------------------------------------------------------
*/
IVL *
SymbFac_MPI_initFromDInpMtx (
   ETree      *frontETree,
   DInpMtx    *inpmtx,
   IV         *frontOwnersIV,
   int        firsttag,
   int        stats[],
   int        msglvl,
   FILE       *msgFile,
   MPI_Comm   comm
) {
DPencil   pencil ;
IVL       *symbfacIVL ;

DPencil_setDefaultFields(&pencil) ;
DPencil_init(&pencil, inpmtx, 0.0, NULL) ;
symbfacIVL = SymbFac_MPI_initFromDInpMtx(frontETree, inpmtx, 
                frontOwnersIV, firsttag, stats, msglvl, msgFile, comm) ;

return(symbfacIVL) ; }

/*--------------------------------------------------------------------*/
/*
   --------------------------------------------------------------------
   perform the symbolic factorization in parallel

   input --
      frontETree    -- front tree for the factorization
      frontOwnersIV -- map from fronts to owning process
      pencil        -- matrix pencil
      firsttag      -- first tag to be used for messages,
                       will use tag, ..., tag + nfront - 1
      msglvl        -- message level
      msgFile       -- message file
      comm          -- MPI communicator

   return value --
      symbfacIVL -- contains front indices for supported fronts

   created -- 97jul29, cca
   --------------------------------------------------------------------
*/
IVL *
SymbFac_MPI_initFromDPencil (
   ETree      *frontETree,
   DPencil    *pencil,
   IV         *frontOwnersIV,
   int        firsttag,
   int        stats[],
   int        msglvl,
   FILE       *msgFile,
   MPI_Comm   comm
) {
char         *frontSupported, *status, *supportTable ;
Ideq         *dequeue ;
int          flag, iproc, J, K, myid, nDJ, nfront, nownedFronts, nproc, 
             nfrontSupported, nvtx, sizeJ, sizeK, tag_bound ;
int          *bndwghts, *fch, *frontOwners, *indicesJ, *indicesK, 
             *nactiveChild, *nLeft, *nodwghts, *par, *sib, *sizes, 
             *vtxToFront ;
IVL          *symbfacIVL ;
Msg          *firstMsgSent ;
Msg          **p_msg ;
/*
   ---------------
   check the input
   ---------------
*/
if ( frontETree == NULL || frontOwnersIV == NULL
   || pencil == NULL || (msglvl > 0 && msgFile == NULL) ) {
   fprintf(stderr, "\n fatal error in SymbFac_MPI_initFromDPencil()"
           "\n comm = %p, frontETree = %p, frontOwnersIV = %p"
           "\n pencil = %p, firsttag = %d, msglvl = %d, msgFile = %p"
           "\n bad input\n", comm, frontETree, frontOwnersIV, pencil,
           firsttag, msglvl, msgFile) ;
   exit(-1) ;
}
nfront = ETree_nfront(frontETree) ;
MPI_Attr_get(MPI_COMM_WORLD, MPI_TAG_UB, &tag_bound, &flag) ;
if ( firsttag < 0 || firsttag + nfront > tag_bound ) {
   fprintf(stderr, "\n fatal error in SymbFac_MPI_initFromDPencil()"
           "\n range of tags is [%d,%d], tag_bound = %d",
           firsttag, firsttag + nfront - 1, tag_bound) ;
   exit(-1) ;
}
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n\n inside SymbFac_MPI_initFromDPencil()") ;
   ETree_writeForHumanEye(frontETree, msgFile) ;
   fprintf(msgFile, "\n\n front owners IV object") ;
   IV_writeForHumanEye(frontOwnersIV, msgFile) ;
   fprintf(msgFile, "\n\n matrix pencil") ;
   DPencil_writeForHumanEye(pencil, msgFile) ;
   fflush(msgFile) ;
}
/*
   -------------------------------
   extract pointers and dimensions
   -------------------------------
*/
nvtx       = ETree_nvtx(frontETree) ;
nodwghts   = ETree_nodwghts(frontETree),
bndwghts   = ETree_bndwghts(frontETree),
vtxToFront = ETree_vtxToFront(frontETree),
par        = ETree_par(frontETree) ;
fch        = ETree_fch(frontETree) ;
sib        = ETree_sib(frontETree) ;
IV_sizeAndEntries(frontOwnersIV, &nfront, &frontOwners) ;
MPI_Comm_rank(comm, &myid) ;
MPI_Comm_size(comm, &nproc) ;
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n\n myid = %d, nproc = %d", myid, nproc) ;
   fflush(msgFile) ;
}
for ( J = 0, nownedFronts = 0 ; J < nfront ; J++ ) {
   if ( frontOwners[J] == myid ) {
      nownedFronts++ ;
   }
}
/*
   --------------------------
   generate the support table
   --------------------------
*/
supportTable = CVinit(nfront*nproc, 'N') ;
for ( J = 0 ; J < nfront ; J++ ) {
   iproc = frontOwners[J] ;
   frontSupported = supportTable + iproc*nfront ;
   for ( K = J ; K != -1 && frontSupported[K] == 'N' ; K = par[K] ) {
      frontSupported[K] = 'Y' ;
   }
}
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n\n supportTable") ;
   for ( J = 0 ; J < nfront ; J++ ) {
      fprintf(msgFile, "\n") ;
      for ( iproc = 0 ; iproc < nproc ; iproc++ ) {
         fprintf(msgFile, " %c", supportTable[J + iproc*nfront]) ;
      }
   }
   fflush(msgFile) ;
}
frontSupported = supportTable + myid*nfront ;
for ( J = 0, nfrontSupported = 0 ; J < nfront ; J++ ) {
   if ( frontSupported[J] == 'Y' ) {
      nfrontSupported++ ;
   }
}
#if MYDEBUG > 0
fprintf(stdout, "\n proc %d : nfrontSupported = %d", 
        myid, nfrontSupported) ;
fflush(stdout) ;
#endif
/*
   -----------------------------------------------
   initialize the local symbolic factorization IVL
   -----------------------------------------------
*/
symbfacIVL = IVL_new() ;
sizes = IVinit(nfront, 0) ;
for ( J = 0 ; J < nfront ; J++ ) {
   if ( frontSupported[J] == 'Y' ) {
      sizes[J] = nodwghts[J] + bndwghts[J] ;
   }
}
IVL_init3(symbfacIVL, IVL_CHUNKED, nfront, sizes) ;
IVfree(sizes) ;
/*
   ---------------------------------------------------------------
   fill the index lists of the owned fronts with the internal
   vertices and boundary vertices available from the matrix pencil
   ---------------------------------------------------------------
*/
nLeft = IVinit(nfront, -1) ;
loadInternalIndices(myid, nfront, nvtx, frontOwners, vtxToFront, 
                    symbfacIVL, pencil, nLeft, msglvl, msgFile) ;
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n\n after loading internal indices") ;
   IVL_writeForHumanEye(symbfacIVL, msgFile) ;
   fprintf(msgFile, "\n\n nLeft") ;
   IVfprintf(msgFile, nfront, nLeft) ;
   fflush(msgFile) ;
}
/*
   -------------------------------------------------
   initialize the status vector and load the dequeue
   -------------------------------------------------
*/
dequeue = Ideq_new() ;
Ideq_resize(dequeue, nfront) ;
status = CVinit(nfront, 'F') ;
for ( J = 0 ; J < nfront ; J++ ) {
   if ( status[J] == 'F' && supportTable[J + myid*nfront] == 'Y' ) {
      status[J] = 'W' ;
      K = par[J] ;
      while ( K != -1 && status[K] == 'F' ) {
         status[K] = 'W' ;
         K = par[K] ;
      }
      if ( msglvl > 1 ) {
         fprintf(msgFile, "\n inserting front %d at tail", J) ;
         fflush(msgFile) ;
      }
      Ideq_insertAtTail(dequeue, J) ;
   }
}
nactiveChild = IVinit(nfront, 0) ;
for ( J = 0 ; J < nfront ; J++ ) {
   if ( status[J] == 'W' ) {
      if ( (K = par[J]) != -1 ) {
         nactiveChild[K]++ ;
      }
   }
}
/*
   ----------------------------------
   loop while the dequeue is nonempty
   ----------------------------------
*/
firstMsgSent = NULL ;
ALLOCATE(p_msg, struct _Msg *, nfront) ;
for ( J = 0 ; J < nfront ; J++ ) {
   p_msg[J] = NULL ;
}
while ( (J = Ideq_removeFromHead(dequeue)) != -1 ) {
   if ( msglvl > 1 ) {
      fprintf(msgFile, "\n\n ### popping %d from dequeue", J) ;
      fflush(msgFile) ;
   }
   if ( status[J] == 'W' ) {
/*
      --------------------------------------------
      wake up the front, post its receive messages
      --------------------------------------------
*/
      p_msg[J] = wakeupFront(J, myid, frontOwners, supportTable,
                             frontETree, symbfacIVL, firsttag, 
                             stats, comm, msglvl, msgFile) ;
      status[J] = 'A' ;
   }
   if ( p_msg[J] != NULL ) {
/*
      -----------------------------
      try to receive messages for J
      -----------------------------
*/
      p_msg[J] = checkRecvMessages(J, p_msg[J], nLeft, nodwghts,  
                                   symbfacIVL, msglvl, msgFile) ;
   }
   if ( p_msg[J] == NULL ) {
/*
      ---------------------
      all messages received
      ---------------------
*/
      status[J] = 'D' ;
      if ( frontOwners[J] == myid ) {
         firstMsgSent = sendMessages(J, myid, frontOwners, nfront, 
                                     nproc, supportTable, par, 
                                     symbfacIVL, firstMsgSent, firsttag,
                                     stats, comm, msglvl, msgFile) ;
      }
      if ( (K = par[J]) != -1 ) {
         if ( frontOwners[K] == myid && nLeft[K] > 0 ) {
/*
            ------------------------------
            merge bnd{J} into K cup bnd{K}
            ------------------------------
*/
            nDJ = nodwghts[J] ;
            IVL_listAndSize(symbfacIVL, J, &sizeJ, &indicesJ) ;
            IVL_listAndSize(symbfacIVL, K, &sizeK, &indicesK) ;
            nLeft[K] = mergeIndices(sizeJ - nDJ, indicesJ + nDJ, 
                                    sizeK, indicesK, nLeft[K]) ;
         }
         if ( --nactiveChild[K] == 0 ) {
/*
            ----------------------
            put K on head of queue
            ----------------------
*/
            Ideq_insertAtHead(dequeue, K) ;
         }
      }
   } else {
/*
      ----------------------
      put J on tail of queue
      ----------------------
*/
      Ideq_insertAtTail(dequeue, J) ;
   }
   firstMsgSent = checkSendMessages(firstMsgSent, msglvl, msgFile) ;
}
do {
   firstMsgSent = checkSendMessages(firstMsgSent, msglvl, msgFile) ;
} while ( firstMsgSent != NULL ) ;

return(symbfacIVL) ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------
   constructor

   created -- 97nov14, cca
   -----------------------
*/
static Msg *
Msg_new (
   void
) {
Msg   *msg ;

ALLOCATE(msg, struct _Msg, 1) ;
Msg_setDefaultFields(msg) ;

return(msg) ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------
   set the default fields

   created -- 97nov14, cca
   -----------------------
*/
static void
Msg_setDefaultFields (
   Msg   *msg
) {
msg->id   =  -1  ;
msg->size =   0  ;
msg->base = NULL ;
msg->next = NULL ;

return ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------
   clear the data

   created -- 97nov14, cca
   -----------------------
*/
static void
Msg_clearData (
   Msg   *msg
) {

Msg_setDefaultFields(msg) ;

return ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------
   free the object

   created -- 97nov14, cca
   -----------------------
*/
static void
Msg_free (
   Msg   *msg
) {

Msg_clearData(msg) ;
FREE(msg) ;

return ; }

/*--------------------------------------------------------------------*/
/*
   -------------------------------------------------------------
   when front J is first examined, post Irecv's for its messages
  
   created -- 97nov14, cca
   -------------------------------------------------------------
*/
static Msg *
wakeupFront (
   int        J,
   int        myid,
   int        owners[],
   char       supportTable[],
   ETree      *frontETree,
   IVL        *symbfacIVL,
   int        firsttag,
   int        stats[],
   MPI_Comm   comm,
   int        msglvl,
   FILE       *msgFile
) {
Msg   *first, *msg ;
int   I, nfront ;
int   *bndwghts, *fch, *indices, *nodwghts, *sib ;

nfront = ETree_nfront(frontETree) ;
first = NULL ;
if ( owners[J] == myid ) {
/*
   ---------------------------------------------------
   owned front, post messages for unsupported children
   ---------------------------------------------------
*/
   if ( msglvl > 1 ) {
      fprintf(msgFile, "\n\n waking up owned front %d", J) ;
      fflush(msgFile) ;
   }
   fch      = ETree_fch(frontETree) ;
   sib      = ETree_sib(frontETree) ;
   nodwghts = ETree_nodwghts(frontETree) ;
   bndwghts = ETree_bndwghts(frontETree) ;
   for ( I = fch[J] ; I != -1 ; I = sib[I] ) {
      if ( supportTable[I + myid*nfront] != 'Y' ) {
         if ( msglvl > 1 ) {
            fprintf(msgFile, "\n    unsupported child %d", I) ;
            fflush(msgFile) ;
         }
/*
         --------------------------------------------------------
         create a message, allocate temporary storage for indices
         --------------------------------------------------------
*/
         msg = Msg_new() ;
         msg->id = I ;
         msg->size = nodwghts[I] + bndwghts[I] ;
         msg->base = (void *) IVinit(msg->size, -1) ;
         msg->next = first ;
         first     = msg ;
         if ( msglvl > 1 ) {
            fprintf(msgFile, 
              "\n    1. posting Irecv, msg %p, source = %d, tag = %d",
              msg, owners[I], firsttag + I) ;
            fflush(msgFile) ;
         }
         stats[2]++ ;
         stats[3] += msg->size * sizeof(int) ;
         MPI_Irecv(msg->base, msg->size, MPI_INT, owners[I], 
                   firsttag + I, comm, &msg->req) ;
      }
   }
} else if ( supportTable[J + myid*nfront] == 'Y' ) {
/*
   --------------------------------------------------------
   supported but not owned. create a message but point 
   into storage for the indices in the symbolic IVL object.
   --------------------------------------------------------
*/
   if ( msglvl > 1 ) {
      fprintf(msgFile, "\n\n waking up supported front %d", J) ;
      fflush(msgFile) ;
   }
   msg = Msg_new() ;
   msg->id = J ;
   IVL_listAndSize(symbfacIVL, J, &msg->size, &indices) ;
   msg->base = (void *) indices ;
   msg->next = first ;
   first     = msg ;
   if ( msglvl > 1 ) {
      fprintf(msgFile, 
        "\n    1. posting Irecv, msg %p, source = %d, tag = %d",
        msg, owners[J], firsttag + J) ;
      fflush(msgFile) ;
   }
   stats[2]++ ;
   stats[3] += msg->size * sizeof(int) ;
   MPI_Irecv(msg->base, msg->size, MPI_INT, owners[J], firsttag + J,
             comm, &msg->req) ;
}
return(first) ; }
   
/*--------------------------------------------------------------------*/
/*
   -------------------------------------------------------------
   check the messages waiting to be received for front J

   return value -- pointer to first message still to be received

   created -- 97nov14, cca
   -------------------------------------------------------------
*/
static Msg *
checkRecvMessages (
   int    J,
   Msg    *firstmsg,
   int    nLeft[],
   int    nodwghts[],
   IVL    *symbfacIVL,
   int    msglvl,
   FILE   *msgFile
) {
int          flag, I, nDI, sizeI, sizeJ ;
int          *indicesI, *indicesJ ;
MPI_Status   status ;
Msg          *msg, *nextmsg ;

if ( msglvl > 1 ) {
   fprintf(msgFile, "\n checking for received messages") ;
   fflush(msgFile) ;
}
for ( msg = firstmsg, firstmsg = NULL ; msg != NULL ; msg = nextmsg ) {
   nextmsg   = msg->next ;
   msg->next = NULL ;
   if ( msglvl > 1 ) {
      fprintf(msgFile, "\n    checking msg %p : id %d, size %d",
              msg, msg->id, msg->size) ;
      fflush(msgFile) ;
   }
   MPI_Test(&msg->req, &flag, &status) ;
   if ( msglvl > 1 ) {
      fprintf(msgFile, ", flag = %d", flag) ;
      fflush(msgFile) ;
   }
   if ( flag != 0 ) {
      if ( msg->id == J ) {
/*
         --------------------------------
         J is supported but not owned, 
         indices are in the symbolic IVL, 
         free the message
         --------------------------------
*/
         if ( msglvl > 1 ) {
            fprintf(msgFile, "\n    supported front %d", J) ;
            fflush(msgFile) ;
         }
         Msg_free(msg) ;
      } else {
/*
         ---------------------------------------------------
         message is from an unsupported child
         merge indices, then free indices, then free message
         ---------------------------------------------------
*/
         if ( msglvl > 1 ) {
            fprintf(msgFile, 
                    "\n    unsupported child front %d", msg->id) ;
            fflush(msgFile) ;
         }
         if ( nLeft[J] > 0 ) {
            I        = msg->id ;
            nDI      = nodwghts[I] ;
            sizeI    = msg->size - nDI ;
            indicesI = (int *) msg->base + nDI ;
            IVL_listAndSize(symbfacIVL, J, &sizeJ, &indicesJ) ;
            nLeft[J] = mergeIndices(sizeI, indicesI, 
                                    sizeJ, indicesJ, nLeft[J]) ;
         }
         IVfree(msg->base) ;
         Msg_free(msg) ;
      }
   } else {
/*
      ----------------------------
      keep the message on the list
      ----------------------------
*/
      msg->next = firstmsg ;
      firstmsg  = msg ;
   }
}
return(firstmsg) ; }

/*--------------------------------------------------------------------*/
/*
*/
static Msg *
sendMessages (
   int        J, 
   int        myid, 
   int        owners[], 
   int        nfront, 
   int        nproc, 
   char       supportTable[], 
   int        par[], 
   IVL        *symbfacIVL,
   Msg        *firstMsgSent,
   int        firsttag, 
   int        stats[], 
   MPI_Comm   comm,
   int        msglvl, 
   FILE       *msgFile
) {
int   destination, K, sizeJ ;
int   *indicesJ ;
Msg   *msg ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n ready to send messages") ;
   fflush(msgFile) ;
}
/*
   -------------------------------
   get size and pointer to indices
   -------------------------------
*/
IVL_listAndSize(symbfacIVL, J, &sizeJ, &indicesJ) ;
/*
   -----------------------------------
   send indices to supported processes
   -----------------------------------
*/
for ( destination = 0 ; destination < nproc ; destination++ ) {
   if (  destination != myid 
      && supportTable[J + destination*nfront] == 'Y' ) {
      msg          = Msg_new() ;
      msg->id      = J ;
      msg->size    = sizeJ ;
      msg->base    = (void *) indicesJ ;
      msg->next    = firstMsgSent ;
      firstMsgSent = msg ;
      if ( msglvl > 1 ) {
         fprintf(msgFile, 
           "\n    1. posting Isend, msg %p, destination = %d, tag = %d",
           msg, destination, firsttag + J) ;
         fflush(msgFile) ;
      }
      stats[0]++ ;
      stats[1] += msg->size * sizeof(int) ;
      MPI_Isend(msg->base, msg->size, MPI_INT, destination, 
                firsttag + J, comm, &msg->req) ;
   }
}
if ( (K = par[J]) != -1 && supportTable[J + owners[K]*nfront] != 'Y' ) {
/*
   ----------------------------------------------------
   send indices to unsupported process that owns parent
   ----------------------------------------------------
*/
   msg          = Msg_new() ;
   msg->id      = J ;
   msg->size    = sizeJ ;
   msg->base    = (void *) indicesJ ;
   msg->next    = firstMsgSent ;
   firstMsgSent = msg ;
   if ( msglvl > 1 ) {
      fprintf(msgFile, 
        "\n    2. posting Isend, msg %p, destination = %d, tag = %d",
        msg, owners[K], firsttag + J) ;
      fflush(msgFile) ;
   }
   stats[0]++ ;
   stats[1] += msg->size * sizeof(int) ;
   MPI_Isend(msg->base, msg->size, MPI_INT, owners[K], 
             firsttag + J, comm, &msg->req) ;
}
return(firstMsgSent) ; }
  
/*--------------------------------------------------------------------*/
/*
   ---------------------------------------------------------------
   merge indices in indJ[sizeJ] into indK[sizeK],
   there are nleft free locations at the end of indK[].
   both indJ[sizeJ] and indK[sizeK] are in ascending order.
   return value is the number of present free locations in indK[].
 
   created -- 97mar01, cca
   ---------------------------------------------------------------
*/
static int
mergeIndices (
   int   sizeJ,
   int   indJ[],
   int   sizeK,
   int   indK[],
   int   nleft
) {
int   ii, jj, jlast, kk, v ;
 
kk = jlast = sizeK - nleft ;
ii = 0, jj = 0 ; 
while ( ii < sizeJ && jj < jlast ) {
   v = indJ[ii] ;
   if ( v < indK[jj] ) {
      indK[kk++] = v ;
      ii++ ;
   } else if ( v == indK[jj] ) {
      ii++, jj++ ;
   } else {
      jj++ ;
   }
}
while ( ii < sizeJ ) {
   indK[kk++] = indJ[ii++] ;
}
if ( kk > jlast ) {
   IVqsortUp(kk, indK) ;
}
return(sizeK - kk) ; }
 
/*--------------------------------------------------------------------*/
/*
   ------------------------------------------------------------------
   load the owned index lists with information from the matrix pencil
 
   created -- 97jul29, cca
   ------------------------------------------------------------------
*/
static void
loadInternalIndices (
   int       myid,
   int       nfront,
   int       nvtx,
   int       frontOwners[],
   int       vtxToFront[],
   IVL       *symbfacIVL,
   DPencil   *pencil,
   int       nLeft[],
   int       msglvl,
   FILE      *msgFile
) {
DInpMtx   *inpmtxA, *inpmtxB ;
double    *vent ;
int       count, ii, J, off, sizeJ, v, vsize, w ;
int       *head, *indicesJ, *link, *mark, *vind ;
/*
   -------------------------------------------------
   set up the lists of vertices for each owned front
   -------------------------------------------------
*/
head = IVinit(nfront, -1) ;
link = IVinit(nvtx,   -1) ;
mark = IVinit(nvtx,  -1) ;
for ( v = 0 ; v < nvtx ; v++ ) {
   J = vtxToFront[v] ;
   if ( frontOwners[J] == myid ) {
      link[v] = head[J] ;
      head[J] = v ;
   }
}
/*
   --------------------------------------------------------
   load the internal indices into the owned adjacency lists
   --------------------------------------------------------
*/
inpmtxA = pencil->inpmtxA ;
inpmtxB = pencil->inpmtxB ;
for ( J = 0 ; J < nfront ; J++ ) {
   if ( frontOwners[J] == myid ) {
      IVL_listAndSize(symbfacIVL, J, &sizeJ, &indicesJ) ;
      count = 0 ;
/*
      ---------------------
      load internal indices
      ---------------------
*/
      for ( v = head[J] ; v != -1 ; v = link[v] ) {
         mark[v] = J ;
         indicesJ[count++] = v ;
      }
      if ( inpmtxA != NULL ) {
/*
         -------------------
         load indices from A
         -------------------
*/
         for ( v = head[J] ; v != -1 && count < sizeJ ; v = link[v] ) {
            DInpMtx_vector(inpmtxA, v, &vsize, &vind, &vent) ;
            for ( ii = 0 ; ii < vsize && count < sizeJ ; ii++ ) {
               off = vind[ii] ;
               if ( off >= 0 ) {
                  w = v + off ;
               } else {
                  w = v - off ;
               }
               if ( vtxToFront[w] < J ) {
                  fprintf(msgFile,
                          "\n error, J = %d, w = %d, map[%d] = %d",
                          J, w, w, vtxToFront[w]) ;
                  exit(-1) ;
               }
               if ( mark[w] != J ) {
                  mark[w] = J ;
                  indicesJ[count++] = w ;
               }
            }
         }
      }
      if ( inpmtxB != NULL ) {
/*
         -------------------
         load indices from B
         -------------------
*/
         for ( v = head[J] ; v != -1 && count < sizeJ ; v = link[v] ) {
            DInpMtx_vector(inpmtxB, v, &vsize, &vind, &vent) ;
            for ( ii = 0 ; ii < vsize && count < sizeJ ; ii++ ) {
               off = vind[ii] ;
               if ( off >= 0 ) {
                  w = v + off ;
               } else {
                  w = v - off ;
               }
               if ( vtxToFront[w] < J ) {
                  fprintf(msgFile,
                          "\n error, J = %d, w = %d, map[%d] = %d",
                          J, w, w, vtxToFront[w]) ;
                  exit(-1) ;
               }
               if ( mark[w] != J ) {
                  mark[w] = J ;
                  indicesJ[count++] = w ;
               }
            }
         }
      }
/*
      ---------------------------------------
      sort the indices in ascending order and
      set the number remaining to be filled
      ---------------------------------------
*/
      IVqsortUp(count, indicesJ) ;
      nLeft[J] = sizeJ - count ;
   }
}
/*
   ------------------------
   free the working storage
   ------------------------
*/
IVfree(head) ;
IVfree(link) ;
IVfree(mark) ;
 
return ; }
 
/*--------------------------------------------------------------------*/
/*
*/
static Msg * 
checkSendMessages (
   Msg    *firstMsgSent,
   int    msglvl,
   FILE   *msgFile
) {
int          flag ;
MPI_Status   status ;
Msg          *msg, *nextmsg ;

if ( firstMsgSent != NULL ) {
   if ( msglvl > 1 ) {
      fprintf(msgFile, "\n checking for sent messages") ;
      fflush(msgFile) ;
   }
   for ( msg = firstMsgSent, firstMsgSent = NULL ; 
         msg != NULL ; 
         msg = nextmsg ) {
      nextmsg = msg->next ;
      if ( msglvl > 1 ) {
         fprintf(msgFile, "\n    checking msg %p : id %d, size %d",
                 msg, msg->id, msg->size) ;
         fflush(msgFile) ;
      }
      MPI_Test(&msg->req, &flag, &status) ;
      if ( msglvl > 1 ) {
         fprintf(msgFile, ", flag = %d", flag) ;
         fflush(msgFile) ;
      }
      if ( flag != 0 ) {
         if ( msglvl > 1 ) {
            fprintf(msgFile, ", free'ing object") ;
            fflush(msgFile) ;
         }
         Msg_free(msg) ;
      }
   }
}

return(firstMsgSent) ; }

/*--------------------------------------------------------------------*/
