/*  splitDLinSystem.c  */

#include "../spoolesMPI.h"
#include "../../timings.h"

/*--------------------------------------------------------------------*/
/*
   ------------------------------------------------------------------
   purpose -- to split a distributed DLinSystem object

   linsys     -- pointer to the local DLinSystem object
   mapIV      -- pointer to the map from vertices to processes
   firsttag   -- first tag value, four will be used, tag through tag+3
   stats[4]    -- statistics vector
      stats[0] -- # of messages sent
      stats[1] -- # of messages received
      stats[2] -- # of bytes sent
      stats[3] -- # of bytes received
   msglvl     -- local message level
   msgFile    -- local message file
   comm       -- MPI communication structure
 
   created  -- 97jul18, cca
   modified -- 97oct17, cca
      stats added
   ------------------------------------------------------------------
*/
void
DLinSystem_MPI_split (
   DLinSystem   *linsys,
   IV           *mapIV,
   int          firsttag,
   int          stats[],
   int          msglvl,
   FILE         *msgFile,
   MPI_Comm     comm
) {
DDenseMtx   *keepmtx ;
int         flag, tag, tag_bound ;
/*
   -----------------------
   check the range of tags
   -----------------------
*/
MPI_Attr_get(MPI_COMM_WORLD, MPI_TAG_UB, &tag_bound, &flag) ;
if ( firsttag < 0 || firsttag + 3 > tag_bound ) {
   fprintf(stderr, "\n fatal error in DLinSystem_MPI_split()"
           "\n tag range = [%d,%d], tag_bound = %d",
           firsttag, firsttag + 3, tag_bound) ;
   exit(-1) ;
}
/*
   ---------------------------------------
   split the DLinSystem object into pieces
   ---------------------------------------
*/
tag = firsttag ;
if ( linsys->pencil != NULL ) {
   DPencil_MPI_split(linsys->pencil, mapIV, tag, stats,
                     msglvl, msgFile, comm) ;
   tag += 2 ;
}
if ( linsys->solmtx != NULL ) {
   keepmtx = DDenseMtx_MPI_splitByRows(linsys->solmtx, mapIV, tag, 
                            stats, msglvl, msgFile, comm) ;
   DDenseMtx_free(linsys->solmtx) ;
   linsys->solmtx = keepmtx ;
   tag++ ;
}
if ( linsys->rhsmtx != NULL ) {
   keepmtx = DDenseMtx_MPI_splitByRows(linsys->rhsmtx, mapIV, tag, 
                            stats, msglvl, msgFile, comm) ;
   DDenseMtx_free(linsys->rhsmtx) ;
   linsys->rhsmtx = keepmtx ;
   tag++ ;
}

return ; }

/*--------------------------------------------------------------------*/
