/*  solveMPI.c  */

#include "../spoolesMPI.h"
#include "../../timings.h"

/*--------------------------------------------------------------------*/
static void globalToLocal ( DFrontMtx *frontmtx, DDenseMtx *solmtx,
   DDenseMtx *rhsmtx, IV *frontOwnersIV, int myid, int msglvl,
   FILE *msgFile ) ;
static void localToGlobal ( DFrontMtx *frontmtx, DDenseMtx *solmtx,
   DDenseMtx *rhsmtx, IV *frontOwnersIV, int myid, int msglvl,
   FILE *msgFile ) ;
static void loadNearestOwnedDescendents ( Tree *tree, int seed,
   int owners[], int myid, Ideq *dequeue, int msglvl, FILE *msgFile ) ;
/*--------------------------------------------------------------------*/
typedef struct _SolveMsg   SolveMsg ;
struct _SolveMsg {
   int           info[3] ; /* type, frontid, nbytes */
   void          *base   ;
   DDenseMtx     *mtx    ;
   SolveMsg      *next   ;
   MPI_Request   req     ;
} ;
SolveMsg * SolveMsg_new ( void ) ;
SolveMsg * SolveMsg_setDefaultFields ( SolveMsg *msg ) ;
SolveMsg * SolveMsg_clearData ( SolveMsg *msg ) ;
SolveMsg * SolveMsg_free ( SolveMsg *msg ) ;
static SolveMsg * wakeup ( DFrontMtx *frontmtx, int J, char step,
   int myid, int owners[], int firsttag, DDenseMtxManager *manager,
   MPI_Comm comm, int msglvl, FILE *msgFile ) ;
static SolveMsg * checkMessages ( int J, int nfront, SolveMsg *first,
   DDenseMtxManager *manager, DDenseMtxList *mtxList, int stats[],
   MPI_Comm comm, int msglvl, FILE *msgFile ) ;
static SolveMsg * sendMessages ( DFrontMtx *frontmtx, int J, int myid,
   int owners[], char step, DDenseMtxList *mtxList, int stats[],
   int firsttag, SolveMsg *firstmsgsent, MPI_Comm comm, 
   int msglvl, FILE *msgFile ) ;
static SolveMsg * checkSentMessages ( SolveMsg *first,
   DDenseMtxManager *manager, int msglvl, FILE *msgFile ) ;
/*--------------------------------------------------------------------*/
/*
   ----------------------------------------
   solve the linear system

   cpus[] -- breakdown of cpu times
     cpus[0] -- initialize submatrices
     cpus[1] -- load rhs and solution
     cpus[2] -- assemble children/parent
     cpus[3] -- solve and update
     cpus[4] -- store rhs and solution
     cpus[5] -- store updates
     cpus[6] -- post initial receives
     cpus[7] -- check for received messages
     cpus[8] -- post sends
     cpus[9] -- check for sent messages

   created -- 97aug23, cca
   ----------------------------------------
*/
void
DFrontMtx_MPI_solve ( 
   DFrontMtx          *frontmtx,
   DDenseMtx          *solmtx,
   DDenseMtx          *rhsmtx,
   DDenseMtxManager   *manager,
   IV                 *frontOwnersIV,
   int                firsttag,
   int                stats[],
   double             cpus[],
   int                msglvl,
   FILE               *msgFile,
   MPI_Comm           comm
) {
char            *status ;
DA2             *rhsDA2, *solDA2 ;
DDenseMtx       **p_mtxBJ, **p_mtxJ ;
DDenseMtxList   *mtxList ;
double          t1, t2 ;
Ideq            *dequeue ;
int             flag, J, K, myid, neqns, nfront, nproc, nrhs, 
                ownerJ, tag_bound ;
int             *fch, *frontOwners, *ndescLeft, *par, 
                *sib ;
SolveMsg        *firstmsgsent ;
SolveMsg        **p_msg ;
Tree            *tree ;
/*
   ------------------------------
   get id and number of processes
   ------------------------------
*/
MPI_Comm_rank(comm, &myid)  ;
MPI_Comm_size(comm, &nproc) ;
IV_sizeAndEntries(frontOwnersIV, &nfront, &frontOwners) ;
tree  = frontmtx->frontETree->tree ;
par   = tree->par ;
fch   = tree->fch ;
sib   = tree->sib ;
nrhs  = rhsmtx->ncol ;
neqns = frontmtx->neqns ;
MPI_Attr_get(MPI_COMM_WORLD, MPI_TAG_UB, &tag_bound, &flag) ;
if ( firsttag + 4*nfront + 2 > tag_bound ) {
   fprintf(stderr, "\n fatal error in DFrontMtx_MPI_solve()"
           "\n tag range is [%d,%d], tag_bound = %d",
           firsttag, firsttag + 4*nfront + 2, tag_bound) ;
   exit(-1) ;
}
/*
   ----------------------
   set up the list object
   ----------------------
*/
mtxList = DFrontMtx_solveList(frontmtx, frontOwnersIV, 0) ;
/*
   --------------------------------------------
   allocate the working storage of the pointers
   --------------------------------------------
*/
ALLOCATE(p_mtxJ,  struct _DDenseMtx *, nfront) ;
ALLOCATE(p_mtxBJ, struct _DDenseMtx *, nfront) ;
ALLOCATE(p_msg,   struct _SolveMsg *,  nfront) ;
for ( J = 0 ; J < nfront ; J++ ) {
   p_mtxJ[J] = p_mtxBJ[J] = NULL ;
   p_msg[J]  = NULL ;
}
/*
   ------------------------------------
   compute the local ndescLeft[] vector
   that synchronizes the forward solve
   ------------------------------------
*/
ndescLeft = IVinit(nfront, 0) ;
for ( J = 0 ; J < nfront ; J++ ) {
   ownerJ = frontOwners[J] ;
   for ( K = par[J] ; K != -1 ; K = par[K] ) {
      if ( frontOwners[K] == ownerJ ) {
         ndescLeft[K]++ ;
         break ;
      }
   }
}
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n\n ndescLeft[]") ;
   IVfprintf(msgFile, nfront, ndescLeft) ;
   fflush(msgFile) ;
}
/*
   --------------------------------------------------------------
   overwrite the internal row and column indices with the indices
   local to the owned right hand side and solution matrix
   --------------------------------------------------------------
*/
globalToLocal(frontmtx, solmtx, rhsmtx, frontOwnersIV, myid,
              msglvl, msgFile) ;
/*
   ---------------------------------------
   set up the rhs and solution DA2 objects
   ---------------------------------------
*/
rhsDA2 = DA2_new() ;
DDenseMtx_setDA2(rhsmtx, rhsDA2) ;
solDA2 = DA2_new() ;
DDenseMtx_setDA2(solmtx, solDA2) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n rhsDA2 and solDA2 initialized") ;
   fflush(msgFile) ;
}
/*
   ----------------------------------------
   set up the dequeue for the forward solve
   ----------------------------------------
*/
status = DFrontMtx_status(frontmtx, frontOwnersIV, myid) ;
dequeue = DFrontMtx_setUpDequeue(frontmtx, frontOwnersIV, status, myid);
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n initial status vector") ;
   CVfprintf(msgFile, nfront, status) ;
   fflush(msgFile) ;
}
/*
   -------------------------
   perform the forward solve
   -------------------------
*/
firstmsgsent = NULL ;
while ( (J = Ideq_removeFromHead(dequeue)) != -1 ) {
   if ( msglvl > 1 ) {
      fprintf(msgFile, "\n\n ### forward solve on %d, status %c", 
              J, status[J]) ;
      fflush(msgFile) ;
   }
   if ( status[J] == 'W' ) {
/*
      ----------------------------------
      wake up front J, post any receives
      ----------------------------------
*/
      MARKTIME(t1) ;
      p_msg[J] = wakeup(frontmtx, J, 'L', myid, frontOwners, 
                        firsttag, manager, comm, msglvl, msgFile) ;
      MARKTIME(t2) ;
      cpus[6] += t2 - t1 ;
   }
/*
   ---------------------------
   check for received messages
   ---------------------------
*/
   MARKTIME(t1) ;
   p_msg[J] = checkMessages(J, nfront, p_msg[J], manager, mtxList, 
                            stats, comm, msglvl, msgFile) ;
   MARKTIME(t2) ;
   cpus[7] += t2 - t1 ;
/*
   ------------------------------------------
   try to do the forward solve for this front
   ------------------------------------------
*/
   DFrontMtx_parallelForwardSolve(frontmtx, rhsDA2, J, manager, 
       mtxList, p_mtxJ, p_mtxBJ, status, cpus, msglvl, msgFile) ;
   if ( status[J] == 'F' ) {
/*
      --------------------------
      check for messages to send
      --------------------------
*/
      MARKTIME(t1) ;
      firstmsgsent = sendMessages(frontmtx, J, myid, frontOwners, 'L', 
                                  mtxList, stats, firsttag, 
                                  firstmsgsent, comm, msglvl, msgFile) ;
      MARKTIME(t2) ;
      cpus[8] += t2 - t1 ;
/*
      --------------------------------
      look for the next owned ancestor
      --------------------------------
*/
      K = par[J] ;
      while ( K != -1 && frontOwners[K] != myid ) {
         K = par[K] ;
      }
      if ( K != -1 && --ndescLeft[K] == 0 ) {
/*
         -----------------------------------------------------------
         next owned ancestor exists and all of its owned descendents 
         are finished, place K on the head of the dequeue
         -----------------------------------------------------------
*/
         if ( msglvl > 1 ) {
            fprintf(msgFile, "\n placing %d on head of dequeue", K) ;
            fflush(msgFile) ;
         }
         Ideq_insertAtHead(dequeue, K) ;
      }
   } else {
/*
      -------------------------------------------
      front J is not done, place on tail of queue
      -------------------------------------------
*/
      if ( msglvl > 1 ) {
         fprintf(msgFile, "\n placing %d on tail of dequeue", J) ;
         fflush(msgFile) ;
      }
      Ideq_insertAtTail(dequeue, J) ;
   }
/*
   -------------------------
   check for completed sends
   -------------------------
*/
   MARKTIME(t1) ;
   firstmsgsent = checkSentMessages(firstmsgsent, manager, 
                                    msglvl, msgFile) ;
   MARKTIME(t2) ;
   cpus[9] += t2 - t1 ;
}
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n after the forward solve") ;
   DDenseMtx_writeForHumanEye(rhsmtx, msgFile) ;
   fflush(msgFile) ;
}
/*
   ------------------------------------------------------------
   load the dequeue with the fronts to start the backward solve
   ------------------------------------------------------------
*/
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n loading highest owned ancestors") ;
   fflush(msgFile) ;
}
loadNearestOwnedDescendents(tree, -1, frontOwners, myid, dequeue,
                            msglvl, msgFile) ;
CVfill(nfront, status, 'W') ;
while ( (J = Ideq_removeFromHead(dequeue)) != -1 ) {
   if ( msglvl > 1 ) {
      fprintf(msgFile, "\n\n ### backward solve for %d, status = %c",
              J, status[J]) ;
      fflush(msgFile) ;
   }
   if ( status[J] == 'W' ) {
/*
      ----------------------------------
      wake up front J, post any receives
      ----------------------------------
*/
      MARKTIME(t1) ;
      p_msg[J] = wakeup(frontmtx, J, 'U', myid, frontOwners, 
                        firsttag, manager, comm, msglvl, msgFile) ;
      MARKTIME(t2) ;
      cpus[6] += t2 - t1 ;
   }
/*
   ---------------------------
   check for received messages
   ---------------------------
*/
   MARKTIME(t1) ;
   p_msg[J] = checkMessages(J, nfront, p_msg[J], manager, mtxList, 
                            stats, comm, msglvl, msgFile) ;
   MARKTIME(t2) ;
   cpus[7] += t2 - t1 ;
/*
   -------------------------------------------
   try to do the backward solve for this front
   -------------------------------------------
*/
   DFrontMtx_parallelBackwardSolve(frontmtx, rhsDA2, solDA2, J, 
       manager, mtxList, p_mtxJ, status, cpus, msglvl, msgFile) ;
   if ( status[J] == 'F' ) {
/*
      --------------------------
      check for messages to send
      --------------------------
*/
      MARKTIME(t1) ;
      firstmsgsent = sendMessages(frontmtx, J, myid, frontOwners, 'U', 
                                  mtxList, stats, firsttag, 
                                  firstmsgsent, comm, msglvl, msgFile) ;
      MARKTIME(t2) ;
      cpus[8] += t2 - t1 ;
/*
      --------------------------------------------
      place nearest owned descendents on the queue
      --------------------------------------------
*/
      if ( msglvl > 1 ) {
         fprintf(msgFile, 
                 "\n\n loading nearest owned descendents of %d", J) ;
         fflush(msgFile) ;
      }
      loadNearestOwnedDescendents(tree, J, frontOwners, myid,
                                  dequeue, msglvl, msgFile) ;
   } else {
/*
      -------------------------------------------------
      front is not yet complete, put J on tail of queue
      -------------------------------------------------
*/
      if ( msglvl > 1 ) {
         fprintf(msgFile, 
            "\n front %d not complete, move to tail of dequeue", J) ;
         fflush(msgFile) ;
      }
      Ideq_insertAtTail(dequeue, J) ;
   }
/*
   -------------------------
   check for completed sends
   -------------------------
*/
   MARKTIME(t1) ;
   firstmsgsent = checkSentMessages(firstmsgsent, manager, 
                                    msglvl, msgFile) ;
   MARKTIME(t2) ;
   cpus[9] += t2 - t1 ;
}
/*
   ----------------------------------------------------------
   map the row and column indices back to their global values
   ----------------------------------------------------------
*/
localToGlobal(frontmtx, solmtx, rhsmtx, frontOwnersIV, myid,
              msglvl, msgFile) ;
/*
   ------------------------
   free the working storage
   ------------------------
*/
Ideq_free(dequeue) ;
CVfree(status) ;
FREE(p_mtxJ) ;
FREE(p_mtxBJ) ;
DDenseMtxList_free(mtxList) ;
IVfree(ndescLeft) ;

return ; }

/*--------------------------------------------------------------------*/
/*
   --------------------------------------------------------------
   overwrite the internal row and column indices with the indices
   local to the owned right hand side and solution matrix

   created -- 97sep12, cca
   --------------------------------------------------------------
*/
static void
globalToLocal (
   DFrontMtx   *frontmtx,
   DDenseMtx   *solmtx,
   DDenseMtx   *rhsmtx,
   IV          *frontOwnersIV, 
   int         myid,
   int         msglvl,
   FILE        *msgFile
) {
int   ii, J, ncolJ, nDJ, neqns, nfront, nrow, nrowJ ;
int   *colindJ, *frontOwners, *map, *rowind, *rowindJ ;

IV_sizeAndEntries(frontOwnersIV, &nfront, &frontOwners) ;
neqns  = frontmtx->neqns  ;
/*
   -----------------------------------------------------------------
   overwrite the internal row and column indices of the owned fronts
   with the indices local to the owned rhs and solution entries
   -----------------------------------------------------------------
*/
map = IVinit(neqns, -1) ;
DDenseMtx_rowIndices(solmtx, &nrow, &rowind) ;
for ( ii = 0 ; ii < nrow ; ii++ ) {
   map[rowind[ii]] = ii ;
}
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n\n map[] for solmtx") ;
   IVfprintf(msgFile, neqns, map) ;
   fflush(msgFile) ;
}
for ( J = 0 ; J < nfront ; J++ ) {
   if ( msglvl > 2 ) {
      fprintf(msgFile, "\n\n front %d, owner %d", J, frontOwners[J]) ;
      fflush(msgFile) ;
   }
   if ( frontOwners[J] == myid ) {
      nDJ = DFrontMtx_frontSize(frontmtx, J) ;
      DFrontMtx_columnIndices(frontmtx, J, &ncolJ, &colindJ) ;
      if ( msglvl > 2 ) {
         fprintf(msgFile, "\n column indices for front %d", J) ;
         fprintf(msgFile, ", nDJ = %d, colindJ = %p", nDJ, colindJ) ;
         fflush(msgFile) ;
         IVfprintf(msgFile, nDJ, colindJ) ;
         fflush(msgFile) ;
      }
      IVgather(nDJ, colindJ, map, colindJ) ;
      if ( msglvl > 2 ) {
         fprintf(msgFile, "\n front %d, mapped colindJ", J) ;
         IVfprintf(msgFile, nDJ, colindJ) ;
         fflush(msgFile) ;
      }
   }
}
if ( frontmtx->symmetryflag == 2 && frontmtx->pivotingflag == 1 ) {
   DDenseMtx_rowIndices(rhsmtx, &nrow, &rowind) ;
   for ( ii = 0 ; ii < nrow ; ii++ ) {
      map[rowind[ii]] = ii ;
   }
   if ( msglvl > 2 ) {
      fprintf(msgFile, "\n\n map[] for rhsmtx") ;
      IVfprintf(msgFile, neqns, map) ;
      fflush(msgFile) ;
   }
   for ( J = 0 ; J < nfront ; J++ ) {
      if ( frontOwners[J] == myid ) {
         nDJ = DFrontMtx_frontSize(frontmtx, J) ;
         DFrontMtx_rowIndices(frontmtx, J, &nrowJ, &rowindJ) ;
         IVgather(nDJ, rowindJ, map, rowindJ) ;
         if ( msglvl > 2 ) {
            fprintf(msgFile, "\n front %d, mapped rowindJ", J) ;
            IVfprintf(msgFile, nDJ, rowindJ) ;
            fflush(msgFile) ;
         }
      }
   }
}
IVfree(map) ;

return ; }

/*--------------------------------------------------------------------*/
/*
   ----------------------------------------------------------
   map the row and column indices back to their global values

   created -- 97sep12, cca
   ----------------------------------------------------------
*/
static void
localToGlobal (
   DFrontMtx   *frontmtx,
   DDenseMtx   *solmtx,
   DDenseMtx   *rhsmtx,
   IV          *frontOwnersIV, 
   int         myid,
   int         msglvl,
   FILE        *msgFile
) {
int   ii, J, ncolJ, nDJ, neqns, nfront, nrow, nrowJ ;
int   *colindJ, *frontOwners, *map, *rowind, *rowindJ ;

IV_sizeAndEntries(frontOwnersIV, &nfront, &frontOwners) ;
neqns  = frontmtx->neqns  ;
map = IVinit(neqns, -1) ;
if ( frontmtx->symmetryflag == 2 && frontmtx->pivotingflag == 1 ) {
   DDenseMtx_rowIndices(rhsmtx, &nrow, &rowind) ;
   for ( ii = 0 ; ii < nrow ; ii++ ) {
      map[rowind[ii]] = ii ;
   }
   for ( J = 0 ; J < nfront ; J++ ) {
      if ( frontOwners[J] == myid ) {
         nDJ = DFrontMtx_frontSize(frontmtx, J) ;
         DFrontMtx_rowIndices(frontmtx, J, &nrowJ, &rowindJ) ;
         IVgather(nDJ, rowindJ, rowind, rowindJ) ;
      }
   }
   for ( ii = 0 ; ii < nrow ; ii++ ) {
      map[rowind[ii]] = -1 ;
   }
}
DDenseMtx_rowIndices(solmtx, &nrow, &rowind) ;
for ( ii = 0 ; ii < nrow ; ii++ ) {
   map[rowind[ii]] = ii ;
}
for ( J = 0 ; J < nfront ; J++ ) {
   if ( frontOwners[J] == myid ) {
      nDJ = DFrontMtx_frontSize(frontmtx, J) ;
      DFrontMtx_columnIndices(frontmtx, J, &ncolJ, &colindJ) ;
      IVgather(nDJ, colindJ, rowind, colindJ) ;
   }
}
for ( ii = 0 ; ii < nrow ; ii++ ) {
   map[rowind[ii]] = -1 ;
}
IVfree(map) ;

return ; }

/*--------------------------------------------------------------------*/
/*
   ---------------------------------------------
   load the dequeue with the closest descendents
   of seed that are owned by thread myid
 
   created -- 97jun27, cca
   ---------------------------------------------
*/
static void
loadNearestOwnedDescendents (
   Tree   *tree,
   int    seed,
   int    owners[],
   int    myid,
   Ideq   *dequeue,
   int    msglvl,
   FILE   *msgFile
) {
int   I ;
int   *fch = tree->fch ;
int   *par = tree->par ;
int   *sib = tree->sib ;
 
if ( seed != -1 ) {
   I = fch[seed] ;
} else {
   I = tree->root ;
}
while ( I != -1 ) {
   if ( owners[I] == myid ) {
      if ( msglvl > 1 ) {
         fprintf(msgFile, "\n loading descendent %d onto queue", I) ;
         fflush(msgFile) ;
      }
      Ideq_insertAtHead(dequeue, I) ;
      while ( sib[I] == -1 && par[I] != seed ) {
         I = par[I] ;
      }
      I = sib[I] ;
   } else {
      if ( fch[I] != -1 ) {
         I = fch[I] ;
      } else {
         while ( sib[I] == -1 && par[I] != seed ) {
            I = par[I] ;
         }
         I = sib[I] ;
      }
   }
}
return ; }
 
/*--------------------------------------------------------------------*/
/*
   -----------------------
   constructor

   created -- 97nov15, cca
   -----------------------
*/
SolveMsg *
SolveMsg_new (
   void
) {
SolveMsg   *msg ;

ALLOCATE(msg, struct _SolveMsg, 1) ;
SolveMsg_setDefaultFields(msg) ;

return(msg) ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------
   set the default fields

   created -- 97nov15, cca
   -----------------------
*/
SolveMsg *
SolveMsg_setDefaultFields (
   SolveMsg   *msg
) {
msg->info[0] =   0  ;
msg->info[1] =  -1  ;
msg->info[2] =   0  ;
msg->base    = NULL ;
msg->mtx     = NULL ;
msg->next    = NULL ;
msg->req     = NULL ;

return(msg) ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------
   clear the data

   created -- 97nov15, cca
   -----------------------
*/
SolveMsg *
SolveMsg_clearData (
   SolveMsg   *msg
) {
SolveMsg_setDefaultFields(msg) ;

return(msg) ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------
   free the object

   created -- 97nov15, cca
   -----------------------
*/
SolveMsg *
SolveMsg_free (
   SolveMsg   *msg
) {
SolveMsg_clearData(msg) ;
FREE(msg) ;

return(msg) ; }

/*--------------------------------------------------------------------*/
/*
*/
static SolveMsg *
wakeup (
   DFrontMtx          *frontmtx,
   int                J,
   char               step,
   int                myid,
   int                owners[],
   int                firsttag,
   DDenseMtxManager   *manager,
   MPI_Comm           comm,
   int                msglvl,
   FILE               *msgFile
) {
int        I, K, source, tag ;
int        *fch, *par, *sib ;
SolveMsg   *first, *msg ;

first = NULL ;
if ( step == 'L' ) {
   if ( msglvl > 1 ) {
      fprintf(msgFile, "\n wakeup %d in forward mode", J) ;
      fflush(msgFile) ;
   }
/*
   -------------------------------------------------------
   post receives for partial updates from unowned children
   -------------------------------------------------------
*/
   fch = ETree_fch(frontmtx->frontETree) ;
   sib = ETree_sib(frontmtx->frontETree) ;
   for ( I = fch[J] ; I != -1 ; I = sib[I] ) {
      if ( msglvl > 1 ) {
         fprintf(msgFile, "\n    child %d, owner %d", I, owners[I]) ;
         fflush(msgFile) ;
      }
      if ( (source = owners[I]) != myid ) {
         msg = SolveMsg_new() ;
         msg->info[0] = 1 ;
         msg->info[1] = I ;
         msg->info[2] = 0 ;
         msg->base    = (void *) msg->info ;
         msg->next    = first ;
         first        = msg ;
         tag          = firsttag + I ;
         if ( msglvl > 1 ) {
            fprintf(msgFile, 
                   "\n    posting Irecv, msg %p, type %d, I %d, tag %d",
                   msg, 1, I, tag) ;
            fflush(msgFile) ;
         }
         MPI_Irecv(msg->base, 3, MPI_INT, source, tag,
                   comm, &msg->req) ;
         if ( msglvl > 1 ) {
            fprintf(msgFile, ", return") ;
            fflush(msgFile) ;
         }
      }
   }
} else if ( step == 'U' ) {
   par = ETree_par(frontmtx->frontETree) ;
   if ( (K = par[J]) != -1 && (source = owners[K]) != myid ) {
/*
      -------------------------------------
      post receive for solution from parent
      -------------------------------------
*/
      msg = SolveMsg_new() ;
      msg->info[0] = 3 ;
      msg->info[1] = K ;
      msg->info[2] = 0 ;
      msg->base    = (void *) msg->info ;
      msg->next    = first ;
      first        = msg ;
      tag          = firsttag + 2*frontmtx->nfront + K ;
      if ( msglvl > 1 ) {
         fprintf(msgFile, 
                 "\n    posting Irecv, msg %p, type %d, K %d, tag %d",
                 msg, 1, K, tag) ;
         fflush(msgFile) ;
      }
      MPI_Irecv(msg->base, 3, MPI_INT, source, tag, comm, &msg->req) ;
      if ( msglvl > 1 ) {
         fprintf(msgFile, ", return") ;
         fflush(msgFile) ;
      }
   }
}
return(first) ; }

/*--------------------------------------------------------------------*/
/*
*/
static SolveMsg *
checkMessages (
   int                J,
   int                nfront,
   SolveMsg           *first,
   DDenseMtxManager   *manager,
   DDenseMtxList      *mtxList,
   int                stats[],
   MPI_Comm           comm,
   int                msglvl,
   FILE               *msgFile
) {
int          flag, I, K, nbytes, source, tag, type ;
MPI_Status   status ;
SolveMsg     *msg, *nextmsg ;

for ( msg = first, first = NULL ; msg != NULL ; msg = nextmsg ) {
/*
   ------------------------
   set link to next message
   ------------------------
*/
   nextmsg   = msg->next ;
   msg->next = NULL ;
/*
   --------------------------------------
   extract the message type and frontid, 
   test to see if the receive is complete 
   --------------------------------------
*/
   type = msg->info[0] ;
   if ( msglvl > 1 ) {
      fprintf(msgFile, 
              "\n    checking message %p : type %d, frontid %d", 
              msg, type, msg->info[1]) ;
      fflush(msgFile) ;
   }
   MPI_Test(&msg->req, &flag, &status) ;
   if ( msglvl > 1 ) {
      fprintf(msgFile, ", flag %d", flag) ;
      fflush(msgFile) ;
   }
   if ( flag != 0 ) {
/*
      ------------------------------------------------
      message has been received, increment statistics,
      extract source, tag, and # of bytes
      ------------------------------------------------
*/
      nbytes = msg->info[2]      ;
      source = status.MPI_SOURCE ;
      tag    = status.MPI_TAG    ;
      if ( msglvl > 1 ) {
         fprintf(msgFile, 
                 "\n    message received, source %d, tag %d, nbytes %d"
                 "\n    info = [ %d %d %d ] ",
                 source, tag, nbytes,
                 msg->info[0], msg->info[1], msg->info[2]) ;
         fflush(msgFile) ;
      }
      switch ( type ) {
      case 1 :
/*
         --------------------
         notification message
         --------------------
*/
         I = msg->info[1] ;
         stats[2]++ ;
         stats[3] += 3*sizeof(int) ;
         if ( nbytes > 0 ) {
/*
            -------------------------------------
            post new receive for DDenseMtx object
            -------------------------------------
*/
            msg->info[0] = 2 ;
            msg->mtx  = DDenseMtxManager_newObjectOfSizeNbytes(manager, 
                                                               nbytes) ;
            msg->base = (void *) DDenseMtx_workspace(msg->mtx) ;
            msg->next = nextmsg ;
            nextmsg   = msg ;
            tag += nfront ;
            if ( msglvl > 1 ) {
               fprintf(msgFile, 
                 "\n    posting Irecv, msg %p, type %d, I %d, tag %d",
                 msg, 2, I, tag) ;
               fflush(msgFile) ;
            }
            MPI_Irecv(msg->base, nbytes, MPI_BYTE, source, tag,
                      comm, &msg->req) ;
            if ( msglvl > 1 ) {
               fprintf(msgFile, ", return") ;
               fflush(msgFile) ;
            }
         }
         break ;
      case 2 :
/*
         --------------
         matrix message
         --------------
*/
         stats[2]++ ;
         stats[3] += nbytes ;
         DDenseMtx_initFromBuffer(msg->mtx) ;
         if ( msglvl > 1 ) {
            fprintf(msgFile, "\n    adding mtx to list %d", J) ;
            fflush(msgFile) ;
         }
         DDenseMtxList_addObjectToList(mtxList, msg->mtx, J) ;
         SolveMsg_free(msg) ;
         break ;
      case 3 :
/*
         --------------------
         notification message
         --------------------
*/
         K = msg->info[1] ;
         stats[2]++ ;
         stats[3] += 3*sizeof(int) ;
         if ( nbytes > 0 ) {
/*
            -------------------------------------
            post new receive for DDenseMtx object
            -------------------------------------
*/
            msg->info[0] = 4 ;
            msg->mtx  = DDenseMtxManager_newObjectOfSizeNbytes(manager, 
                                                               nbytes) ;
            msg->base = (void *) DDenseMtx_workspace(msg->mtx) ;
            msg->next = nextmsg ;
            nextmsg   = msg ;
            tag += nfront ;
            if ( msglvl > 1 ) {
               fprintf(msgFile, 
                 "\n    posting Irecv, msg %p, type %d, K %d, tag %d",
                 msg, 2, K, tag) ;
               fflush(msgFile) ;
            }
            MPI_Irecv(msg->base, nbytes, MPI_BYTE, source, tag,
                      comm, &msg->req) ;
            if ( msglvl > 1 ) {
               fprintf(msgFile, ", return") ;
               fflush(msgFile) ;
            }
         }
         break ;
      case 4 :
/*
         --------------
         matrix message
         --------------
*/
         stats[2]++ ;
         stats[3] += nbytes ;
         DDenseMtx_initFromBuffer(msg->mtx) ;
         if ( msglvl > 1 ) {
            fprintf(msgFile, "\n    adding mtx to list %d", J) ;
            fflush(msgFile) ;
         }
         DDenseMtxList_addObjectToList(mtxList, msg->mtx, J) ;
         SolveMsg_free(msg) ;
         break ;
      default :
         break ;
      }
   } else {
      msg->next = first ;
      first = msg ;
   }
}
return(first) ; }

/*--------------------------------------------------------------------*/
/*
*/
static SolveMsg *
sendMessages (
   DFrontMtx       *frontmtx,
   int             J,
   int             myid,
   int             owners[],
   char            step,
   DDenseMtxList   *mtxList,
   int             stats[],
   int             firsttag,
   SolveMsg        *firstmsg,
   MPI_Comm        comm,
   int             msglvl,
   FILE            *msgFile
) {
SolveMsg   *msg ;

if ( step == 'L' ) {
   DDenseMtx   *mtx ;
   int         destination, K, nbytes, ncol, nrow, tag ;
   int         *par ;

   par = ETree_par(frontmtx->frontETree) ;
   if ( (K = par[J]) != -1 && (destination = owners[K]) != myid ) {
/*
      -----------------------------------
      get the matrix object from K's list
      -----------------------------------
*/
      mtx = DDenseMtxList_getList(mtxList, K) ;
      DDenseMtx_dimensions(mtx, &nrow, &ncol) ;
      nbytes = DDenseMtx_nbytesNeeded(nrow, ncol) ;
/*
      -------------------------
      send notification message
      -------------------------
*/
      msg = SolveMsg_new() ;
      msg->info[0] = 1 ;
      msg->info[1] = J ;
      msg->info[2] = nbytes ;
      msg->base    = (void *) msg->info ;
      msg->mtx     = mtx ;
      tag          = firsttag + J ;
      msg->next    = firstmsg ;
      firstmsg     = msg   ;
      if ( msglvl > 1 ) {
         fprintf(msgFile, 
"\n    posting Isend, msg %p, type %d, J %d, dest %d, tag %d, nbytes %d"
"\n    info = [ %d %d %d ]",
          msg, 1, J, destination, tag, nbytes,
          msg->info[0], msg->info[1], msg->info[2]) ;
         fflush(msgFile) ;
      }
      MPI_Isend(msg->base, 3, MPI_INT, destination, tag, 
                comm, &msg->req) ;
      stats[0]++ ;
      stats[1] += 3*sizeof(int) ;
/*
      -------------------------
      send message with matrix
      -------------------------
*/
      msg = SolveMsg_new() ;
      msg->info[0] = 2 ;
      msg->info[1] = J ;
      msg->info[2] = nbytes ;
      msg->mtx     = mtx ;
      msg->base    = (void *) DDenseMtx_workspace(msg->mtx) ;
      tag          = tag + frontmtx->nfront ;
      msg->next    = firstmsg ;
      firstmsg     = msg   ;
      if ( msglvl > 1 ) {
         fprintf(msgFile, 
"\n    posting Isend, msg %p, type %d, J %d, dest %d, tag %d, nbytes %d"
"\n    info = [ %d %d %d ]",
          msg, 2, J, destination, tag, nbytes,
          msg->info[0], msg->info[1], msg->info[2]) ;
         fflush(msgFile) ;
      }
      MPI_Isend(msg->base, nbytes, MPI_BYTE, destination, tag, 
                comm, &msg->req) ;
      stats[0]++ ;
      stats[1] += nbytes ;
   }
} else if ( step == 'U' ) {
   DDenseMtx   *mtx ;
   int         destination, I, nbytes, ncol, nrow, tag ;
   int         *fch, *sib ;

   fch = ETree_fch(frontmtx->frontETree) ;
   sib = ETree_sib(frontmtx->frontETree) ;
   for ( I = fch[J] ; I != -1 ; I = sib[I] ) {
      if ( (destination = owners[I]) != myid ) {
/*
         -----------------------------------
         get the matrix object from I's list
         -----------------------------------
*/
         mtx = DDenseMtxList_getList(mtxList, I) ;
         DDenseMtx_dimensions(mtx, &nrow, &ncol) ;
         nbytes = DDenseMtx_nbytesNeeded(nrow, ncol) ;
/*
         -------------------------
         send notification message
         -------------------------
*/
         msg = SolveMsg_new() ;
         msg->info[0] = 3 ;
         msg->info[1] = J ;
         msg->info[2] = nbytes ;
         msg->base    = (void *) msg->info ;
         msg->mtx     = mtx ;
         tag          = firsttag + 2*frontmtx->nfront + J ;
         msg->next    = firstmsg ;
         firstmsg     = msg   ;
         if ( msglvl > 1 ) {
            fprintf(msgFile, 
"\n    posting Isend, msg %p, type %d, J %d, dest %d, tag %d, nbytes %d"
"\n    info = [ %d %d %d ]",
          msg, 3, J, destination, tag, nbytes,
          msg->info[0], msg->info[1], msg->info[2]) ;
            fflush(msgFile) ;
         }
         MPI_Isend(msg->base, 3, MPI_INT, destination, tag, 
                   comm, &msg->req) ;
         stats[0]++ ;
         stats[1] += 3*sizeof(int) ;
/*
         -------------------------
         send message with matrix
         -------------------------
*/
         msg = SolveMsg_new() ;
         msg->info[0] = 4 ;
         msg->info[1] = J ;
         msg->info[2] = nbytes ;
         msg->mtx     = mtx ;
         msg->base    = (void *) DDenseMtx_workspace(msg->mtx) ;
         tag          = tag + frontmtx->nfront ;
         msg->next    = firstmsg ;
         firstmsg     = msg   ;
         if ( msglvl > 1 ) {
            fprintf(msgFile, 
"\n    posting Isend, msg %p, type %d, J %d, dest %d, tag %d, nbytes %d"
"\n    info = [ %d %d %d ]",
          msg, 4, J, destination, tag, nbytes,
          msg->info[0], msg->info[1], msg->info[2]) ;
            fflush(msgFile) ;
         }
         MPI_Isend(msg->base, nbytes, MPI_BYTE, destination, tag, 
                   comm, &msg->req) ;
         stats[0]++ ;
         stats[1] += nbytes ;
      }
   }
}
return(firstmsg) ; }

/*--------------------------------------------------------------------*/
/*
*/
static SolveMsg *
checkSentMessages (
   SolveMsg           *first,
   DDenseMtxManager   *manager,
   int                msglvl,
   FILE               *msgFile
) {
int          flag, frontid, nbytes, type ;
MPI_Status   status ;
SolveMsg     *msg, *nextmsg ;

for ( msg = first, first = NULL ; msg != NULL ; msg = nextmsg ) {
   nextmsg = msg->next ;
   msg->next = NULL ;
/*
   -------------------
   test for completion
   -------------------
*/
   type    = msg->info[0] ;
   frontid = msg->info[1] ;
   nbytes  = msg->info[2] ;
   if ( msglvl > 1 ) {
      fprintf(msgFile, 
              "\n checking sent msg %p : type %d, front %d, nbytes %d",
              msg, type, frontid, nbytes) ;
      fflush(msgFile) ;
   }
   MPI_Test(&msg->req, &flag, &status) ;
   if ( msglvl > 1 ) {
      fprintf(msgFile, ", flag %d", flag) ;
      fflush(msgFile) ;
   }
   if ( flag == 1 ) {
      switch ( type ) {
      case 1 :
      case 3 :
         SolveMsg_free(msg) ;
         break ;
      case 2 :
      case 4 :
         if ( msg->mtx != NULL ) {
            if ( msglvl > 1 ) {
               fprintf(msgFile, "\n mtx %p release", msg->mtx) ;
               fflush(msgFile) ;
            }
            DDenseMtxManager_releaseObject(manager, msg->mtx) ;
         }
         SolveMsg_free(msg) ;
         break ;
      default :
         break ;
      }
   } else {
      msg->next = first ;
      first     = msg   ;
   }
}
return(first) ; }

/*--------------------------------------------------------------------*/
