/*  MPIorderViaMMD.c  */

#include "../spoolesMPI.h"
#include "../../misc.h"

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------------------------------
   this is the MPI version of the orderViaMMD() method.
   the difference is that we start from a distributed DInpMtx object
   instead of a global Graph object.

   created -- 97dec17, cca
   -----------------------------------------------------------------
*/
ETree *
DInpMtx_MPI_orderViaMMD (
   DInpMtx    *inpmtx,
   int        maxzeros,
   int        maxfrontsize,
   int        seed,
   int        stats[],
   int        msglvl,
   FILE       *msgFile,
   MPI_Comm   comm
) {
DPencil   pencil ;
ETree     *frontETree ;
/*
   ---------------
   check the input
   ---------------
*/
if ( inpmtx == NULL || maxzeros < 0 || maxfrontsize < 0
   || stats == NULL || (msglvl > 0 && msgFile == NULL) ) {
   fprintf(stderr, "\n fatal error in DInpMtx_MPI_orderViaMMD()"
           "\n bad input"
           "\n inpmtx = %p, maxzeros = %d, maxfrontsize = %d"
           "\n stats = %p, msglvl = %d, msgFile = %p\n",
           inpmtx, maxzeros, maxfrontsize, stats, msglvl, msgFile) ;
   exit(-1) ;
}

DPencil_setDefaultFields(&pencil) ;
DPencil_init(&pencil, inpmtx, 0.0, NULL) ;
frontETree = DPencil_MPI_orderViaMMD(&pencil, maxzeros, maxfrontsize,
                                   seed, stats, msglvl, msgFile, comm) ;

return(frontETree) ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------------------------------
   this is the MPI version of the orderViaMMD() method.
   the difference is that we start from a distributed DPencil object
   instead of a global Graph object.

   created -- 97dec17, cca
   -----------------------------------------------------------------
*/
ETree *
DPencil_MPI_orderViaMMD (
   DPencil    *pencil,
   int        maxzeros,
   int        maxfrontsize,
   int        seed,
   int        stats[],
   int        msglvl,
   FILE       *msgFile,
   MPI_Comm   comm
) {
double   minops ;
double   *opcounts ;
ETree    *frontETree, *newFrontETree ;
Graph    *graph ;
int      jproc, myid, nfront, nproc, nvtx, root ;
int      *buffer ;
IVL      *adjIVL ;
/*
   ---------------
   check the input
   ---------------
*/
if ( pencil == NULL || maxzeros < 0 || maxfrontsize < 0
   || stats == NULL || (msglvl > 0 && msgFile == NULL) ) {
   fprintf(stderr, "\n fatal error in DPencil_MPI_orderViaMMD()"
           "\n bad input"
           "\n pencil = %p, maxzeros = %d, maxfrontsize = %d"
           "\n stats = %p, msglvl = %d, msgFile = %p\n",
           pencil, maxzeros, maxfrontsize, stats, msglvl, msgFile) ;
   exit(-1) ;
}
/*
   --------------------------------------
   get id of self and number of processes
   --------------------------------------
*/
MPI_Comm_rank(comm, &myid)  ;
MPI_Comm_size(comm, &nproc) ;
/*
   --------------------------------
   get the full adjacency structure
   --------------------------------
*/
adjIVL = DPencil_MPI_fullAdjacency(pencil, stats, 
                                   msglvl, msgFile, comm) ;
nvtx = IVL_nlist(adjIVL) ;
/*
   ---------------------------
   initialize the graph object
   ---------------------------
*/
graph = Graph_new() ;
Graph_init2(graph, 0, nvtx, 0, adjIVL->tsize, nvtx, adjIVL->tsize,
           adjIVL, NULL, NULL) ;
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n\n Graph object") ;
   Graph_writeForHumanEye(graph, msgFile) ;
   fflush(msgFile) ;
}
/*
   ------------------------
   order via minimum degree
   ------------------------
*/
frontETree = orderViaMMD(graph, seed, msglvl, msgFile) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, 
          "\n\n original front tree, nfront = %d, nzf = %d, ops = %.0f",
          ETree_nfront(frontETree),
          ETree_nFactorEntries(frontETree, 2),
          ETree_nFactorOps(frontETree, 2)) ;
   fflush(msgFile) ;
}
if ( msglvl > 2 ) {
   ETree_writeForHumanEye(frontETree, msgFile) ;
   fflush(msgFile) ;
}
/*
   ---------------------
   free the Graph object
   ---------------------
*/
Graph_free(graph) ;
/*
   ------------------------------
   get the transformed front tree
   ------------------------------
*/
newFrontETree = ETree_transform(frontETree, NULL, 
                                maxzeros, maxfrontsize, seed) ;
ETree_free(frontETree) ;
frontETree = newFrontETree ;
nvtx = ETree_nvtx(frontETree) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, 
       "\n\n transformed front tree, nfront = %d, nzf = %d, ops = %.0f",
       ETree_nfront(frontETree),
       ETree_nFactorEntries(frontETree, 2),
       ETree_nFactorOps(frontETree, 2)) ;
   fflush(msgFile) ;
}
if ( msglvl > 2 ) {
   ETree_writeForHumanEye(frontETree, msgFile) ;
   fflush(msgFile) ;
}
/*
   -----------------------------------------------
   exchange op counts with all the other processes
   -----------------------------------------------
*/
opcounts = DVinit(nproc, 0.0) ;
opcounts[myid] = ETree_nFactorOps(frontETree, 2) ;
MPI_Allgather((void *) &opcounts[myid], 1, MPI_DOUBLE,
              (void *) opcounts, 1, MPI_DOUBLE, comm) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n opcounts") ;
   DVfprintf(msgFile, nproc, opcounts) ;
   fflush(msgFile) ;
}
/*
   --------------------------------------------
   determine which process has the best opcount
   --------------------------------------------
*/
minops = DVmin(nproc, opcounts, &jproc) ;
if ( jproc == myid ) {
   nfront = ETree_nfront(frontETree) ;
   root   = ETree_root(frontETree) ;
}
MPI_Bcast((void *) &nfront, 1, MPI_INT, jproc, comm) ;
MPI_Bcast((void *) &root,   1, MPI_INT, jproc, comm) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n jproc = %d, nfront = %d, root = %d",
           jproc, nfront, root) ;
   fflush(msgFile) ;
}
buffer = IVinit(5*nfront + nvtx, -1) ;
if ( jproc == myid ) {
   IVcopy(nfront, buffer,            ETree_par(frontETree)) ;
   IVcopy(nfront, buffer + nfront,   ETree_fch(frontETree)) ;
   IVcopy(nfront, buffer + 2*nfront, ETree_sib(frontETree)) ;
   IVcopy(nfront, buffer + 3*nfront, ETree_nodwghts(frontETree)) ;
   IVcopy(nfront, buffer + 4*nfront, ETree_bndwghts(frontETree)) ;
   IVcopy(nvtx,   buffer + 5*nfront, ETree_vtxToFront(frontETree)) ;
}
MPI_Bcast((void *) buffer, 5*nfront+nvtx, MPI_INT, jproc, comm) ;
if ( jproc != myid ) {
   ETree_free(frontETree) ;
   frontETree = ETree_new() ;
   ETree_init1(frontETree, nfront, nvtx) ;
   frontETree->tree->root = root ;
   IVcopy(nfront, ETree_par(frontETree),        buffer) ;
   IVcopy(nfront, ETree_fch(frontETree),        buffer + nfront) ;
   IVcopy(nfront, ETree_sib(frontETree),        buffer + 2*nfront) ;
   IVcopy(nfront, ETree_nodwghts(frontETree),   buffer + 3*nfront) ;
   IVcopy(nfront, ETree_bndwghts(frontETree),   buffer + 4*nfront) ;
   IVcopy(nvtx,   ETree_vtxToFront(frontETree), buffer + 5*nfront) ;
}
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n\n best front tree") ;
   ETree_writeForHumanEye(frontETree, msgFile) ;
   fflush(msgFile) ;
}
/*
   --------------------
   free working storage
   --------------------
*/
DVfree(opcounts) ;
IVfree(buffer)   ;

return(frontETree) ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------------------------------
   this is the MPI version of the orderViaND() method.
   the difference is that we start from a distributed DInpMtx object
   instead of a global Graph object.

   created -- 97dec17, cca
   -----------------------------------------------------------------
*/
ETree *
DInpMtx_MPI_orderViaND (
   DInpMtx    *inpmtx,
   int        maxdomainsize,
   int        maxzeros,
   int        maxfrontsize,
   int        seed,
   int        stats[],
   int        msglvl,
   FILE       *msgFile,
   MPI_Comm   comm
) {
DPencil   pencil ;
ETree     *frontETree ;
/*
   ---------------
   check the input
   ---------------
*/
if ( inpmtx == NULL || maxdomainsize < 0 
   || maxzeros < 0 || maxfrontsize < 0
   || stats == NULL || (msglvl > 0 && msgFile == NULL) ) {
   fprintf(stderr, "\n fatal error in DInpMtx_MPI_orderViaND()"
           "\n bad input"
           "\n inpmtx = %p, maxdomainsize = %d,"
           "\n maxzeros = %d, maxfrontsize = %d"
           "\n stats = %p, msglvl = %d, msgFile = %p\n",
           inpmtx, maxdomainsize, maxzeros, maxfrontsize, 
           stats, msglvl, msgFile) ;
   exit(-1) ;
}

DPencil_setDefaultFields(&pencil) ;
DPencil_init(&pencil, inpmtx, 0.0, NULL) ;
frontETree = DPencil_MPI_orderViaND(&pencil, maxdomainsize, maxzeros, 
                     maxfrontsize, seed, stats, msglvl, msgFile, comm) ;

return(frontETree) ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------------------------------
   this is the MPI version of the orderViaND() method.
   the difference is that we start from a distributed DPencil object
   instead of a global Graph object.

   created -- 97dec17, cca
   -----------------------------------------------------------------
*/
ETree *
DPencil_MPI_orderViaND (
   DPencil    *pencil,
   int        maxdomainsize,
   int        maxzeros,
   int        maxfrontsize,
   int        seed,
   int        stats[],
   int        msglvl,
   FILE       *msgFile,
   MPI_Comm   comm
) {
double   minops ;
double   *opcounts ;
ETree    *frontETree, *newFrontETree ;
Graph    *graph ;
int      jproc, myid, nfront, nproc, nvtx, root ;
int      *buffer ;
IVL      *adjIVL ;
/*
   ---------------
   check the input
   ---------------
*/
if ( pencil == NULL || maxdomainsize < 0 
   || maxzeros < 0 || maxfrontsize < 0
   || stats == NULL || (msglvl > 0 && msgFile == NULL) ) {
   fprintf(stderr, "\n fatal error in DPencil_MPI_orderViaND()"
           "\n bad input"
           "\n pencil = %p, maxdomainsize = %d,"
           "\n maxzeros = %d, maxfrontsize = %d"
           "\n stats = %p, msglvl = %d, msgFile = %p\n",
           pencil, maxdomainsize, maxzeros, maxfrontsize, 
           stats, msglvl, msgFile) ;
   exit(-1) ;
}
/*
   --------------------------------------
   get id of self and number of processes
   --------------------------------------
*/
MPI_Comm_rank(comm, &myid)  ;
MPI_Comm_size(comm, &nproc) ;
/*
   --------------------------------
   get the full adjacency structure
   --------------------------------
*/
adjIVL = DPencil_MPI_fullAdjacency(pencil, stats, 
                                   msglvl, msgFile, comm) ;
nvtx = IVL_nlist(adjIVL) ;
/*
   ---------------------------
   initialize the graph object
   ---------------------------
*/
graph = Graph_new() ;
Graph_init2(graph, 0, nvtx, 0, adjIVL->tsize, nvtx, adjIVL->tsize,
           adjIVL, NULL, NULL) ;
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n\n Graph object") ;
   Graph_writeForHumanEye(graph, msgFile) ;
   fflush(msgFile) ;
}
/*
   ---------------------------
   order via nested dissection
   ---------------------------
*/
frontETree = orderViaND(graph, maxdomainsize, seed, msglvl, msgFile) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, 
          "\n\n original front tree, nfront = %d, nzf = %d, ops = %.0f",
          ETree_nfront(frontETree),
          ETree_nFactorEntries(frontETree, 2),
          ETree_nFactorOps(frontETree, 2)) ;
   fflush(msgFile) ;
}
if ( msglvl > 2 ) {
   ETree_writeForHumanEye(frontETree, msgFile) ;
   fflush(msgFile) ;
}
/*
   ---------------------
   free the Graph object
   ---------------------
*/
Graph_free(graph) ;
/*
   ------------------------------
   get the transformed front tree
   ------------------------------
*/
newFrontETree = ETree_transform(frontETree, NULL, 
                                maxzeros, maxfrontsize, seed) ;
ETree_free(frontETree) ;
frontETree = newFrontETree ;
nvtx = ETree_nvtx(frontETree) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, 
       "\n\n transformed front tree, nfront = %d, nzf = %d, ops = %.0f",
       ETree_nfront(frontETree),
       ETree_nFactorEntries(frontETree, 2),
       ETree_nFactorOps(frontETree, 2)) ;
   fflush(msgFile) ;
}
if ( msglvl > 2 ) {
   ETree_writeForHumanEye(frontETree, msgFile) ;
   fflush(msgFile) ;
}
/*
   -----------------------------------------------
   exchange op counts with all the other processes
   -----------------------------------------------
*/
opcounts = DVinit(nproc, 0.0) ;
opcounts[myid] = ETree_nFactorOps(frontETree, 2) ;
MPI_Allgather((void *) &opcounts[myid], 1, MPI_DOUBLE,
              (void *) opcounts, 1, MPI_DOUBLE, comm) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n opcounts") ;
   DVfprintf(msgFile, nproc, opcounts) ;
   fflush(msgFile) ;
}
/*
   --------------------------------------------
   determine which process has the best opcount
   --------------------------------------------
*/
minops = DVmin(nproc, opcounts, &jproc) ;
if ( jproc == myid ) {
   nfront = ETree_nfront(frontETree) ;
   root   = ETree_root(frontETree) ;
}
MPI_Bcast((void *) &nfront, 1, MPI_INT, jproc, comm) ;
MPI_Bcast((void *) &root,   1, MPI_INT, jproc, comm) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n jproc = %d, nfront = %d, root = %d",
           jproc, nfront, root) ;
   fflush(msgFile) ;
}
buffer = IVinit(5*nfront + nvtx, -1) ;
if ( jproc == myid ) {
   IVcopy(nfront, buffer,            ETree_par(frontETree)) ;
   IVcopy(nfront, buffer + nfront,   ETree_fch(frontETree)) ;
   IVcopy(nfront, buffer + 2*nfront, ETree_sib(frontETree)) ;
   IVcopy(nfront, buffer + 3*nfront, ETree_nodwghts(frontETree)) ;
   IVcopy(nfront, buffer + 4*nfront, ETree_bndwghts(frontETree)) ;
   IVcopy(nvtx,   buffer + 5*nfront, ETree_vtxToFront(frontETree)) ;
}
MPI_Bcast((void *) buffer, 5*nfront+nvtx, MPI_INT, jproc, comm) ;
if ( jproc != myid ) {
   ETree_free(frontETree) ;
   frontETree = ETree_new() ;
   ETree_init1(frontETree, nfront, nvtx) ;
   frontETree->tree->root = root ;
   IVcopy(nfront, ETree_par(frontETree),        buffer) ;
   IVcopy(nfront, ETree_fch(frontETree),        buffer + nfront) ;
   IVcopy(nfront, ETree_sib(frontETree),        buffer + 2*nfront) ;
   IVcopy(nfront, ETree_nodwghts(frontETree),   buffer + 3*nfront) ;
   IVcopy(nfront, ETree_bndwghts(frontETree),   buffer + 4*nfront) ;
   IVcopy(nvtx,   ETree_vtxToFront(frontETree), buffer + 5*nfront) ;
}
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n\n best front tree") ;
   ETree_writeForHumanEye(frontETree, msgFile) ;
   fflush(msgFile) ;
}
/*
   --------------------
   free working storage
   --------------------
*/
DVfree(opcounts) ;
IVfree(buffer)   ;

return(frontETree) ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------------------------------
   this is the MPI version of the orderViaMS() method.
   the difference is that we start from a distributed DInpMtx object
   instead of a global Graph object.

   created -- 97dec17, cca
   -----------------------------------------------------------------
*/
ETree *
DInpMtx_MPI_orderViaMS (
   DInpMtx    *inpmtx,
   int        maxdomainsize,
   int        maxzeros,
   int        maxfrontsize,
   int        seed,
   int        stats[],
   int        msglvl,
   FILE       *msgFile,
   MPI_Comm   comm
) {
DPencil   pencil ;
ETree     *frontETree ;
/*
   ---------------
   check the input
   ---------------
*/
if ( inpmtx == NULL || maxdomainsize < 0 
   || maxzeros < 0 || maxfrontsize < 0
   || stats == NULL || (msglvl > 0 && msgFile == NULL) ) {
   fprintf(stderr, "\n fatal error in DInpMtx_MPI_orderViaMS()"
           "\n bad input"
           "\n inpmtx = %p, maxdomainsize = %d,"
           "\n maxzeros = %d, maxfrontsize = %d"
           "\n stats = %p, msglvl = %d, msgFile = %p\n",
           inpmtx, maxdomainsize, maxzeros, maxfrontsize, 
           stats, msglvl, msgFile) ;
   exit(-1) ;
}

DPencil_setDefaultFields(&pencil) ;
DPencil_init(&pencil, inpmtx, 0.0, NULL) ;
frontETree = DPencil_MPI_orderViaMS(&pencil, maxdomainsize, maxzeros, 
                     maxfrontsize, seed, stats, msglvl, msgFile, comm) ;

return(frontETree) ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------------------------------
   this is the MPI version of the orderViaMS() method.
   the difference is that we start from a distributed DPencil object
   instead of a global Graph object.

   created -- 97dec17, cca
   -----------------------------------------------------------------
*/
ETree *
DPencil_MPI_orderViaMS (
   DPencil    *pencil,
   int        maxdomainsize,
   int        maxzeros,
   int        maxfrontsize,
   int        seed,
   int        stats[],
   int        msglvl,
   FILE       *msgFile,
   MPI_Comm   comm
) {
double   minops ;
double   *opcounts ;
ETree    *frontETree, *newFrontETree ;
Graph    *graph ;
int      jproc, myid, nfront, nproc, nvtx, root ;
int      *buffer ;
IVL      *adjIVL ;
/*
   ---------------
   check the input
   ---------------
*/
if ( pencil == NULL || maxdomainsize < 0 
   || maxzeros < 0 || maxfrontsize < 0
   || stats == NULL || (msglvl > 0 && msgFile == NULL) ) {
   fprintf(stderr, "\n fatal error in DPencil_MPI_orderViaMS()"
           "\n bad input"
           "\n pencil = %p, maxdomainsize = %d,"
           "\n maxzeros = %d, maxfrontsize = %d"
           "\n stats = %p, msglvl = %d, msgFile = %p\n",
           pencil, maxdomainsize, maxzeros, maxfrontsize, 
           stats, msglvl, msgFile) ;
   exit(-1) ;
}
/*
   --------------------------------------
   get id of self and number of processes
   --------------------------------------
*/
MPI_Comm_rank(comm, &myid)  ;
MPI_Comm_size(comm, &nproc) ;
/*
   --------------------------------
   get the full adjacency structure
   --------------------------------
*/
adjIVL = DPencil_MPI_fullAdjacency(pencil, stats, 
                                   msglvl, msgFile, comm) ;
nvtx = IVL_nlist(adjIVL) ;
/*
   ---------------------------
   initialize the graph object
   ---------------------------
*/
graph = Graph_new() ;
Graph_init2(graph, 0, nvtx, 0, adjIVL->tsize, nvtx, adjIVL->tsize,
           adjIVL, NULL, NULL) ;
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n\n Graph object") ;
   Graph_writeForHumanEye(graph, msgFile) ;
   fflush(msgFile) ;
}
/*
   ---------------------------
   order via nested dissection
   ---------------------------
*/
frontETree = orderViaMS(graph, maxdomainsize, seed, msglvl, msgFile) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, 
          "\n\n original front tree, nfront = %d, nzf = %d, ops = %.0f",
          ETree_nfront(frontETree),
          ETree_nFactorEntries(frontETree, 2),
          ETree_nFactorOps(frontETree, 2)) ;
   fflush(msgFile) ;
}
if ( msglvl > 2 ) {
   ETree_writeForHumanEye(frontETree, msgFile) ;
   fflush(msgFile) ;
}
/*
   ---------------------
   free the Graph object
   ---------------------
*/
Graph_free(graph) ;
/*
   ------------------------------
   get the transformed front tree
   ------------------------------
*/
newFrontETree = ETree_transform(frontETree, NULL, 
                                maxzeros, maxfrontsize, seed) ;
ETree_free(frontETree) ;
frontETree = newFrontETree ;
nvtx = ETree_nvtx(frontETree) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, 
       "\n\n transformed front tree, nfront = %d, nzf = %d, ops = %.0f",
       ETree_nfront(frontETree),
       ETree_nFactorEntries(frontETree, 2),
       ETree_nFactorOps(frontETree, 2)) ;
   fflush(msgFile) ;
}
if ( msglvl > 2 ) {
   ETree_writeForHumanEye(frontETree, msgFile) ;
   fflush(msgFile) ;
}
/*
   -----------------------------------------------
   exchange op counts with all the other processes
   -----------------------------------------------
*/
opcounts = DVinit(nproc, 0.0) ;
opcounts[myid] = ETree_nFactorOps(frontETree, 2) ;
MPI_Allgather((void *) &opcounts[myid], 1, MPI_DOUBLE,
              (void *) opcounts, 1, MPI_DOUBLE, comm) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n opcounts") ;
   DVfprintf(msgFile, nproc, opcounts) ;
   fflush(msgFile) ;
}
/*
   --------------------------------------------
   determine which process has the best opcount
   --------------------------------------------
*/
minops = DVmin(nproc, opcounts, &jproc) ;
if ( jproc == myid ) {
   nfront = ETree_nfront(frontETree) ;
   root   = ETree_root(frontETree) ;
}
MPI_Bcast((void *) &nfront, 1, MPI_INT, jproc, comm) ;
MPI_Bcast((void *) &root,   1, MPI_INT, jproc, comm) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n jproc = %d, nfront = %d, root = %d",
           jproc, nfront, root) ;
   fflush(msgFile) ;
}
buffer = IVinit(5*nfront + nvtx, -1) ;
if ( jproc == myid ) {
   IVcopy(nfront, buffer,            ETree_par(frontETree)) ;
   IVcopy(nfront, buffer + nfront,   ETree_fch(frontETree)) ;
   IVcopy(nfront, buffer + 2*nfront, ETree_sib(frontETree)) ;
   IVcopy(nfront, buffer + 3*nfront, ETree_nodwghts(frontETree)) ;
   IVcopy(nfront, buffer + 4*nfront, ETree_bndwghts(frontETree)) ;
   IVcopy(nvtx,   buffer + 5*nfront, ETree_vtxToFront(frontETree)) ;
}
MPI_Bcast((void *) buffer, 5*nfront+nvtx, MPI_INT, jproc, comm) ;
if ( jproc != myid ) {
   ETree_free(frontETree) ;
   frontETree = ETree_new() ;
   ETree_init1(frontETree, nfront, nvtx) ;
   frontETree->tree->root = root ;
   IVcopy(nfront, ETree_par(frontETree),        buffer) ;
   IVcopy(nfront, ETree_fch(frontETree),        buffer + nfront) ;
   IVcopy(nfront, ETree_sib(frontETree),        buffer + 2*nfront) ;
   IVcopy(nfront, ETree_nodwghts(frontETree),   buffer + 3*nfront) ;
   IVcopy(nfront, ETree_bndwghts(frontETree),   buffer + 4*nfront) ;
   IVcopy(nvtx,   ETree_vtxToFront(frontETree), buffer + 5*nfront) ;
}
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n\n best front tree") ;
   ETree_writeForHumanEye(frontETree, msgFile) ;
   fflush(msgFile) ;
}
/*
   --------------------
   free working storage
   --------------------
*/
DVfree(opcounts) ;
IVfree(buffer)   ;

return(frontETree) ; }

/*--------------------------------------------------------------------*/
