/*  fullAdjMPI.c  */

#include "../spoolesMPI.h"

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------------------------
   purpose -- given a distributed DInpMtx object,
      gather all the indices onto each process and create
      an IVL object that contains the full adjacency structure

   created -- 97dec17, cca
   -----------------------------------------------------------
*/
IVL *
DInpMtx_MPI_fullAdjacency (
   DInpMtx    *inpmtx,
   int        stats[],
   int        msglvl,
   FILE       *msgFile,
   MPI_Comm   comm
) {
DInpMtx   *adjmtx ;
int       ierr, iproc, maxnent, myid, nent, nproc, oldtype, totalnent ;
int       *buffer, *counts, *ivec1, *ivec2 ;
IVL       *adjIVL ;
/*
   --------------------------------------
   get id of self and number of processes
   --------------------------------------
*/
MPI_Comm_rank(comm, &myid)  ;
MPI_Comm_size(comm, &nproc) ;
/*
   ----------------------
   convert to row storage
   ----------------------
*/
oldtype = DInpMtx_coordType(inpmtx) ;
DInpMtx_changeCoordType(inpmtx, 1) ;
nent  = DInpMtx_nent(inpmtx) ;
ivec1 = DInpMtx_ivec1(inpmtx) ;
ivec2 = DInpMtx_ivec2(inpmtx) ;
if ( msglvl > 0 ) {
   fprintf(msgFile, "\n\n %d internal entries", nent) ;
   fflush(msgFile) ;
}
/*
   -------------------------------------------
   find out how many entries each process owns
   -------------------------------------------
*/
counts = IVinit(nproc, 0) ;
counts[myid] = nent ;
MPI_Allgather((void *) &counts[myid], 1, MPI_INT,
              (void *) counts, 1, MPI_INT, comm) ;
totalnent = IVsum(nproc, counts) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n %d total entries", totalnent) ;
   fprintf(msgFile, "\n\n counts vector") ;
   IVfp80(msgFile, nproc, counts, 80, &ierr) ;
   fflush(msgFile) ;
}
/*
   -------------------------------------------------
   allocate a new DInpMtx object to hold the indices
   -------------------------------------------------
*/
adjmtx = DInpMtx_new() ;
DInpMtx_init(adjmtx, 1, 1, totalnent, 0) ;
/*
   -----------------------------------------
   allocate a buffer to send/receive entries
   -----------------------------------------
*/
maxnent = IVmax(nproc, counts, &iproc) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n %d maximum entries", maxnent) ;
   fflush(msgFile) ;
}
buffer = IVinit(2*maxnent, -1) ;
/*
   ----------------------------
   now send and receive entries
   ----------------------------
*/
for ( iproc = 0 ; iproc < nproc ; iproc++ ) {
   nent = counts[iproc] ;
   if ( iproc == myid ) {
/*
      -----------------------
      load the entries buffer
      -----------------------
*/
      IVcopy(nent, buffer, ivec1) ;
      IVcopy(nent, buffer + nent, ivec2) ;
      if ( msglvl > 1 ) {
         fprintf(msgFile, "\n\n owned entries in buffer") ;
         fflush(msgFile) ;
      }
      if ( msglvl > 2 ) {
         IVfprintf(msgFile, 2*nent, buffer) ;
         fflush(msgFile) ;
      }
      stats[0]++ ;
      stats[2] += 2*nent*sizeof(int) ;
   } else {
      stats[1]++ ;
      stats[3] += 2*nent*sizeof(int) ;
   }
/*
   ----------------------------------------
   broadcast the entries from process iproc
   ----------------------------------------
*/
   MPI_Bcast((void *) buffer, 2*nent, MPI_INT, iproc, comm) ;
   if ( msglvl > 2 ) {
      fprintf(msgFile, "\n\n after bcast, buffer") ;
      IVfprintf(msgFile, 2*nent, buffer) ;
      fflush(msgFile) ;
   }
/*
   ------------------------------------------
   load the entries into the adjacency matrix
   ------------------------------------------
*/
   DInpMtx_inputTriples(adjmtx, nent, buffer, buffer + nent, NULL) ;
}
/*
   ---------------------------------------
   sort and compress the adjacency entries
   ---------------------------------------
*/
DInpMtx_sortAndCompress(adjmtx) ;
/*
   ------------------------------------------------------
   create the IVL object that contains the full adjacency
   ------------------------------------------------------
*/
adjIVL = DInpMtx_fullAdjacency(adjmtx) ;
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n\n full adjacency object") ;
   IVL_writeForHumanEye(adjIVL, msgFile) ;
   fflush(msgFile) ;
}
/*
   --------------------------------
   convert back to original storage
   --------------------------------
*/
DInpMtx_changeCoordType(inpmtx, oldtype) ;
/*
   ------------------------
   free the working storage
   ------------------------
*/
IVfree(counts) ;
IVfree(buffer) ;
DInpMtx_free(adjmtx) ;

return(adjIVL) ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------------------------
   purpose -- given a distributed DPencil object,
      gather all the indices onto each process and create
      an IVL object that contains the full adjacency structure

   created -- 97dec18, cca
   -----------------------------------------------------------
*/
IVL *
DPencil_MPI_fullAdjacency (
   DPencil    *pencil,
   int        stats[],
   int        msglvl,
   FILE       *msgFile,
   MPI_Comm   comm
) {
DInpMtx   *adjmtx, *inpmtxA, *inpmtxB ;
int       ierr, iproc, maxnent, myid, nent, nentA, nentB,
          nproc, oldtypeA, oldtypeB, totalnent ;
int       *buffer, *counts, *ivec1, *ivec2, *tempbuffer ;
IVL       *adjIVL ;
/*
   ------------------------------------------------------
   check for a simple pencil (only one nontrivial matrix)
   ------------------------------------------------------
*/
inpmtxA = pencil->inpmtxA ;
inpmtxB = pencil->inpmtxB ;
if ( inpmtxA == NULL ) {
   if ( pencil->sigma != 0.0 && inpmtxB != NULL ) {
      adjIVL = DInpMtx_MPI_fullAdjacency(inpmtxB, stats, 
                                         msglvl, msgFile, comm) ;
   } else {
      adjIVL = NULL ;
   }
   return(adjIVL) ;
} else if ( pencil->sigma == 0.0 || inpmtxB == NULL ) {
   adjIVL = DInpMtx_MPI_fullAdjacency(inpmtxA, 
                                      stats, msglvl, msgFile, comm) ;
   return(adjIVL) ;
}
/*
   --------------------------------------
   get id of self and number of processes
   --------------------------------------
*/
MPI_Comm_rank(comm, &myid)  ;
MPI_Comm_size(comm, &nproc) ;
/*
   ---------------------------------------------
   allocate a send buffer, fill with the indices 
   from the two matrices, sort and compress
   ---------------------------------------------
*/
oldtypeA = DInpMtx_coordType(inpmtxA) ;
DInpMtx_changeCoordType(inpmtxA, 1) ;
oldtypeB = DInpMtx_coordType(inpmtxB) ;
DInpMtx_changeCoordType(inpmtxB, 1) ;
nentA  = DInpMtx_nent(inpmtxA)  ;
nentB  = DInpMtx_nent(inpmtxB)  ;
tempbuffer = IVinit(2*(nentA + nentB), -1) ;
ivec1 = tempbuffer ;
ivec2 = ivec1 + nentA + nentB ;
IVcopy(nentA, ivec1,         DInpMtx_ivec1(inpmtxA)) ;
IVcopy(nentB, ivec1 + nentA, DInpMtx_ivec1(inpmtxB)) ;
IVcopy(nentA, ivec2,         DInpMtx_ivec2(inpmtxA)) ;
IVcopy(nentB, ivec2 + nentA, DInpMtx_ivec2(inpmtxB)) ;
nent = IV2sortUpAndCompress(nentA + nentB, ivec1, ivec2) ;
if ( msglvl > 0 ) {
   fprintf(msgFile, "\n\n %d internal entries", nent) ;
   fflush(msgFile) ;
}
ivec2 = ivec1 + nent ;
/*
   -------------------------------------------
   find out how many entries each process owns
   -------------------------------------------
*/
counts = IVinit(nproc, 0) ;
counts[myid] = nent ;
MPI_Allgather((void *) &counts[myid], 1, MPI_INT,
              (void *) counts, 1, MPI_INT, comm) ;
totalnent = IVsum(nproc, counts) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n %d total entries", totalnent) ;
   fprintf(msgFile, "\n\n counts vector") ;
   IVfp80(msgFile, nproc, counts, 80, &ierr) ;
   fflush(msgFile) ;
}
/*
   -------------------------------------------------
   allocate a new DInpMtx object to hold the indices
   -------------------------------------------------
*/
adjmtx = DInpMtx_new() ;
DInpMtx_init(adjmtx, 1, 1, totalnent, 0) ;
/*
   -----------------------------------------
   allocate a buffer to send/receive entries
   and load with the owned entries
   -----------------------------------------
*/
maxnent = IVmax(nproc, counts, &iproc) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n %d maximum entries", maxnent) ;
   fflush(msgFile) ;
}
buffer = IVinit(2*maxnent, -1) ;
/*
   ----------------------------
   now send and receive entries
   ----------------------------
*/
for ( iproc = 0 ; iproc < nproc ; iproc++ ) {
   nent = counts[iproc] ;
   if ( iproc == myid ) {
/*
      -----------------------
      load the entries buffer
      -----------------------
*/
      IVcopy(nent, buffer, ivec1) ;
      IVcopy(nent, buffer + nent, ivec2) ;
      if ( msglvl > 1 ) {
         fprintf(msgFile, "\n\n owned entries in buffer") ;
         fflush(msgFile) ;
      }
      if ( msglvl > 2 ) {
         IVfprintf(msgFile, 2*nent, buffer) ;
         fflush(msgFile) ;
      }
      stats[0]++ ;
      stats[2] += 2*nent*sizeof(int) ;
   } else {
      stats[1]++ ;
      stats[3] += 2*nent*sizeof(int) ;
   }
/*
   ----------------------------------------
   broadcast the entries from process iproc
   ----------------------------------------
*/
   MPI_Bcast((void *) buffer, 2*nent, MPI_INT, iproc, comm) ;
   if ( msglvl > 2 ) {
      fprintf(msgFile, "\n\n after bcast, buffer") ;
      IVfprintf(msgFile, 2*nent, buffer) ;
      fflush(msgFile) ;
   }
/*
   ------------------------------------------
   load the entries into the adjacency matrix
   ------------------------------------------
*/
   DInpMtx_inputTriples(adjmtx, nent, buffer, buffer + nent, NULL) ;
}
/*
   ---------------------------------------
   sort and compress the adjacency entries
   ---------------------------------------
*/
DInpMtx_sortAndCompress(adjmtx) ;
/*
   ------------------------------------------------------
   create the IVL object that contains the full adjacency
   ------------------------------------------------------
*/
adjIVL = DInpMtx_fullAdjacency(adjmtx) ;
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n\n full adjacency object") ;
   IVL_writeForHumanEye(adjIVL, msgFile) ;
   fflush(msgFile) ;
}
/*
   --------------------------------
   convert back to original storage
   --------------------------------
*/
DInpMtx_changeCoordType(inpmtxA, oldtypeA) ;
DInpMtx_changeCoordType(inpmtxB, oldtypeB) ;
/*
   ------------------------
   free the working storage
   ------------------------
*/
IVfree(counts) ;
IVfree(tempbuffer) ;
IVfree(buffer) ;
DInpMtx_free(adjmtx) ;

return(adjIVL) ; }

/*--------------------------------------------------------------------*/
